﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/fssystem/fs_ZeroBitmapFile.h>
#include <nn/util/util_BitUtil.h>

#include <nn/nn_SdkLog.h>

namespace nn { namespace fssystem {

Result ZeroBitmapFile::Initialize(std::unique_ptr<fs::fsa::IFile> pBaseFile, std::unique_ptr<Bit8[], nn::fs::detail::Deleter> pBitmap, int64_t bitmapSize, int64_t blockSize, fs::OpenMode mode) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pBaseFile);
    NN_SDK_REQUIRES_NOT_NULL(pBitmap);
    NN_SDK_REQUIRES_EQUAL(nn::util::ispow2(blockSize), true);

    m_pBaseFile = std::move(pBaseFile);
    m_BlockSize = blockSize;
    m_Mode = mode;

    int64_t size;
    NN_RESULT_DO(m_pBaseFile->GetSize(&size));
    NN_SDK_REQUIRES_EQUAL(bitmapSize, nn::util::DivideUp(nn::util::DivideUp(size, m_BlockSize), BitsPerByte));

    m_pBitmap = std::move(pBitmap);
    m_BitmapSize = bitmapSize;

    NN_RESULT_SUCCESS;
}

Result ZeroBitmapFile::DoRead(size_t* outValue, int64_t offset, void* buffer, size_t size, const fs::ReadOption& option) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pBaseFile);
    NN_SDK_REQUIRES_NOT_NULL(m_pBitmap);

    NN_RESULT_DO(m_pBaseFile->Read(outValue, offset, buffer, size, option));

    int64_t start = offset;
    int64_t end = offset + *outValue;
    NN_SDK_REQUIRES_LESS((end - 1) / m_BlockSize / BitsPerByte, m_BitmapSize);
    while(start < end)
    {
        int64_t clearSize = std::min(m_BlockSize - (start % m_BlockSize), end - start);
        if(IsFree(start))
        {
            memset(reinterpret_cast<char*>(buffer) + start - offset, 0, static_cast<size_t>(clearSize));
        }
        start += clearSize;
    }

    NN_RESULT_SUCCESS;
}

Result ZeroBitmapFile::DoWrite(int64_t offset, const void* buffer, size_t size, const fs::WriteOption& option) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pBaseFile);
    NN_SDK_REQUIRES_NOT_NULL(m_pBitmap);

    int64_t fileSize;
    NN_RESULT_DO(m_pBaseFile->GetSize(&fileSize));
    if(offset + static_cast<int64_t>(size) > fileSize)
    {
        if(m_Mode & nn::fs::OpenMode_AllowAppend)
        {
            NN_RESULT_THROW(nn::fs::ResultUsableSpaceNotEnough());
        }
        else
        {
            NN_RESULT_THROW(nn::fs::ResultFileExtensionWithoutOpenModeAllowAppend());
        }
    }

    NN_RESULT_DO(m_pBaseFile->Write(offset, buffer, size, nn::fs::WriteOption::MakeValue(option.flags & ~nn::fs::WriteOptionFlag_Flush)));

    int64_t start = offset;
    int64_t end = offset + size;
    NN_SDK_REQUIRES_LESS((end - 1) / m_BlockSize / BitsPerByte, m_BitmapSize);

    {
        int64_t alignDown = nn::util::align_down(start, static_cast<size_t>(m_BlockSize));
        int64_t clearSize = start - alignDown;
        if(clearSize > 0 && IsFree(start))
        {
            std::unique_ptr<Bit8[], nn::fs::detail::Deleter> clearBuffer;
            clearBuffer = fs::detail::MakeUnique<Bit8[]>(static_cast<size_t>(clearSize));
            NN_RESULT_THROW_UNLESS(clearBuffer, nn::fs::ResultAllocationMemoryFailedMakeUnique());
            memset(clearBuffer.get(), 0, static_cast<size_t>(clearSize));
            NN_RESULT_DO(m_pBaseFile->Write(alignDown, clearBuffer.get(), static_cast<size_t>(clearSize), nn::fs::WriteOption::MakeValue(option.flags & ~nn::fs::WriteOptionFlag_Flush)));
        }
    }

    {
        int64_t alignUp = nn::util::align_up(end, static_cast<size_t>(m_BlockSize));
        int64_t clearSize = std::min(alignUp, fileSize) - end;
        if(clearSize > 0 && IsFree(end))
        {
            std::unique_ptr<Bit8[], nn::fs::detail::Deleter> clearBuffer;
            clearBuffer = fs::detail::MakeUnique<Bit8[]>(static_cast<size_t>(clearSize));
            NN_RESULT_THROW_UNLESS(clearBuffer, nn::fs::ResultAllocationMemoryFailedMakeUnique());
            memset(clearBuffer.get(), 0, static_cast<size_t>(clearSize));
            NN_RESULT_DO(m_pBaseFile->Write(end, clearBuffer.get(), static_cast<size_t>(clearSize), nn::fs::WriteOption::MakeValue(option.flags & ~nn::fs::WriteOptionFlag_Flush)));
        }
    }

    if ((option.flags & nn::fs::WriteOptionFlag_Flush) != 0)
    {
        NN_RESULT_DO(m_pBaseFile->Flush());
    }

    while(start < end)
    {
        int64_t clearSize = std::min(m_BlockSize - (start % m_BlockSize), end - start);
        Clear(start);
        start += clearSize;
    }

    NN_RESULT_SUCCESS;
}

Result ZeroBitmapFile::DoFlush() NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pBaseFile);

    NN_RESULT_DO(m_pBaseFile->Flush());
    NN_RESULT_SUCCESS;
}

Result ZeroBitmapFile::DoSetSize(int64_t size) NN_NOEXCEPT
{
    NN_UNUSED(size);

    NN_RESULT_THROW(nn::fs::ResultUnsupportedOperation());
}

Result ZeroBitmapFile::DoGetSize(int64_t* outValue) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pBaseFile);

    NN_RESULT_DO(m_pBaseFile->GetSize(outValue));
    NN_RESULT_SUCCESS;
}

Result ZeroBitmapFile::DoOperateRange(
    void* outBuffer,
    size_t outBufferSize,
    fs::OperationId operationId,
    int64_t offset,
    int64_t size,
    const void* inBuffer,
    size_t inBufferSize) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pBaseFile);

    NN_RESULT_DO(m_pBaseFile->OperateRange(outBuffer,
                                           outBufferSize,
                                           operationId,
                                           offset,
                                           size,
                                           inBuffer,
                                           inBufferSize));
    NN_RESULT_SUCCESS;
}

}}
