﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/


#include <nn/fssystem/fs_StorageFile.h>
#include <nn/fs/fs_IStorage.h>
#include <algorithm>

namespace nn { namespace fssystem {

StorageFile::StorageFile(nn::fs::IStorage* pBaseStorage, nn::fs::OpenMode mode) NN_NOEXCEPT
    : m_pBaseStorage(pBaseStorage),
      m_Mode(mode)
{
    NN_SDK_REQUIRES_NOT_NULL(pBaseStorage);
}

Result StorageFile::DoRead(size_t* outValue, int64_t offset, void* buffer, size_t size, const fs::ReadOption& option) NN_NOEXCEPT
{
    NN_FSP_REQUIRES(m_pBaseStorage != nullptr, nn::fs::ResultNullptrArgument());

    size_t readSize = 0;
    NN_RESULT_DO(DryRead(&readSize, offset, size, option, m_Mode));

    NN_RESULT_DO(m_pBaseStorage->Read(offset, buffer, readSize));
    *outValue = readSize;

    NN_RESULT_SUCCESS;
}

Result StorageFile::DoWrite(int64_t offset, const void* buffer, size_t size, const fs::WriteOption& option) NN_NOEXCEPT
{
    NN_FSP_REQUIRES(m_pBaseStorage != nullptr, nn::fs::ResultNullptrArgument());
    NN_FSP_REQUIRES((m_Mode & nn::fs::OpenMode_Write) != 0, nn::fs::ResultInvalidOperationForOpenMode());
    NN_FSP_REQUIRES(offset >= 0, nn::fs::ResultInvalidOffset());

    // IFile::Write() でハンドリングしている想定
    NN_SDK_REQUIRES(size > 0);

    int64_t fileSize;
    NN_RESULT_DO(GetSize(&fileSize));

    // 書き込み開始位置をチェック
    // offset がファイルサイズより大きく、且つ自動拡張が不許可ならエラーを返します
    if( offset > fileSize )
    {
        if( (m_Mode & nn::fs::OpenMode_AllowAppend) == 0 )
        {
            return nn::fs::ResultInvalidOffset();
        }
    }

    // 書き込むサイズを調整する
    if( static_cast<int64_t>(offset + size) > fileSize || static_cast<int64_t>(offset + size) < offset)
    {
        if( (m_Mode & nn::fs::OpenMode_AllowAppend) == 0 )
        {
            // 自動伸長が有効になっていません。
            return nn::fs::ResultFileExtensionWithoutOpenModeAllowAppend();
        }
        else
        {
            NN_RESULT_DO(SetSize(offset + size));
        }
    }

    // データの書き込みを実行
    NN_RESULT_DO(m_pBaseStorage->Write(offset, buffer, size));

    if ((option.flags & nn::fs::WriteOptionFlag_Flush) != 0)
    {
        NN_RESULT_DO(Flush());
    }

    NN_RESULT_SUCCESS;
}

Result StorageFile::DoFlush() NN_NOEXCEPT
{
    NN_FSP_REQUIRES(m_pBaseStorage != nullptr, nn::fs::ResultNullptrArgument());
    if( (m_Mode & nn::fs::OpenMode_Write) == 0 )
    {
        NN_RESULT_SUCCESS;
    }
    NN_RESULT_DO(m_pBaseStorage->Flush());
    NN_RESULT_SUCCESS;
}

Result StorageFile::DoSetSize(int64_t size) NN_NOEXCEPT
{
    NN_FSP_REQUIRES(m_pBaseStorage != nullptr, nn::fs::ResultNullptrArgument());
    NN_RESULT_DO(DrySetSize(size, m_Mode));
    return m_pBaseStorage->SetSize(size);
}

Result StorageFile::DoGetSize(int64_t* outValue) NN_NOEXCEPT
{
    NN_FSP_REQUIRES(m_pBaseStorage != nullptr, nn::fs::ResultNullptrArgument());

    return m_pBaseStorage->GetSize(outValue);
}

Result StorageFile::DoOperateRange(
    void* outBuffer,
    size_t outBufferSize,
    fs::OperationId operationId,
    int64_t offset,
    int64_t size,
    const void* inBuffer,
    size_t inBufferSize) NN_NOEXCEPT
{
    switch(operationId)
    {
    case fs::OperationId::Invalidate:
        NN_FSP_REQUIRES(
            (m_Mode & nn::fs::OpenMode_Read) != 0,
            nn::fs::ResultInvalidOperationForOpenMode());
        break;

    case fs::OperationId::QueryRange:
        break;

    default:
        return nn::fs::ResultUnsupportedOperation();
    }

    NN_FSP_REQUIRES(
        m_pBaseStorage != nullptr,
        nn::fs::ResultNullptrArgument());
    NN_FSP_REQUIRES(
        offset >= 0,
        nn::fs::ResultInvalidOffset());

    // 操作サイズを調整する
    int64_t fileSize = 0;
    NN_RESULT_DO(GetSize(&fileSize));

    const auto operableSize = std::max<int64_t>(fileSize - offset, 0);
    const auto operateSize = std::min(operableSize, size);

    NN_RESULT_DO(m_pBaseStorage->OperateRange(
        outBuffer,
        outBufferSize,
        operationId,
        offset,
        operateSize,
        inBuffer,
        inBufferSize));

    NN_RESULT_SUCCESS;
}

}}
