﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/os/os_Mutex.h>
#include <nn/fs/fs_SaveDataTypes.h>
#include <nn/fssystem/fs_SaveDataFileSystemCacheManager.h>

namespace nn { namespace fssystem {

void SaveDataFileSystemCacheManager::Cache::Register( std::shared_ptr<nn::fssystem::SaveDataFileSystem>&& pFileSystem ) NN_NOEXCEPT
{
    m_CachedSpaceId = pFileSystem->GetSaveDataSpaceId();
    m_CachedId = pFileSystem->GetSaveDataId();
    m_pSaveDataFileSystem = std::move(pFileSystem);
}

bool SaveDataFileSystemCacheManager::Cache::IsCached(nn::fs::SaveDataSpaceId spaceId, nn::fs::SaveDataId id) const NN_NOEXCEPT
{
    return m_pSaveDataFileSystem != nullptr && spaceId == m_CachedSpaceId && id == m_CachedId;
}

std::shared_ptr<nn::fssystem::SaveDataFileSystem> SaveDataFileSystemCacheManager::Cache::Move() NN_NOEXCEPT
{
    return std::move(m_pSaveDataFileSystem);
}

void SaveDataFileSystemCacheManager::Cache::Unregister() NN_NOEXCEPT
{
    m_pSaveDataFileSystem.reset();
}

SaveDataFileSystemCacheManager::SaveDataFileSystemCacheManager() NN_NOEXCEPT
    : m_Mutex(true)
    , m_pCachedFileSystems(nullptr)
    , m_NextCacheIndex(0)
{
}

void SaveDataFileSystemCacheManager::Initialize(int maxCacheCount) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_GREATER_EQUAL(maxCacheCount, 0);
    std::lock_guard<nn::os::Mutex> scopedLock(m_Mutex);
    NN_SDK_ASSERT(m_pCachedFileSystems.get() == nullptr);

    m_MaxCachedFileSystemCount = maxCacheCount;
    if( maxCacheCount > 0 )
    {
        m_pCachedFileSystems.reset(new Cache[m_MaxCachedFileSystemCount]);
        NN_ABORT_UNLESS_NOT_NULL(m_pCachedFileSystems);
    }
}

bool SaveDataFileSystemCacheManager::GetCache(
    std::shared_ptr<nn::fssystem::SaveDataFileSystem>* pOutFileSystem,
    nn::fs::SaveDataSpaceId spaceId,
    nn::fs::SaveDataId id
) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_GREATER_EQUAL(m_MaxCachedFileSystemCount, 0);

    std::lock_guard<nn::os::Mutex> scopedLock(m_Mutex);
    for( int i = 0; i < m_MaxCachedFileSystemCount; ++i )
    {
        if( m_pCachedFileSystems[i].IsCached(spaceId, id) )
        {
            *pOutFileSystem = m_pCachedFileSystems[i].Move();
            return true;
        }
    }
    return false;
}

void SaveDataFileSystemCacheManager::Register(
    std::shared_ptr<nn::fssystem::SaveDataFileSystem>&& pFileSystem
) NN_NOEXCEPT
{
    if( m_MaxCachedFileSystemCount <= 0 )
    {
        return;
    }

    NN_SDK_REQUIRES_GREATER_EQUAL(m_NextCacheIndex, 0);
    NN_SDK_REQUIRES_GREATER(m_MaxCachedFileSystemCount, m_NextCacheIndex);

    // https://spdlybra.nintendo.co.jp/jira/browse/SIGLO-51235?focusedCommentId=1951415&page=com.atlassian.jira.plugin.system.issuetabpanels:comment-tabpanel#comment-1951415
    // ns が sd カードのセーブをクリンナップする暫定措置を施すことがある（SIGLO-46600、SIGLO-46492）ため、SdSystem はキャッシュしない。
    // Sd 上の CacheStorage (SdUser) はキャッシュする。
    if( pFileSystem->GetSaveDataSpaceId() == nn::fs::SaveDataSpaceId::SdSystem )
    {
        // SaveDataExtraData の読み出しで TargetLocked になるのを防ぐため、ロック取る
        std::lock_guard<nn::os::Mutex> scopedLock(m_Mutex);
        pFileSystem.reset();
        return;
    }

    // Commit されていないデータは巻き戻しておく
    nn::Result result = pFileSystem->Rollback();
    if( result.IsSuccess() )
    {
        std::lock_guard<nn::os::Mutex> scopedLock(m_Mutex);
        m_pCachedFileSystems[m_NextCacheIndex].Register(std::move(pFileSystem));
        ++m_NextCacheIndex;
        m_NextCacheIndex %= m_MaxCachedFileSystemCount;
    }
}

void SaveDataFileSystemCacheManager::Register(
    std::shared_ptr<nn::fssystem::ApplicationTemporaryFileSystem>&& pFileSystem
) NN_NOEXCEPT
{
    // 一時ストレージはキャッシュしない
    // SaveDataExtraData の読み出しで TargetLocked になるのを防ぐため、ロック取る
    std::lock_guard<nn::os::Mutex> scopedLock(m_Mutex);
    pFileSystem.reset();
}

void SaveDataFileSystemCacheManager::Unregister(
        nn::fs::SaveDataSpaceId spaceId,
        nn::fs::SaveDataId id
    ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_GREATER_EQUAL(m_MaxCachedFileSystemCount, 0);

    std::lock_guard<nn::os::Mutex> scopedLock(m_Mutex);
    for( int i = 0; i < m_MaxCachedFileSystemCount; ++i )
    {
        if( m_pCachedFileSystems[i].IsCached(spaceId, id) )
        {
            m_pCachedFileSystems[i].Unregister();
        }
    }
}

}}
