﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/crypto/crypto_Compare.h>
#include <nn/crypto/crypto_AesDecryptor.h>
#include <nn/crypto/crypto_RsaPssSha256Verifier.h>
#include <nn/fs/fs_SubStorage.h>
#include <nn/fssystem/fs_Assert.h>
#include <nn/fssystem/fs_AesCtrStorage.h>
#include <nn/fssystem/fs_AesXtsStorage.h>
#include <nn/fssystem/fs_AllocatorUtility.h>
#include <nn/fssystem/fs_NcaFileSystemDriver.h>
#include <nn/ncm/ncm_ContentMeta.h>
#if defined(NN_BUILD_CONFIG_OS_WIN)
#include <nn/fssystem/utilTool/fs_NcaSparseStorage.h>
#endif

namespace nn { namespace fssystem {

namespace {

const char AcidFileName[] NN_IS_UNUSED_MEMBER = "/ACID"; // TBD
const uint32_t SdkAddonVersionMin = 0x000b0000; // Addon 0.11.0_0

enum class Signature : uint32_t
{
    V0 = NN_UTIL_CREATE_SIGNATURE_4('N', 'C', 'A', '0'),
    V1 = NN_UTIL_CREATE_SIGNATURE_4('N', 'C', 'A', '1'),
    V2 = NN_UTIL_CREATE_SIGNATURE_4('N', 'C', 'A', '2'),
};

}

NcaReader::NcaReader() NN_NOEXCEPT
    : m_pSharedBaseStorage()
    , m_pHeaderStorage()
    , m_pBodyStorage(nullptr)
    , m_pDecryptAesCtr(nullptr)
    , m_pDecryptAesCtrForExternalKey(nullptr)
    , m_IsSwAesPrioritized(false)
    , m_HeaderEncryptionType(NcaHeader::EncryptionType::Auto)
{
    std::memset(&m_Data, 0, sizeof(m_Data));
    std::memset(m_DataDecryptionKeys, 0, sizeof(m_DataDecryptionKeys));
    std::memset(m_ExternalDataDecryptionKey, 0, sizeof(m_ExternalDataDecryptionKey));
}

NcaReader::~NcaReader() NN_NOEXCEPT
{
}

Result NcaReader::Initialize(std::shared_ptr<fs::IStorage> pBaseStorage, const NcaCryptoConfiguration& cryptoConfiguration) NN_NOEXCEPT
{
    m_pSharedBaseStorage = pBaseStorage;
    return Initialize(m_pSharedBaseStorage.get(), cryptoConfiguration);
}

Result NcaReader::Initialize(fs::IStorage* pBaseStorage, const NcaCryptoConfiguration& cryptoConfiguration) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pBaseStorage);
    NN_SDK_REQUIRES(m_pBodyStorage == nullptr);
    NN_FSP_REQUIRES(cryptoConfiguration.pGenerateKey != nullptr, nn::fs::ResultInvalidArgument());

    // 平文でヘッダ署名 1 の検証が通った場合、ヘッダを復号化せずに使う
    std::unique_ptr<fs::IStorage> pHeaderStorage;

    char commonDecryptionKeys[NcaCryptoConfiguration::EncryptionKeyCount][NcaCryptoConfiguration::Aes128KeySize];

    // 共通鍵を復号
    for( int i = 0; i < NcaCryptoConfiguration::EncryptionKeyCount; ++i )
    {
        cryptoConfiguration.pGenerateKey(
            commonDecryptionKeys[i],
            AesXtsStorage::KeySize,
            cryptoConfiguration.headerEncryptedEncryptionKey[i],
            AesXtsStorage::KeySize,
            static_cast<int>(KeyType::NcaHeaderKey),
            cryptoConfiguration
        );
    }

    // (nca,fs) ヘッダ向けのストレージを作成
    const char iv[AesXtsStorage::IvSize] = {};
    pHeaderStorage.reset(new AesXtsStorage(
        pBaseStorage,
        commonDecryptionKeys[0],
        commonDecryptionKeys[1],
        AesXtsStorage::KeySize,
        iv,
        AesXtsStorage::IvSize,
        NcaHeader::XtsBlockSize
    ));
    NN_RESULT_THROW_UNLESS(pHeaderStorage != nullptr, fs::ResultAllocationMemoryFailedInNcaReaderA());

    // nca ヘッダ読み込み
    NN_RESULT_DO(pHeaderStorage->Read(0, &m_Data, sizeof(m_Data)));

    // シグネチャのチェック
    auto checkSignature = [=]() NN_NOEXCEPT -> Result
    {
        if( m_Data.signature == static_cast<uint32_t>(Signature::V0) ||
            m_Data.signature == static_cast<uint32_t>(Signature::V1) ||
            m_Data.signature == static_cast<uint32_t>(Signature::V2) )
        {
            return fs::ResultUnsupportedSdkVersion();
        }
        NN_RESULT_THROW_UNLESS(m_Data.signature == NcaHeader::Signature, fs::ResultInvalidNcaSignature());
        NN_RESULT_SUCCESS;
    };

    auto checkSignatureResult = checkSignature();
    if (checkSignatureResult.IsFailure())
    {
        if (cryptoConfiguration.isAvailableRawHeader)
        {
            NN_RESULT_DO(pBaseStorage->Read(0, &m_Data, sizeof(m_Data)));
            if (checkSignature().IsFailure())
            {
                return checkSignatureResult;
            }
            int64_t baseStorageSize = 0;
            NN_RESULT_DO(pBaseStorage->GetSize(&baseStorageSize));
            pHeaderStorage.reset(new fs::SubStorage(pBaseStorage, 0, baseStorageSize));
            NN_RESULT_THROW_UNLESS(pHeaderStorage != nullptr, fs::ResultAllocationMemoryFailedInNcaReaderA());
            m_HeaderEncryptionType = NcaHeader::EncryptionType::None;
        }
        else
        {
            return checkSignatureResult;
        }
    }

    // ヘッダ署名 1 による検証
    NN_RESULT_THROW_UNLESS(
        crypto::VerifyRsa2048PssSha256(
            m_Data.headerSign1,
            NcaHeader::HeaderSignSize,
            cryptoConfiguration.headerSign1KeyModulus,
            NcaCryptoConfiguration::Rsa2048KeyModulusSize,
            cryptoConfiguration.headerSign1KeyPublicExponent,
            NcaCryptoConfiguration::Rsa2048KeyPublicExponentSize,
            &m_Data.signature,
            NcaHeader::Size - NcaHeader::HeaderSignSize * NcaHeader::HeaderSignCount
        ),
        fs::ResultNcaHeaderSignature1VerificationFailed()
    );

    // バージョンチェック
    NN_RESULT_THROW_UNLESS(
        m_Data.sdkAddonVersion >= SdkAddonVersionMin, fs::ResultUnsupportedSdkVersion());

    NN_RESULT_THROW_UNLESS(
        m_Data.keyIndex < NcaCryptoConfiguration::KeyAreaEncryptionKeyIndexCount, fs::ResultInvalidNcaKeyIndex());

    const char ZeroRightsId[NcaHeader::RightIdSize] = {};
    // 内部鍵 = rightsId が 0
    if( nn::crypto::IsSameBytes(ZeroRightsId, m_Data.rightsId, NcaHeader::RightIdSize) )
    {
        // 必要最小限の領域のみ復号
        cryptoConfiguration.pGenerateKey(
            m_DataDecryptionKeys[NcaHeader::DecryptionKey_AesCtr],
            crypto::AesDecryptor128::BlockSize,
            m_Data.encryptedKey + NcaHeader::DecryptionKey_AesCtr * crypto::AesDecryptor128::BlockSize,
            crypto::AesDecryptor128::BlockSize,
            GetKeyTypeValue(m_Data.keyIndex, m_Data.GetProperKeyGeneration()),
            cryptoConfiguration
        );

        // AesCtrHw 鍵はそのままコピー
        std::memcpy(
            m_DataDecryptionKeys[NcaHeader::DecryptionKey_AesCtrHw],
            m_Data.encryptedKey + NcaHeader::DecryptionKey_AesCtrHw * crypto::AesDecryptor128::BlockSize,
            crypto::AesDecryptor128::BlockSize
        );
    }

    std::memset(m_ExternalDataDecryptionKey, 0x00, sizeof(m_ExternalDataDecryptionKey));

    m_pDecryptAesCtr               = cryptoConfiguration.pDecryptAesCtr;
    m_pDecryptAesCtrForExternalKey = cryptoConfiguration.pDecryptAesCtrForExternalKey;

    m_pHeaderStorage = std::move(pHeaderStorage);
    m_pBodyStorage = pBaseStorage;

    NN_RESULT_SUCCESS;
} // NOLINT(impl/function_size)

Result NcaReader::ReadHeader(NcaFsHeader* pOutHeader, int index) const NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutHeader);
    NN_SDK_REQUIRES_RANGE(index, 0, NcaHeader::FsCountMax);

    const int64_t offset = sizeof(NcaHeader) + sizeof(NcaFsHeader) * index;
    return m_pHeaderStorage->Read(offset, pOutHeader, sizeof(NcaFsHeader));
}

Result NcaReader::VerifyHeaderSign2(const void* pHeaderSign2KeyModulus, size_t headerSign2KeyModulusSize) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pBodyStorage);
    const char HeaderSign2KeyPublicExponent[] = { 0x01, 0x00, 0x01 };

    // ヘッダ署名 2 による検証
    NN_RESULT_THROW_UNLESS(
        crypto::VerifyRsa2048PssSha256(
            m_Data.headerSign2, NcaHeader::HeaderSignSize,
            pHeaderSign2KeyModulus, headerSign2KeyModulusSize,
            HeaderSign2KeyPublicExponent, sizeof(HeaderSign2KeyPublicExponent),
            &m_Data.signature, NcaHeader::Size - NcaHeader::HeaderSignSize * 2
        ),
        nn::fs::ResultNcaHeaderSignature2VerificationFailed()
    );

    NN_RESULT_SUCCESS;
}

fs::IStorage* NcaReader::GetBodyStorage() NN_NOEXCEPT
{
    return m_pBodyStorage;
}

uint32_t NcaReader::GetSignature() const NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pBodyStorage);
    return m_Data.signature;
}

NcaHeader::DistributionType NcaReader::GetDistributionType() const NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pBodyStorage);
    return m_Data.distributionType;
}

NcaHeader::ContentType NcaReader::GetContentType() const NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pBodyStorage);
    return m_Data.contentType;
}

uint8_t NcaReader::GetKeyGeneration() const NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pBodyStorage);
    return m_Data.GetProperKeyGeneration();
}

uint8_t NcaReader::GetKeyIndex() const NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pBodyStorage);
    return m_Data.keyIndex;
}

uint64_t NcaReader::GetContentSize() const NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pBodyStorage);
    return m_Data.contentSize;
}

uint64_t NcaReader::GetProgramId() const NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pBodyStorage);
    return m_Data.programId;
}

uint32_t NcaReader::GetContentIndex() const NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pBodyStorage);
    return m_Data.contentIndex;
}

uint32_t NcaReader::GetSdkAddonVersion() const NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pBodyStorage);
    return m_Data.sdkAddonVersion;
}

void NcaReader::GetRightsId(uint8_t* pOutBuffer, size_t outBufferSize) const NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutBuffer);
    NN_SDK_REQUIRES(outBufferSize >= NcaHeader::RightIdSize);
    NN_UNUSED(outBufferSize);
    std::memcpy(pOutBuffer, m_Data.rightsId, NcaHeader::RightIdSize);
}

bool NcaReader::HasFsInfo(int index) const NN_NOEXCEPT
{
    NN_SDK_REQUIRES_RANGE(index, 0, NcaHeader::FsCountMax);
    return m_Data.fsInfo[index].startSector != 0 || m_Data.fsInfo[index].endSector != 0;
}

// TODO: 将来的に削除すべき
int NcaReader::GetFsCount() const NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pBodyStorage);
    for( int i = 0; i < NcaHeader::FsCountMax; ++i )
    {
        if( HasFsInfo(i) == false )
        {
            return i;
        }
    }
    return NcaHeader::FsCountMax;
}

const Hash& NcaReader::GetFsHeaderHash(int index) const NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pBodyStorage);
    NN_SDK_REQUIRES_RANGE(index, 0, NcaHeader::FsCountMax);
    return m_Data.fsHeaderHash[index];
}

void NcaReader::GetFsHeaderHash(Hash* outValue, int index) const NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pBodyStorage);
    NN_SDK_REQUIRES_NOT_NULL(outValue);
    NN_SDK_REQUIRES_RANGE(index, 0, NcaHeader::FsCountMax);
    std::memcpy(outValue, &m_Data.fsHeaderHash[index], sizeof(Hash));
}

void NcaReader::GetFsInfo(NcaHeader::FsInfo* outValue, int index) const NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pBodyStorage);
    NN_SDK_REQUIRES_NOT_NULL(outValue);
    NN_SDK_REQUIRES_RANGE(index, 0, NcaHeader::FsCountMax);
    std::memcpy(outValue, &m_Data.fsInfo[index], sizeof(NcaHeader::FsInfo));
}

uint64_t NcaReader::GetFsOffset(int index) const NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pBodyStorage);
    NN_SDK_REQUIRES_RANGE(index, 0, NcaHeader::FsCountMax);
    return NcaHeader::SectorToByte(m_Data.fsInfo[index].startSector);
}

uint64_t NcaReader::GetFsEndOffset(int index) const NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pBodyStorage);
    NN_SDK_REQUIRES_RANGE(index, 0, NcaHeader::FsCountMax);
    return NcaHeader::SectorToByte(m_Data.fsInfo[index].endSector);
}

uint64_t NcaReader::GetFsSize(int index) const NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pBodyStorage);
    NN_SDK_REQUIRES_RANGE(index, 0, NcaHeader::FsCountMax);
    return NcaHeader::SectorToByte(m_Data.fsInfo[index].endSector - m_Data.fsInfo[index].startSector);
}

void NcaReader::GetEncryptedKey(void* outValue, size_t size) const NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pBodyStorage);
    NN_SDK_REQUIRES_NOT_NULL(outValue);
    NN_SDK_REQUIRES_GREATER_EQUAL(size, static_cast<size_t>(NcaHeader::EncryptedKeyAreaSize));
    NN_UNUSED(size);
    std::memcpy(outValue, m_Data.encryptedKey, NcaHeader::EncryptedKeyAreaSize);
}

const void* NcaReader::GetDecryptionKey(int index) const NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pBodyStorage);
    NN_SDK_REQUIRES_RANGE(index, 0, NcaHeader::DecryptionKey_Count);
    return m_DataDecryptionKeys[index];
}

bool NcaReader::HasValidInternalKey() const NN_NOEXCEPT
{
    const char ZeroKey[crypto::AesDecryptor128::BlockSize] = {};
    for( int i = 0; i < NcaHeader::DecryptionKey_Count; ++i )
    {
        if (!nn::crypto::IsSameBytes(ZeroKey, m_Data.encryptedKey + i * crypto::AesDecryptor128::BlockSize, crypto::AesDecryptor128::BlockSize))
        {
            return true;
        }
    }
    return false;
}

bool NcaReader::HasInternalDecryptionKeyForAesHw() const NN_NOEXCEPT
{
    const char ZeroKey[crypto::AesDecryptor128::BlockSize] = {};
    return !nn::crypto::IsSameBytes(ZeroKey, GetDecryptionKey(NcaHeader::DecryptionKey_AesCtrHw), crypto::AesDecryptor128::BlockSize);
}

void NcaReader::SetExternalDecryptionKey(const void* pKey, size_t keySize) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pKey);
    NN_SDK_REQUIRES_EQUAL(sizeof(m_ExternalDataDecryptionKey), keySize);
    std::memcpy(m_ExternalDataDecryptionKey, pKey, keySize);
}

bool NcaReader::IsSwAesPrioritized() const NN_NOEXCEPT
{
    return m_IsSwAesPrioritized;
}

void NcaReader::PrioritizeSwAes() NN_NOEXCEPT
{
    m_IsSwAesPrioritized = true;
}

const void* NcaReader::GetExternalDecryptionKey() const NN_NOEXCEPT
{
    return m_ExternalDataDecryptionKey;
}

bool NcaReader::HasExternalDecryptionKey() const NN_NOEXCEPT
{
    const char ZeroKey[crypto::AesDecryptor128::BlockSize] = {};
    return !nn::crypto::IsSameBytes(ZeroKey, GetExternalDecryptionKey(), crypto::AesDecryptor128::BlockSize);
}

void NcaReader::GetRawData(void* pOutValue, size_t outSize) const NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pBodyStorage);
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);
    NN_SDK_REQUIRES_LESS_EQUAL(sizeof(NcaHeader), outSize);
    NN_UNUSED(outSize);

    std::memcpy(pOutValue, &m_Data, sizeof(NcaHeader));
}

DecryptAesCtrFunction NcaReader::GetExternalDecryptAesCtrFunction() const NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pDecryptAesCtr);
    return m_pDecryptAesCtr;
}

DecryptAesCtrFunction NcaReader::GetExternalDecryptAesCtrFunctionForExternalKey() const NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pDecryptAesCtrForExternalKey);
    return m_pDecryptAesCtrForExternalKey;
}

Result NcaReader::Verify() NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pBodyStorage);

    NN_RESULT_DO(m_Data.Verify());

    for( int i = 0; i < NcaHeader::FsCountMax; ++i )
    {
        NcaFsHeaderReader reader;

        if( HasFsInfo(i) )
        {
            NN_RESULT_DO(reader.Initialize(*this, i));
            NN_RESULT_DO(reader.Verify(m_Data.contentType));
        }
        else
        {
            NcaFsHeader header;
            NN_RESULT_DO(ReadHeader(&header, i));

            const char Zero[sizeof(header)] = {};
            NN_RESULT_THROW_UNLESS(
                nn::crypto::IsSameBytes(&header, Zero, sizeof(Zero)), fs::ResultInvalidNcaFsHeader());
        }
    }

    NN_RESULT_SUCCESS;
}

NcaHeader::EncryptionType NcaReader::GetEncryptionType() const NN_NOEXCEPT
{
    return m_HeaderEncryptionType;
}

NcaFsHeaderReader::NcaFsHeaderReader() NN_NOEXCEPT
    : m_FsIndex(-1)
{
    std::memset(&m_Data, 0, sizeof(m_Data));
}

Result NcaFsHeaderReader::Initialize(const NcaReader& reader, int index) NN_NOEXCEPT
{
    m_FsIndex = -1;

    // index のチェックは ReadHeader に任せる
    NN_RESULT_DO(reader.ReadHeader(&m_Data, index));

    Hash hash;
    crypto::GenerateSha256Hash(&hash, sizeof(Hash), &m_Data, sizeof(NcaFsHeader));

    // fs ヘッダの完全性検証
    NN_RESULT_THROW_UNLESS(
        nn::crypto::IsSameBytes(&reader.GetFsHeaderHash(index), &hash, sizeof(Hash)),
        fs::ResultNcaFsHeaderHashVerificationFailed()
    );

    m_FsIndex = index;

    NN_RESULT_SUCCESS;
}

NcaFsHeader::HashData& NcaFsHeaderReader::GetHashData() NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsInitialized());
    return m_Data.hashData;
}

const NcaFsHeader::HashData& NcaFsHeaderReader::GetHashData() const NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsInitialized());
    return m_Data.hashData;
}

uint16_t NcaFsHeaderReader::GetVersion() const NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsInitialized());
    return m_Data.version;
}

int NcaFsHeaderReader::GetFsIndex() const NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsInitialized());
    return m_FsIndex;
}

NcaFsHeader::FsType NcaFsHeaderReader::GetFsType() const NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsInitialized());
    return m_Data.fsType;
}

NcaFsHeader::HashType NcaFsHeaderReader::GetHashType() const NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsInitialized());
    return m_Data.hashType;
}

NcaFsHeader::EncryptionType NcaFsHeaderReader::GetEncryptionType() const NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsInitialized());
    return m_Data.encryptionType;
}

NcaPatchInfo& NcaFsHeaderReader::GetPatchInfo() NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsInitialized());
    return m_Data.patchInfo;
}

const NcaPatchInfo& NcaFsHeaderReader::GetPatchInfo() const NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsInitialized());
    return m_Data.patchInfo;
}

const NcaAesCtrUpperIv NcaFsHeaderReader::GetAesCtrUpperIv() const NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsInitialized());
    return m_Data.aesCtrUpperIv;
}

bool NcaFsHeaderReader::ExistsSparseLayer() const NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsInitialized());
    return m_Data.sparseInfo.generation != 0;
}

NcaSparseInfo& NcaFsHeaderReader::GetSparseInfo() NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsInitialized());
    return m_Data.sparseInfo;
}

const NcaSparseInfo& NcaFsHeaderReader::GetSparseInfo() const NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsInitialized());
    return m_Data.sparseInfo;
}

void NcaFsHeaderReader::GetRawData(void* pOutValue, size_t outSize) const NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsInitialized());
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);
    NN_SDK_REQUIRES_LESS_EQUAL(sizeof(NcaFsHeader), outSize);
    NN_UNUSED(outSize);

    std::memcpy(pOutValue, &m_Data, sizeof(NcaFsHeader));
}

Result NcaFsHeaderReader::Verify(NcaHeader::ContentType contentType) const NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsInitialized());
    NN_SDK_REQUIRES_MINMAX(int(contentType), int(NcaHeader::ContentType::Program), int(NcaHeader::ContentType::PublicData));

    NN_RESULT_DO(m_Data.Verify());

    // NOTE: 適宜修正する
    static const uint32_t ProgramCode = 1;
    static const uint32_t ProgramData = 2;
    static const uint32_t HtmlDocument = static_cast<uint32_t>(ncm::ContentType::HtmlDocument);
    static const uint32_t LegalInformation = static_cast<uint32_t>(ncm::ContentType::LegalInformation);

    const auto secureValue = m_Data.aesCtrUpperIv.part.secureValue & ~NcaFsHeader::ProgramIdOffsetMaskOnSecureValue;

    if (GetEncryptionType() == NcaFsHeader::EncryptionType::None)
    {
        NN_RESULT_THROW_UNLESS(secureValue == 0, fs::ResultInvalidNcaFsHeader());
        NN_RESULT_SUCCESS;
    }

    // secureValue のチェック
    switch( contentType )
    {
    case NcaHeader::ContentType::Program:
        switch( m_FsIndex )
        {
        case 0:
            NN_RESULT_THROW_UNLESS(secureValue == ProgramCode, fs::ResultInvalidNcaFsHeader());
            break;

        case 1:
            NN_RESULT_THROW_UNLESS(secureValue == ProgramData, fs::ResultInvalidNcaFsHeader());
            break;

        default:
            NN_RESULT_THROW_UNLESS(secureValue == 0, fs::ResultInvalidNcaFsHeader());
            break;
        }
        break;

    case NcaHeader::ContentType::Manual:
        NN_RESULT_THROW_UNLESS(
            secureValue == HtmlDocument || secureValue == LegalInformation,
            fs::ResultInvalidNcaFsHeader()
        );
        break;

    default:
        NN_RESULT_THROW_UNLESS(secureValue == 0, fs::ResultInvalidNcaFsHeader());
        break;
    }

    NN_RESULT_SUCCESS;
}

}}
