﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include <mutex>

#include <nn/nn_Abort.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_SdkLog.h>
#include <nn/util/util_FormatString.h>
#include <nn/os/os_Mutex.h>
#include <nn/os/os_Thread.h>
#include <nn/nn_TimeSpan.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/fs/fs_Result.h>
#include <nn/fs/fs_ResultPrivate.h>
#include <nn/fs/fsa/fs_IFile.h>
#include <nn/fs/fsa/fs_IDirectory.h>
#include <nn/fssystem/fs_DirectorySaveDataFileSystem.h>
#include <nn/fssystem/fs_PathTool.h>
#include <nn/fssystem/fs_Assert.h>
#include <nn/fssystem/fs_Utility.h>

using namespace nn;
using namespace nn::fs;
using namespace nn::fs::fsa;
using namespace nn::fssystem;

namespace nn { namespace fssystem {

    namespace
    {
        const int PathLengthMax = nn::fs::EntryNameLengthMax;

        const char CommittedDirectoryName[]     = "/0/";
        const char SynchronizingDirectoryName[] = "/_/";
        const char ModifiedDirectoryName[]      = "/1/";

        const size_t RequestWorkBufferSize = 1024 * 1024;


        // SourcePath ファイルを destinationParentPath/pEntry->name ファイルとしてコピー
        Result CopyFile(IFileSystem* pFileSystem, const char* destinationParentPath, const char* SourcePath, const DirectoryEntry* pEntry, char* workBuffer, size_t workBufferSize)
        {
            std::unique_ptr<IFile> pSrcFile;
            NN_RESULT_DO(pFileSystem->OpenFile(&pSrcFile, SourcePath, OpenMode_Read));

            std::unique_ptr<IFile> pDstFile;
            char path[nn::fs::EntryNameLengthMax + 1];
            const auto untruncatedSize = nn::util::SNPrintf(path, sizeof(path), "%s%s", destinationParentPath, pEntry->name);
            NN_SDK_ASSERT_LESS(static_cast<size_t>(untruncatedSize), sizeof(path));
            NN_RESULT_DO(pFileSystem->CreateFile(path, pEntry->fileSize));
            NN_RESULT_DO(pFileSystem->OpenFile(&pDstFile, path, OpenMode_Write));
            NN_UNUSED(untruncatedSize);

            int64_t restSize = pEntry->fileSize;
            int64_t offset = 0;
            while(restSize > 0)
            {
                size_t readSize;
                NN_RESULT_DO(pSrcFile->Read(&readSize, offset, workBuffer, workBufferSize, ReadOption()));
                NN_RESULT_DO(pDstFile->Write(offset, workBuffer, readSize, WriteOption()));

                restSize -= readSize;
                offset   += readSize;
            }

            NN_RESULT_SUCCESS;
        }

        Result CopyDirectoryRecursive(IFileSystem* pFileSystem, const char* destinationPath, const char* sourcePath, char* workBuffer, size_t workBufferSize)
        {
            char destinationPathBuffer[nn::fs::EntryNameLengthMax + 1];
            const auto untruncatedSize = nn::util::SNPrintf(destinationPathBuffer, sizeof(destinationPathBuffer), "%s", destinationPath);
            NN_SDK_ASSERT_LESS(static_cast<size_t>(untruncatedSize), sizeof(destinationPathBuffer));
            NN_UNUSED(untruncatedSize);

            return IterateDirectoryRecursive(pFileSystem, sourcePath,
                [&](const char* path, const DirectoryEntry& entry) -> Result
                {
                    NN_UNUSED(path);
                    strncat(destinationPathBuffer, entry.name, sizeof(destinationPathBuffer) - strnlen(destinationPathBuffer, sizeof(destinationPathBuffer) - 1) - 1);
                    strncat(destinationPathBuffer, "/", sizeof(destinationPathBuffer) - strnlen(destinationPathBuffer, sizeof(destinationPathBuffer) - 1) - 1);
                    return pFileSystem->CreateDirectory(destinationPathBuffer);
                },
                [&](const char* path, const DirectoryEntry& entry) -> Result
                {
                    NN_UNUSED(path);
                    NN_UNUSED(entry);
                    const auto length = strnlen(destinationPathBuffer, sizeof(destinationPathBuffer));
                    if( length < 2 )
                    {
                        NN_SDK_ASSERT(false);
                        return nn::fs::ResultInvalidPathFormat();
                    }
                    char* p = &destinationPathBuffer[length - 2];
                    while(*p != '/' && p > destinationPathBuffer)
                    {
                        p--;
                    }
                    *(p + 1) = '\0';
                    NN_RESULT_SUCCESS;
                },
                [&](const char* path, const DirectoryEntry& entry) -> Result
                {
                    return CopyFile(pFileSystem, destinationPathBuffer, path, &entry, workBuffer, workBufferSize);
                }
            );
        }


        template <typename FUNC>
        Result RetryFinitelyForTargetLocked(FUNC func)
        {
            const int RetryCountMax = 10;
            const int RetryIntervalMs = 100;

            int retryRestCount = RetryCountMax;
            while( NN_STATIC_CONDITION(true) )
            {
                NN_RESULT_TRY(func())
                    NN_RESULT_CATCH(ResultTargetLocked)
                    {
                        if( --retryRestCount < 0)
                        {
                            NN_RESULT_RETHROW;
                        }
                        else
                        {
                            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(RetryIntervalMs));
                            continue;
                        }
                    }
                NN_RESULT_END_TRY

                NN_RESULT_SUCCESS;
            }
        }


        class DirectorySaveDataFile: public nn::fs::fsa::IFile, public nn::fs::detail::Newable
        {
        public:
            NN_IMPLICIT DirectorySaveDataFile(std::unique_ptr<IFile>&& pBaseFile, DirectorySaveDataFileSystem* pParent, int mode) NN_NOEXCEPT
                : m_pBaseFile(std::move(pBaseFile)), m_pParent(pParent), m_Mode(mode)
            {
            }

            virtual ~DirectorySaveDataFile() NN_NOEXCEPT
            {
                if( (m_Mode & nn::fs::OpenMode_Write) != 0 )
                {
                    m_pParent->DecrementWriteOpenFileCount();
                }
            }

            virtual Result DoRead(size_t* outValue, int64_t offset, void *buffer, size_t size, const nn::fs::ReadOption& option) NN_NOEXCEPT NN_OVERRIDE
            {
                return m_pBaseFile->Read(outValue, offset, buffer, size, option);
            }
            virtual Result DoWrite(int64_t offset, const void *buffer, size_t size, const nn::fs::WriteOption& option) NN_NOEXCEPT NN_OVERRIDE
            {
                return m_pBaseFile->Write(offset, buffer, size, option);
            }
            virtual Result DoFlush() NN_NOEXCEPT NN_OVERRIDE
            {
                return m_pBaseFile->Flush();
            }
            virtual Result DoSetSize(int64_t size) NN_NOEXCEPT NN_OVERRIDE
            {
                return m_pBaseFile->SetSize(size);
            }
            virtual Result DoGetSize(int64_t *outValue) NN_NOEXCEPT NN_OVERRIDE
            {
                return m_pBaseFile->GetSize(outValue);
            }
            virtual Result DoOperateRange(
                void* outBuffer, size_t outBufferSize, OperationId operationId, int64_t offset, int64_t size, const void* inBuffer, size_t inBufferSize
            ) NN_NOEXCEPT NN_OVERRIDE
            {
                NN_RESULT_DO(m_pBaseFile->OperateRange(
                    outBuffer,
                    outBufferSize,
                    operationId,
                    offset,
                    size,
                    inBuffer,
                    inBufferSize));
                NN_RESULT_SUCCESS;
            }

        private:
            std::unique_ptr<IFile>       m_pBaseFile;
            DirectorySaveDataFileSystem* m_pParent;
            int m_Mode;
        };

    }


    DirectorySaveDataFileSystem::DirectorySaveDataFileSystem(IFileSystem* baseFileSystem, nn::MemoryResource* pAllocator) NN_NOEXCEPT
        : m_pBaseFileSystem(baseFileSystem),
          m_Mutex(false),
          m_pAllocator(pAllocator),
          m_WriteOpenFileCount(0)
    {
    }

    DirectorySaveDataFileSystem::DirectorySaveDataFileSystem(std::unique_ptr<nn::fs::fsa::IFileSystem>&& baseFileSystem, nn::MemoryResource* pAllocator) NN_NOEXCEPT
        : m_pBaseFileSystem(baseFileSystem.get()),
          m_Mutex(false),
          m_pDelegatedBaseFileSystem(std::move(baseFileSystem)),
          m_pAllocator(pAllocator),
          m_WriteOpenFileCount(0)
    {
    }

    Result DirectorySaveDataFileSystem::Initialize() NN_NOEXCEPT
    {
        nn::fs::DirectoryEntryType type;

        // 初回起動時セットアップ
        NN_RESULT_TRY(m_pBaseFileSystem->GetEntryType(&type, ModifiedDirectoryName))
            NN_RESULT_CATCH(ResultPathNotFound)
            {
                NN_RESULT_DO(m_pBaseFileSystem->CreateDirectory(ModifiedDirectoryName));
                NN_RESULT_DO(m_pBaseFileSystem->CreateDirectory(CommittedDirectoryName));
            }
        NN_RESULT_END_TRY;


        // 適切な内容に同期
        auto result = m_pBaseFileSystem->GetEntryType(&type, CommittedDirectoryName);
        if( result.IsSuccess() )
        {
            // modified を committed まで巻き戻す
            return SynchronizeDirectory(ModifiedDirectoryName, CommittedDirectoryName);
        }
        else if( nn::fs::ResultPathNotFound::Includes(result) )
        {
            // コミット処理の続きを行う
            NN_RESULT_DO(SynchronizeDirectory(SynchronizingDirectoryName, ModifiedDirectoryName));
            NN_RESULT_DO(m_pBaseFileSystem->RenameDirectory(SynchronizingDirectoryName, CommittedDirectoryName));
            NN_RESULT_SUCCESS;
        }
        else
        {
            return result;
        }
    }

    DirectorySaveDataFileSystem::~DirectorySaveDataFileSystem() NN_NOEXCEPT
    {
    }

    Result ResolveFullPath(char* pathBuffer, int bufferSize, const char* path) NN_NOEXCEPT
    {
        NN_FSP_REQUIRES( strnlen(path, EntryNameLengthMax + 1) < EntryNameLengthMax + 1, ResultTooLongPath() );
        strncpy(pathBuffer, ModifiedDirectoryName, bufferSize);
        pathBuffer[bufferSize - 1] = '\0';

        const auto ModifiedDirectoryNameLength = sizeof(ModifiedDirectoryName) - 1;

        NN_FSP_REQUIRES( *path == '/', ResultInvalidPath() );
        size_t normalizedPathLength;
        NN_RESULT_DO(PathTool::Normalize(pathBuffer + ModifiedDirectoryNameLength - 1, &normalizedPathLength, path, bufferSize - (ModifiedDirectoryNameLength - 1)));

        NN_RESULT_SUCCESS;
    }

    Result DirectorySaveDataFileSystem::DoCreateFile(const char *path, int64_t size, int option) NN_NOEXCEPT
    {
        char pathBuffer[PathLengthMax + 1];
        NN_RESULT_DO(ResolveFullPath(pathBuffer, sizeof(pathBuffer), path));

        std::lock_guard<os::Mutex> scopedLock(m_Mutex);
        return m_pBaseFileSystem->CreateFile(pathBuffer, size, option);
    }

    Result DirectorySaveDataFileSystem::DoDeleteFile(const char *path) NN_NOEXCEPT
    {
        char pathBuffer[PathLengthMax + 1];
        NN_RESULT_DO(ResolveFullPath(pathBuffer, sizeof(pathBuffer), path));

        std::lock_guard<os::Mutex> scopedLock(m_Mutex);
        return m_pBaseFileSystem->DeleteFile(pathBuffer);
    }

    Result DirectorySaveDataFileSystem::DoCreateDirectory(const char *path) NN_NOEXCEPT
    {
        char pathBuffer[PathLengthMax + 1];
        NN_RESULT_DO(ResolveFullPath(pathBuffer, sizeof(pathBuffer), path));

        std::lock_guard<os::Mutex> scopedLock(m_Mutex);
        return m_pBaseFileSystem->CreateDirectory(pathBuffer);
    }

    Result DirectorySaveDataFileSystem::DoDeleteDirectory(const char *path) NN_NOEXCEPT
    {
        char pathBuffer[PathLengthMax + 1];
        NN_RESULT_DO(ResolveFullPath(pathBuffer, sizeof(pathBuffer), path));

        std::lock_guard<os::Mutex> scopedLock(m_Mutex);
        return m_pBaseFileSystem->DeleteDirectory(pathBuffer);
    }

    Result DirectorySaveDataFileSystem::DoDeleteDirectoryRecursively(const char *path) NN_NOEXCEPT
    {
        char pathBuffer[PathLengthMax + 1];
        NN_RESULT_DO(ResolveFullPath(pathBuffer, sizeof(pathBuffer), path));

        std::lock_guard<os::Mutex> scopedLock(m_Mutex);
        return m_pBaseFileSystem->DeleteDirectoryRecursively(pathBuffer);
    }

    Result DirectorySaveDataFileSystem::DoCleanDirectoryRecursively(const char* path) NN_NOEXCEPT
    {
        char pathBuffer[PathLengthMax + 1];
        NN_RESULT_DO(ResolveFullPath(pathBuffer, sizeof(pathBuffer), path));

        std::lock_guard<os::Mutex> scopedLock(m_Mutex);
        return m_pBaseFileSystem->CleanDirectoryRecursively(pathBuffer);
    }

    Result DirectorySaveDataFileSystem::DoRenameFile(const char *currentPath, const char *newPath) NN_NOEXCEPT
    {
        char currentPathBuffer[PathLengthMax + 1];
        char newPathBuffer[PathLengthMax + 1];
        NN_RESULT_DO(ResolveFullPath(currentPathBuffer, sizeof(currentPathBuffer), currentPath));
        NN_RESULT_DO(ResolveFullPath(newPathBuffer,     sizeof(newPathBuffer),     newPath));

        std::lock_guard<os::Mutex> scopedLock(m_Mutex);
        return m_pBaseFileSystem->RenameFile(currentPathBuffer, newPathBuffer);
    }

    Result DirectorySaveDataFileSystem::DoRenameDirectory(const char *currentPath, const char *newPath) NN_NOEXCEPT
    {
        char currentPathBuffer[PathLengthMax + 1];
        char newPathBuffer[PathLengthMax + 1];
        NN_RESULT_DO(ResolveFullPath(currentPathBuffer, sizeof(currentPathBuffer), currentPath));
        NN_RESULT_DO(ResolveFullPath(newPathBuffer,     sizeof(newPathBuffer),     newPath));

        std::lock_guard<os::Mutex> scopedLock(m_Mutex);
        return m_pBaseFileSystem->RenameDirectory(currentPathBuffer, newPathBuffer);
    }

    Result DirectorySaveDataFileSystem::DoGetEntryType(nn::fs::DirectoryEntryType* outValue, const char *path) NN_NOEXCEPT
    {
        char pathBuffer[PathLengthMax + 1];
        NN_RESULT_DO(ResolveFullPath(pathBuffer, sizeof(pathBuffer), path));

        std::lock_guard<os::Mutex> scopedLock(m_Mutex);
        return m_pBaseFileSystem->GetEntryType(outValue, pathBuffer);
    }

    Result DirectorySaveDataFileSystem::DoOpenFile(std::unique_ptr<nn::fs::fsa::IFile>* outValue, const char* path, nn::fs::OpenMode mode) NN_NOEXCEPT
    {
        char pathBuffer[PathLengthMax + 1];
        NN_RESULT_DO(ResolveFullPath(pathBuffer, sizeof(pathBuffer), path));

        std::lock_guard<os::Mutex> scopedLock(m_Mutex);
        std::unique_ptr<IFile> pBaseFile;
        NN_RESULT_DO(m_pBaseFileSystem->OpenFile(&pBaseFile, pathBuffer, mode));

        std::unique_ptr<DirectorySaveDataFile> pFile(new DirectorySaveDataFile(std::move(pBaseFile), this, mode));
        NN_RESULT_THROW_UNLESS(pFile != nullptr, ResultAllocationMemoryFailedInDirectorySaveDataFileSystemA());

        if( (mode & nn::fs::OpenMode_Write) != 0 )
        {
            ++m_WriteOpenFileCount;
        }
        *outValue = std::move(pFile);

        NN_RESULT_SUCCESS;
    }

    Result DirectorySaveDataFileSystem::DoOpenDirectory(std::unique_ptr<nn::fs::fsa::IDirectory>* outValue, const char* path, nn::fs::OpenDirectoryMode mode) NN_NOEXCEPT
    {
        char pathBuffer[PathLengthMax + 1];
        NN_RESULT_DO(ResolveFullPath(pathBuffer, sizeof(pathBuffer), path));

        std::lock_guard<os::Mutex> scopedLock(m_Mutex);
        return m_pBaseFileSystem->OpenDirectory(outValue, pathBuffer, mode);
    }

    Result DirectorySaveDataFileSystem::AllocateBuffer(char** pOutBuffer, size_t* pOutBufferSize, size_t requestSize) NN_NOEXCEPT
    {
        size_t tryRequestSize = requestSize;

        while( tryRequestSize > 512 )
        {
            char* pBuffer = static_cast<char*>(m_pAllocator->allocate(tryRequestSize));
            if( pBuffer != nullptr )
            {
                *pOutBuffer = pBuffer;
                *pOutBufferSize = tryRequestSize;
                NN_RESULT_SUCCESS;
            }

            tryRequestSize >>= 1;
        }

        // win 版にしか残らない且つ想定外の挙動なので ABORT する
        NN_ABORT("Failed to allocate buffer for CommitSaveData().\n");
    }


    // destinationPath の内容を sourcePath の内容と同一にする
    Result DirectorySaveDataFileSystem::SynchronizeDirectory(const char* destinationPath, const char* sourcePath) NN_NOEXCEPT
    {
        // TODO: アトミックになっていない対策
        // TODO: 更新がないものもコピーしなおしている点の改善
        NN_RESULT_DO(m_pBaseFileSystem->DeleteDirectoryRecursively(destinationPath));
        NN_RESULT_DO(m_pBaseFileSystem->CreateDirectory(destinationPath));

        char*  workBuffer = nullptr;
        size_t workBufferSize = 0;
        NN_RESULT_DO(AllocateBuffer(&workBuffer, &workBufferSize, RequestWorkBufferSize));
        NN_UTIL_SCOPE_EXIT
        {
            m_pAllocator->deallocate(workBuffer, workBufferSize);
        };

        NN_RESULT_DO(CopyDirectoryRecursive(m_pBaseFileSystem, destinationPath, sourcePath, workBuffer, workBufferSize));

        NN_RESULT_SUCCESS;
    }

    Result DirectorySaveDataFileSystem::DoCommit() NN_NOEXCEPT
    {
        std::lock_guard<os::Mutex> scopedLock(m_Mutex);

        NN_RESULT_THROW_UNLESS(m_WriteOpenFileCount == 0, nn::fs::ResultPreconditionViolation());

        NN_RESULT_DO(RetryFinitelyForTargetLocked([&]()
            {
                return m_pBaseFileSystem->RenameDirectory(CommittedDirectoryName, SynchronizingDirectoryName);
            }
        ));

        // この時点でコミット確定

        // TODO: エラー発生時のケア
        NN_RESULT_DO(RetryFinitelyForTargetLocked([&]()
            {
                return SynchronizeDirectory(SynchronizingDirectoryName, ModifiedDirectoryName);
            }
        ));

        // TODO: エラー発生時のケア
        NN_RESULT_DO(RetryFinitelyForTargetLocked([&]()
            {
                return m_pBaseFileSystem->RenameDirectory(SynchronizingDirectoryName, CommittedDirectoryName);
            }
        ));

        NN_RESULT_SUCCESS;
    }

    Result DirectorySaveDataFileSystem::DoCommitProvisionally(int64_t counter) NN_NOEXCEPT
    {
        NN_UNUSED(counter);
        NN_RESULT_SUCCESS;
    }

    Result DirectorySaveDataFileSystem::DoRollback() NN_NOEXCEPT
    {
        NN_RESULT_DO(Initialize());
        NN_RESULT_SUCCESS;
    }

    void DirectorySaveDataFileSystem::DecrementWriteOpenFileCount() NN_NOEXCEPT
    {
        std::lock_guard<os::Mutex> scopedLock(m_Mutex);
        --m_WriteOpenFileCount;
    }


}}
