﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/


#include <cstring>
#include <nn/fs_Base.h>
#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/nn_SdkAssert.h>
#include <nn/result/result_HandlingUtility.h>

#include <nn/fssystem/fs_DbmRomTypes.h>
#include <nn/fssystem/fs_DbmRomPathTool.h>

namespace nn { namespace fssystem {

/*!
    @brief コンストラクタです。
*/
RomPathTool::PathParser::PathParser()
    : m_pPrevStartPath(NULL)
    , m_pPrevEndPath(NULL)
    , m_pNextPath(NULL)
    ,m_bParseFinished(false)
{
}

/*!
    @brief パーサを初期化します。

    @param[in] pFullPath フルパス

    @return 関数の処理結果を返します。
*/
Result RomPathTool::PathParser::Initialize(const RomPathChar* pFullPath)
{
    NN_SDK_ASSERT_NOT_NULL(pFullPath);

    // フルパスがルートディレクトリから始まっているかどうかチェックします。
    if (! IsSeparator(pFullPath[0]))
    {
        // 不正なパスです。
        return ResultDbmInvalidPathFormat();
    }
    // セパレーターが連続していたら読み進めておきます
    while (IsSeparator(pFullPath[1]))
    {
        pFullPath ++;
    }

    // 初回のみ先頭のセパレーターを指し示すようにします。
    m_pPrevStartPath = pFullPath;

    // 初回のみ先頭のセパレーターを指し示すようにします。
    m_pPrevEndPath = m_pPrevStartPath;

    // m_pNextPath は常に区切り文字の次の文字を指すようにします。
    m_pNextPath = &pFullPath[1];
    while (IsSeparator(m_pNextPath[0]))
    {
        m_pNextPath ++;
    }

    return ResultSuccess();
}

/*!
    @brief パーサを破棄します。
*/
void RomPathTool::PathParser::Finalize()
{
}

/*!
    @brief パスのパースが完了したかどうかを取得します。
*/
bool RomPathTool::PathParser::IsParseFinished() const
{
    return m_bParseFinished;
}

/*!
    @brief  パスがディレクトリだったかどうかを取得します。

    @pre    IsParseFinished が true の時のみ使用できます。
*/
bool RomPathTool::PathParser::IsDirectoryPath() const
{
    NN_SDK_ASSERT_NOT_NULL(m_pNextPath);
    // パスの最後が区切り文字かどうかで判定します。
    if ((m_pNextPath[0] == NUL) && (m_pNextPath[-1] == DIR_SEPARATOR))
    {
        return true;
    }

    // カレントディレクトリを示す文字であれば true を返します。
    if (IsCurrentDirectory(m_pNextPath))
    {
        return true;
    }

    // 親ディレクトリを示す文字であれば true を返します。
    if (IsParentDirectory(m_pNextPath))
    {
        return true;
    }

    return false;
}

/*!
    @brief 最上位ディレクトリ部分を取得します。

    @param[out] pDirName ディレクトリ名

    @return 関数の処理結果を返します。
*/
Result RomPathTool::PathParser::GetNextDirectoryName(RomEntryName* pDirName)
{
    NN_SDK_ASSERT_NOT_NULL(m_pPrevStartPath);
    NN_SDK_ASSERT_NOT_NULL(m_pPrevEndPath);
    NN_SDK_ASSERT_NOT_NULL(m_pNextPath);
    NN_SDK_ASSERT_NOT_NULL(pDirName);

    // 現在のディレクトリ名をコピーします。
    pDirName->length = (m_pPrevEndPath - m_pPrevStartPath);
    pDirName->path = m_pPrevStartPath;

    // 次のパスの始点へ進めます
    m_pPrevStartPath = m_pNextPath;

    // 次の区切り文字を探索します。
    const RomPathChar* p = m_pNextPath;
    for (size_t dirNameLength = 0; ; dirNameLength++)
    {
        // 区切り文字が見つかれば探索終了です。
        if (IsSeparator(p[dirNameLength]))
        {
            if (dirNameLength >= MAX_PATH_LENGTH)
            {
                // ディレクトリ名が長すぎます。
                return ResultDbmDirectoryNameTooLong();
            }

            // 区切り文字の一文字手前を指すようにします。
            m_pPrevEndPath = &p[dirNameLength];
            m_pNextPath = m_pPrevEndPath + 1;

            // 連続区切り文字の次の文字を指すようにします。
            while (IsSeparator(*m_pNextPath))
            {
                m_pNextPath ++;
            }
            if (*m_pNextPath == NUL)
            {
                m_bParseFinished = true;
            }
            break;
        }

        // 終端文字が見つかれば探索終了です。
        if (p[dirNameLength] == NUL)
        {
            m_bParseFinished = true;
            m_pPrevEndPath = m_pNextPath = &p[dirNameLength];
            break;
        }
    }

    return ResultSuccess();
}

/*!
    @brief 文字列の先頭から次の１節をディレクトリ名として取得します。

    @param[out] pName ディレクトリ名

    @return 関数の処理結果を返します。
*/
Result RomPathTool::PathParser::GetAsDirectoryName(RomEntryName* pName) const
{
    NN_SDK_ASSERT_NOT_NULL(m_pNextPath);
    NN_SDK_ASSERT_NOT_NULL(m_pPrevStartPath);
    NN_SDK_ASSERT_NOT_NULL(m_pPrevEndPath);
    NN_SDK_ASSERT_NOT_NULL(pName);

    size_t numChar = m_pPrevEndPath - m_pPrevStartPath;
    if (numChar > MAX_PATH_LENGTH)
    {
        // ディレクトリ名が長すぎます。
        return ResultDbmDirectoryNameTooLong();
    }

    // ディレクトリ名データへの参照を設定します。
    pName->length = numChar;
    pName->path = m_pPrevStartPath;

    return ResultSuccess();
}

/*!
    @brief 文字列の先頭から次の１節をファイル名として取得します。

    @param[out] pName ファイル名

    @return 関数の処理結果を返します。
*/
Result RomPathTool::PathParser::GetAsFileName(RomEntryName* pName) const
{
    NN_SDK_ASSERT_NOT_NULL(m_pNextPath);
    NN_SDK_ASSERT_NOT_NULL(m_pPrevStartPath);
    NN_SDK_ASSERT_NOT_NULL(m_pPrevEndPath);
    NN_SDK_ASSERT_NOT_NULL(pName);

    size_t numChar = m_pPrevEndPath - m_pPrevStartPath;
    if (numChar > MAX_PATH_LENGTH)
    {
        // ファイル名が長すぎます。
        return ResultDbmFileNameTooLong();
    }

    // ファイル名データへの参照を設定します。
    pName->length = numChar;
    pName->path = m_pPrevStartPath;

    return ResultSuccess();
}

/*!
    @brief 親ディレクトリ名を抽出します。

           フルパス:"/AAA/BBB/CCC" 開始:"AAA" => 親ディレクトリ:"" (ルートディレクトリ)
           フルパス:"/AAA/BBB/CCC" 開始:"BBB" => 親ディレクトリ:"AAA"
           フルパス:"/AAA/BBB/CCC/." 開始:"." => 親ディレクトリ:"BBB"
           フルパス:"/AAA/BBB/CCC/.." 開始:".." => 親ディレクトリ:"AAA"
           フルパス:"/AAA/.." 開始:".." => ResultDbmInvalidPathFormat()

    @param[out] pOut 親ディレクトリ名
    @param[in] base カレントディレクトリ名。
                    フルパス内の文字列を指している必要があります。
    @param[in] pFullPath フルパス

    @return 関数の処理結果を返します。
*/
Result RomPathTool::GetParentDirectoryName(RomEntryName* pOut, const RomEntryName& base, const RomPathChar* pHead)
{
    const RomPathChar* pStart = base.path;
    const RomPathChar* pEnd = base.path + base.length - 1;

    // スキップするセパレータの個数を決定します。
    int32_t depth = 1;

    // カレントディレクトリに当たる部分が親ディレクトリを
    // 指していればセパレータを 1 個余分にスキップします。
    if (IsParentDirectory(base))
    {
        depth++;
    }

    if (base.path > pHead)
    {
        // パスの先頭に向かって親ディレクトリの探索を行います。
        size_t length = 0;
        const RomPathChar* p = base.path - 1;
        while (p >= pHead)
        {
            if (IsSeparator(*p))
            {
                // カレントディレクトリであればもう 1 つ親のディレクトリまで
                // 探索を行います。
                if (IsCurrentDirectory(p + 1, length))
                {
                    depth++;
                }

                // 親ディレクトリであればもう 2 つ親のディレクトリまで
                // 探索を行います。
                if (IsParentDirectory(p + 1, length))
                {
                    depth += 2;
                }

                // 親ディレクトリに到達しました。
                if (depth == 0)
                {
                    pStart = p + 1;
                    break;
                }

                // 連続するセパレータをスキップします。
                while (IsSeparator(*p))
                {
                    p--;
                }

                // 抽出する範囲の末尾を更新します。
                pEnd = p;
                length = 0;
                depth--;
            }

            length++;
            p--;
        }

        // ルートディレクトリの親ディレクトリは取得できません。
        if (depth != 0)
        {
            return ResultDbmInvalidPathFormat();
        }

        // 先頭のセパレータまで到達すれば、セパレータの次の文字を
        // 開始地点とします。
        if (p == pHead)
        {
            pStart = pHead + 1;
        }
    }

    if (pEnd <= pHead)
    {
        // ルートディレクトリです。
        pOut->path = pHead;
        pOut->length = 0;
    }
    else
    {
        // 開始地点から範囲の末尾までを抽出します。
        pOut->path = pStart;
        pOut->length = (pEnd - pStart + 1);
    }

    return ResultSuccess();
}

}}
