﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

// ビルドスイッチのため、例外的に先頭でインクルードする
#include <nn/fs/detail/fs_ResultHandlingUtilitySuppressRecordingEventOnUnsupportedPlatforms.h>

#include <new>
#include <cstring>
#include <vector>
#include <algorithm>

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>

#include <nn/nn_SdkAssert.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_FormatString.h>

#include <nn/fs/fsa/fs_IFileSystem.h>
#include <nn/fs/fs_Result.h>
#include <nn/fs/fs_ResultPrivate.h>
#include <nn/fs/fs_QueryRange.h>
#include <nn/fs/detail/fs_Newable.h>
#include <nn/fs/fsa/fs_IFile.h>
#include <nn/fs/fsa/fs_IDirectory.h>
#include <nn/fat/fat_FatFileSystem.h>
#include <nn/fssystem/fs_ConcatenationFileSystem.h>
#include <nn/fssystem/fs_Assert.h>
#include <nn/fssystem/fs_Utility.h>
#include <nn/fs/fs_FileSystemPrivate.h>


using namespace nn::fat;
using namespace nn::fs;
using namespace nn::fs::fsa;
using namespace nn::fs::detail;


namespace nn { namespace fssystem {


namespace {

    template <typename T>
    class Allocator : public std::allocator<T>
    {
    public:
        Allocator() NN_NOEXCEPT{}
        Allocator(const Allocator&) NN_NOEXCEPT{}
        template<typename U>
        Allocator(const Allocator<U>&) NN_NOEXCEPT{}

        template<typename U>
        struct rebind
        {
            typedef Allocator<U> other;
        };

        T* allocate(size_t size, const T* hint = 0) NN_NOEXCEPT
        {
            // TODO: fssrv のアロケータを使う
            NN_UNUSED(hint);
            return static_cast<T*>(nn::fs::detail::Allocate(sizeof(T) * size));
        }
        void deallocate(T* p, size_t size) NN_NOEXCEPT
        {
            nn::fs::detail::Deallocate(p, sizeof(T) * size);
        }
    };


    typedef std::vector<std::unique_ptr<IFile>, Allocator<std::unique_ptr<IFile>>> FileArray;

    Result GenerateInternalFilePath(char* internalFilePathBuffer, size_t internalFilePathBufferSize, const char* path, int index) NN_NOEXCEPT
    {
        const auto untruncatedSize = nn::util::SNPrintf(internalFilePathBuffer, internalFilePathBufferSize, "%s/%02d", path, index);
        if( internalFilePathBufferSize <= static_cast<size_t>(untruncatedSize) )
        {
            return nn::fs::ResultTooLongPath();
        }
        NN_RESULT_SUCCESS;
    }

    bool IsConcatenationFileAttribute(int attribute) NN_NOEXCEPT
    {
        return (attribute & nn::fat::Attribute::Attribute_Archive) != 0 && (attribute & nn::fat::Attribute::Attribute_Directory) != 0;
    }

    class ConcatenationFile: public IFile, public Newable
    {
    public:
        static size_t GetPathBufferSize() NN_NOEXCEPT
        {
            return EntryNameLengthMax + 1;
        }

        ConcatenationFile(OpenMode mode, FileArray&& fileArray, int64_t divisionSize, FatFileSystem* pBaseFileSystem, const char* parentDirectoryPath) NN_NOEXCEPT
            : m_Mode(mode),
              m_FileArray(std::move(fileArray)),
              m_DivisionSize(divisionSize),
              m_pBaseFileSystem(pBaseFileSystem)
        {
            NN_SDK_REQUIRES_LESS(
                strnlen(parentDirectoryPath, sizeof(m_ParentDirectoryPath)),
                sizeof(m_ParentDirectoryPath));
            strncpy(m_ParentDirectoryPath, parentDirectoryPath, sizeof(m_ParentDirectoryPath));
            m_ParentDirectoryPath[sizeof(m_ParentDirectoryPath) - 1] = '\0';
        }

        virtual ~ConcatenationFile() NN_NOEXCEPT
        {
        }

        virtual Result DoRead(size_t* outValue, int64_t offset, void *buffer, size_t size, const ReadOption& option) NN_NOEXCEPT NN_OVERRIDE;
        virtual Result DoWrite(int64_t offset, const void *buffer, size_t size, const WriteOption& option) NN_NOEXCEPT NN_OVERRIDE;
        virtual Result DoFlush() NN_NOEXCEPT NN_OVERRIDE;
        virtual Result DoSetSize(int64_t size) NN_NOEXCEPT NN_OVERRIDE;
        virtual Result DoGetSize(int64_t *outValue) NN_NOEXCEPT NN_OVERRIDE;
        virtual Result DoOperateRange(
            void* outBuffer,
            size_t outBufferSize,
            OperationId operationId,
            int64_t offset,
            int64_t size,
            const void* inBuffer,
            size_t inBufferSize) NN_NOEXCEPT NN_OVERRIDE;

    private:
        int     GetInternalFileIndex(int64_t offset) NN_NOEXCEPT;
        int     GetInternalFileCount(int64_t size) NN_NOEXCEPT;
        int64_t GetInternalFileSize(int64_t wholeSize, int index) NN_NOEXCEPT;

        template<typename TFunc>
        Result DoOperateRangeImpl(int64_t offset, int64_t size, TFunc func) NN_NOEXCEPT;

    private:
        const OpenMode       m_Mode;
        FileArray            m_FileArray;
        const int64_t        m_DivisionSize;
        FatFileSystem* const m_pBaseFileSystem;
        char                 m_ParentDirectoryPath[EntryNameLengthMax + 1];
    };

    //! サイズに対し分割ファイルの数を返す
    int ConcatenationFile::GetInternalFileCount(int64_t size) NN_NOEXCEPT
    {
        if( size == 0 ) return 1;
        return static_cast<int>((size + m_DivisionSize - 1) / m_DivisionSize);
    }

    //! オフセットに対し該当する分割ファイルのインデックスを返す
    int ConcatenationFile::GetInternalFileIndex(int64_t offset) NN_NOEXCEPT
    {
        return static_cast<int>(offset / m_DivisionSize);
    }

    //! 全体サイズに対して index 番目の分割ファイルサイズを返す
    int64_t ConcatenationFile::GetInternalFileSize(int64_t wholeSize, int index) NN_NOEXCEPT
    {
        auto tailIndex = GetInternalFileIndex(wholeSize);
        if( index < tailIndex )
        {
            return m_DivisionSize;
        }
        else
        {
            NN_SDK_ASSERT(index == tailIndex);
            return wholeSize % m_DivisionSize;
        }
    }

    Result ConcatenationFile::DoRead(size_t* outValue, int64_t offset, void *buffer, size_t size, const ReadOption& option) NN_NOEXCEPT
    {
        size_t restSize = 0;
        NN_RESULT_DO(DryRead(&restSize, offset, size, option, m_Mode));

        size_t doneSize = 0;

        while(restSize > 0)
        {
            int index = GetInternalFileIndex(offset);

            int64_t internalFileRestSize = m_DivisionSize - GetInternalFileSize(offset, index);
            int64_t internalFileOffset = offset - index * m_DivisionSize;
            size_t accessSize = static_cast<size_t>(std::min<int64_t>(restSize, internalFileRestSize));

            size_t readSize;
            NN_SDK_ASSERT(static_cast<size_t>(index) < m_FileArray.size());
            NN_RESULT_DO(m_FileArray[index]->Read(&readSize, internalFileOffset, static_cast<char*>(buffer) + doneSize, accessSize, option));

            restSize -= readSize;
            doneSize += readSize;
            offset   += readSize;

            if( readSize < accessSize )
            {
                break;
            }
        }

        *outValue = doneSize;
        NN_RESULT_SUCCESS;
    }

    Result ConcatenationFile::DoWrite(int64_t offset, const void *buffer, size_t size, const WriteOption& option) NN_NOEXCEPT
    {
        NN_FSP_REQUIRES( (m_Mode & nn::fs::OpenMode_Write) != 0, ResultInvalidOperationForOpenMode() );

        if( offset < 0 )
        {
            return ResultOutOfRange();
        }

        int64_t currentSize;
        NN_RESULT_DO(GetSize(&currentSize));

        if ((m_Mode & OpenMode_AllowAppend) == 0)
        {
            NN_FSP_REQUIRES(static_cast<int64_t>(offset + size) <= currentSize, ResultFileExtensionWithoutOpenModeAllowAppend(), "OpenMode_AllowAppend is required for implicit extension of file size by WriteFile().");
        }

        if (size != 0 && static_cast<int64_t>(offset + size) > currentSize)
        {
            NN_RESULT_DO(SetSize(offset + size));
        }

        size_t restSize = size;
        size_t doneSize = 0;

        while(restSize > 0)
        {
            int index = GetInternalFileIndex(offset);

            int64_t internalFileRestSize = m_DivisionSize - GetInternalFileSize(offset, index);
            int64_t internalFileOffset = offset - index * m_DivisionSize;
            size_t accessSize = static_cast<size_t>(std::min<int64_t>(restSize, internalFileRestSize));

            NN_SDK_ASSERT(static_cast<size_t>(index) < m_FileArray.size());
            NN_RESULT_DO(m_FileArray[index]->Write(internalFileOffset, static_cast<const char*>(buffer) + doneSize, accessSize, option));

            restSize -= accessSize;
            doneSize += accessSize;
            offset   += accessSize;
        }

        if ((option.flags & WriteOptionFlag_Flush) != 0)
        {
            NN_RESULT_DO(Flush());
        }


        NN_RESULT_SUCCESS;
    }

    Result ConcatenationFile::DoFlush() NN_NOEXCEPT
    {
        if( (m_Mode & nn::fs::OpenMode_Write) == 0 )
        {
            NN_RESULT_SUCCESS;
        }

        for(int index = 0; index < static_cast<int>(m_FileArray.size()); ++index)
        {
            NN_SDK_ASSERT(m_FileArray[index] != nullptr);
            m_FileArray[index]->Flush();
        }

        NN_RESULT_SUCCESS;
    }

    Result ConcatenationFile::DoSetSize(int64_t size) NN_NOEXCEPT
    {
        NN_RESULT_DO(DrySetSize(size, m_Mode));

        int64_t currentSize;
        NN_RESULT_DO(GetSize(&currentSize));

        const int currentTailFileIndex = GetInternalFileCount(currentSize) - 1;
        const int newTailFileIndex     = GetInternalFileCount(size) - 1;

        if( size > currentSize )
        {
            // extend

            // 現在の末尾ファイルの伸長
            NN_RESULT_DO(m_FileArray[currentTailFileIndex]->SetSize(GetInternalFileSize(size, currentTailFileIndex)));

            // 分割ファイルの追加
            for(int index = currentTailFileIndex + 1; index <= newTailFileIndex; ++index)
            {
                char internalFilePath[EntryNameLengthMax + 1];
                NN_RESULT_DO(GenerateInternalFilePath(internalFilePath, sizeof(internalFilePath), m_ParentDirectoryPath, index));

                NN_RESULT_DO(m_pBaseFileSystem->CreateFile(internalFilePath, GetInternalFileSize(size, index), 0));

                std::unique_ptr<IFile> file;

                NN_RESULT_DO(m_pBaseFileSystem->OpenFile(&file, internalFilePath, m_Mode));
                m_FileArray.push_back(std::move(file));
            }

        }
        else
        {
            // truncate

            for(int index = currentTailFileIndex; index >= newTailFileIndex + 1; --index)
            {
                // close
                m_FileArray.pop_back();

                char internalFilePath[EntryNameLengthMax + 1];
                NN_RESULT_DO(GenerateInternalFilePath(internalFilePath, sizeof(internalFilePath), m_ParentDirectoryPath, index));

                NN_RESULT_DO(m_pBaseFileSystem->DeleteFile(internalFilePath));
            }

            NN_RESULT_DO(m_FileArray[newTailFileIndex]->SetSize(GetInternalFileSize(size, newTailFileIndex)));

        }

        NN_RESULT_SUCCESS;
    }

    Result ConcatenationFile::DoGetSize(int64_t *outValue) NN_NOEXCEPT
    {
        int64_t totalSize = 0;
        for(int index = 0; index < static_cast<int>(m_FileArray.size()); ++index)
        {
            int64_t size;
            NN_RESULT_DO(m_FileArray[index]->GetSize(&size));
            totalSize += size;
        }

        *outValue = totalSize;
        NN_RESULT_SUCCESS;
    }

    Result ConcatenationFile::DoOperateRange(
        void* outBuffer,
        size_t outBufferSize,
        OperationId operationId,
        int64_t offset,
        int64_t size,
        const void* inBuffer,
        size_t inBufferSize) NN_NOEXCEPT
    {
        switch( operationId )
        {
        case OperationId::Invalidate:
            {
                NN_FSP_REQUIRES(
                    (m_Mode & nn::fs::OpenMode_Read) != 0,
                    ResultInvalidOperationForOpenMode());

                NN_RESULT_DO(DoOperateRangeImpl(
                    offset,
                    size,
                    [=](nn::fs::fsa::IFile* pFile,
                        int64_t internalFileOffset,
                        int64_t accessSize) NN_NOEXCEPT -> nn::Result
                    {
                        NN_RESULT_DO(pFile->OperateRange(
                            outBuffer,
                            outBufferSize,
                            operationId,
                            internalFileOffset,
                            accessSize,
                            inBuffer,
                            inBufferSize));
                        NN_RESULT_SUCCESS;
                    }
                ));
                NN_RESULT_SUCCESS;
            }

        case OperationId::QueryRange:
            {
                NN_FSP_REQUIRES(outBuffer != nullptr, nn::fs::ResultNullptrArgument());
                NN_FSP_REQUIRES(outBufferSize == sizeof(QueryRangeInfo), nn::fs::ResultInvalidSize());

                QueryRangeInfo infoMerged;
                infoMerged.Clear();

                NN_RESULT_DO(DoOperateRangeImpl(
                    offset,
                    size,
                    [=, &infoMerged](nn::fs::fsa::IFile* pFile,
                        int64_t internalFileOffset,
                        int64_t accessSize) NN_NOEXCEPT->nn::Result
                    {
                        QueryRangeInfo infoEntry;
                        NN_RESULT_DO(pFile->OperateRange(
                            &infoEntry,
                            sizeof(infoEntry),
                            operationId,
                            internalFileOffset,
                            accessSize,
                            inBuffer,
                            inBufferSize));
                        infoMerged.Merge(infoEntry);
                        NN_RESULT_SUCCESS;
                    }
                ));
                *reinterpret_cast<QueryRangeInfo*>(outBuffer) = infoMerged;
                NN_RESULT_SUCCESS;
            }

        default:
            return nn::fs::ResultUnsupportedOperation();
        }
    }

    template<typename TFunc>
    Result ConcatenationFile::DoOperateRangeImpl(
        int64_t offset,
        int64_t size,
        TFunc func) NN_NOEXCEPT
    {
        NN_RESULT_THROW_UNLESS(0 <= offset, nn::fs::ResultOutOfRange());

        int64_t currentSize;
        NN_RESULT_DO(GetSize(&currentSize));
        NN_RESULT_THROW_UNLESS(offset <= currentSize, nn::fs::ResultOutOfRange());

        auto currentOffset = offset;
        auto restSize = std::max<int64_t>(std::min(size, currentSize - offset), 0);

        while( 0 < restSize )
        {
            const auto fileIndex = GetInternalFileIndex(currentOffset);

            const auto internalFileRestSize
                = m_DivisionSize - GetInternalFileSize(currentOffset, fileIndex);
            const auto internalFileOffset
                = currentOffset - fileIndex * m_DivisionSize;
            const auto accessSize
                = std::min(restSize, internalFileRestSize);

            NN_SDK_ASSERT(static_cast<size_t>(fileIndex) < m_FileArray.size());
            NN_RESULT_DO(func(m_FileArray[fileIndex].get(), internalFileOffset, accessSize));

            restSize -= accessSize;
            currentOffset += accessSize;
        }

        NN_RESULT_SUCCESS;
    }

    class ConcatenationDirectory : public IDirectory, public Newable
    {
    public:
        static size_t GetPathBufferSize() NN_NOEXCEPT
        {
            return EntryNameLengthMax + 1;
        }

        ConcatenationDirectory(OpenDirectoryMode mode, std::unique_ptr<IDirectory>&& baseDirectory, ConcatenationFileSystem* pParentFileSystem, FatFileSystem* pBaseFileSystem, const char* path) NN_NOEXCEPT
            : m_Mode(mode),
                m_BaseDirectory(std::move(baseDirectory)),
                m_pBaseFileSystem(pBaseFileSystem),
                m_pParentFileSystem(pParentFileSystem)
        {
            // TODO: パスの正規化

            // m_Path は必ずスラッシュ終わりであるとする
            strncpy(m_Path, path, sizeof(m_Path));
            m_Path[sizeof(m_Path) - 1] = '\0';
            const auto length = strnlen(m_Path, sizeof(m_Path));
            if( length == 0 || m_Path[length - 1] != '/' )
            {
                NN_ABORT_UNLESS(length < sizeof(m_Path) - 1);
                strncat(m_Path, "/", 2);
                m_Path[sizeof(m_Path) - 1] = '\0';
            }
        }

        virtual ~ConcatenationDirectory() NN_NOEXCEPT
        {
        }

        virtual Result DoRead(int64_t *outValue, DirectoryEntry *entryBuffer, int64_t entryBufferCount) NN_NOEXCEPT;
        virtual Result DoGetEntryCount(int64_t *outValue) NN_NOEXCEPT;

    private:
        const OpenDirectoryMode        m_Mode;
        std::unique_ptr<IDirectory>    m_BaseDirectory;
        char                           m_Path[EntryNameLengthMax + 1];
        FatFileSystem* const           m_pBaseFileSystem;
        ConcatenationFileSystem* const m_pParentFileSystem;

        bool IsReadTarget(DirectoryEntry* pEntry) const NN_NOEXCEPT;
    };

    bool ConcatenationDirectory::IsReadTarget(DirectoryEntry* pEntry) const NN_NOEXCEPT
    {
        return
            ((m_Mode & OpenDirectoryMode_File)      && (pEntry->directoryEntryType == DirectoryEntryType_File      ||  IsConcatenationFileAttribute(pEntry->reserved1[0]))) ||
            ((m_Mode & OpenDirectoryMode_Directory) && (pEntry->directoryEntryType == DirectoryEntryType_Directory && !IsConcatenationFileAttribute(pEntry->reserved1[0]))) ;
    }

    Result ConcatenationDirectory::DoRead(int64_t *outValue, DirectoryEntry* entryBuffer, int64_t entryBufferCount) NN_NOEXCEPT
    {
        int64_t entryCount = 0;
        while( entryCount < entryBufferCount )
        {
            DirectoryEntry entry;
            int64_t readCount;
            NN_RESULT_DO(m_BaseDirectory->Read(&readCount, &entry, 1));

            if( readCount == 0 )
            {
                break;
            }

            if( !IsReadTarget(&entry) )
            {
                continue;
            }

            if( IsConcatenationFileAttribute(entry.reserved1[0]) )
            {
                entry.directoryEntryType = DirectoryEntryType_File;

                if( (m_Mode & OpenDirectoryModePrivate_NotRequireFileSize) == 0 )
                {
                    char internalFilePath[EntryNameLengthMax + 1];
                    util::SNPrintf(internalFilePath, sizeof(internalFilePath), "%s%s", m_Path, entry.name);

                    NN_RESULT_DO(m_pParentFileSystem->GetFileSize(&entry.fileSize, internalFilePath));
                }
            }
            entry.reserved1[0] = 0;
            entryBuffer[entryCount] = entry;
            ++entryCount;
        }

        *outValue = entryCount;
        NN_RESULT_SUCCESS;
    }

    Result ConcatenationDirectory::DoGetEntryCount(int64_t *outValue) NN_NOEXCEPT
    {
        std::unique_ptr<IDirectory> directory;
        NN_RESULT_DO(m_pBaseFileSystem->OpenDirectory(&directory, m_Path, static_cast<OpenDirectoryMode>(OpenDirectoryMode_All | OpenDirectoryModePrivate_NotRequireFileSize)));

        int64_t entryCount = 0;
        while(NN_STATIC_CONDITION(true))
        {
            DirectoryEntry entry;
            int64_t readCount;
            NN_RESULT_DO(directory->Read(&readCount, &entry, 1));

            if( readCount == 0 )
            {
                *outValue = entryCount;
                NN_RESULT_SUCCESS;
            }

            if( !IsReadTarget(&entry) )
            {
                continue;
            }

            ++entryCount;
        }
    }



} // namespace

    Result ConcatenationFileSystem::GetInternalFileCount(int* outValue, const char* path) NN_NOEXCEPT
    {
        for(int index=0;; ++index)
        {
            char internalFilePath[EntryNameLengthMax + 1];
            NN_RESULT_DO(GenerateInternalFilePath(internalFilePath, sizeof(internalFilePath), path, index));

            DirectoryEntryType type;
            NN_RESULT_TRY(m_pBaseFileSystem->GetEntryType(&type, internalFilePath))
                NN_RESULT_CATCH(ResultPathNotFound)
                {
                    *outValue = index;
                    NN_RESULT_SUCCESS;
                }
            NN_RESULT_END_TRY
        }
    }

    bool ConcatenationFileSystem::IsConcatenationFile(const char* path) NN_NOEXCEPT
    {
        int attribute = 0;
        if( m_pBaseFileSystem->GetAttribute(&attribute, path).IsFailure() )
        {
            return false;
        }

        return IsConcatenationFileAttribute(attribute);
    }

    ConcatenationFileSystem::ConcatenationFileSystem(std::unique_ptr<nn::fat::FatFileSystem>&& pBaseFileSystem) NN_NOEXCEPT
        : m_pBaseFileSystem(std::move(pBaseFileSystem)),
          m_DivisionSize(0xFFFF0000)
    {
    }

    ConcatenationFileSystem::~ConcatenationFileSystem() NN_NOEXCEPT
    {
    }

    Result ResolveParentPath(char* outPath, size_t outPathSize, const char* relativePath)
    {
        const auto untruncatedSize = nn::util::SNPrintf(outPath, outPathSize, "%s", relativePath);
        if( outPathSize <= static_cast<size_t>(untruncatedSize) )
        {
            return nn::fs::ResultTooLongPath();
        }
        char* pTail = outPath + strnlen(outPath, outPathSize) - 1;
        while(*pTail != '/' && pTail > outPath)
        {
            --pTail;
        }
        *(pTail + 1) = '\0';
        NN_RESULT_SUCCESS;
    }

    Result ConcatenationFileSystem::DoGetEntryType(nn::fs::DirectoryEntryType* outValue, const char* path) NN_NOEXCEPT
    {
        if(IsConcatenationFile(path))
        {
            *outValue = DirectoryEntryType_File;
            NN_RESULT_SUCCESS;
        }
        else
        {
            return m_pBaseFileSystem->GetEntryType(outValue, path);
        }
    }

    Result ConcatenationFileSystem::DoGetFreeSpaceSize(int64_t* outValue, const char* path) NN_NOEXCEPT
    {
        return m_pBaseFileSystem->GetFreeSpaceSize(outValue, path);
    }

    Result ConcatenationFileSystem::DoGetTotalSpaceSize(int64_t* outValue, const char* path) NN_NOEXCEPT
    {
        return m_pBaseFileSystem->GetTotalSpaceSize(outValue, path);
    }

    Result ConcatenationFileSystem::DoGetFileTimeStampRaw(nn::fs::FileTimeStampRaw* outTimeStamp, const char* path) NN_NOEXCEPT
    {
        return m_pBaseFileSystem->GetFileTimeStampRaw(outTimeStamp, path);
    }

    Result ConcatenationFileSystem::DoFlush() NN_NOEXCEPT
    {
        return m_pBaseFileSystem->Flush();
    }

    Result ConcatenationFileSystem::DoOpenFile(std::unique_ptr<nn::fs::fsa::IFile>* outValue, const char* path, nn::fs::OpenMode mode) NN_NOEXCEPT
    {
        if(IsConcatenationFile(path))
        {
            int count;
            NN_RESULT_DO(GetInternalFileCount(&count, path));

            FileArray fileArray;
            for(int index = 0; index < count; ++index)
            {
                char internalFilePath[EntryNameLengthMax + 1];
                NN_RESULT_DO(GenerateInternalFilePath(internalFilePath, sizeof(internalFilePath), path, index));

                std::unique_ptr<IFile> file;
                NN_RESULT_DO(m_pBaseFileSystem->OpenFile(&file, internalFilePath, mode));
                fileArray.push_back(std::move(file));
            }

            const auto bufferSize = ConcatenationFile::GetPathBufferSize();
            NN_RESULT_THROW_UNLESS(
                strnlen(path, bufferSize) < bufferSize,
                nn::fs::ResultTooLongPath());
            std::unique_ptr<IFile> file(new ConcatenationFile(mode, std::move(fileArray), m_DivisionSize, m_pBaseFileSystem.get(), path));

            if( file == nullptr )
            {
                return ResultAllocationMemoryFailedInConcatenationFileSystemA();
            }

            *outValue = std::move(file);
            NN_RESULT_SUCCESS;
        }
        else
        {
            // ベースの IFile をそのまま返す
            return m_pBaseFileSystem->OpenFile(outValue, path, mode);
        }
    }

    Result ConcatenationFileSystem::DoOpenDirectory(std::unique_ptr<nn::fs::fsa::IDirectory>* outValue, const char* path, nn::fs::OpenDirectoryMode mode) NN_NOEXCEPT
    {

        if(IsConcatenationFile(path))
        {
            return ResultPathNotFound();
        }
        else
        {
            std::unique_ptr<IDirectory> baseDirectory;
            NN_RESULT_DO(m_pBaseFileSystem->OpenDirectory(&baseDirectory, path, OpenDirectoryMode_All)); // All で開いて自前でフィルタかける

            const auto bufferSize = ConcatenationDirectory::GetPathBufferSize();
            NN_RESULT_THROW_UNLESS(
                strnlen(path, bufferSize) < bufferSize,
                nn::fs::ResultTooLongPath());
            std::unique_ptr<IDirectory> directory(new ConcatenationDirectory(mode, std::move(baseDirectory), this, m_pBaseFileSystem.get(), path));

            if( directory == nullptr )
            {
                return ResultAllocationMemoryFailedInConcatenationFileSystemB();
            }

            *outValue = std::move(directory);

            NN_RESULT_SUCCESS;
        }
    }

    Result ConcatenationFileSystem::DoCreateFile(const char *path, int64_t size, int flag) NN_NOEXCEPT
    {
        if( size < 0 )
        {
            return ResultOutOfRange();
        }

        int flagForBaseFileSystem = flag & ~CreateFileOptionFlag_BigFile;

        if( flag & nn::fs::CreateFileOptionFlag_BigFile )
        {
            char parentPath[EntryNameLengthMax + 1];
            NN_RESULT_DO(ResolveParentPath(parentPath, sizeof(parentPath), path));
            if(IsConcatenationFile(parentPath))
            {
                // 親ディレクトリとして指定されたパスが連結ファイルである
                return ResultPathNotFound();
            }

            NN_RESULT_DO(m_pBaseFileSystem->CreateDirectory(path, nn::fat::Attribute::Attribute_Archive));

            if( size == 0 )
            {
                char internalFilePath[EntryNameLengthMax + 1];

                // size が 0 の場合はサイズ 0 のファイルを 1 つ作成
                NN_RESULT_DO(GenerateInternalFilePath(internalFilePath, sizeof(internalFilePath), path, 0));
                return m_pBaseFileSystem->CreateFile(internalFilePath, 0, flagForBaseFileSystem);
            }


            int64_t restSize = size;
            for(int index = 0; restSize > 0; ++index)
            {
                char internalFilePath[EntryNameLengthMax + 1];
                NN_RESULT_DO(GenerateInternalFilePath(internalFilePath, sizeof(internalFilePath), path, index));

                int64_t fileSize = std::min(restSize, m_DivisionSize);

                NN_RESULT_TRY(m_pBaseFileSystem->CreateFile(internalFilePath, fileSize, flagForBaseFileSystem))
                    NN_RESULT_CATCH_ALL
                    {
                        // ロールバックを試みる
                        for(int deleteIndex = index - 1; deleteIndex >= 0; --deleteIndex)
                        {
                            (void)GenerateInternalFilePath(internalFilePath, sizeof(internalFilePath), path, deleteIndex);
                            if(m_pBaseFileSystem->DeleteFile(internalFilePath).IsFailure())
                            {
                                break;
                            }
                        }
                        (void)m_pBaseFileSystem->DeleteDirectoryRecursively(path);

                        NN_RESULT_RETHROW;
                    }
                NN_RESULT_END_TRY

                restSize -= fileSize;
            }

            NN_RESULT_SUCCESS;
        }
        else
        {
            return m_pBaseFileSystem->CreateFile(path, size, flagForBaseFileSystem);
        }
    }

    Result ConcatenationFileSystem::DoDeleteFile(const char *path) NN_NOEXCEPT
    {
        if(IsConcatenationFile(path))
        {
            int count;
            NN_RESULT_DO(GetInternalFileCount(&count, path));
            for(int index = count - 1; index >= 0; --index)
            {
                char internalFilePath[EntryNameLengthMax + 1];
                NN_RESULT_DO(GenerateInternalFilePath(internalFilePath, sizeof(internalFilePath), path, index));

                NN_RESULT_DO(m_pBaseFileSystem->DeleteFile(internalFilePath));
            }

            NN_RESULT_DO(m_pBaseFileSystem->DeleteDirectoryRecursively(path));
            NN_RESULT_SUCCESS;
        }
        else
        {
            return m_pBaseFileSystem->DeleteFile(path);
        }
    }

    Result ConcatenationFileSystem::DoCreateDirectory(const char *path) NN_NOEXCEPT
    {
        char parentPath[EntryNameLengthMax + 1];
        NN_RESULT_DO(ResolveParentPath(parentPath, sizeof(parentPath), path));
        if(IsConcatenationFile(parentPath))
        {
            // 親ディレクトリとして指定されたパスが連結ファイルである
            return ResultPathNotFound();
        }

        return m_pBaseFileSystem->IFileSystem::CreateDirectory(path);
    }

    Result ConcatenationFileSystem::DoDeleteDirectory(const char *path) NN_NOEXCEPT
    {
        if(IsConcatenationFile(path))
        {
            return fs::ResultPathNotFound();
        }
        else
        {
            return m_pBaseFileSystem->DeleteDirectory(path);
        }

    }

    Result ConcatenationFileSystem::CleanDirectoryRecursivelyImpl(const char* path) NN_NOEXCEPT
    {
        return CleanupDirectoryRecursive(this, path,
            [&](const char* path, const DirectoryEntry& entry) -> Result
            {
                NN_UNUSED(path);
                NN_UNUSED(entry);
                NN_RESULT_SUCCESS;
            },
            [&](char* path, const DirectoryEntry& entry) -> Result
            {
                NN_UNUSED(entry);
                NN_RESULT_DO(DeleteDirectory(path));
                NN_RESULT_SUCCESS;
            },
            [&](const char* path, const DirectoryEntry& entry) -> Result
            {
                NN_UNUSED(entry);
                return DeleteFile(path);
            }
        );
    }

    Result ConcatenationFileSystem::DoDeleteDirectoryRecursively(const char *path) NN_NOEXCEPT
    {
        if(IsConcatenationFile(path))
        {
            return fs::ResultPathNotFound();
        }
        else
        {
            // ルートディレクトリは削除出来ない
            if(path[0] == '/' && path[1] == '\0')
            {
                NN_RESULT_THROW(ResultDirectoryUndeletable());
            }

            NN_RESULT_DO(CleanDirectoryRecursivelyImpl(path));

            return m_pBaseFileSystem->DeleteDirectory(path);
        }
    }

    Result ConcatenationFileSystem::DoCleanDirectoryRecursively(const char* path) NN_NOEXCEPT
    {
        if(IsConcatenationFile(path))
        {
            return fs::ResultPathNotFound();
        }
        else
        {
            NN_RESULT_THROW(CleanDirectoryRecursivelyImpl(path));
        }
    }

    Result ConcatenationFileSystem::DoRenameFile(const char *currentPath, const char *newPath) NN_NOEXCEPT
    {
        if(IsConcatenationFile(currentPath))
        {
            return m_pBaseFileSystem->RenameDirectory(currentPath, newPath);
        }
        else
        {
            return m_pBaseFileSystem->RenameFile(currentPath, newPath);
        }
    }

    Result ConcatenationFileSystem::DoRenameDirectory(const char *currentPath, const char *newPath) NN_NOEXCEPT
    {
        if(IsConcatenationFile(currentPath))
        {
            return fs::ResultPathNotFound();
        }
        else
        {
            return m_pBaseFileSystem->RenameDirectory(currentPath, newPath);
        }
    }

    Result ConcatenationFileSystem::GetFileSize(int64_t* outValue, const char *path) NN_NOEXCEPT
    {
        int64_t totalSize = 0;
        for(int index=0;; ++index)
        {
            char internalFilePath[EntryNameLengthMax + 1];
            NN_RESULT_DO(GenerateInternalFilePath(internalFilePath, sizeof(internalFilePath), path, index));

            int64_t size;
            NN_RESULT_TRY(m_pBaseFileSystem->GetFileSize(&size, internalFilePath))
                NN_RESULT_CATCH(ResultPathNotFound)
                {
                    *outValue = totalSize;
                    NN_RESULT_SUCCESS;
                }
            NN_RESULT_END_TRY

            totalSize += size;
        }
    }

    Result ConcatenationFileSystem::DoQueryEntry(char* outBuffer, size_t outBufferSize, const char* inBuffer, size_t inBufferSize, QueryId queryId, const char* path) NN_NOEXCEPT
    {
        NN_UNUSED(outBuffer);
        NN_UNUSED(outBufferSize);
        NN_UNUSED(inBuffer);
        NN_UNUSED(inBufferSize);

        switch (queryId)
        {
            case QueryId::SetConcatenationFileAttribute:
            {
                NN_RESULT_DO(m_pBaseFileSystem->SetAttribute(path, nn::fat::Attribute::Attribute_Archive));
                NN_RESULT_SUCCESS;
            }
            default:
            {
                return ResultNotImplemented();
            }
        }
    }

    Result ConcatenationFileSystem::DoCommit() NN_NOEXCEPT
    {
        return m_pBaseFileSystem->Commit();
    }

}}



