﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/fssystem/fs_BucketTreeUtility.h>
#include <nn/fssystem/fs_BucketTreeBuilder.h>

namespace nn { namespace fssystem {

namespace {

/**
 * @brief   エントリのイテレータです。
 */
class EntryIterator
{
public:
    typedef std::random_access_iterator_tag iterator_category;
    typedef ptrdiff_t difference_type;
    typedef int64_t value_type;
    typedef value_type* pointer;
    typedef value_type& reference;

public:
    EntryIterator() NN_NOEXCEPT
        : m_Ptr(nullptr)
        , m_Stride(1)
    {
    }

    EntryIterator(void* ptr, size_t stride) NN_NOEXCEPT
        : m_Ptr(reinterpret_cast<char*>(ptr))
        , m_Stride(static_cast<int>(stride))
    {
    }

    EntryIterator& operator++() NN_NOEXCEPT
    {
        m_Ptr += m_Stride;
        return *this;
    }

    EntryIterator operator++(int) NN_NOEXCEPT
    {
        EntryIterator iter(*this);
        m_Ptr += m_Stride;
        return iter;
    }

    EntryIterator& operator--() NN_NOEXCEPT
    {
        m_Ptr -= m_Stride;
        return *this;
    }

    EntryIterator operator--(int) NN_NOEXCEPT
    {
        EntryIterator iter(*this);
        m_Ptr -= m_Stride;
        return iter;
    }

    difference_type operator-(const EntryIterator& iter) const NN_NOEXCEPT
    {
        return (m_Ptr - iter.m_Ptr) / m_Stride;
    }

    EntryIterator operator+(difference_type offset) const NN_NOEXCEPT
    {
        EntryIterator iter(m_Ptr, m_Stride);
        iter += offset;
        return iter;
    }

    EntryIterator operator-(difference_type offset) const NN_NOEXCEPT
    {
        EntryIterator iter(m_Ptr, m_Stride);
        iter -= offset;
        return iter;
    }

    EntryIterator& operator+=(difference_type offset) NN_NOEXCEPT
    {
        m_Ptr += offset * m_Stride;
        return *this;
    }

    EntryIterator& operator-=(difference_type offset) NN_NOEXCEPT
    {
        m_Ptr -= offset * m_Stride;
        return *this;
    }

    bool operator==(const EntryIterator& iter) const NN_NOEXCEPT
    {
        return m_Ptr == iter.m_Ptr;
    }

    bool operator!=(const EntryIterator& iter) const NN_NOEXCEPT
    {
        return m_Ptr != iter.m_Ptr;
    }

    bool operator<(const EntryIterator& iter) const NN_NOEXCEPT
    {
        return m_Ptr < iter.m_Ptr;
    }

    bool operator<=(const EntryIterator& iter) const NN_NOEXCEPT
    {
        return m_Ptr <= iter.m_Ptr;
    }

    bool operator>(const EntryIterator& iter) const NN_NOEXCEPT
    {
        return m_Ptr > iter.m_Ptr;
    }

    bool operator>=(const EntryIterator& iter) const NN_NOEXCEPT
    {
        return m_Ptr >= iter.m_Ptr;
    }

    void* Get() const NN_NOEXCEPT
    {
        return m_Ptr;
    }

private:
    char* m_Ptr;
    int m_Stride;
};

typedef detail::BucketTreeNode<int64_t*> Node;
typedef detail::BucketTreeNode<EntryIterator> EntrySet;
NN_STATIC_ASSERT(std::is_pod<Node>::value);
NN_STATIC_ASSERT(std::is_pod<EntrySet>::value);

NN_STATIC_ASSERT(sizeof(Node) == sizeof(BucketTree::NodeHeader));
NN_STATIC_ASSERT(sizeof(EntrySet) == sizeof(BucketTree::NodeHeader));
NN_STATIC_ASSERT(std::is_pod<Node>::value);
NN_STATIC_ASSERT(std::is_pod<EntrySet>::value);

}

/**
 * @brief   コンストラクタです。
 */
BucketTreeBuilder::BucketTreeBuilder() NN_NOEXCEPT
    : m_NodeL1()
    , m_NodeL2()
    , m_EntrySet()
    , m_NodeStorage()
    , m_EntryStorage()
    , m_NodeSize(0)
    , m_EntrySize(0)
    , m_TotalEntryCount(0)
    , m_EntryCount(0)
    , m_OffsetCount(0)
    , m_NodeL2Index(0)
    , m_EntryIndex(0)
    , m_EntryOffset(-1)
{
}

/**
 * @brief   初期化します。
 *
 * @param[in]   pAllocator      アロケータのポインタ
 * @param[in]   headerStorage   ヘッダストレージ
 * @param[in]   nodeStorage     ノードストレージ
 * @param[in]   entryStorage    エントリストレージ
 * @param[in]   nodeSize        １ノードのサイズ
 * @param[in]   entrySize       １エントリのサイズ
 * @param[in]   entryCount      エントリの総数
 *
 * @retval  ResultSuccess                   正常に処理が終了しました。
 * @retval  ResultBufferAllocationFailed    メモリの確保に失敗しました。
 * @retval  上記以外                        ストレージの書き込みに失敗しました。
 *
 * @pre
 *      - pAllocator != nullptr
 *      - pOutput != nullptr
 *      - sizeof(int64_t) <= entrySize
 *      - entrySize + sizeof(BucketTree::NodeHeader) <= nodeSize
 *      - BucketTree::NodeSizeMin <= nodeSize && nodeSize <= BucketTree::NodeSizeMax
 *      - nodeSize が２の累乗
 *      - 0 <= entryCount
 *      - 未初期化
 */
Result BucketTreeBuilder::Initialize(
                              IAllocator* pAllocator,
                              fs::SubStorage headerStorage,
                              fs::SubStorage nodeStorage,
                              fs::SubStorage entryStorage,
                              size_t nodeSize,
                              size_t entrySize,
                              int entryCount
                          ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pAllocator);
    NN_SDK_REQUIRES_LESS_EQUAL(sizeof(int64_t), entrySize);
    NN_SDK_REQUIRES_LESS_EQUAL(entrySize + sizeof(NodeHeader), nodeSize);
    NN_SDK_REQUIRES_MINMAX(nodeSize, BucketTree::NodeSizeMin, BucketTree::NodeSizeMax);
    NN_SDK_REQUIRES(util::ispow2(nodeSize));
    NN_SDK_REQUIRES_LESS_EQUAL(0, entryCount);
    NN_SDK_REQUIRES(!IsInitialized());

    if( entryCount <= 0 )
    {
        m_NodeSize = nodeSize;
        m_EntrySize = entrySize;

        Header header;
        header.Format(0);
        NN_RESULT_DO(headerStorage.Write(0, &header, static_cast<size_t>(QueryHeaderStorageSize())));

        NN_RESULT_SUCCESS;
    }

    Result result = fs::ResultBufferAllocationFailed();

    if( m_EntrySet.Allocate(pAllocator, nodeSize) )
    {
        const auto offsetCount = BucketTree::GetOffsetCount(nodeSize);
        const auto entrySetCount = BucketTree::GetEntrySetCount(nodeSize, entrySize, entryCount);

        // L2 は場合によっては確保しない
        if( entrySetCount <= offsetCount || m_NodeL2.Allocate(pAllocator, nodeSize) )
        {
            // NOTE: メモリ不足を早めに検出するため、L1 の確保を最後にする
            if( m_NodeL1.Allocate(pAllocator, nodeSize) )
            {
                Header header;
                header.Format(entryCount);

                // ヘッダの書き出し
                result = headerStorage.Write(
                            0, &header, static_cast<size_t>(QueryHeaderStorageSize()));
                if( result.IsSuccess() )
                {
                    m_NodeStorage = nodeStorage;
                    m_EntryStorage = entryStorage;
                    m_NodeSize = nodeSize;
                    m_EntrySize = entrySize;
                    m_TotalEntryCount = entryCount;
                    m_EntryCount = BucketTree::GetEntryCount(nodeSize, entrySize);
                    m_OffsetCount = offsetCount;
                    m_NodeL2Index = BucketTree::GetNodeL2Count(nodeSize, entrySize, entryCount);

                    m_NodeL1.FillZero(nodeSize);
                    m_NodeL2.FillZero(nodeSize);
                    m_EntrySet.FillZero(nodeSize);

                    NN_RESULT_SUCCESS;
                }
                m_NodeL1.Free(nodeSize);
            }
            m_NodeL2.Free(nodeSize);
        }
        m_EntrySet.Free(nodeSize);
    }

    // 成功時はここに来ない
    NN_RESULT_THROW(result);
}

/**
 * @brief   終了処理をします。
 */
void BucketTreeBuilder::Finalize() NN_NOEXCEPT
{
    if( IsInitialized() )
    {
        m_NodeL1.Free(m_NodeSize);
        m_NodeL2.Free(m_NodeSize);
        m_EntrySet.Free(m_NodeSize);
        m_NodeStorage = fs::SubStorage();
        m_EntryStorage = fs::SubStorage();
        m_NodeSize = 0;
        m_EntrySize = 0;
        m_TotalEntryCount = 0;
        m_EntryCount = 0;
        m_OffsetCount = 0;
        m_NodeL2Index = 0;
        m_EntryIndex = 0;
        m_EntryOffset = -1;
    }
}

/**
 * @brief   エントリを書き出します。
 *
 * @param[in]   pEntry  書き出すエントリの先頭アドレス
 *
 * @return  関数の処理結果を返します。
 *
 * @pre
 *      - pEntry != nullptr
 *      - 初期化済み
 */
Result BucketTreeBuilder::WriteImpl(const void* pEntry) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pEntry);
    NN_SDK_REQUIRES(IsInitialized());

    if( m_TotalEntryCount <= m_EntryIndex )
    {
        // 書き出すエントリが多過ぎる
        return fs::ResultOutOfRange();
    }

    const auto entryOffset = detail::SafeValue::GetInt64(pEntry);
    if( entryOffset <= m_EntryOffset )
    {
        // 昇順に並んでいない
        return fs::ResultInvalidOffset();
    }

    // キャッシュ書き出し
    NN_RESULT_DO(FlushCache(entryOffset));

    // オフセット書き出し
    UpdateOffset(entryOffset);

    // エントリをキャッシュへコピー
    {
        auto iter = m_EntrySet.Get<EntrySet>()->GetBegin(m_EntrySize);
        iter += (m_EntryIndex % m_EntryCount);

        std::memcpy(iter.Get(), pEntry, m_EntrySize);

        m_EntryOffset = entryOffset;
    }
    ++m_EntryIndex;

    NN_RESULT_SUCCESS;
}

/**
 * @brief   オフセットを書き出します。
 */
void BucketTreeBuilder::UpdateOffset(int64_t entryOffset) NN_NOEXCEPT
{
    // ノードにオフセットを設定
    if( (m_EntryIndex % m_EntryCount) == 0 )
    {
        auto* const pNodeL1 = m_NodeL1.Get<Node>();
        auto nodeL1Iter = pNodeL1->GetBegin();

        // L1 のみ
        if( m_NodeL2Index == 0 )
        {
            nodeL1Iter[m_EntryIndex / m_EntryCount] = entryOffset;
        }
        // L1,L2 両方
        else
        {
            // L1 ノード内の L2 部分に設定
            if( m_NodeL2Index < m_OffsetCount )
            {
                nodeL1Iter[m_NodeL2Index] = entryOffset;
            }
            // L2 ノードに設定
            else
            {
                auto* const pNodeL2 = m_NodeL2.Get<Node>();
                auto nodeL2Iter = pNodeL2->GetBegin();

                nodeL2Iter[m_NodeL2Index % m_OffsetCount] = entryOffset;

                // L1 ノードも設定
                if( (m_NodeL2Index % m_OffsetCount) == 0 )
                {
                    nodeL1Iter[m_NodeL2Index / m_OffsetCount - 1] = entryOffset;
                }
            }

            ++m_NodeL2Index;
        }
    }
}

/**
 * @brief   いっぱいになったキャッシュを書き出します。
 *
 * @return  関数の処理結果を返します。
 */
Result BucketTreeBuilder::FlushCache(int64_t entryOffset) NN_NOEXCEPT
{
    // エントリセットのキャッシュ書き出し
    if( 0 < m_EntryIndex && (m_EntryIndex % m_EntryCount) == 0 )
    {
        auto* const pEntrySet = m_EntrySet.Get<EntrySet>();
        const auto entrySetIndex = (m_EntryIndex / m_EntryCount) - 1;

        pEntrySet->header.index = entrySetIndex;
        pEntrySet->header.count = m_EntryCount;
        pEntrySet->header.offset = entryOffset;

        // ストレージへ書き出し
        NN_RESULT_DO(m_EntryStorage.Write(
            entrySetIndex * static_cast<int64_t>(m_NodeSize), pEntrySet, m_NodeSize));

        m_EntrySet.FillZero(m_NodeSize);

        // L2 ノードのキャッシュ書き出し
        if( m_OffsetCount < m_NodeL2Index && (m_NodeL2Index % m_OffsetCount) == 0 )
        {
            auto* const pNode = m_NodeL2.Get<Node>();
            const auto nodeIndex = (m_NodeL2Index / m_OffsetCount) - 2;

            pNode->header.index = nodeIndex;
            pNode->header.count = m_OffsetCount;
            pNode->header.offset = entryOffset;

            // ストレージへ書き出し
            NN_RESULT_DO(m_NodeStorage.Write(
                (1 + nodeIndex) * static_cast<int64_t>(m_NodeSize), pNode, m_NodeSize));

            m_NodeL2.FillZero(m_NodeSize);
        }
    }

    NN_RESULT_SUCCESS;
}

/**
 * @brief   エントリの書き出しを確定します。
 *
 * @param[in]   endOffset   エントリの最大オフセット
 *
 * @return  関数の処理結果を返します。
 *
 * @pre
 *      - 0 <= endOffset
 *      - 初期化済み
 */
Result BucketTreeBuilder::Commit(int64_t endOffset) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_LESS_EQUAL(0, endOffset);
    NN_SDK_REQUIRES(IsInitialized());

    if( endOffset <= m_EntryOffset )
    {
        // 昇順に並んでいない
        return fs::ResultInvalidOffset();
    }
    if( m_TotalEntryCount != m_EntryIndex )
    {
        // 書き出すエントリが少なすぎる
        return fs::ResultOutOfRange();
    }

    // NOTE: きりのいいインデックスならこの処理でキャッシュを書き出せる
    NN_RESULT_DO(FlushCache(endOffset));

    // 残ったエントリセットのキャッシュ書き出し
    if( (m_EntryIndex % m_EntryCount) != 0 )
    {
        auto* const pEntrySet = m_EntrySet.Get<EntrySet>();
        const auto entrySetIndex = m_EntryIndex / m_EntryCount;

        pEntrySet->header.index = entrySetIndex;
        pEntrySet->header.count = m_EntryIndex % m_EntryCount;
        pEntrySet->header.offset = endOffset;

        // ストレージへ書き出し
        NN_RESULT_DO(m_EntryStorage.Write(
            entrySetIndex * static_cast<int64_t>(m_NodeSize), pEntrySet, m_NodeSize));
    }

    // 残った L2 ノードのキャッシュ書き出し
    if( m_OffsetCount < m_NodeL2Index &&
        ((m_EntryIndex % m_EntryCount) != 0 || (m_NodeL2Index % m_OffsetCount) != 0) )
    {
        auto* const pNode = m_NodeL2.Get<Node>();
        const auto nodeIndex = (m_NodeL2Index + m_OffsetCount - 1) / m_OffsetCount - 2;
        const auto nodeCount = m_NodeL2Index % m_OffsetCount;

        pNode->header.index = nodeIndex;
        pNode->header.count = nodeCount == 0 ? m_OffsetCount : nodeCount;
        pNode->header.offset = endOffset;

        // ストレージへ書き出し
        NN_RESULT_DO(m_NodeStorage.Write(
            (1 + nodeIndex) * static_cast<int64_t>(m_NodeSize), pNode, m_NodeSize));
    }

    // L1 ノードのキャッシュ書き出し
    {
        int nodeCount = (m_NodeL2Index == 0)
                            ? util::DivideUp(m_EntryIndex, m_EntryCount)        // L1 のみ
                            : util::DivideUp(m_NodeL2Index, m_OffsetCount) - 1; // L2 あり

        auto* const pNode = m_NodeL1.Get<Node>();

        pNode->header.index = 0;
        pNode->header.count = nodeCount;
        pNode->header.offset = endOffset;

        // ストレージへ書き出し
        NN_RESULT_DO(m_NodeStorage.Write(0, pNode, m_NodeSize));
    }

    m_EntryOffset = std::numeric_limits<int64_t>::max(); // コミット済みをマーク

    NN_RESULT_SUCCESS;
}

}}
