﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>
#include <nn/fssystem/fs_AsynchronousAccess.h>
#include <nn/fssystem/fs_BucketTreeUtility.h>

namespace nn { namespace fssystem {

namespace {

typedef detail::BucketTreeNode<const int64_t*> Node;

NN_STATIC_ASSERT(sizeof(Node) == sizeof(BucketTree::NodeHeader));
NN_STATIC_ASSERT(std::is_pod<Node>::value);

const int NodeHeaderSize = sizeof(BucketTree::NodeHeader);

/**
 * @brief   ストレージ上のノードを表すクラスです。
 */
class StorageNode
{
public:
    //! コンストラクタです。
    StorageNode(size_t dataSize, int dataCount) NN_NOEXCEPT
        : m_Begin(NodeHeaderSize, static_cast<int>(dataSize))
        , m_Count(dataCount)
        , m_Index(-1)
    {
    }

    //! コンストラクタです。
    StorageNode(int64_t nodeOffset, size_t dataSize, int dataCount) NN_NOEXCEPT
        : m_Begin(nodeOffset + NodeHeaderSize, static_cast<int>(dataSize))
        , m_Count(dataCount)
        , m_Index(-1)
    {
    }

    //! 指定したアドレスを含むデータのインデックスを検索します。
    void Find(const char* buffer, int64_t virtualAddress) NN_NOEXCEPT
    {
        int count = m_Count;
        auto pos = m_Begin;

        while( 0 < count )
        {
            auto half = count / 2;
            auto center = pos + half;

            int64_t offset = 0;
            std::memcpy(&offset, buffer + center.Get(), sizeof(int64_t));

            if( offset <= virtualAddress )
            {
                pos = center + 1;
                count -= half + 1;
            }
            else
            {
                count = half;
            }
        }

        // std::upper_bound() 相当の処理なので結果を１つ下へ丸め込む
        m_Index = static_cast<int>(pos - m_Begin) - 1;
    }

    //! 指定したアドレスを含むデータのインデックスを検索します。
    Result Find(fs::SubStorage* pStorage, int64_t virtualAddress) NN_NOEXCEPT
    {
        int count = m_Count;
        auto pos = m_Begin;

        while( 0 < count )
        {
            auto half = count / 2;
            auto center = pos + half;

            int64_t offset = 0;
            NN_RESULT_DO(pStorage->Read(center.Get(), &offset, sizeof(int64_t)));

            if( offset <= virtualAddress )
            {
                pos = center + 1;
                count -= half + 1;
            }
            else
            {
                count = half;
            }
        }

        // std::upper_bound() 相当の処理なので結果を１つ下へ丸め込む
        m_Index = static_cast<int>(pos - m_Begin) - 1;

        NN_RESULT_SUCCESS;
    }

    //! 見つかったデータのインデックスを取得します。
    int GetIndex() const NN_NOEXCEPT
    {
        return m_Index;
    }

private:
    //! ストレージ上のオフセットを表すクラスです。
    class Offset
    {
    public:
        typedef int64_t difference_type;

    public:
        Offset(int64_t offset, int stride) NN_NOEXCEPT
            : m_Offset(offset)
            , m_Stride(stride)
        {
        }

        Offset& operator++() NN_NOEXCEPT
        {
            m_Offset += m_Stride;
            return *this;
        }

        Offset operator++(int) NN_NOEXCEPT
        {
            Offset obj(*this);
            m_Offset += m_Stride;
            return obj;
        }

        Offset& operator--() NN_NOEXCEPT
        {
            m_Offset -= m_Stride;
            return *this;
        }

        Offset operator--(int) NN_NOEXCEPT
        {
            Offset obj(*this);
            m_Offset -= m_Stride;
            return obj;
        }

        difference_type operator-(const Offset& obj) const NN_NOEXCEPT
        {
            return (m_Offset - obj.m_Offset) / m_Stride;
        }

        Offset operator+(difference_type offset) const NN_NOEXCEPT
        {
            return Offset(m_Offset + offset * m_Stride, m_Stride);
        }

        Offset operator-(difference_type offset) const NN_NOEXCEPT
        {
            return Offset(m_Offset - offset * m_Stride, m_Stride);
        }

        Offset& operator+=(difference_type offset) NN_NOEXCEPT
        {
            m_Offset += offset * m_Stride;
            return *this;
        }

        Offset& operator-=(difference_type offset) NN_NOEXCEPT
        {
            m_Offset -= offset * m_Stride;
            return *this;
        }

        bool operator==(const Offset& obj) const NN_NOEXCEPT
        {
            return m_Offset == obj.m_Offset;
        }

        bool operator!=(const Offset& obj) const NN_NOEXCEPT
        {
            return m_Offset != obj.m_Offset;
        }

        int64_t Get() const NN_NOEXCEPT
        {
            return m_Offset;
        }

    private:
        int64_t m_Offset;
        int m_Stride;
    };

private:
    const Offset m_Begin;
    const int m_Count;
    int m_Index;
};

}

// NOTE: バージョン変更ごとに include している cpp の再コンパイルが走らないようにここで定義
const uint32_t BucketTree::Version = 0x00000001;

NN_DEFINE_STATIC_CONSTANT(const uint32_t BucketTree::Signature);
NN_DEFINE_STATIC_CONSTANT(const size_t BucketTree::NodeSizeMin);
NN_DEFINE_STATIC_CONSTANT(const size_t BucketTree::NodeSizeMax);


/**
 * @brief   フォーマットします。
 *
 * @param[in]   argEntryCount   エントリの総数
 *
 * @pre
 *      - 0 <= argEntryCount
 *
 * @note
 *      引数名が関数定義時と異なるが VS2015 での警告対策のためこのままにする
 */
void BucketTree::Header::Format(int argEntryCount) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_LESS_EQUAL(0, argEntryCount);

    this->signature = Signature;
    this->version = Version;
    this->entryCount = argEntryCount;
    this->_reserved = 0;
}

/**
 * @brief   エラーチェックをします。
 *
 * @retval  ResultSuccess   データは正常です。
 * @retval  上記以外        エラーを検出しました。
 */
Result BucketTree::Header::Verify() const NN_NOEXCEPT
{
    if( this->signature != Signature )
    {
        return fs::ResultInvalidBucketTreeSignature();
    }

    if( this->entryCount < 0 )
    {
        return fs::ResultInvalidBucketTreeEntryCount();
    }

    if( Version < this->version )
    {
        return fs::ResultUnsupportedVersion();
    }

    NN_RESULT_SUCCESS;
}

/**
 * @brief   エラーチェックをします。
 *
 * @param[in]   nodeIndex   ノードのインデックス。
 * @param[in]   nodeSize    １ノードのサイズ
 * @param[in]   entrySize   １エントリのサイズ
 *
 * @retval  ResultSuccess   データは正常です。
 * @retval  上記以外        エラーを検出しました。
 */
Result BucketTree::NodeHeader::Verify(
                                   int nodeIndex,
                                   size_t nodeSize,
                                   size_t entrySize
                               ) const NN_NOEXCEPT
{
    if( this->index != nodeIndex )
    {
        return fs::ResultInvalidArgument();
    }

    if( entrySize == 0 || nodeSize < entrySize + NodeHeaderSize )
    {
        return fs::ResultInvalidSize();
    }

    const size_t entryCountMax = (nodeSize - NodeHeaderSize) / entrySize;
    if( this->count <= 0 || entryCountMax < static_cast<size_t>(this->count) )
    {
        return fs::ResultInvalidBucketTreeNodeEntryCount();
    }
    else if( this->offset < 0 )
    {
        return fs::ResultInvalidBucketTreeNodeOffset();
    }

    NN_RESULT_SUCCESS;
}

/**
 * @brief   エントリセットの数を取得します。
 *
 * @param[in]   nodeSize    １ノードのサイズ
 * @param[in]   entrySize   １エントリのサイズ
 * @param[in]   entryCount  エントリの総数
 *
 * @return  エントリセットの数を返します。
 */
int BucketTree::GetEntrySetCount(
                    size_t nodeSize,
                    size_t entrySize,
                    int entryCount
                ) NN_NOEXCEPT
{
    // NOTE: 引数は事前検証済みのものを使うためここでは検証しない
    const auto entryCountPerNode = GetEntryCount(nodeSize, entrySize);
    return nn::util::DivideUp(entryCount, entryCountPerNode);
}

/**
 * @brief   L2 のノード数を取得します。
 *
 * @param[in]   nodeSize    １ノードのサイズ
 * @param[in]   entrySize   １エントリのサイズ
 * @param[in]   entryCount  エントリの総数
 *
 * @return  L2 のノード数を返します。
 */
int BucketTree::GetNodeL2Count(
                    size_t nodeSize,
                    size_t entrySize,
                    int entryCount
                ) NN_NOEXCEPT
{
    // NOTE: 引数は事前検証済みのものを使うためここでは検証しない
    const auto offsetCountPerNode = GetOffsetCount(nodeSize);
    const auto entrySetCount = GetEntrySetCount(nodeSize, entrySize, entryCount);

    // L1 のみで対応可能
    if( entrySetCount <= offsetCountPerNode )
    {
        return 0;
    }

    /**
     *  L1 ノードに空きがある場合、L2 ノードの一部を L1 に移動させて、省メモリ化を図ります。
     *
     *       ＜最適化前＞
     *  L1 ■■□□□□□□
     *     ｜└───────┐
     *  L2 ◆◆◆◆◆◆◆◆／◆◆◆◆◆◇◇◇
     *
     *       ＜最適化（前詰め）後＞
     *  L1 ■◆◆◆◆◆◆◆  // TODO: ◆◆◆◆◆◆◆■
     *     ｜                //       こう配置すればアルゴリズムが簡素になった...
     *  L2 ◆◆◆◆◆◆◇◇
     */
    const auto nodeL2Count = nn::util::DivideUp(entrySetCount, offsetCountPerNode);

    // L3 が必要かチェック
    // L3 が必要な場合は未対応なので ABORT します。
    NN_ABORT_UNLESS_LESS_EQUAL(nodeL2Count, offsetCountPerNode);

    return nn::util::DivideUp( // L2 上のオフセットを L1 のノードへ移す数
               entrySetCount - (offsetCountPerNode - (nodeL2Count - 1)),
               offsetCountPerNode
           );
}

/**
 * @brief   ノード部分のストレージのサイズを取得します。
 *
 * @param[in]   nodeSize    １ノードのサイズ
 * @param[in]   entrySize   １エントリのサイズ
 * @param[in]   entryCount  エントリの総数
 *
 * @return  ノード部分のストレージサイズを返します。
 *
 * @pre
 *      - sizeof(int64_t) <= entrySize
 *      - entrySize + sizeof(NodeHeader) <= nodeSize
 *      - NodeSizeMin <= nodeSize && nodeSize <= NodeSizeMax
 *      - nodeSize が２の累乗
 *      - 0 <= entryCount
 */
int64_t BucketTree::QueryNodeStorageSize(
                        size_t nodeSize,
                        size_t entrySize,
                        int entryCount
                    ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_LESS_EQUAL(sizeof(int64_t), entrySize);
    NN_SDK_REQUIRES_LESS_EQUAL(entrySize + sizeof(NodeHeader), nodeSize);
    NN_SDK_REQUIRES_MINMAX(nodeSize, NodeSizeMin, NodeSizeMax);
    NN_SDK_REQUIRES(nn::util::ispow2(nodeSize));
    NN_SDK_REQUIRES_LESS_EQUAL(0, entryCount);

    if( entryCount <= 0 )
    {
        return 0;
    }
    return (1 + GetNodeL2Count(nodeSize, entrySize, entryCount)) * static_cast<int64_t>(nodeSize);
}

/**
 * @brief   エントリ部分のストレージのサイズを取得します。
 *
 * @param[in]   nodeSize    １ノードのサイズ
 * @param[in]   entrySize   １エントリのサイズ
 * @param[in]   entryCount  エントリの総数
 *
 * @return  エントリ部分のストレージサイズを返します。
 *
 * @pre
 *      - sizeof(int64_t) <= entrySize
 *      - entrySize + sizeof(NodeHeader) <= nodeSize
 *      - NodeSizeMin <= nodeSize && nodeSize <= NodeSizeMax
 *      - nodeSize が２の累乗
 *      - 0 <= entryCount
 */
int64_t BucketTree::QueryEntryStorageSize(
                        size_t nodeSize,
                        size_t entrySize,
                        int entryCount
                    ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_LESS_EQUAL(sizeof(int64_t), entrySize);
    NN_SDK_REQUIRES_LESS_EQUAL(entrySize + sizeof(NodeHeader), nodeSize);
    NN_SDK_REQUIRES_MINMAX(nodeSize, NodeSizeMin, NodeSizeMax);
    NN_SDK_REQUIRES(nn::util::ispow2(nodeSize));
    NN_SDK_REQUIRES_LESS_EQUAL(0, entryCount);

    if( entryCount <= 0 )
    {
        return 0;
    }
    return GetEntrySetCount(nodeSize, entrySize, entryCount) * static_cast<int64_t>(nodeSize);
}

/**
 * @brief   コンストラクタです。
 */
BucketTree::BucketTree() NN_NOEXCEPT
    : m_NodeStorage()
    , m_EntryStorage()
    , m_NodeL1()
    , m_NodeSize(0)
    , m_EntrySize(0)
    , m_EntryCount(0)
    , m_OffsetCount(0)
    , m_EntrySetCount(0)
    , m_BeginOffset(0)
    , m_EndOffset(0)
{
}

/**
 * @brief   初期化します。
 *
 * @param[in]   pAllocator      アロケータのポインタ
 * @param[in]   nodeStorage     ノードストレージ
 * @param[in]   entryStorage    エントリストレージ
 * @param[in]   nodeSize        １ノードのサイズ
 * @param[in]   entrySize       １エントリのサイズ
 * @param[in]   entryCount      エントリの総数
 *
 * @retval  ResultSuccess                   正常に処理が終了しました。
 * @retval  ResultInvalidArgument           引数の指定が不適切です
 * @retval  ResultBufferAllocationFailed    メモリの確保に失敗しました。
 * @retval  上記以外                        ストレージの読み込みに失敗しました。
 *
 * @pre
 *      - pAllocator != nullptr
 *      - sizeof(int64_t) <= entrySize
 *      - entrySize + sizeof(NodeHeader) <= nodeSize
 *      - NodeSizeMin <= nodeSize && nodeSize <= NodeSizeMax
 *      - nodeSize が２の累乗
 *      - 0 < entryCount
 *      - 未初期化
 */
Result BucketTree::Initialize(
                       IAllocator* pAllocator,
                       fs::SubStorage nodeStorage,
                       fs::SubStorage entryStorage,
                       size_t nodeSize,
                       size_t entrySize,
                       int entryCount
                   ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pAllocator);
    NN_SDK_REQUIRES_LESS_EQUAL(sizeof(int64_t), entrySize);
    NN_SDK_REQUIRES_LESS_EQUAL(entrySize + sizeof(NodeHeader), nodeSize);
    NN_SDK_REQUIRES_MINMAX(nodeSize, NodeSizeMin, NodeSizeMax);
    NN_SDK_REQUIRES(nn::util::ispow2(nodeSize));
    NN_SDK_REQUIRES(!IsInitialized());

    NN_RESULT_THROW_UNLESS(0 < entryCount, fs::ResultInvalidArgument());

    Result result = fs::ResultBufferAllocationFailed();

    if( m_NodeL1.Allocate(pAllocator, nodeSize) )
    {
        // L1 読み込み
        result = nodeStorage.Read(0, m_NodeL1.Get(), nodeSize);
        if( result.IsSuccess() )
        {
            result = m_NodeL1->Verify(0, nodeSize, sizeof(int64_t));
            if( result.IsSuccess() )
            {
                const auto offsetCount = GetOffsetCount(nodeSize);
                const auto entrySetCount = GetEntrySetCount(nodeSize, entrySize, entryCount);
                const auto* const pNode = m_NodeL1.Get<Node>();

                int64_t beginOffset;
                // L1 ノード内に L2 ノードのオフセットが存在する
                if( offsetCount < entrySetCount && pNode->GetCount() < offsetCount )
                {
                    beginOffset = *pNode->GetEnd();
                }
                else
                {
                    beginOffset = *pNode->GetBegin();
                }

                if( 0 <= beginOffset && beginOffset <= pNode->GetBeginOffset() )
                {
                    const auto endOffset = pNode->GetEndOffset();

                    if( beginOffset < endOffset )
                    {
                        m_NodeStorage = nodeStorage;
                        m_EntryStorage = entryStorage;
                        m_NodeSize = nodeSize;
                        m_EntrySize = entrySize;
                        m_EntryCount = entryCount;
                        m_OffsetCount = offsetCount;
                        m_EntrySetCount = entrySetCount;
                        m_BeginOffset = beginOffset;
                        m_EndOffset = endOffset;

                        NN_RESULT_SUCCESS;
                    }
                }
                result = fs::ResultInvalidBucketTreeEntryOffset();
            }
        }
        m_NodeL1.Free(nodeSize);
    }

    return result;
}

/**
 * @brief   初期化します。
 *
 * @param[in]   nodeSize    １ノードのサイズ
 * @param[in]   endOffset   管理する領域の終了オフセット
 *
 * @return  処理結果を返します。
 *
 * @pre
 *      - NodeSizeMin <= nodeSize && nodeSize <= NodeSizeMax
 *      - nodeSize が２の累乗
 *      - 0 <= endOffset
 *      - 未初期化
 */
void BucketTree::Initialize(size_t nodeSize, int64_t endOffset) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_MINMAX(nodeSize, NodeSizeMin, NodeSizeMax);
    NN_SDK_REQUIRES(nn::util::ispow2(nodeSize));
    NN_SDK_REQUIRES_LESS_EQUAL(0, endOffset);
    NN_SDK_REQUIRES(!IsInitialized());

    m_NodeSize = nodeSize;
    m_EndOffset = endOffset;
}

/**
 * @brief   終了処理をします。
 */
void BucketTree::Finalize() NN_NOEXCEPT
{
    if( IsInitialized() )
    {
        m_NodeStorage = fs::SubStorage();
        m_EntryStorage = fs::SubStorage();
        m_NodeL1.Free(m_NodeSize);
        m_NodeSize = 0;
        m_EntrySize = 0;
        m_EntryCount = 0;
        m_OffsetCount = 0;
        m_EntrySetCount = 0;
        m_BeginOffset = 0;
        m_EndOffset = 0;
    }
}

/**
 * @brief   仮想アドレスからエントリを検索します。
 *
 * @param[in,out]   pVisitor        巡回オブジェクトのポインタ
 * @param[in]       virtualAddress  エントリ検索用の仮想アドレス
 *
 * @retval  ResultSuccess                   正常に処理が終了しました。
 * @retval  ResultOutOfRange                エントリが見つかりませんでした。
 * @retval  ResultBufferAllocationFailed    メモリの確保に失敗しました。
 * @retval  上記以外                        ストレージの読み込みに失敗しました。
 *
 * @pre
 *      - pVisitor != nullptr
 *      - 0 <= virtualAddress
 *      - 初期化済み
 */
Result BucketTree::Find(Visitor* pVisitor, int64_t virtualAddress) const NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pVisitor);
    NN_SDK_REQUIRES(IsInitialized());

    NN_RESULT_THROW_UNLESS(0 <= virtualAddress, fs::ResultInvalidOffset());
    NN_RESULT_THROW_UNLESS(!IsEmpty(), fs::ResultOutOfRange());

    NN_RESULT_DO(pVisitor->Initialize(this));

    return pVisitor->Find(virtualAddress);
}

/**
 * @brief   キャッシュを無効化します。
 *
 * @return  処理結果を返します。
 */
Result BucketTree::InvalidateCache() NN_NOEXCEPT
{
    {
        int64_t storageSize = 0;
        NN_RESULT_DO(m_NodeStorage.GetSize(&storageSize));
        NN_RESULT_DO(m_NodeStorage.OperateRange(
            fs::OperationId::Invalidate,
            0,
            storageSize));
    }

    {
        NN_RESULT_DO(m_NodeStorage.Read(0, m_NodeL1.Get(), m_NodeSize));
        NN_RESULT_DO(m_NodeL1->Verify(0, m_NodeSize, sizeof(int64_t)));

        const auto* const pNode = m_NodeL1.Get<Node>();

        int64_t beginOffset = 0;
        // L1 ノード内に L2 ノードのオフセットが存在する
        if( m_OffsetCount < m_EntrySetCount && pNode->GetCount() < m_OffsetCount )
        {
            beginOffset = *pNode->GetEnd();
        }
        else
        {
            beginOffset = *pNode->GetBegin();
        }
        NN_RESULT_THROW_UNLESS(
            0 <= beginOffset && beginOffset <= pNode->GetBeginOffset(),
            fs::ResultInvalidBucketTreeEntryOffset());

        const auto endOffset = pNode->GetEndOffset();
        NN_RESULT_THROW_UNLESS(
            beginOffset < endOffset,
            fs::ResultInvalidBucketTreeEntryOffset());

        m_BeginOffset = beginOffset;
        m_EndOffset = endOffset;
    }

    {
        int64_t storageSize = 0;
        NN_RESULT_DO(m_EntryStorage.GetSize(&storageSize));
        NN_RESULT_DO(m_EntryStorage.OperateRange(
            fs::OperationId::Invalidate,
            0,
            storageSize));
    }

    NN_RESULT_SUCCESS;
}

/**
 * @brief   コンストラクタです。
 */
BucketTree::Visitor::Visitor() NN_NOEXCEPT
    : m_pTree(nullptr)
    , m_pEntry(nullptr)
    , m_EntryIndex(-1)
    , m_EntrySetCount(0)
{
    m_EntrySet.info.count = 0;
    m_EntrySet.info.index = 0;
    m_EntrySet.info.end = 0;
    m_EntrySet.info.begin = 0;
}

/**
 * @brief   デストラクタです。
 */
BucketTree::Visitor::~Visitor() NN_NOEXCEPT
{
    if( m_pEntry != nullptr )
    {
        m_pTree->GetAllocator()->deallocate(m_pEntry, m_pTree->m_EntrySize);

        m_pTree = nullptr;
        m_pEntry = nullptr;
    }
}

/**
 * @brief   初期化します。
 *
 * @param[in]   pTree   巡回するツリーのポインタ
 *
 * @return  実行結果を返します。
 */
Result BucketTree::Visitor::Initialize(const BucketTree* pTree) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pTree);
    NN_SDK_REQUIRES(m_pTree == nullptr || m_pTree == pTree);

    if( m_pEntry == nullptr )
    {
        m_pEntry = pTree->GetAllocator()->allocate(pTree->m_EntrySize);
        NN_RESULT_THROW_UNLESS(m_pEntry != nullptr, fs::ResultBufferAllocationFailed());

        m_pTree = pTree;
    }
    NN_RESULT_SUCCESS;
}

/**
 * @brief   エントリを検索します。
 *
 * @param[in]   virtualAddress  エントリ検索用の仮想アドレス
 *
 * @retval  ResultSuccess       エントリが見つかりました。
 * @retval  ResultOutOfRange    エントリが見つかりませんでした。
 * @retval  上記以外            ストレージの読み込みに失敗しました。
 *
 * @pre
 *      - 初期化済み
 */
Result BucketTree::Visitor::Find(int64_t virtualAddress) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pTree);
    // NOTE: 0 <= virtualAddress は BucketTree::Find() で保障されている

    const auto* const pNode = m_pTree->m_NodeL1.Get<Node>();
    NN_RESULT_THROW_UNLESS(virtualAddress < pNode->GetEndOffset(), fs::ResultOutOfRange());

    int entrySetIndex = 0;

    // L1 ノード内の L2 部分を調査
    if( m_pTree->IsExistOffsetL2OnL1() && virtualAddress < pNode->GetBeginOffset() )
    {
        const auto begin = pNode->GetEnd();
        const auto end = pNode->GetBegin() + m_pTree->m_OffsetCount;

        auto pos = std::upper_bound(begin, end, virtualAddress);
        // L2 の範囲外
        NN_RESULT_THROW_UNLESS(begin < pos, fs::ResultOutOfRange());

        --pos;

        // (pos - begin) <= NodeSizeMax なのでそのままキャストする
        entrySetIndex = static_cast<int>(pos - begin);
    }
    // L1 ノードを調査
    else
    {
        const auto begin = pNode->GetBegin();
        const auto end = pNode->GetEnd();

        auto pos = std::upper_bound(begin, end, virtualAddress);
        // L1 の範囲外
        NN_RESULT_THROW_UNLESS(begin < pos, fs::ResultOutOfRange());

        --pos;

        // L2 あり
        if( m_pTree->IsExistL2() )
        {
            const auto nodeIndex = static_cast<int>(pos - begin);
            NN_RESULT_THROW_UNLESS(
                0 <= nodeIndex && nodeIndex < m_pTree->m_OffsetCount,
                fs::ResultInvalidBucketTreeNodeOffset()
            );

            NN_RESULT_DO(FindEntrySet(&entrySetIndex, virtualAddress, nodeIndex));
        }
        // L1 のみ
        else
        {
            entrySetIndex = static_cast<int>(pos - begin);
        }
    }

    NN_RESULT_THROW_UNLESS(
        0 <= entrySetIndex && entrySetIndex < m_pTree->m_EntrySetCount,
        fs::ResultInvalidBucketTreeNodeOffset()
    );

    NN_RESULT_DO(FindEntry(virtualAddress, entrySetIndex));

    m_EntrySetCount = m_pTree->m_EntrySetCount;

    NN_RESULT_SUCCESS;
}

/**
 * @brief   ストレージからエントリセットを検索します。
 *
 * @param[out]  pOutIndex       見つけたインデックスの出力先
 * @param[in]   virtualAddress  エントリ検索用仮想アドレス
 * @param[in]   nodeIndex       検索する L2 ノードのインデックス
 *
 * @retval  ResultSuccess       インデックスが見つかりました。
 * @retval  ResultOutOfRange    インデックスが見つかりませんでした。
 * @retval  上記以外            ストレージの読み込みに失敗しました。
 */
Result BucketTree::Visitor::FindEntrySet(
                                int* pOutIndex,
                                int64_t virtualAddress,
                                int nodeIndex
                            ) NN_NOEXCEPT
{
    // 内部関数なので事前条件はチェックしない
    const auto nodeSize = m_pTree->m_NodeSize;

    PooledBuffer pool(nodeSize, 1);
    if( nodeSize <= pool.GetSize() )
    {
        // 内部関数なのでバッファサイズは渡さない
        return FindEntrySetWithBuffer(pOutIndex, virtualAddress, nodeIndex, pool.GetBuffer());
    }
    else
    {
        pool.Deallocate();

        return FindEntrySetWithoutBuffer(pOutIndex, virtualAddress, nodeIndex);
    }
}

Result BucketTree::Visitor::FindEntrySetWithBuffer(
                                int* pOutIndex,
                                int64_t virtualAddress,
                                int nodeIndex,
                                char* buffer
                            ) NN_NOEXCEPT
{
    // 内部関数なので事前条件はチェックしない
    const auto nodeSize = m_pTree->m_NodeSize;
    const auto nodeOffset = (1 + nodeIndex) * static_cast<int64_t>(nodeSize); // L1 ノード分を +1 する
    fs::SubStorage& storage = m_pTree->m_NodeStorage;

    // L2 ノードの読み込み
    NN_RESULT_DO(storage.Read(nodeOffset, buffer, nodeSize));

    NodeHeader header;
    std::memcpy(&header, buffer, NodeHeaderSize);
    NN_RESULT_DO(header.Verify(nodeIndex, nodeSize, sizeof(int64_t)));

    StorageNode node(sizeof(int64_t), header.count);
    node.Find(buffer, virtualAddress);

    if( 0 <= node.GetIndex() )
    {
        *pOutIndex = m_pTree->GetEntrySetIndex(header.index, node.GetIndex());

        NN_RESULT_SUCCESS;
    }
    NN_RESULT_THROW(fs::ResultOutOfRange());
}

Result BucketTree::Visitor::FindEntrySetWithoutBuffer(
                                int* pOutIndex,
                                int64_t virtualAddress,
                                int nodeIndex
                            ) NN_NOEXCEPT
{
    // 内部関数なので事前条件はチェックしない
    const auto nodeSize = m_pTree->m_NodeSize;
    const auto nodeOffset = (1 + nodeIndex) * static_cast<int64_t>(nodeSize); // L1 ノード分を +1 する
    fs::SubStorage& storage = m_pTree->m_NodeStorage;

    // L2 ノードの読み込み
    NodeHeader header;
    NN_RESULT_DO(storage.Read(nodeOffset, &header, NodeHeaderSize));
    NN_RESULT_DO(header.Verify(nodeIndex, nodeSize, sizeof(int64_t)));

    StorageNode node(nodeOffset, sizeof(int64_t), header.count);
    NN_RESULT_DO(node.Find(&storage, virtualAddress));

    if( 0 <= node.GetIndex() )
    {
        *pOutIndex = m_pTree->GetEntrySetIndex(header.index, node.GetIndex());

        NN_RESULT_SUCCESS;
    }
    NN_RESULT_THROW(fs::ResultOutOfRange());
}

/**
 * @brief   エントリセットを読み込んでエントリを検索します。
 *
 * @param[in]   virtualAddress  エントリ検索用仮想アドレス
 * @param[in]   entrySetIndex   検索するエントリセットのインデックス
 *
 * @retval  ResultSuccess       エントリが見つかりました。
 * @retval  ResultOutOfRange    エントリが見つかりませんでした。
 * @retval  上記以外            ストレージの読み込みに失敗しました。
 */
Result BucketTree::Visitor::FindEntry(
                                int64_t virtualAddress,
                                int entrySetIndex
                            ) NN_NOEXCEPT
{
    // 内部関数なので事前条件はチェックしない
    const auto entrySetSize = m_pTree->m_NodeSize;

    PooledBuffer pool(entrySetSize, 1);
    if( entrySetSize <= pool.GetSize() )
    {
        // 内部関数なのでバッファサイズは渡さない
        return FindEntryWithBuffer(virtualAddress, entrySetIndex, pool.GetBuffer());
    }
    else
    {
        pool.Deallocate();

        return FindEntryWithoutBuffer(virtualAddress, entrySetIndex);
    }
}

Result BucketTree::Visitor::FindEntryWithBuffer(
                                int64_t virtualAddress,
                                int entrySetIndex,
                                char* buffer
                            ) NN_NOEXCEPT
{
    // 内部関数なので事前条件はチェックしない
    const auto entrySetSize = m_pTree->m_NodeSize;
    const auto entrySetOffset = entrySetIndex * static_cast<int64_t>(entrySetSize);
    const auto entrySize = m_pTree->m_EntrySize;
    fs::SubStorage& storage = m_pTree->m_EntryStorage;

    // エントリセットの読み込み
    NN_RESULT_DO(storage.Read(entrySetOffset, buffer, entrySetSize));

    EntrySetHeader entrySet;
    std::memcpy(&entrySet, buffer, sizeof(EntrySetHeader));
    NN_RESULT_DO(entrySet.header.Verify(entrySetIndex, entrySetSize, entrySize));

    StorageNode node(entrySize, entrySet.info.count);
    node.Find(buffer, virtualAddress);

    if( 0 <= node.GetIndex() )
    {
        const auto entryIndex = node.GetIndex();
        const auto entryOffset = detail::GetBucketTreeEntryOffset(0, entrySize, entryIndex);

        std::memcpy(m_pEntry, buffer + entryOffset, entrySize);

        m_EntrySet = entrySet;
        m_EntryIndex = entryIndex;

        NN_RESULT_SUCCESS;
    }
    NN_RESULT_THROW(nn::fs::ResultOutOfRange());
}

Result BucketTree::Visitor::FindEntryWithoutBuffer(
                                int64_t virtualAddress,
                                int entrySetIndex
                            ) NN_NOEXCEPT
{
    // 内部関数なので事前条件はチェックしない
    const auto entrySetSize = m_pTree->m_NodeSize;
    const auto entrySetOffset = entrySetIndex * static_cast<int64_t>(entrySetSize);
    const auto entrySize = m_pTree->m_EntrySize;
    fs::SubStorage& storage = m_pTree->m_EntryStorage;

    // エントリセットの読み込み
    EntrySetHeader entrySet;
    NN_RESULT_DO(storage.Read(entrySetOffset, &entrySet, sizeof(EntrySetHeader)));
    NN_RESULT_DO(entrySet.header.Verify(entrySetIndex, entrySetSize, entrySize));

    StorageNode node(entrySetOffset, entrySize, entrySet.info.count);
    NN_RESULT_DO(node.Find(&storage, virtualAddress));

    if( 0 <= node.GetIndex() )
    {
        m_EntryIndex = -1; // 読み込みに失敗したときに無効になるようにする

        const auto entryIndex = node.GetIndex();
        const auto entryOffset =
            detail::GetBucketTreeEntryOffset(entrySetOffset, entrySize, entryIndex);

        NN_RESULT_DO(storage.Read(entryOffset, m_pEntry, entrySize));

        m_EntrySet = entrySet;
        m_EntryIndex = entryIndex;

        NN_RESULT_SUCCESS;
    }
    NN_RESULT_THROW(nn::fs::ResultOutOfRange());
}

/**
 * @brief   巡回位置を次のエントリに進めます。
 *
 * @retval  ResultSuccess       正常に処理が終了しました。
 * @retval  ResultOutOfRange    ツリー範囲外に移動しようとしました。
 * @retval  上記以外            ストレージの読み込みに失敗しました。
 */
Result BucketTree::Visitor::MoveNext() NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(IsValid(), fs::ResultOutOfRange());

    auto entryIndex = m_EntryIndex + 1;

    // エントリの末尾なので次のエントリセットへ
    if( entryIndex == m_EntrySet.info.count )
    {
        const auto entrySetIndex = m_EntrySet.info.index + 1;

        // エントリセットの末尾なのでエラー
        NN_RESULT_THROW_UNLESS(entrySetIndex < m_EntrySetCount, fs::ResultOutOfRange());

        const auto end = m_EntrySet.info.end;

        m_EntryIndex = -1; // 読み込みに失敗したときに無効になるようにする

        const auto entrySetSize = m_pTree->m_NodeSize;
        const auto entrySetOffset = entrySetIndex * static_cast<int64_t>(entrySetSize);

        NN_RESULT_DO(m_pTree->m_EntryStorage.Read(
                        entrySetOffset, &m_EntrySet, sizeof(EntrySetHeader)));
        NN_RESULT_DO(m_EntrySet.header.Verify(
                        entrySetIndex, entrySetSize, m_pTree->m_EntrySize));

        NN_RESULT_THROW_UNLESS(
            end == m_EntrySet.info.begin && m_EntrySet.info.begin < m_EntrySet.info.end,
            fs::ResultInvalidBucketTreeEntrySetOffset()
        );

        entryIndex = 0;
    }
    else
    {
        m_EntryIndex = -1;
    }

    const auto entrySize = m_pTree->m_EntrySize;
    const auto entryOffset = detail::GetBucketTreeEntryOffset(
                                 m_EntrySet.info.index,
                                 m_pTree->m_NodeSize,
                                 entrySize,
                                 entryIndex
                             );
    NN_RESULT_DO(m_pTree->m_EntryStorage.Read(entryOffset, m_pEntry, entrySize));

    m_EntryIndex = entryIndex;

    NN_RESULT_SUCCESS;
}

/**
 * @brief   巡回位置を前のエントリに戻します。
 *
 * @retval  ResultSuccess       正常に処理が終了しました。
 * @retval  ResultOutOfRange    ツリー範囲外に移動しようとしました。
 * @retval  上記以外            ストレージの読み込みに失敗しました。
 */
Result BucketTree::Visitor::MovePrevious() NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(IsValid(), fs::ResultOutOfRange());

    int entryIndex = m_EntryIndex;

    // エントリの先頭なので前のエントリセットへ
    if( entryIndex == 0 )
    {
        // エントリセットの先頭なのでエラー
        NN_RESULT_THROW_UNLESS(0 < m_EntrySet.info.index, fs::ResultOutOfRange());

        const auto begin = m_EntrySet.info.begin;

        m_EntryIndex = -1; // 読み込みに失敗したときに無効になるようにする

        const auto entrySetSize = m_pTree->m_NodeSize;
        const auto entrySetIndex = m_EntrySet.info.index - 1;
        const auto entrySetOffset = entrySetIndex * static_cast<int64_t>(entrySetSize);

        NN_RESULT_DO(m_pTree->m_EntryStorage.Read(
                        entrySetOffset, &m_EntrySet, sizeof(EntrySetHeader)));
        NN_RESULT_DO(m_EntrySet.header.Verify(
                        entrySetIndex, entrySetSize, m_pTree->m_EntrySize));

        NN_RESULT_THROW_UNLESS(
            begin == m_EntrySet.info.end && m_EntrySet.info.begin < m_EntrySet.info.end,
            fs::ResultInvalidBucketTreeEntrySetOffset()
        );

        entryIndex = m_EntrySet.info.count;
    }
    else
    {
        m_EntryIndex = -1;
    }

    --entryIndex;

    const auto entrySize = m_pTree->m_EntrySize;
    const auto entryOffset = detail::GetBucketTreeEntryOffset(
                                 m_EntrySet.info.index,
                                 m_pTree->m_NodeSize,
                                 entrySize,
                                 entryIndex
                             );
    NN_RESULT_DO(m_pTree->m_EntryStorage.Read(entryOffset, m_pEntry, entrySize));

    m_EntryIndex = entryIndex;

    NN_RESULT_SUCCESS;
}

}}
