﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

// ビルドスイッチのため、例外的に先頭でインクルードする
#include <nn/fs/detail/fs_ResultHandlingUtilitySuppressRecordingEventOnUnsupportedPlatforms.h>

#include <nn/nn_SdkLog.h>
#include <nn/fs/fs_Directory.h>
#include <nn/fs/fsa/fs_IDirectory.h>
#include <nn/fssystem/fs_AesXtsFileSystem.h>
#include <nn/fssystem/fs_AesXtsFile.h>
#include <nn/fssystem/fs_PathTool.h>
#include <nn/fssystem/fs_Utility.h>

namespace nn { namespace fssystem {

NN_DEFINE_STATIC_CONSTANT(const size_t AesXtsFileSystem::AesBlockSize);
NN_DEFINE_STATIC_CONSTANT(const size_t AesXtsFileSystem::EncryptionKeyGenerationKeySize);
NN_DEFINE_STATIC_CONSTANT(const size_t AesXtsFileSystem::MacKeySize);
NN_DEFINE_STATIC_CONSTANT(const size_t AesXtsFileSystem::IvSize);

namespace detail {

    namespace
    {
        void CheckPathIsNormalized(const char* path) NN_NOEXCEPT
        {
            NN_UNUSED(path);

#if defined(NN_DETAIL_ENABLE_SDK_ASSERT)
            bool isNormalized;
            bool isPathValid = (
                path[0] == '\0'
                || (PathTool::IsNormalized(&isNormalized, path).IsSuccess() && isNormalized)
            );
            if( !isPathValid )
            {
                NN_SDK_LOG("unnormalized path: %s\n", path);
            }
            NN_SDK_ASSERT(isPathValid);
#endif // defined(NN_DETAIL_ENABLE_SDK_ASSERT)
        }
    }

    class AesXtsFileSystemDirectory : public fs::fsa::IDirectory, public fs::detail::Newable
    {
    public:
        AesXtsFileSystemDirectory(fs::fsa::IFileSystem* pBaseFileSystem, std::unique_ptr<fs::fsa::IDirectory>&& pBaseDirectory, const char* path, fs::OpenDirectoryMode mode) NN_NOEXCEPT
            : m_pBaseFileSystem(pBaseFileSystem),
              m_pBaseDirectory(std::move(pBaseDirectory)),
              m_Mode(mode)
        {
            std::strncpy(m_Path, path, sizeof(m_Path) - 1);
            m_Path[sizeof(m_Path) - 1] = '\0';

            m_PathLength = static_cast<int>(strnlen(m_Path, fs::EntryNameLengthMax));
            NN_SDK_ASSERT_LESS(0, m_PathLength);
            if (m_Path[m_PathLength - 1] == '/')
            {
                m_Path[m_PathLength - 1] = '\0';
                --m_PathLength;
            }

            CheckPathIsNormalized(m_Path);
        }

        virtual ~AesXtsFileSystemDirectory() NN_NOEXCEPT NN_OVERRIDE;

    private:
        virtual Result DoRead(int64_t* outValue, fs::DirectoryEntry* outEntries, int64_t count) NN_NOEXCEPT NN_OVERRIDE;

        virtual Result DoGetEntryCount(int64_t* outValue) NN_NOEXCEPT NN_OVERRIDE;

    private:
        fs::fsa::IFileSystem* m_pBaseFileSystem;
        std::unique_ptr<fs::fsa::IDirectory> m_pBaseDirectory;
        fs::OpenDirectoryMode m_Mode;
        char m_Path[fs::EntryNameLengthMax + 1];
        int m_PathLength;
    };

    AesXtsFileSystemDirectory::~AesXtsFileSystemDirectory() NN_NOEXCEPT
    {
        m_pBaseDirectory.reset();
    }

    Result AesXtsFileSystemDirectory::DoRead(int64_t* outValue, fs::DirectoryEntry* outEntries, int64_t count) NN_NOEXCEPT
    {
        NN_RESULT_DO(m_pBaseDirectory->Read(outValue, outEntries, count));

        m_Path[m_PathLength] = '/';
        for (auto i = 0; i < *outValue; ++i)
        {
            if (outEntries[i].directoryEntryType == fs::DirectoryEntryType::DirectoryEntryType_File)
            {
                if ((m_Mode & fs::OpenDirectoryModePrivate_NotRequireFileSize) != 0)
                {
                    outEntries[i].fileSize = 0;
                }
                else
                {
                    // ここではヘッダーの検証はしない。シグネチャーは確認する。
                    // 読み出せないファイルは非暗号化ファイルと見てエラーにはせず、スキップする
                    // 非暗号化ファイルのサイズは 0 と扱う

                    if (m_PathLength + 1 + strnlen(outEntries[i].name, fs::EntryNameLengthMax + 1) > fs::EntryNameLengthMax)
                    {
                        return fs::ResultTooLongPath();
                    }

                    std::strncpy(m_Path + m_PathLength + 1, outEntries[i].name, sizeof(m_Path) - 1 - (m_PathLength + 1));
                    NN_SDK_ASSERT(m_Path[sizeof(m_Path) - 1] == '\0');

                    int64_t fileSize = 0;

                    std::unique_ptr<nn::fs::fsa::IFile> pFile;
                    auto result = m_pBaseFileSystem->OpenFile(&pFile, m_Path, nn::fs::OpenMode::OpenMode_Read);
                    if (result.IsSuccess())
                    {
                        size_t readSize;
                        uint32_t signature;
                        result = pFile->Read(&readSize, offsetof(AesXtsFileHeader, signature), &signature, sizeof(signature), nn::fs::ReadOption::MakeValue(0));
                        if (result.IsFailure() || readSize != sizeof(signature) || !AesXtsFileHeader::IsValidSignature(signature))
                        {
                            fileSize = 0;
                        }
                        else
                        {
                            result = pFile->Read(&readSize, offsetof(AesXtsFileHeader, actualFileSize), &fileSize, sizeof(fileSize), nn::fs::ReadOption::MakeValue(0));
                            if (result.IsFailure() || readSize != sizeof(fileSize))
                            {
                                fileSize = 0;
                            }
                        }
                    }

                    outEntries[i].fileSize = fileSize;
                }
            }
        }
        NN_RESULT_SUCCESS;
    }

    Result AesXtsFileSystemDirectory::DoGetEntryCount(int64_t* outValue) NN_NOEXCEPT
    {
        return m_pBaseDirectory->GetEntryCount(outValue);
    }

    template<typename Function>
    Result CheckRecursively(fs::fsa::IFileSystem* pBaseFileSystem, const char* path, char* pathBuffer, size_t sizePathBuffer, fs::DirectoryEntry* pDirectoryEntry, Function doFile) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_EQUAL(static_cast<size_t>(fs::EntryNameLengthMax + 1), sizePathBuffer);
        return fssystem::IterateDirectoryRecursive(
            pBaseFileSystem,
            path,
            pathBuffer,
            sizePathBuffer,
            pDirectoryEntry,
            [&](const char* path, const fs::DirectoryEntry& entry) NN_NOEXCEPT -> Result
            {
                NN_UNUSED(path);
                NN_UNUSED(entry);
                NN_RESULT_SUCCESS;
            },
            [&](const char* path, const fs::DirectoryEntry& entry) NN_NOEXCEPT -> Result
            {
                NN_UNUSED(path);
                NN_UNUSED(entry);
                NN_RESULT_SUCCESS;
            },
            [&](const char* path, const fs::DirectoryEntry& entry) NN_NOEXCEPT -> Result
            {
                NN_UNUSED(entry);
                return doFile(path);
            });
    }

}

AesXtsFileSystem::AesXtsFileSystem() NN_NOEXCEPT
    : m_pBaseFileSystem(),
      m_XtsBlockSize(0),
      m_pGenerateRandom(nullptr)
{
    std::memset(m_EncryptionKeyGenerationKey, 0, EncryptionKeyGenerationKeySize);
    std::memset(m_MacKey, 0, MacKeySize);
}

AesXtsFileSystem::~AesXtsFileSystem() NN_NOEXCEPT
{
}

Result AesXtsFileSystem::Initialize(
        std::shared_ptr<fs::fsa::IFileSystem> pBaseFileSystem,
        const void* pEncryptionKeyGenerationKey, size_t encryptionKeyGenerationKeySize,
        const void* pMacKey, size_t macKeySize,
        GenerateRandomFunction pGenerateRandom,
        size_t xtsBlockSize
    ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pBaseFileSystem);
    NN_SDK_REQUIRES_NOT_NULL(pEncryptionKeyGenerationKey);
    NN_SDK_REQUIRES_EQUAL(encryptionKeyGenerationKeySize, EncryptionKeyGenerationKeySize);
    NN_SDK_REQUIRES_NOT_NULL(pMacKey);
    NN_SDK_REQUIRES_EQUAL(macKeySize, MacKeySize);
    NN_SDK_REQUIRES_NOT_NULL(pGenerateRandom);

    // AesXtsStorage を利用するので、xtsBlockSize は AesBlockSize の倍数
    NN_STATIC_ASSERT(AesBlockSize == AesXtsStorage::AesBlockSize);
    NN_SDK_REQUIRES(nn::util::is_aligned(xtsBlockSize, AesBlockSize));

    m_pBaseFileSystem = pBaseFileSystem;
    std::memcpy(m_EncryptionKeyGenerationKey, pEncryptionKeyGenerationKey, encryptionKeyGenerationKeySize);
    std::memcpy(m_MacKey, pMacKey, macKeySize);
    m_XtsBlockSize = xtsBlockSize;
    m_pGenerateRandom = pGenerateRandom;

    NN_RESULT_SUCCESS;
}

Result AesXtsFileSystem::DoCreateFile(const char* path, int64_t size, int option) NN_NOEXCEPT
{
    detail::CheckPathIsNormalized(path);

    // 作成
    int64_t fileSize = AesXtsFileHeader::Size + nn::util::align_up(size, AesXtsStorage::KeySize);
    NN_RESULT_DO(m_pBaseFileSystem->CreateFile(path, fileSize, option));

    {
        nn::fssystem::AesXtsFileHeader header;
        header.Create(path, size, m_EncryptionKeyGenerationKey, sizeof(m_EncryptionKeyGenerationKey), m_MacKey, sizeof(m_MacKey), m_pGenerateRandom);

        auto result = EncryptHeader(&header, path, path);
        if (result.IsFailure())
        {
            // 書き込みに失敗したらファイルを消す
            NN_RESULT_DO(m_pBaseFileSystem->DeleteFile(path));
            return result;
        }
    }

    NN_RESULT_SUCCESS;
}

Result AesXtsFileSystem::DoDeleteFile(const char* path) NN_NOEXCEPT
{
    return m_pBaseFileSystem->DeleteFile(path);
}

Result AesXtsFileSystem::DoCreateDirectory(const char* path) NN_NOEXCEPT
{
    return m_pBaseFileSystem->CreateDirectory(path);
}

Result AesXtsFileSystem::DoDeleteDirectory(const char* path) NN_NOEXCEPT
{
    return m_pBaseFileSystem->DeleteDirectory(path);
}

Result AesXtsFileSystem::DoDeleteDirectoryRecursively(const char* path) NN_NOEXCEPT
{
    return m_pBaseFileSystem->DeleteDirectoryRecursively(path);
}

Result AesXtsFileSystem::DoCleanDirectoryRecursively(const char* path) NN_NOEXCEPT
{
    return m_pBaseFileSystem->CleanDirectoryRecursively(path);
}

Result AesXtsFileSystem::DoRenameFile(const char* currentPath, const char* newPath) NN_NOEXCEPT
{
    detail::CheckPathIsNormalized(currentPath);
    detail::CheckPathIsNormalized(newPath);

    nn::fssystem::AesXtsFileHeader header;

    NN_RESULT_DO(DecryptHeader(&header, currentPath, currentPath));
    NN_RESULT_DO(m_pBaseFileSystem->RenameFile(currentPath, newPath));

    if (header.HasValidSignature())
    {
        auto result = EncryptHeader(&header, newPath, newPath);

        // 失敗時は Rename を巻き戻す
        if (result.IsFailure())
        {
            // エラーは無視して元の値を書き直す
            char decryptKeys[AesXtsFileHeader::CryptoKeySize * AesXtsFileHeader::CryptoKeyCount];
            header.DecryptKeysAndVerify(decryptKeys, sizeof(decryptKeys), newPath, m_EncryptionKeyGenerationKey, sizeof(m_EncryptionKeyGenerationKey), m_MacKey, sizeof(m_MacKey));
            EncryptHeader(&header, newPath, currentPath);

            // 元のエラーを返せば十分なので、以下の Result は無視する
            m_pBaseFileSystem->RenameFile(newPath, currentPath);
            return result;
        }
    }

    NN_RESULT_SUCCESS;
}

Result AesXtsFileSystem::DoRenameDirectory(const char* currentPath, const char* newPath) NN_NOEXCEPT
{
    detail::CheckPathIsNormalized(currentPath);
    detail::CheckPathIsNormalized(newPath);

    auto pDirectoryEntry = fs::detail::MakeUnique<fs::DirectoryEntry>();
    if (!pDirectoryEntry)
    {
        return fs::ResultAllocationMemoryFailedInAesXtsFileF();
    }
    auto pathLocal = fs::detail::MakeUnique<char[]>(fs::EntryNameLengthMax + 1);
    if (!pathLocal)
    {
        return fs::ResultAllocationMemoryFailedInAesXtsFileG();
    }
    int lengthCurrent = static_cast<int>(strnlen(currentPath, fs::EntryNameLengthMax));
    NN_SDK_ASSERT_LESS(0, lengthCurrent);
    if (currentPath[lengthCurrent - 1] == '/')
    {
        --lengthCurrent;
    }
    int lengthNew = static_cast<int>(strnlen(newPath, fs::EntryNameLengthMax));
    NN_SDK_ASSERT_LESS(0, lengthNew);
    if (newPath[lengthNew - 1] == '/')
    {
        --lengthNew;
    }

    NN_RESULT_DO(detail::CheckRecursively(m_pBaseFileSystem.get(), currentPath, pathLocal.get(), fs::EntryNameLengthMax + 1, pDirectoryEntry.get(), [&](const char* path) NN_NOEXCEPT -> Result
    {
        nn::fssystem::AesXtsFileHeader header;
        NN_RESULT_DO(DecryptHeader(&header, path, path));
        NN_RESULT_SUCCESS;
    }));

    NN_RESULT_DO(m_pBaseFileSystem->RenameDirectory(currentPath, newPath));

    {
        auto lambdaSetOldPath = [&](const char* path) NN_NOEXCEPT -> Result
        {
            if (lengthCurrent + strnlen(path, fs::EntryNameLengthMax + 1) - lengthNew > fs::EntryNameLengthMax)
            {
                return fs::ResultTooLongPath();
            }

            std::memcpy(pDirectoryEntry->name, currentPath, lengthCurrent);
            pDirectoryEntry->name[lengthCurrent] = '/';
            std::strncpy(pDirectoryEntry->name + lengthCurrent + 1, path + lengthNew + 1, fs::EntryNameLengthMax - (lengthNew + 1));
            pDirectoryEntry->name[fs::EntryNameLengthMax] = '\0';

            NN_RESULT_SUCCESS;
        };

        auto result = detail::CheckRecursively(m_pBaseFileSystem.get(), newPath, pathLocal.get(), fs::EntryNameLengthMax + 1, pDirectoryEntry.get(), [&](const char* path) NN_NOEXCEPT -> Result
        {
            // 以前のヘッダーの読み出し
            nn::fssystem::AesXtsFileHeader header;

            // 復号
            NN_RESULT_DO(lambdaSetOldPath(path));

            auto result = DecryptHeader(&header, path, pDirectoryEntry->name);
            if (result.IsSuccess() && header.HasValidSignature())
            {
                // 更新
                NN_RESULT_DO(EncryptHeader(&header, path, path));
            }

            NN_RESULT_SUCCESS;
        });
        if (result.IsFailure())
        {
            // 書き込みに失敗したら、そこまで書いたファイル全てを巻き戻して、リネームを巻き戻す
            detail::CheckRecursively(m_pBaseFileSystem.get(), newPath, pathLocal.get(), fs::EntryNameLengthMax + 1, pDirectoryEntry.get(), [&](const char* path) NN_NOEXCEPT -> Result
            {
                auto result = lambdaSetOldPath(path);
                if (result.IsSuccess())
                {
                    // 以前のヘッダーの読み出し
                    nn::fssystem::AesXtsFileHeader header;

                    // 新しい鍵で復号できるものを古い鍵に戻す
                    result = DecryptHeader(&header, path, path);
                    if (result.IsSuccess() && header.HasValidSignature())
                    {
                        // 更新 (なるだけ多くのファイルを救いたいのでエラーは無視する)
                        EncryptHeader(&header, path, pDirectoryEntry->name);
                    }
                }

                NN_RESULT_SUCCESS;
            });

            // 元のエラーを返せば十分なので、以下の Result は無視する
            m_pBaseFileSystem->RenameDirectory(newPath, currentPath);
            return result;
        }
    }

    NN_RESULT_SUCCESS;
}

Result AesXtsFileSystem::DoGetEntryType(fs::DirectoryEntryType* outValue, const char* path) NN_NOEXCEPT
{
    return m_pBaseFileSystem->GetEntryType(outValue, path);
}

Result AesXtsFileSystem::DoGetFreeSpaceSize(int64_t* outValue, const char* path) NN_NOEXCEPT
{
    return m_pBaseFileSystem->GetFreeSpaceSize(outValue, path);
}

Result AesXtsFileSystem::DoGetTotalSpaceSize(int64_t* outValue, const char* path) NN_NOEXCEPT
{
    return m_pBaseFileSystem->GetTotalSpaceSize(outValue, path);
}

Result AesXtsFileSystem::DoOpenFile(std::unique_ptr<fs::fsa::IFile>* outValue, const char* path, fs::OpenMode mode) NN_NOEXCEPT
{
    detail::CheckPathIsNormalized(path);

    fs::OpenMode baseFileMode = mode;
    if (mode & nn::fs::OpenMode_Write)
    {
        // AlignmentMatchingStorage を使用するため、Write 時は Read 必須
        baseFileMode = static_cast<nn::fs::OpenMode>(baseFileMode | nn::fs::OpenMode_Read);
    }

    std::unique_ptr<fs::fsa::IFile> baseFile;
    NN_RESULT_DO(m_pBaseFileSystem->OpenFile(&baseFile, path, baseFileMode));

    // 初期ベクトルは0で固定
    static const char iv[IvSize] = {0};

    // AesXtsFile を生成して返す
    std::unique_ptr<AesXtsFile> file(new AesXtsFile());
    if (file == nullptr)
    {
        return nn::fs::ResultAllocationMemoryFailedInAesXtsFileSystemA();
    }
    NN_RESULT_DO(file->Initialize(
        mode,
        std::move(baseFile),
        path,
        m_EncryptionKeyGenerationKey, sizeof(m_EncryptionKeyGenerationKey),
        m_MacKey, sizeof(m_MacKey),
        iv, IvSize,
        m_XtsBlockSize));
    *outValue = std::move(file);

    NN_RESULT_SUCCESS;
}

Result AesXtsFileSystem::DoOpenDirectory(std::unique_ptr<fs::fsa::IDirectory>* outValue, const char* path, fs::OpenDirectoryMode mode) NN_NOEXCEPT
{
    std::unique_ptr<fs::fsa::IDirectory> baseDirectory;
    NN_RESULT_DO(m_pBaseFileSystem->OpenDirectory(&baseDirectory, path, mode));
    outValue->reset(new detail::AesXtsFileSystemDirectory(m_pBaseFileSystem.get(), std::move(baseDirectory), path, mode));
    NN_RESULT_THROW_UNLESS(outValue != nullptr, fs::ResultAllocationMemoryFailedInAesXtsFileSystemB());
    NN_RESULT_SUCCESS;
}

Result AesXtsFileSystem::DoCommit() NN_NOEXCEPT
{
    return m_pBaseFileSystem->Commit();
}

Result AesXtsFileSystem::DoCommitProvisionally(int64_t counter) NN_NOEXCEPT
{
    return m_pBaseFileSystem->CommitProvisionally(counter);
}

Result AesXtsFileSystem::DoRollback() NN_NOEXCEPT
{
    return m_pBaseFileSystem->Rollback();
}

Result AesXtsFileSystem::DoQueryEntry(char* outBuffer, size_t outBufferSize, const char* inBuffer, size_t inBufferSize, fs::fsa::QueryId queryId, const char* path) NN_NOEXCEPT
{
    return m_pBaseFileSystem->QueryEntry(outBuffer, outBufferSize, inBuffer, inBufferSize, queryId, path);
}


Result AesXtsFileSystem::DecryptHeader(AesXtsFileHeader* pOutHeader, const char* path, const char* decryptPath) NN_NOEXCEPT
{
    // 以下のファイルは非暗号化ファイルと扱う
    // ・ファイルサイズが足りない
    // ・シグネチャーが暗号化ファイルのものでない
    // 非暗号化ファイルは復号成功と扱う (この関数の後の処理を中止しない)

    // 以前のヘッダーの読み出し
    std::unique_ptr<nn::fs::fsa::IFile> pFile;
    NN_RESULT_DO(m_pBaseFileSystem->OpenFile(&pFile, path, nn::fs::OpenMode::OpenMode_Read));

    size_t readSize;
    NN_RESULT_DO(pFile->Read(&readSize, 0, pOutHeader, sizeof(*pOutHeader), nn::fs::ReadOption::MakeValue(0)));
    if (readSize != sizeof(*pOutHeader) || !pOutHeader->HasValidSignature())
    {
        pOutHeader->signature = 0;
        NN_RESULT_SUCCESS;
    }

    // 検証
    uint8_t decryptKeys[AesXtsFileHeader::CryptoKeySize * AesXtsFileHeader::CryptoKeyCount];
    if (! pOutHeader->DecryptKeysAndVerify(decryptKeys, sizeof(decryptKeys), decryptPath, m_EncryptionKeyGenerationKey, sizeof(m_EncryptionKeyGenerationKey), m_MacKey, sizeof(m_MacKey)))
    {
        return nn::fs::ResultAesXtsFileSystemFileHeaderCorruptedOnRename();
    }

    NN_RESULT_SUCCESS;
}

Result AesXtsFileSystem::EncryptHeader(AesXtsFileHeader* pHeader, const char* path, const char* encryptPath) NN_NOEXCEPT
{
    // 署名と暗号化
    pHeader->SignAndEncryptKeys(encryptPath, m_EncryptionKeyGenerationKey, sizeof(m_EncryptionKeyGenerationKey), m_MacKey, sizeof(m_MacKey));

    // ヘッダー更新
    std::unique_ptr<nn::fs::fsa::IFile> pFile;
    NN_RESULT_DO(m_pBaseFileSystem->OpenFile(&pFile, path, nn::fs::OpenMode::OpenMode_Write));

    NN_RESULT_DO(pFile->Write(0, pHeader, sizeof(*pHeader), nn::fs::WriteOption::MakeValue(0)));
    NN_RESULT_DO(pFile->Flush());

    NN_RESULT_SUCCESS;
}

}}
