﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/util/util_BitUtil.h>
#include <nn/util/util_BytePtr.h>
#include <nn/util/util_Endian.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/crypto/crypto_Aes128CtrEncryptor.h>
#include <nn/crypto/crypto_Aes128CtrDecryptor.h>

#include <nn/fssystem/fs_AesCtrStorage.h>
#include <nn/fssystem/fs_AsynchronousAccess.h>
#include <nn/fssystem/fs_ThreadPriorityChanger.h>
#include <nn/fssystem/fs_Utility.h>
#include <nn/fs/fs_QueryRange.h>

#include <nn/os/os_ThreadApi.h>

namespace nn { namespace fssystem {

NN_DEFINE_STATIC_CONSTANT(const size_t AesCtrStorage::BlockSize);
NN_DEFINE_STATIC_CONSTANT(const size_t AesCtrStorage::KeySize);
NN_DEFINE_STATIC_CONSTANT(const size_t AesCtrStorage::IvSize);

NN_STATIC_ASSERT(AesCtrStorage::BlockSize == crypto::Aes128CtrDecryptor::BlockSize);
NN_STATIC_ASSERT(AesCtrStorage::IvSize == crypto::Aes128CtrDecryptor::IvSize);

void AesCtrStorage::MakeIv(
                        void* pOutIv,
                        size_t outSize,
                        uint64_t upperValue,
                        int64_t offset
                    ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutIv);
    NN_SDK_REQUIRES_EQUAL(outSize, IvSize);
    NN_SDK_REQUIRES_GREATER_EQUAL(offset, 0);
    NN_UNUSED(outSize);

    util::BytePtr ptr(pOutIv);

    util::StoreBigEndian(ptr.Get<uint64_t>(), upperValue);
    ptr.Advance(sizeof(uint64_t));

    util::StoreBigEndian<int64_t>(ptr.Get<int64_t>(), offset / BlockSize);
}

AesCtrStorage::AesCtrStorage(
                   fs::IStorage* pBaseStorage,
                   const void* pKey,
                   size_t keySize,
                   const void* pIv,
                   size_t ivSize
               ) NN_NOEXCEPT
    : m_pBaseStorage(pBaseStorage)
{
    NN_SDK_REQUIRES_NOT_NULL(pBaseStorage);
    NN_SDK_REQUIRES_NOT_NULL(pKey);
    NN_SDK_REQUIRES_EQUAL(keySize, KeySize);
    NN_SDK_REQUIRES_NOT_NULL(pIv);
    NN_SDK_REQUIRES_EQUAL(ivSize, IvSize);
    NN_UNUSED(keySize);
    NN_UNUSED(ivSize);

    std::memcpy(m_Key, pKey, KeySize);
    std::memcpy(m_Iv, pIv, IvSize);
}

Result AesCtrStorage::Read(int64_t offset, void* buffer, size_t size) NN_NOEXCEPT
{
    if( size == 0 )
    {
        NN_RESULT_SUCCESS;
    }
    NN_RESULT_THROW_UNLESS(buffer != nullptr, fs::ResultNullptrArgument());

    // BlockSize 単位アクセスしかこない前提
    NN_FSP_REQUIRES(nn::util::is_aligned(offset, BlockSize), nn::fs::ResultInvalidArgument());
    NN_FSP_REQUIRES(nn::util::is_aligned(size, BlockSize), nn::fs::ResultInvalidArgument());

    NN_RESULT_DO(m_pBaseStorage->Read(offset, buffer, size));

    ScopedThreadPriorityChanger changePriority(+1, ScopedThreadPriorityChanger::Mode::Relative);

    char counter[IvSize];
    std::memcpy(counter, m_Iv, IvSize);
    AddCounter(counter, IvSize, offset / BlockSize);

    auto decryptedSize = crypto::DecryptAes128Ctr(
        buffer,
        size,
        m_Key,
        KeySize,
        counter,
        IvSize,
        buffer,
        size
    );
    NN_ABORT_UNLESS_EQUAL(decryptedSize, size);
    NN_UNUSED(decryptedSize);

    NN_RESULT_SUCCESS;
}

Result AesCtrStorage::Write(int64_t offset, const void* buffer, size_t size) NN_NOEXCEPT
{
    if( size == 0 )
    {
        NN_RESULT_SUCCESS;
    }
    NN_RESULT_THROW_UNLESS(buffer != nullptr, fs::ResultNullptrArgument());

    // BlockSize 単位アクセスしかこない前提
    NN_FSP_REQUIRES(nn::util::is_aligned(offset, BlockSize), nn::fs::ResultInvalidArgument());
    NN_FSP_REQUIRES(nn::util::is_aligned(size, BlockSize), nn::fs::ResultInvalidArgument());

    PooledBuffer pooledBuffer;
    const auto useWorkBuffer = !IsPooledBuffer(buffer);

    if( useWorkBuffer )
    {
        pooledBuffer.Allocate(size, BlockSize);
    }

    char counter[IvSize];
    std::memcpy(counter, m_Iv, IvSize);
    AddCounter(counter, IvSize, offset / BlockSize);

    size_t restSize = size;
    int64_t currentOffset = 0;

    while( 0 < restSize )
    {
        size_t writeSize = useWorkBuffer ? std::min(pooledBuffer.GetSize(), restSize) : restSize;
        auto writeBuffer = useWorkBuffer ? pooledBuffer.GetBuffer() : const_cast<void*>(buffer);

        {
            ScopedThreadPriorityChanger changePriority(
                +1,
                ScopedThreadPriorityChanger::Mode::Relative);

            auto encryptedSize = crypto::EncryptAes128Ctr(
                writeBuffer,
                writeSize,
                m_Key,
                KeySize,
                counter,
                IvSize,
                reinterpret_cast<const char*>(buffer) + currentOffset,
                writeSize
            );
            NN_ABORT_UNLESS_EQUAL(encryptedSize, writeSize);
            NN_UNUSED(encryptedSize);
        }

        NN_RESULT_DO(m_pBaseStorage->Write(offset + currentOffset, writeBuffer, writeSize));

        currentOffset += writeSize;
        restSize -= writeSize;

        if( restSize > 0 )
        {
            AddCounter(counter, IvSize, writeSize / BlockSize);
        }
    }

    NN_RESULT_SUCCESS;
}

Result AesCtrStorage::GetSize(int64_t* pOutValue) NN_NOEXCEPT
{
    return m_pBaseStorage->GetSize(pOutValue);
}

Result AesCtrStorage::OperateRange(
                          void* outBuffer,
                          size_t outBufferSize,
                          fs::OperationId operationId,
                          int64_t offset,
                          int64_t size,
                          const void* inBuffer,
                          size_t inBufferSize
                      ) NN_NOEXCEPT
{
    if( size == 0 )
    {
        if( operationId == nn::fs::OperationId::QueryRange )
        {
            NN_FSP_REQUIRES(outBuffer != nullptr, nn::fs::ResultNullptrArgument());
            NN_FSP_REQUIRES(outBufferSize == sizeof(nn::fs::QueryRangeInfo), nn::fs::ResultInvalidSize());

            reinterpret_cast<nn::fs::QueryRangeInfo*>(outBuffer)->Clear();
        }

        NN_RESULT_SUCCESS;
    }

    // BlockSize 単位アクセスしかこない前提
    NN_FSP_REQUIRES(nn::util::is_aligned(offset, BlockSize), nn::fs::ResultInvalidArgument());
    NN_FSP_REQUIRES(nn::util::is_aligned(size, BlockSize), nn::fs::ResultInvalidArgument());

    switch( operationId )
    {
    case nn::fs::OperationId::QueryRange:
        {
            NN_FSP_REQUIRES(outBuffer != nullptr, nn::fs::ResultNullptrArgument());
            NN_FSP_REQUIRES(outBufferSize == sizeof(nn::fs::QueryRangeInfo), nn::fs::ResultInvalidSize());

            NN_RESULT_DO(m_pBaseStorage->OperateRange(
                outBuffer,
                outBufferSize,
                operationId,
                offset,
                size,
                inBuffer,
                inBufferSize));

            nn::fs::QueryRangeInfo info;
            info.Clear();
            info.aesCtrKeyTypeFlag = static_cast<int32_t>(
                nn::fs::AesCtrKeyTypeFlag::InternalKeyForSoftwareAes);

            reinterpret_cast<nn::fs::QueryRangeInfo*>(outBuffer)->Merge(info);
            break;
        }

    default:
        NN_RESULT_DO(m_pBaseStorage->OperateRange(
            outBuffer,
            outBufferSize,
            operationId,
            offset,
            size,
            inBuffer,
            inBufferSize));
    }

    NN_RESULT_SUCCESS;
}

}}
