﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/nn_SdkAssert.h>
#include <nn/result/result_HandlingUtility.h>

#include <nn/fssystem/dbm/fs_FileSystemControlArea.h>

namespace nn { namespace fssystem { namespace dbm {

/**
* @brief        コンストラクタです。
*
* @details      コンストラクタです。
*/
FileSystemControlArea::FileSystemControlArea() NN_NOEXCEPT
    : m_Storage()
{
}

/**
* @brief        管理領域をマウントします。
*
* @param[in]    storage        ストレージ
*
* @details      管理領域をマウントします。
*/
void FileSystemControlArea::Initialize(
         fs::SubStorage storage
     ) NN_NOEXCEPT
{
    m_Storage = storage;
}

/**
* @brief        管理領域をアンマウントします。
*
* @details      管理領域をアンマウントします。
*/
void FileSystemControlArea::Finalize() NN_NOEXCEPT
{
}

/**
* @brief        拡張したアロケーションテーブルの情報を設定します。
*
* @param[in]    count           拡張後のアロケーションテーブルの個数
*
* @return       関数の処理結果を返します。
* @retval       ResultSuccess   正常に拡張できました。
* @retval       上記以外        ストレージでの読み書きに失敗しました。
*
* @details      拡張したアロケーションテーブルの情報を設定します。
*/
Result FileSystemControlArea::ExpandAllocationTableInfo(uint32_t count) NN_NOEXCEPT
{
    int64_t blockSize;
    StorageInfo tableInfo;
    StorageInfo bodyInfo;

    // 取得
    NN_RESULT_DO(ReadBlockSize(&blockSize));
    NN_RESULT_DO(ReadAllocationTableInfo(&tableInfo.offset, &tableInfo.size));
    NN_RESULT_DO(ReadDataBodyInfo(&bodyInfo.offset, &bodyInfo.size));

    // 設定
    NN_RESULT_DO(WriteAllocationTableInfo(tableInfo.offset, count));
    NN_RESULT_DO(WriteDataBodyInfo(bodyInfo.offset, count));

    NN_RESULT_SUCCESS;
}

/**
* @brief        ストレージ上のデータ情報を更新します。

* @param[in]    offset          ストレージ上のデータのオフセット
* @param[in]    size            ストレージ上のデータのサイズ
* @param[in]    accessOffset    管理領域の先頭からのオフセット
*
* @return       関数の処理結果を返します。
* @retval       ResultSuccess   正常に更新できました。
* @retval       上記以外        ストレージへの書き込みに失敗しました。
*
* @pre          マウントしている。
*
* @details      ストレージ上のデータ情報を更新します。
*/
Result FileSystemControlArea::WriteStorageInfo(
           int64_t offset,
           uint32_t size,
           uint32_t accessOffset
       ) NN_NOEXCEPT
{
    StorageInfo info;
    info.offset = offset;
    info.size = size;
    return m_Storage.Write(
               accessOffset,
               &info,
               sizeof(StorageInfo)
           );
}

/**
* @brief        ストレージ上のデータ情報を取得します。
*
* @param[out]   outOffset       ストレージ上のデータのオフセット
* @param[out]   outSize         ストレージ上のデータのサイズ
* @param[in]    accessOffset    管理領域の先頭からのオフセット
*
* @return       関数の処理結果を返します。
* @retval       ResultSuccess   正常に取得できました。
* @retval       上記以外        ストレージからの読み込みに失敗しました。
*
* @pre          マウントしている。
* @pre          outOffset が NULL ではない。
* @pre          outSize が NULL ではない。
*
* @details      ストレージ上のデータ情報を取得します。
*/
Result FileSystemControlArea::ReadStorageInfo(
           int64_t* outOffset,
           uint32_t* outSize,
           uint32_t accessOffset
       ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outOffset);
    NN_SDK_REQUIRES_NOT_NULL(outSize);

    StorageInfo info;
    NN_RESULT_DO(
        m_Storage.Read(
            accessOffset,
            &info,
            sizeof(StorageInfo)
        )
    );

    *outOffset = info.offset;
    *outSize = info.size;

    NN_RESULT_SUCCESS;
}

/**
* @brief        ストレージ上の可変長データ情報を更新します。
*
* @param[in]    index           ストレージ上のデータの開始インデックス
* @param[in]    accessOffset    管理領域の先頭からのオフセット
*
* @return       関数の処理結果を返します。
* @retval       ResultSuccess   正常に更新できました。
* @retval       上記以外        ストレージへの書き込みに失敗しました。
*
* @pre          マウントしている。
*
* @details      ストレージ上の可変長データ情報を更新します。
*/
Result FileSystemControlArea::WriteAllocationInfo(
           uint32_t index,
           uint32_t accessOffset
       ) NN_NOEXCEPT
{
    AllocationInfo info;
    info.index = index;
    return m_Storage.Write(
               accessOffset,
               &info,
               sizeof(AllocationInfo)
           );
}

/**
* @brief        ストレージ上の可変長データ情報を取得します。
*
* @param[out]   outIndex        ストレージ上のデータの開始インデックス
* @param[in]    accessOffset    管理領域の先頭からのオフセット
*
* @return       関数の処理結果を返します。
* @retval       ResultSuccess   正常に取得できました。
* @retval       上記以外        ストレージからの読み込みに失敗しました。
*
* @pre          マウントしている。
* @pre          outIndex が NULL ではない。
*
* @details      ストレージ上の可変長データ情報を取得します。
*/
Result FileSystemControlArea::ReadAllocationInfo(
           uint32_t* outIndex,
           uint32_t accessOffset
       ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outIndex);

    AllocationInfo info;
    NN_RESULT_DO(
        m_Storage.Read(
            accessOffset,
            &info,
            sizeof(AllocationInfo)
        )
    );

    *outIndex = info.index;

    NN_RESULT_SUCCESS;
}

}}}

