﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <mutex>
#include <algorithm>

#include <nn/nn_Common.h>
#include <nn/nn_SdkLog.h>
#include <nn/nn_Abort.h>
#include <nn/result/result_HandlingUtility.h>

#include <nn/fs/fs_ResultPrivate.h>
#include <nn/fssystem/buffers/fs_FileSystemBufferManager.h>

namespace nn { namespace fssystem {

/*!
* @brief       コンストラクタです。
*/
FileSystemBufferManager::CacheHandleTable::CacheHandleTable() NN_NOEXCEPT
    : m_InternalEntryBuffer(),
      m_ExternalEntryBuffer(nullptr),
      m_EntryBufferSize(0),
      m_pEntries(nullptr),
      m_EntryCount(0),
      m_EntryCountMax(0),
      m_AttrList(),
      m_ExternalAttrInfoBuffer(nullptr),
      m_ExternalAttrInfoCount(0),
      m_CacheCountMin(0),
      m_CacheSizeMin(0),
      m_TotalCacheSize(0),
      m_CurrentHandle(0LL)
{
}

/*!
* @brief       デストラクタです。
*/
FileSystemBufferManager::CacheHandleTable::~CacheHandleTable() NN_NOEXCEPT
{
    Finalize();
}

/*!
* @brief       初期化します。
*
* @param[in]   cacheCountMax   キャッシュの最大数
*
* @pre
*              - 未初期化である。
*/
nn::Result FileSystemBufferManager::CacheHandleTable::Initialize(int cacheCountMax) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(m_pEntries == nullptr);
    NN_SDK_REQUIRES(m_InternalEntryBuffer == nullptr);

    // バッファを確保します。
    if( m_ExternalEntryBuffer == nullptr )
    {
        m_EntryBufferSize = sizeof(Entry) * cacheCountMax;
        m_InternalEntryBuffer = nn::fs::detail::MakeUnique<char[]>(m_EntryBufferSize);
    }
    if( m_InternalEntryBuffer == nullptr && m_ExternalEntryBuffer == nullptr )
    {
        NN_RESULT_THROW(nn::fs::ResultAllocationMemoryFailedInFileSystemBufferManagerA());
    }

    // エントリ配列を初期化します。
    m_pEntries = reinterpret_cast<Entry*>(
                     m_ExternalEntryBuffer == nullptr ? m_InternalEntryBuffer.get() : m_ExternalEntryBuffer
                 );
    m_EntryCount = 0;
    m_EntryCountMax = cacheCountMax;
    NN_SDK_ASSERT_NOT_NULL(m_pEntries);

    // TORIAEZU: キャッシュ数から適当に決定します。
    m_CacheCountMin = cacheCountMax / 16;
    m_CacheSizeMin = m_CacheCountMin * 256;

    NN_RESULT_SUCCESS;
}

/*!
* @brief       終了処理を行います。
*
* @details     デストラクタで自動的に呼び出されるため、明示的に呼び出す必要はありません。
*/
void FileSystemBufferManager::CacheHandleTable::Finalize() NN_NOEXCEPT
{
    if( m_pEntries != nullptr )
    {
        // デストラクト時にキャッシュエントリが残っている場合はエラー扱いとします。
        NN_SDK_ASSERT(m_EntryCount == 0);

        // 属性の管理領域を破棄します。
        if( m_ExternalAttrInfoBuffer == nullptr )
        {
            auto attrIter = m_AttrList.begin();
            while( attrIter != m_AttrList.end() )
            {
                const auto pAttrInfo = &*attrIter;
                attrIter = m_AttrList.erase(attrIter);
                delete pAttrInfo;
            }
        }

        // 各メンバを破棄します。
        m_InternalEntryBuffer.reset();
        m_ExternalEntryBuffer = nullptr;
        m_EntryBufferSize = 0;
        m_pEntries = nullptr;
        m_TotalCacheSize = 0;
    }
}

/*!
* @brief       キャッシュを登録します。
*
* @param[out]  pOutHandle  登録したキャッシュのハンドルを格納するバッファ
* @param[in]   address     登録するメモリ範囲の先頭アドレス
* @param[in]   size        登録するメモリ範囲のサイズ
* @param[in]   bufAttr     登録するバッファの属性
*
* @return      登録に成功したかどうかを返します。
* @retval      true        登録に成功しました。
* @retval      false       登録に失敗しました。
*
* @pre
*              - 初期化済みである。
*              - pOutHandle != nullptr
*/
bool FileSystemBufferManager::CacheHandleTable::Register(
         CacheHandle* pOutHandle,
         uintptr_t address,
         size_t size,
         const BufferAttribute& bufAttr
     ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pEntries);
    NN_SDK_REQUIRES_NOT_NULL(pOutHandle);

    auto pEntry = AcquireEntry(address, size, bufAttr);
    if( pEntry != nullptr )
    {
        // 属性の管理情報を更新します。
        const auto pAttrInfo = FindAttrInfo(bufAttr);
        if( pAttrInfo != nullptr )
        {
            pAttrInfo->IncrementCacheCount();
            pAttrInfo->AddCacheSize(size);
        }
        else
        {
            AttrInfo* pInfo = nullptr;
            if( m_ExternalAttrInfoBuffer == nullptr )
            {
                pInfo = new AttrInfo(bufAttr.GetLevel(), 1, size);
            }
            else if( 0 <= bufAttr.GetLevel() && bufAttr.GetLevel() < m_ExternalAttrInfoCount )
            {
                const auto buffer = m_ExternalAttrInfoBuffer + bufAttr.GetLevel() * sizeof(AttrInfo);
                pInfo = new (buffer) AttrInfo(bufAttr.GetLevel(), 1, size);
            }
            if( pInfo == nullptr )
            {
                ReleaseEntry(pEntry);
                return false;
            }
            m_AttrList.push_back(*pInfo);
        }

        m_TotalCacheSize += size;
        *pOutHandle = pEntry->GetHandle();
        return true;
    }
    else
    {
        return false;
    }
}

/*!
* @brief       キャッシュの登録を解除します。
*
* @param[out]  pOutAddress 登録を解除したキャッシュのメモリ範囲の先頭アドレスを格納するバッファ
* @param[out]  pOutSize    登録を解除したキャッシュのメモリ範囲のサイズを格納するバッファ
* @param[in]   handle      登録を解除するキャッシュのハンドル
*
* @return      登録の解除に成功したかどうかを返します。
* @retval      true        解除に成功しました。
* @retval      false       解除に失敗しました。
*
* @pre
*              - 初期化済みである。
*              - pOutAddress != nullptr
*              - pOutSize != nullptr
*/
bool FileSystemBufferManager::CacheHandleTable::Unregister(
         uintptr_t* pOutAddress,
         size_t* pOutSize,
         CacheHandle handle
     ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pEntries);
    NN_SDK_REQUIRES_NOT_NULL(pOutAddress);
    NN_SDK_REQUIRES_NOT_NULL(pOutSize);

    const auto pEntry = std::lower_bound(m_pEntries, m_pEntries + m_EntryCount, handle,
        [](const Entry& entry, CacheHandle handle) NN_NOEXCEPT
        {
            return entry.GetHandle() < handle;
        });

    if( pEntry != m_pEntries + m_EntryCount && pEntry->GetHandle() == handle )
    {
        UnregisterCore(pOutAddress, pOutSize, pEntry);
        return true;
    }
    else
    {
        return false;
    }
}

/*!
* @brief       最古のキャッシュの登録を解除します。
*
* @param[out]  pOutAddress     登録を解除したキャッシュのメモリ範囲の先頭アドレスを格納するバッファ
* @param[out]  pOutSize        登録を解除したキャッシュのメモリ範囲のサイズを格納するバッファ
* @param[in]   bufAttr         登録を解除するキャッシュを制限するためのバッファ属性
* @param[in]   requiredSize    必要な空き領域のサイズ
*
* @return      登録の解除に成功したかどうかを返します。
* @retval      true            解除に成功しました。
* @retval      false           解除に失敗しました。
*
* @pre
*              - 初期化済みである。
*              - pOutAddress != nullptr
*              - pOutSize != nullptr
*/
bool FileSystemBufferManager::CacheHandleTable::UnregisterOldest(
         uintptr_t* pOutAddress,
         size_t* pOutSize,
         const BufferAttribute& bufAttr,
         size_t requiredSize
     ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pEntries);
    NN_SDK_REQUIRES_NOT_NULL(pOutAddress);
    NN_SDK_REQUIRES_NOT_NULL(pOutSize);

    NN_UNUSED(bufAttr);
    NN_UNUSED(requiredSize);

    if( m_EntryCount == 0 )
    {
        return false;
    }

    // 引き剥がせるエントリを探索します。
    const auto CanUnregister = [this](const Entry& entry) NN_NOEXCEPT
    {
        const auto pAttrInfo = FindAttrInfo(entry.GetBufferAttribute());
        NN_SDK_ASSERT_NOT_NULL(pAttrInfo);
        int cacheCountMin = GetCacheCountMin(entry.GetBufferAttribute());
        size_t cacheSizeMin = GetCacheSizeMin(entry.GetBufferAttribute());
        return cacheCountMin < pAttrInfo->GetCacheCount()
            && cacheSizeMin + entry.GetSize() <= pAttrInfo->GetCacheSize();
    };
    auto pEntry = std::find_if(m_pEntries, m_pEntries + m_EntryCount, CanUnregister);

    // 引き剥がすのに適したエントリがなければ先頭を引き剥がします。
    if( pEntry == m_pEntries + m_EntryCount )
    {
        pEntry = m_pEntries;
    }

    // 最終的な対象エントリを引き剥がします。
    NN_SDK_ASSERT_NOT_EQUAL(pEntry, m_pEntries + m_EntryCount);
    UnregisterCore(pOutAddress, pOutSize, pEntry);
    return true;
}

/*!
* @brief       キャッシュの登録を解除します。
*
* @param[out]  pOutAddress     登録を解除したキャッシュのメモリ範囲の先頭アドレスを格納するバッファ
* @param[out]  pOutSize        登録を解除したキャッシュのメモリ範囲のサイズを格納するバッファ
* @param[in]   pEntry          登録を解除するエントリ
*
* @pre
*              - 初期化済みである。
*              - pOutAddress != nullptr
*              - pOutSize != nullptr
*              - pEntry != nullptr
*/
void FileSystemBufferManager::CacheHandleTable::UnregisterCore(uintptr_t* pOutAddress, size_t* pOutSize, Entry* pEntry) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pEntries);
    NN_SDK_REQUIRES_NOT_NULL(pOutAddress);
    NN_SDK_REQUIRES_NOT_NULL(pOutSize);
    NN_SDK_REQUIRES_NOT_NULL(pEntry);

    // 属性の管理情報を更新します。
    const auto pAttrInfo = FindAttrInfo(pEntry->GetBufferAttribute());
    NN_SDK_ASSERT_NOT_NULL(pAttrInfo);
    NN_SDK_ASSERT_GREATER(pAttrInfo->GetCacheCount(), 0);
    NN_SDK_ASSERT_GREATER_EQUAL(pAttrInfo->GetCacheSize(), pEntry->GetSize());
    pAttrInfo->DecrementCacheCount();
    pAttrInfo->SubtractCacheSize(pEntry->GetSize());

    NN_SDK_ASSERT_GREATER_EQUAL(m_TotalCacheSize, pEntry->GetSize());
    m_TotalCacheSize -= pEntry->GetSize();

    // キャッシュリストから削除します。
    *pOutAddress = pEntry->GetAddress();
    *pOutSize = pEntry->GetSize();
    ReleaseEntry(pEntry);
}

/*!
* @brief       ハンドルを新規作成します。
*
* @return      作成したハンドルを返します。
*
* @pre
*              - 初期化済みである。
*/
FileSystemBufferManager::CacheHandle
    FileSystemBufferManager::CacheHandleTable::PublishCacheHandle() NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pEntries);
    // オーバーフローはしないはず
    ++m_CurrentHandle;
    return m_CurrentHandle;
}

/*!
* @brief       キャッシュサイズの合計を取得します。
*
* @return      キャッシュサイズの合計を返します。
*/
size_t FileSystemBufferManager::CacheHandleTable::GetTotalCacheSize() const NN_NOEXCEPT
{
    return m_TotalCacheSize;
}

/*!
* @brief       エントリを確保して初期化します。
*
* @param[in]   address エントリに登録するキャッシュのメモリ範囲の先頭アドレス
* @param[in]   size    エントリに登録するキャッシュのメモリ範囲のサイズ
*
* @return      作成したエントリを指すポインタを返します。
* @retval      nullptr メモリ不足のため新規作成できませんでした。
*
* @pre
*              - 初期化済みである。
*/
FileSystemBufferManager::CacheHandleTable::Entry*
    FileSystemBufferManager::CacheHandleTable::AcquireEntry(
        uintptr_t address,
        size_t size,
        const BufferAttribute& bufAttr
    ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pEntries);

    Entry* pEntry = nullptr;

    // 残りエントリ数が制限より多い場合のみ処理します。
    if( m_EntryCount < m_EntryCountMax )
    {
        pEntry = m_pEntries + m_EntryCount;
        pEntry->Initialize(
                    PublishCacheHandle(),
                    address,
                    size,
                    bufAttr
                );
        ++m_EntryCount;
        NN_SDK_ASSERT(m_EntryCount == 1 || (pEntry - 1)->GetHandle() < pEntry->GetHandle());
    }

    return pEntry;
}

/*!
* @brief       エントリを解放します。
*
* @param[in]   pEntry  解放するエントリ
*
* @pre
*              - 初期化済みである。
*              - pEntry != nullptr
*/
void FileSystemBufferManager::CacheHandleTable::ReleaseEntry(Entry* pEntry) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pEntries);
    NN_SDK_REQUIRES_NOT_NULL(pEntry);

    const auto entryBuffer
        = m_ExternalEntryBuffer == nullptr ? m_InternalEntryBuffer.get() : m_ExternalEntryBuffer;
    NN_SDK_ASSERT_LESS_EQUAL(static_cast<void*>(entryBuffer), static_cast<void*>(pEntry));
    NN_SDK_ASSERT_LESS(
        static_cast<void*>(pEntry),
        static_cast<void*>(entryBuffer + m_EntryBufferSize)
    );
    NN_UNUSED(entryBuffer);

    std::memmove(pEntry, pEntry + 1, sizeof(Entry) * (m_EntryCount - (pEntry + 1 - m_pEntries)));
    --m_EntryCount;
}

/*!
* @brief       指定の属性の管理情報を取得します。
*
* @param[in]   bufAttr 管理情報を取得する属性
*
* @return      管理情報をさすポインタを返します。
* @retval      nullptr 指定の属性の管理情報は登録されていません。
*/
FileSystemBufferManager::CacheHandleTable::AttrInfo*
    FileSystemBufferManager::CacheHandleTable::FindAttrInfo(
        const BufferAttribute& bufAttr
    ) NN_NOEXCEPT
{
    const auto HasLevel = [&bufAttr](const AttrInfo& info) NN_NOEXCEPT
    {
        return bufAttr.GetLevel() == info.GetLevel();
    };
    const auto attrIter = std::find_if(m_AttrList.begin(), m_AttrList.end(), HasLevel);
    return attrIter != m_AttrList.end() ? &*attrIter : nullptr;
}

/*!
* @brief       指定のサイズと属性を持つバッファを割り当てます。
*
* @param[in]   size    割り当てるサイズ
* @param[in]   bufAttr 割り当てるバッファの属性
*
* @return      割り当てたメモリ範囲の先頭アドレスとサイズの組を返します。
*
* @details     割り当てに失敗した場合、成功するまで短時間待機して割り当てを再試行することを繰り返します。
*/
const std::pair<uintptr_t, size_t> FileSystemBufferManager::DoAllocateBuffer(
                                       size_t size,
                                       const BufferAttribute& bufAttr
                                   ) NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_Locker);

    std::pair<uintptr_t, size_t> memRange(0, 0);
    const auto order = m_Buddy.GetOrderFromBytes(size);
    NN_SDK_ASSERT(0 <= order);

    for( ; ; )
    {
        if( auto address = m_Buddy.AllocateByOrder(order) )
        {
            const auto allocatedSize = m_Buddy.GetBytesFromOrder(order);
            NN_SDK_ASSERT(size <= allocatedSize);

            memRange.first = reinterpret_cast<uintptr_t>(address);
            memRange.second = allocatedSize;

            const size_t freeSize = m_Buddy.GetTotalFreeSize();
            if( freeSize < m_SizeFreePeak )
            {
                m_SizeFreePeak = freeSize;
            }

            const size_t totalAllocatableSize = freeSize + m_CacheHandleTable.GetTotalCacheSize();
            if( totalAllocatableSize < m_SizeTotalAllocatablePeak )
            {
                m_SizeTotalAllocatablePeak = totalAllocatableSize;
            }
            break;
        }

        uintptr_t deallocateAddress;
        size_t deallocateSize;

        ++m_RetriedCount;
        if( m_CacheHandleTable.UnregisterOldest(&deallocateAddress, &deallocateSize, bufAttr, size) )
        {
            DeallocateBuffer(deallocateAddress, deallocateSize);
        }
        else
        {
            // 引き剥がし可能なエントリが不足している場合
            break;
        }
    }

    return memRange;
}

/*!
* @brief       指定の範囲のバッファを解放します。
*
* @param[in]   address 解放するバッファの先頭アドレス
* @param[in]   size    解放するバッファのサイズ
*
* @pre
*              - 指定の範囲がこのマネージャによって確保された未解放の領域である
*              - 解放するメモリ範囲がキャッシュとして登録されていない
*
* @details     事前条件に違反していたとしても停止しない可能性があります。
*/
void FileSystemBufferManager::DoDeallocateBuffer(
         uintptr_t address,
         size_t size
     ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(nn::util::ispow2(size));

    std::lock_guard<nn::os::Mutex> lock(m_Locker);

    m_Buddy.Free(
                reinterpret_cast<void*>(address),
                m_Buddy.GetOrderFromBytes(size)
            );
}

/*!
* @brief       指定のメモリ範囲をキャッシュとして登録します。
*
* @param[in]   address         登録するバッファの先頭アドレス
* @param[in]   size            登録するバッファのサイズ
* @param[in]   bufAttr         登録するバッファの属性
*
* @return      登録したキャッシュのハンドルを返します。
*
* @pre
*              - 登録するメモリ範囲がキャッシュとして登録されていない
*
* @details     登録したキャッシュは自動的に無効化されることがあります。
*
*              事前条件に違反していたとしても停止しない可能性があります。
*/
FileSystemBufferManager::CacheHandle FileSystemBufferManager::DoRegisterCache(
                                         uintptr_t address,
                                         size_t size,
                                         const BufferAttribute& bufAttr
                                     ) NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_Locker);

    CacheHandle handle = 0;

    for( ; ; )
    {
        if( m_CacheHandleTable.Register(&handle, address, size, bufAttr) )
        {
            break;
        }

        uintptr_t deallocateAddress = 0;
        size_t deallocateSize = 0;

        ++m_RetriedCount;
        if( m_CacheHandleTable.UnregisterOldest(&deallocateAddress, &deallocateSize, bufAttr) )
        {
            DeallocateBuffer(deallocateAddress, deallocateSize);
        }
        else
        {
            // 引き剥がし可能なエントリが不足
            NN_SDK_LOG("%s: No entry is unregisterable\n", NN_CURRENT_FUNCTION_NAME);

            // 渡されたバッファを解放しダミーのハンドルを返す
            DeallocateBuffer(address, size);
            handle = m_CacheHandleTable.PublishCacheHandle();
            break;
        }
    }

    return handle;
}

/*!
* @brief       指定のハンドルに紐付けられたキャッシュを取得します。
*
* @param[in]   handle  取得するキャッシュのハンドル
*
* @return      キャッシュのメモリ範囲の先頭アドレスとサイズの組を返します。
*
* @details     取得に成功したキャッシュの登録は解除されます。
*
*              無効化されたキャッシュのハンドルが指定された場合、
*              キャッシュの取得に失敗し、返される先頭アドレスとサイズはいずれも 0 になります。
*/
const std::pair<uintptr_t, size_t> FileSystemBufferManager::DoAcquireCache(
                                       CacheHandle handle
                                   ) NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_Locker);

    std::pair<uintptr_t, size_t> memRange(0, 0);
    if( m_CacheHandleTable.Unregister(&memRange.first, &memRange.second, handle) )
    {
        const size_t totalAllocatableSize = m_Buddy.GetTotalFreeSize() + m_CacheHandleTable.GetTotalCacheSize();
        if( totalAllocatableSize < m_SizeTotalAllocatablePeak )
        {
            m_SizeTotalAllocatablePeak = totalAllocatableSize;
        }
    }
    else
    {
        memRange.first = 0;
        memRange.second = 0;
    }
    return memRange;
}

/*!
* @brief       バッファマネージャが管理するメモリ全体のサイズを取得します。
*
* @return      バッファマネージャが管理するメモリ全体のサイズを返します。
*/
size_t FileSystemBufferManager::DoGetTotalSize() const NN_NOEXCEPT
{
    return m_TotalSize;
}

/*!
* @brief       バッファマネージャが管理するメモリのうち空き領域のサイズを取得します。
*
* @return      バッファマネージャが管理するメモリのうち空き領域のサイズを返します。
*/
size_t FileSystemBufferManager::DoGetFreeSize() const NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_Locker);

    return m_Buddy.GetTotalFreeSize();
}

/*!
* @brief       バッファマネージャが管理するメモリのうち割り当て可能な領域の合計サイズを取得します。
*
* @return      バッファマネージャが管理するメモリのうち割り当て可能な領域の合計サイズを返します。
*/
size_t FileSystemBufferManager::DoGetTotalAllocatableSize() const NN_NOEXCEPT
{
    return GetFreeSize() + m_CacheHandleTable.GetTotalCacheSize();
}

/*!
* @brief       バッファマネージャが管理するメモリのうち空き領域サイズのピークを取得します。
*
* @return      バッファマネージャが管理するメモリのうち空き領域サイズのピークを返します。
*/
size_t FileSystemBufferManager::DoGetFreeSizePeak() const NN_NOEXCEPT
{
    return m_SizeFreePeak;
}

/*!
* @brief       バッファマネージャが管理するメモリのうち割り当て可能な領域の合計サイズのピークを取得します。
*
* @return      バッファマネージャが管理するメモリのうち割り当て可能な領域の合計サイズのピークを返します。
*/
size_t FileSystemBufferManager::DoGetTotalAllocatableSizePeak() const NN_NOEXCEPT
{
    return m_SizeTotalAllocatablePeak;
}

/*!
* @brief       バッファマネージャが管理するメモリの確保リトライ回数を取得します。
*
* @return      バッファマネージャが管理するメモリの確保リトライ回数を返します。
*/
size_t FileSystemBufferManager::DoGetRetriedCount() const NN_NOEXCEPT
{
    return m_RetriedCount;
}

/*!
* @brief       バッファマネージャが管理するメモリの確保リトライ回数/空き領域サイズのピークをクリアします。
*/
void FileSystemBufferManager::DoClearPeak() NN_NOEXCEPT
{
    m_SizeFreePeak = GetFreeSize();
    m_RetriedCount = 0;
}

}}
