﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <mutex>
#include <nn/nn_SdkLog.h>
#include <nn/nn_Allocator.h>
#include <nn/nn_SdkAssert.h>
#include <nn/fs_Base.h>
#include <nn/fs/fs_SaveDataManagement.h>
#include <nn/fs/fs_SystemSaveData.h>
#include <nn/fs/detail/fs_Newable.h>
#include <nn/fs/fs_ResultPrivate.h>
#include <nn/fssrv/fssrv_SaveDataIndexer.h>
#include <nn/fssrv/fssrv_SaveDataIndexerTypes.h>
#include <nn/fssrv/fssrv_SaveDataInfoReaderImpl.h>
#include <nn/fssystem/fs_AllocatorUtility.h>
#include <nn/util/util_FormatString.h>
#include <nn/util/util_Optional.h>

#include "detail/fssrv_Trace.h"

namespace nn { namespace fssrv {

    void GenerateSaveDataInfo(fs::SaveDataInfo* pOutValue, const SaveDataIndexerKey& key, const SaveDataIndexerValue& value)
    {
        memset(pOutValue, 0, sizeof(fs::SaveDataInfo));
        pOutValue->saveDataId = value.id;
        pOutValue->saveDataSpaceId = value.spaceId;
        pOutValue->saveDataSize = value.size;

        pOutValue->systemSaveDataId = key.staticSaveDataId;
        pOutValue->applicationId.value = key.programId.value;
        pOutValue->saveDataType = key.type;
        pOutValue->saveDataUserId = key.userId;
        pOutValue->index = key.index;
        pOutValue->rank = key.rank;
    }

    namespace {
        const char LastPublishedIdSaveFile[] = "lastPublishedId";
        const int IndexMaxCount = 4 * 1024;
        const int ReservedIndexCount = 128;
        const size_t IndexerSystemSaveDataSize = 768 * 1024;
        const size_t IndexerSystemSaveDataJournalSize = 768 * 1024;
        const int IndexerSystemSaveDataFlags = 0;

        void MakeRootPath(char* outValue, const char* mountName)
        {
            auto length = util::SNPrintf(outValue, SaveDataIndexerInternalPathLengthMax, "%s:/", mountName);
            NN_SDK_ASSERT(static_cast<size_t>(length) < SaveDataIndexerInternalPathLengthMax);
            NN_UNUSED(length);
        }

        void MakeLastPublishedIdSaveFilePath(char* outValue, const char* mountName)
        {
            auto length = util::SNPrintf(outValue, SaveDataIndexerInternalPathLengthMax, "%s:/%s", mountName, LastPublishedIdSaveFile);
            NN_SDK_ASSERT(static_cast<size_t>(length) < SaveDataIndexerInternalPathLengthMax);
            NN_UNUSED(length);
        }

        class ScopedMount
        {
            char m_MountName[nn::fs::MountNameLengthMax + 1];
            bool m_Mounted;
        public:
            ScopedMount()
                : m_Mounted(false)
            {
            }

            ~ScopedMount()
            {
                if(m_Mounted)
                {
                    nn::fs::Unmount(m_MountName);
                }
            }

            Result Mount(const char* name, nn::fs::SaveDataSpaceId spaceId, nn::fs::SaveDataId systemSaveDataId)
            {
                NN_SDK_REQUIRES(!m_Mounted);
                NN_SDK_REQUIRES(strnlen(name, nn::fs::MountNameLengthMax + 1) < sizeof(m_MountName));

                strncpy(m_MountName, name, nn::fs::MountNameLengthMax);
                m_MountName[nn::fs::MountNameLengthMax] = '\0';

                nn::fs::DisableAutoSaveDataCreation();
                NN_RESULT_TRY(nn::fs::MountSystemSaveData(m_MountName, spaceId, systemSaveDataId))
                    NN_RESULT_CATCH(nn::fs::ResultTargetNotFound)
                    {
                        NN_RESULT_DO(nn::fs::CreateSystemSaveData(spaceId, systemSaveDataId, 0, IndexerSystemSaveDataSize, IndexerSystemSaveDataJournalSize, IndexerSystemSaveDataFlags));
                        NN_RESULT_DO(nn::fs::MountSystemSaveData(m_MountName, spaceId, systemSaveDataId));
                    }
                    NN_RESULT_CATCH(nn::fs::ResultSignedSystemPartitionDataCorrupted)
                    {
                        // SIGLO-79618: 無視せず上に返すための W/A
                        NN_RESULT_RETHROW;
                    }
                    NN_RESULT_CATCH(nn::fs::ResultDataCorrupted)
                    {
                        NN_RESULT_DO(nn::fs::DeleteSaveData(spaceId, systemSaveDataId));
                        NN_RESULT_DO(nn::fs::CreateSystemSaveData(spaceId, systemSaveDataId, 0, IndexerSystemSaveDataSize, IndexerSystemSaveDataJournalSize, IndexerSystemSaveDataFlags));
                        NN_RESULT_DO(nn::fs::MountSystemSaveData(m_MountName, spaceId, systemSaveDataId));
                    }
                NN_RESULT_END_TRY;

                m_Mounted = true;
                NN_RESULT_SUCCESS;
            }
        };

    }

    NN_FS_SCOPED_TRACE_CLASSNAME_SPECIALIZATION(Reader)

    class Reader : public SaveDataInfoReaderImpl, public nn::fs::detail::Newable
    {
    public:
        NN_IMPLICIT Reader(SaveDataIndexer* indexer) NN_NOEXCEPT : m_Indexer(indexer)
        {
            NN_FS_SCOPED_TRACE("%p", m_Indexer);
            m_Iterator = m_Indexer->GetBeginIterator();
            m_Handle = m_Indexer->GetHandle();
        }

        ~Reader() NN_NOEXCEPT
        {
            NN_FS_SCOPED_TRACE("%p", m_Indexer);

            auto scopedLock = m_Indexer->GetScopedLock();
            m_Indexer->UnregisterReader();
        }

        virtual Result Read(nn::sf::Out<std::int64_t> outValue, const nn::sf::OutBuffer& outEntries) NN_NOEXCEPT NN_OVERRIDE
        {
            NN_FS_SCOPED_TRACE("%p", m_Indexer);

            auto scopedLock = m_Indexer->GetScopedLock();
            NN_RESULT_THROW_UNLESS(m_Handle == m_Indexer->GetHandle(), nn::fs::ResultInvalidHandle());

            NN_FS_SCOPED_TRACE_CAPTURE_RESULT
            {
                auto infoBuffer = reinterpret_cast<nn::fs::SaveDataInfo*>(outEntries.GetPointerUnsafe());

                int64_t restCount = outEntries.GetSize() / sizeof(nn::fs::SaveDataInfo);
                int64_t count = 0;
                for (; !m_Iterator.IsEnd() && restCount > 0; m_Iterator.Next())
                {
                    auto key = m_Iterator.Get();
                    auto pValue = reinterpret_cast<const SaveDataIndexerValue*>(m_Iterator.GetPointer());
                    GenerateSaveDataInfo(&infoBuffer[count], key, *pValue);

                    --restCount;
                    count++;
                }

                outValue.Set(count);

                NN_RESULT_SUCCESS;
            }
            NN_FS_SCOPED_TRACE_END_CAPTURE_RESULT
        }

        void Fix(const SaveDataIndexerKey& key) NN_NOEXCEPT
        {
            m_Indexer->FixIterator(&m_Iterator, key);
        }

    private:
        SaveDataIndexer* m_Indexer;
        SaveDataIndexDatabaseIterator m_Iterator;
        int m_Handle;
    };

    SaveDataIndexer::SaveDataIndexer(const char* systemSaveDataMountName, nn::fs::SaveDataSpaceId spaceId, nn::fs::SystemSaveDataId systemSaveDataId, MemoryResource* memoryResource) NN_NOEXCEPT
        : m_SystemSaveDataId(systemSaveDataId)
        , m_SpaceId(spaceId)
        , m_MemoryResource(memoryResource)
        , m_Mutex(false)
        , m_IsInitialized(false)
        , m_IsLoaded(false)
        , m_CurrentHandleValue(1)
    {
        auto length = util::SNPrintf(m_SystemSaveDataMountName, SaveDataIndexerInternalPathLengthMax, "%s", systemSaveDataMountName);
        NN_SDK_ASSERT(static_cast<size_t>(length) < SaveDataIndexerInternalPathLengthMax);
        NN_UNUSED(length);
    }

    SaveDataIndexer::~SaveDataIndexer() NN_NOEXCEPT
    {
    }

    Result SaveDataIndexer::TryInitializeDatabase() NN_NOEXCEPT
    {
        if (!m_IsInitialized)
        {
            ScopedMount scopedMount;
            NN_RESULT_DO(scopedMount.Mount(m_SystemSaveDataMountName, m_SpaceId, m_SystemSaveDataId));

            char rootPath[SaveDataIndexerInternalPathLengthMax];
            MakeRootPath(rootPath, m_SystemSaveDataMountName);
            NN_RESULT_DO(m_Database.Initialize(rootPath, IndexMaxCount + ReservedIndexCount, m_MemoryResource));
            m_IsInitialized = true;
        }

        NN_RESULT_SUCCESS;
    }

    Result SaveDataIndexer::TryLoadDatabase(bool isForce) NN_NOEXCEPT
    {
        if (isForce)
        {
            m_IsLoaded = false;
        }

        if (m_IsLoaded)
        {
            NN_RESULT_SUCCESS;
        }

        ScopedMount scopedMount;
        NN_RESULT_DO(scopedMount.Mount(m_SystemSaveDataMountName, m_SpaceId, m_SystemSaveDataId));

        NN_RESULT_DO(m_Database.Load());

        bool isCreated = false;
        NN_UTIL_SCOPE_EXIT
        {
            if (isCreated)
            {
                nn::fs::CommitSaveData(m_SystemSaveDataMountName);
            }
        };

        nn::fs::FileHandle file;
        char lastPublishedIdSaveFilePath[SaveDataIndexerInternalPathLengthMax];
        MakeLastPublishedIdSaveFilePath(lastPublishedIdSaveFilePath, m_SystemSaveDataMountName);
        auto result = nn::fs::OpenFile(&file, lastPublishedIdSaveFilePath, nn::fs::OpenMode_Read);
        if (result.IsFailure())
        {
            if (nn::fs::ResultPathNotFound::Includes(result))
            {
                NN_RESULT_DO(nn::fs::CreateFile(lastPublishedIdSaveFilePath, sizeof(nn::fs::SaveDataId)));
                NN_RESULT_DO(nn::fs::OpenFile(&file, lastPublishedIdSaveFilePath, nn::fs::OpenMode_Read));
                isCreated = true;
            }
            else
            {
                return result;
            }
        }

        NN_UTIL_SCOPE_EXIT
        {
            nn::fs::CloseFile(file);
        };

        if (!isCreated)
        {
            NN_RESULT_DO(nn::fs::ReadFile(file, 0, &m_LastPublishedId, sizeof(nn::fs::SaveDataId)));
        }
        else
        {
            m_LastPublishedId = 0;
        }

        m_IsLoaded = true;

        NN_RESULT_SUCCESS;
    }

    Result SaveDataIndexer::Publish(nn::fs::SaveDataId* outValue, const SaveDataIndexerKey& key) NN_NOEXCEPT
    {
        std::lock_guard<nn::os::Mutex> scopedLock(m_Mutex);

        NN_RESULT_DO(TryInitializeDatabase());
        NN_RESULT_DO(TryLoadDatabase(false));

        NN_SDK_REQUIRES(m_IsLoaded);

        SaveDataIndexerValue value;
        std::memset(&value, 0x0, sizeof(SaveDataIndexerValue));
        size_t valueSize;

        auto result = m_Database.Get(&valueSize, key, &value, sizeof(SaveDataIndexerValue));
        if (result.IsSuccess())
        {
            return nn::fs::ResultAlreadyExists();
        }

        value.id = ++m_LastPublishedId;
        result = m_Database.Put(key, &value, sizeof(SaveDataIndexerValue));
        if (result.IsFailure())
        {
            m_LastPublishedId--;
            return result;
        }
        NN_RESULT_DO(FixReader(key));

        *outValue = value.id;
        NN_RESULT_SUCCESS;
    }

    Result SaveDataIndexer::PutStaticSaveDataIdIndex(const SaveDataIndexerKey& key) NN_NOEXCEPT
    {
        std::lock_guard<nn::os::Mutex> scopedLock(m_Mutex);

        NN_RESULT_DO(TryInitializeDatabase());
        NN_RESULT_DO(TryLoadDatabase(false));

        NN_SDK_REQUIRES(m_IsLoaded);

        NN_SDK_REQUIRES(key.staticSaveDataId != nn::fs::InvalidSystemSaveDataId);
        NN_SDK_REQUIRES(key.userId == nn::fs::InvalidUserId); // userId が不正でない場合は Publish する必要がある

        auto itr = m_Database.GetBeginIterator();
        for (; !itr.IsEnd(); itr.Next())
        {
            auto pValue = reinterpret_cast<const SaveDataIndexerValue*>(itr.GetPointer());
            if (pValue->id == key.staticSaveDataId)
            {
                return nn::fs::ResultAlreadyExists();
            }
        }

        SaveDataIndexerValue value;
        std::memset(&value, 0x0, sizeof(SaveDataIndexerValue));
        value.id = key.staticSaveDataId;

        NN_RESULT_DO(m_Database.Put(key, &value, sizeof(SaveDataIndexerValue)));
        NN_RESULT_DO(FixReader(key));

        NN_RESULT_SUCCESS;
    }

    bool SaveDataIndexer::IsRemainedReservedOnly() NN_NOEXCEPT
    {
        return m_Database.Count() >= IndexMaxCount;
    }

    Result SaveDataIndexer::Delete(nn::fs::SaveDataId id) NN_NOEXCEPT
    {
        std::lock_guard<nn::os::Mutex> scopedLock(m_Mutex);

        NN_RESULT_DO(TryInitializeDatabase());
        NN_RESULT_DO(TryLoadDatabase(false));

        NN_SDK_REQUIRES(m_IsLoaded);

        auto itr = m_Database.GetBeginIterator();
        for (; !itr.IsEnd(); itr.Next())
        {
            auto pValue = reinterpret_cast<const SaveDataIndexerValue*>(itr.GetPointer());
            if (pValue->id == id)
            {
                auto key = itr.Get();
                NN_RESULT_DO(m_Database.Delete(key));
                NN_RESULT_DO(FixReader(key));

                NN_RESULT_SUCCESS;
            }
        }
        return nn::fs::ResultTargetNotFound();
    }

    Result SaveDataIndexer::Commit() NN_NOEXCEPT
    {
        std::lock_guard<nn::os::Mutex> scopedLock(m_Mutex);

        NN_RESULT_DO(TryInitializeDatabase());
        NN_RESULT_DO(TryLoadDatabase(false));

        NN_SDK_REQUIRES(m_IsLoaded);

        ScopedMount scopedMount;
        NN_RESULT_DO(scopedMount.Mount(m_SystemSaveDataMountName, m_SpaceId, m_SystemSaveDataId));

        NN_RESULT_DO(m_Database.Save(true));

        nn::fs::FileHandle file;
        char lastPublishedIdSaveFilePath[SaveDataIndexerInternalPathLengthMax];
        MakeLastPublishedIdSaveFilePath(lastPublishedIdSaveFilePath, m_SystemSaveDataMountName);
        NN_RESULT_DO(nn::fs::OpenFile(&file, lastPublishedIdSaveFilePath, nn::fs::OpenMode_Write));
        bool isClosed = false;
        NN_UTIL_SCOPE_EXIT
        {
            if (!isClosed)
            {
                nn::fs::CloseFile(file);
            }
        };
        NN_RESULT_DO(nn::fs::WriteFile(file, 0, &m_LastPublishedId, sizeof(nn::fs::SaveDataId), nn::fs::WriteOption::MakeValue(0)));
        NN_RESULT_DO(nn::fs::FlushFile(file));
        nn::fs::CloseFile(file);
        isClosed = true;

        return nn::fs::CommitSaveData(m_SystemSaveDataMountName);
    }

    Result SaveDataIndexer::Rollback() NN_NOEXCEPT
    {
        std::lock_guard<nn::os::Mutex> scopedLock(m_Mutex);

        NN_RESULT_DO(TryInitializeDatabase());
        NN_RESULT_DO(TryLoadDatabase(true));
        UpdateHandle();
        NN_RESULT_SUCCESS;
    }

    Result SaveDataIndexer::Reset() NN_NOEXCEPT
    {
        std::lock_guard<nn::os::Mutex> scopedLock(m_Mutex);

        if (m_IsLoaded)
        {
            m_IsLoaded = false;
        }

        auto result = nn::fs::DeleteSaveData(m_SystemSaveDataId);
        if (result.IsFailure())
        {
            if (!nn::fs::ResultTargetNotFound::Includes(result))
            {
                return result;
            }
        }

        UpdateHandle();
        NN_RESULT_SUCCESS;
    }

    Result SaveDataIndexer::Get(SaveDataIndexerValue* outValue, const SaveDataIndexerKey& key) NN_NOEXCEPT
    {
        std::lock_guard<nn::os::Mutex> scopedLock(m_Mutex);

        NN_RESULT_DO(TryInitializeDatabase());
        NN_RESULT_DO(TryLoadDatabase(false));

        NN_SDK_REQUIRES(m_IsLoaded);

        size_t valueSize;

        auto result = m_Database.Get(&valueSize, key, outValue, sizeof(SaveDataIndexerValue));
        if (result.IsFailure())
        {
            return nn::fs::ResultTargetNotFound();
        }

        NN_RESULT_SUCCESS;
    }

    Result SaveDataIndexer::GetKey(SaveDataIndexerKey* outValue, nn::fs::SaveDataId id) NN_NOEXCEPT
    {
        std::lock_guard<nn::os::Mutex> scopedLock(m_Mutex);

        NN_RESULT_DO(TryInitializeDatabase());
        NN_RESULT_DO(TryLoadDatabase(false));

        NN_SDK_REQUIRES(m_IsLoaded);

        auto itr = m_Database.GetBeginIterator();
        for (; !itr.IsEnd(); itr.Next())
        {
            auto pValue = reinterpret_cast<const SaveDataIndexerValue*>(itr.GetPointer());
            if (pValue->id == id)
            {
                *outValue = itr.Get();
                NN_RESULT_SUCCESS;
            }
        }
        return nn::fs::ResultTargetNotFound();
    }

    Result SaveDataIndexer::GetValue(SaveDataIndexerValue* outValue, nn::fs::SaveDataId id) NN_NOEXCEPT
    {
        std::lock_guard<nn::os::Mutex> scopedLock(m_Mutex);

        NN_RESULT_DO(TryInitializeDatabase());
        NN_RESULT_DO(TryLoadDatabase(false));

        NN_SDK_REQUIRES(m_IsLoaded);

        auto itr = m_Database.GetBeginIterator();
        for (; !itr.IsEnd(); itr.Next())
        {
            auto pValue = reinterpret_cast<const SaveDataIndexerValue*>(itr.GetPointer());
            if (pValue->id == id)
            {
                *outValue = *pValue;
                NN_RESULT_SUCCESS;
            }
        }
        return nn::fs::ResultTargetNotFound();
    }

    Result SaveDataIndexer::SetValue(const SaveDataIndexerKey& key, const SaveDataIndexerValue& value) NN_NOEXCEPT
    {
        std::lock_guard<nn::os::Mutex> scopedLock(m_Mutex);

        NN_RESULT_DO(TryInitializeDatabase());
        NN_RESULT_DO(TryLoadDatabase(false));

        NN_SDK_REQUIRES(m_IsLoaded);

        auto itr = m_Database.GetLowerBoundIterator(key);
        NN_RESULT_THROW_UNLESS(!itr.IsEnd(), fs::ResultTargetNotFound());

        auto pValue = reinterpret_cast<SaveDataIndexerValue*>(itr.GetPointer());
        *pValue = value;

        NN_RESULT_SUCCESS;
    }

    int SaveDataIndexer::GetIndexCount() NN_NOEXCEPT
    {
        std::lock_guard<nn::os::Mutex> scopedLock(m_Mutex);
        return m_Database.Count();
    }

    Result SaveDataIndexer::RegisterReader(std::shared_ptr<Reader> reader) NN_NOEXCEPT
    {
        ReaderAccessor* pReaderAccessor = new ReaderAccessor(std::move(reader));
        NN_RESULT_THROW_UNLESS(pReaderAccessor, nn::fs::ResultAllocationMemoryFailedInSaveDataIndexerA());
        m_ReaderList.push_back(*pReaderAccessor);
        NN_RESULT_SUCCESS;
    }

    void SaveDataIndexer::UnregisterReader() NN_NOEXCEPT
    {
        for (auto it = m_ReaderList.begin(); it != m_ReaderList.end();)
        {
            auto& readerAccessor = *it;
            if (readerAccessor.IsExpired())
            {
                it = m_ReaderList.erase(it);
                delete &readerAccessor;
            }
            else
            {
                it++;
            }
        }
    }

    Result SaveDataIndexer::FixReader(const SaveDataIndexerKey& key) NN_NOEXCEPT
    {
        for (auto& accessor : m_ReaderList)
        {
            auto reader = accessor.Lock();
            if (reader != nullptr)
            {
                reader->Fix(key);
            }
        }
        NN_RESULT_SUCCESS;
    }

    Result SaveDataIndexer::OpenSaveDataInfoReader(std::shared_ptr<SaveDataInfoReaderImpl>* outValue) NN_NOEXCEPT
    {
        std::shared_ptr<Reader> reader;
        {
            std::lock_guard<nn::os::Mutex> scopedLock(m_Mutex);

            NN_RESULT_DO(TryInitializeDatabase());
            NN_RESULT_DO(TryLoadDatabase(false));

            reader = fssystem::AllocateShared<Reader>(this);
            NN_RESULT_THROW_UNLESS(reader, nn::fs::ResultAllocationMemoryFailedInSaveDataIndexerA());
            NN_RESULT_DO(RegisterReader(reader));
        }
        *outValue = std::move(reader);
        NN_RESULT_SUCCESS;
    }

    void SaveDataIndexer::FixIterator(SaveDataIndexDatabaseIterator* iterator, const SaveDataIndexerKey& key) NN_NOEXCEPT
    {
        m_Database.FixIterator(iterator, key);
    }

    SaveDataIndexDatabaseIterator SaveDataIndexer::GetBeginIterator() NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(m_IsLoaded);
        return m_Database.GetBeginIterator();
    }

    int SaveDataIndexer::GetHandle() NN_NOEXCEPT
    {
        return m_CurrentHandleValue;
    }

    std::unique_lock<os::Mutex> SaveDataIndexer::GetScopedLock() NN_NOEXCEPT
    {
        return std::unique_lock<os::Mutex>(m_Mutex);
    }

    void SaveDataIndexer::UpdateHandle() NN_NOEXCEPT
    {
        m_CurrentHandleValue++;
    }

    Result SaveDataIndexer::UpdateValueBySaveDataId(nn::fs::SaveDataId id, void* context, std::function<void(SaveDataIndexerValue* pValue, void* context)> delegate) NN_NOEXCEPT
    {
        std::lock_guard<nn::os::Mutex> scopedLock(m_Mutex);

        NN_RESULT_DO(TryInitializeDatabase());
        NN_RESULT_DO(TryLoadDatabase(false));

        NN_SDK_REQUIRES(m_IsLoaded);

        auto itr = m_Database.GetBeginIterator();
        for (; !itr.IsEnd(); itr.Next())
        {
            auto pValue = reinterpret_cast<const SaveDataIndexerValue*>(itr.GetPointer());
            if (pValue->id == id)
            {
                SaveDataIndexerValue value;
                std::memcpy(&value, pValue, sizeof(SaveDataIndexerValue));
                delegate(&value, context);
                NN_RESULT_DO(m_Database.Put(itr.Get(), &value, sizeof(SaveDataIndexerValue)));
                NN_RESULT_SUCCESS;
            }
        }
        return nn::fs::ResultTargetNotFound();
    }

    Result SaveDataIndexer::SetSpaceId(nn::fs::SaveDataId id, nn::fs::SaveDataSpaceId spaceId) NN_NOEXCEPT
    {
        return UpdateValueBySaveDataId(id, &spaceId, [](SaveDataIndexerValue* pValue, void* context){ pValue->spaceId = *(reinterpret_cast<nn::fs::SaveDataSpaceId*>(context)); });
    }

    Result SaveDataIndexer::SetSize(nn::fs::SaveDataId id, int64_t size) NN_NOEXCEPT
    {
        return UpdateValueBySaveDataId(id, &size, [](SaveDataIndexerValue* pValue, void* context){ pValue->size = *(reinterpret_cast<int64_t*>(context)); });
    }

    Result SaveDataIndexer::SetState(nn::fs::SaveDataId id, SaveDataState state) NN_NOEXCEPT
    {
        return UpdateValueBySaveDataId(id, &state, [](SaveDataIndexerValue* pValue, void* context){ pValue->state = *(reinterpret_cast<SaveDataState*>(context)); });
    }

}}
