﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/fssrv/fssrv_MemoryResourceFromExpHeap.h>

namespace nn { namespace fssrv {

    MemoryResourceFromExpHeap::MemoryResourceFromExpHeap(nn::lmem::HeapHandle heapHandle) NN_NOEXCEPT
    {
        m_HeapHandle = heapHandle;
    }

    MemoryResourceFromExpHeap::~MemoryResourceFromExpHeap() NN_NOEXCEPT
    {
    }

    void* MemoryResourceFromExpHeap::do_allocate(std::size_t bytes, std::size_t alignment) NN_NOEXCEPT
    {
        return nn::lmem::AllocateFromExpHeap(m_HeapHandle, bytes, static_cast<int>(alignment));
    }

    void MemoryResourceFromExpHeap::do_deallocate(void* p, std::size_t bytes, std::size_t alignment) NN_NOEXCEPT
    {
        NN_UNUSED(bytes);
        NN_UNUSED(alignment);

        nn::lmem::FreeToExpHeap(m_HeapHandle, p);
    }

    PeakCheckableMemoryResourceFromExpHeap::PeakCheckableMemoryResourceFromExpHeap(std::size_t heapSize) NN_NOEXCEPT
        : m_HeapHandle(nullptr)
        , m_Mutex(false)
        , m_FreeSizePeak(heapSize)
        , m_FreeSizeCurrent(heapSize)
    {
    }

    PeakCheckableMemoryResourceFromExpHeap::~PeakCheckableMemoryResourceFromExpHeap() NN_NOEXCEPT
    {
    }

    void PeakCheckableMemoryResourceFromExpHeap::ClearPeak() NN_NOEXCEPT
    {
        m_FreeSizePeak = m_FreeSizeCurrent;
    }

    namespace
    {
        std::size_t GetUseSize(void* p)
        {
            const auto pBlockHead = reinterpret_cast<nn::lmem::detail::ExpHeapMemoryBlockHead*>(
                reinterpret_cast<uintptr_t>(p) - sizeof(nn::lmem::detail::ExpHeapMemoryBlockHead));

            return pBlockHead->blockSize + ((pBlockHead->attribute >> 8) & 0x7f) + sizeof(nn::lmem::detail::ExpHeapMemoryBlockHead);
        }
    }

    void PeakCheckableMemoryResourceFromExpHeap::OnAllocate(void* p, std::size_t size) NN_NOEXCEPT
    {
        NN_UNUSED(size);
        if( p != nullptr )
        {
            m_FreeSizeCurrent -= GetUseSize(p);
            if( m_FreeSizeCurrent < m_FreeSizePeak )
            {
                m_FreeSizePeak = m_FreeSizeCurrent;
            }
        }
    }
    void PeakCheckableMemoryResourceFromExpHeap::OnDeallocate(void* p, std::size_t size) NN_NOEXCEPT
    {
        NN_UNUSED(size);
        if( p != nullptr )
        {
            m_FreeSizeCurrent += GetUseSize(p);
        }
    }

    void* PeakCheckableMemoryResourceFromExpHeap::do_allocate(std::size_t bytes, std::size_t alignment) NN_NOEXCEPT
    {
        std::lock_guard<os::Mutex> scopedLock(m_Mutex);

        void* p = nn::lmem::AllocateFromExpHeap(m_HeapHandle, bytes, static_cast<int>(alignment));
        OnAllocate(p, bytes);
        return p;
    }

    void PeakCheckableMemoryResourceFromExpHeap::do_deallocate(void* p, std::size_t bytes, std::size_t alignment) NN_NOEXCEPT
    {
        NN_UNUSED(bytes);
        NN_UNUSED(alignment);

        std::lock_guard<os::Mutex> scopedLock(m_Mutex);

        OnDeallocate(p, bytes);
        nn::lmem::FreeToExpHeap(m_HeapHandle, p);
    }

}}
