﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <mutex>
#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/nn_Abort.h>
#include <nn/nn_SdkLog.h>
#include <nn/fssystem/fs_AsynchronousAccess.h>
#include <nn/fssrv/fssrv_MemoryReport.h>
#include <nn/fssrv/fssrv_MemoryResourceFromExpHeap.h>
#include <nn/fssrv/fssrv_MemoryResourceFromStandardAllocator.h>
#include "../fssrv/detail/fssrv_SdmmcStorageService.h"

#include <nn/fs/detail/fs_ResultHandlingUtility.h>

namespace nn { namespace fssrv {

namespace
{

MemoryReportCreator* g_pMemoryReportCreator = nullptr;

}

MemoryReportCreator::MemoryReportCreator(
    MemoryReport* pBufferManagerMemoryReport,
    MemoryReport* pExpHeapMemoryReport,
    MemoryReport* pBufferPoolMemoryReport
) NN_NOEXCEPT
    : m_pBufferManagerMemoryReport(pBufferManagerMemoryReport)
    , m_pExpHeapMemoryReport(pExpHeapMemoryReport)
    , m_pBufferPoolMemoryReport(pBufferPoolMemoryReport)
    , m_Mutex(true, 0)
{
}

MemoryReportCreator::~MemoryReportCreator() NN_NOEXCEPT
{
}

nn::Result MemoryReportCreator::GetMemoryReportInfo(nn::fs::MemoryReportInfo* pOutMemoryReportInfo) NN_NOEXCEPT
{
    NN_FS_RESULT_THROW_UNLESS(pOutMemoryReportInfo != nullptr, nn::fs::ResultNullptrArgument());

    std::lock_guard<nn::os::Mutex> scopedLock(m_Mutex);

    std::memset(pOutMemoryReportInfo, 0x0, sizeof(nn::fs::MemoryReportInfo));

    pOutMemoryReportInfo->pooledBufferPeakFreeSize = nn::fssystem::GetPooledBufferFreeSizePeak();
    pOutMemoryReportInfo->pooledBufferRetriedCount = nn::fssystem::GetPooledBufferRetriedCount();
    pOutMemoryReportInfo->pooledBufferReduceAllocationCount = nn::fssystem::GetPooledBufferReduceAllocationCount();
    nn::fssystem::ClearPooledBufferPeak();

    if( m_pBufferManagerMemoryReport != nullptr )
    {
        pOutMemoryReportInfo->bufferManagerPeakFreeSize = m_pBufferManagerMemoryReport->GetFreeSizePeak();
        pOutMemoryReportInfo->bufferManagerPeakTotalAllocatableSize = m_pBufferManagerMemoryReport->GetTotalAllocatableSizePeak();
        pOutMemoryReportInfo->bufferManagerRetriedCount = m_pBufferManagerMemoryReport->GetRetriedCount();
        m_pBufferManagerMemoryReport->Clear();
    }

    if( m_pExpHeapMemoryReport != nullptr )
    {
        pOutMemoryReportInfo->expHeapPeakFreeSize = m_pExpHeapMemoryReport->GetFreeSizePeak();
        m_pExpHeapMemoryReport->Clear();
    }

    if( m_pBufferPoolMemoryReport != nullptr )
    {
        pOutMemoryReportInfo->bufferPoolPeakFreeSize = m_pBufferPoolMemoryReport->GetFreeSizePeak();
        m_pBufferPoolMemoryReport->Clear();
    }

    nn::fssrv::detail::GetAndClearPatrolReadAllocateBufferCount(
        &pOutMemoryReportInfo->patrolReadAllocateBufferSuccessCount,
        &pOutMemoryReportInfo->patrolReadAllocateBufferFailureCount
    );

    NN_RESULT_SUCCESS;
}

void SetMemoryReportCreator(MemoryReportCreator* pMemoryReportCreator) NN_NOEXCEPT
{
    g_pMemoryReportCreator = pMemoryReportCreator;
}

nn::Result GetMemoryReportInfo(nn::fs::MemoryReportInfo* pOutMemoryReportInfo) NN_NOEXCEPT
{
    if( g_pMemoryReportCreator == nullptr )
    {
        return nn::fs::ResultPreconditionViolation();
    }
    return g_pMemoryReportCreator->GetMemoryReportInfo(pOutMemoryReportInfo);
}

}}
