﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <new>
#include <type_traits>
#include <mutex>
#include <nn/nn_Common.h>
#include <nn/nn_SdkLog.h>
#include <nn/nn_SdkAssert.h>
#include <nn/fs/detail/fs_ServerName.h>
#include <nn/fssrv/fssrv_DeferredProcessManager.h>
#include <nn/fssrv/fssrv_FileSystemProxyServer.h>
#include <nn/fssrv/fssrv_RequestHook.h>
#include <nn/fssrv/detail/fssrv_FileSystemProxyServiceObject.h>
#include <nn/fssrv/sf/fssrv_IFileSystemProxy.h>
#include <nn/os/os_Semaphore.h>
#include <nn/os/os_Mutex.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/sf/sf_HipcServer.h>
#include <nn/sf/sf_HipcServerSessionManagerHandler.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/psc.h>
#if (defined(NN_BUILD_CONFIG_HARDWARE_NX))
    #include <nn/pcv/pcv.h>
    #include <nn/pcv/pcv_Arbitration.h>
#endif

#include <nn/settings/factory/settings_GameCard.h>
#include <nn/fssrv/fssrv_FileSystemProxyImpl.h>
#include <nn/fssystem/fs_AsynchronousAccess.h>
#include <nn/fssystem/fs_ServiceContext.h>
#include "detail/fssrv_AccessLogSdCardWriter.h"
#include "detail/fssrv_LocationResolverSet.h"
#include "detail/fssrv_SdmmcStorageService.h"
#include "detail/fssrv_GameCardManager.h"
#include "detail/fssrv_DeviceBuffer.h"
#include "detail/fssrv_Trace.h"
#include "detail/fssrv_FileSystemProxyCoreImpl.h"

namespace nn { namespace fssrv {

    namespace {

        const int32_t MultiWaitWaitableObjectCountMax = 64;
        const int32_t MultiWaitWaitableObjectCountUsedByServiceFramework = 2;
        // DeferredProcessManager::QueueDeferredProcessForPriority で1つ使用
        const int32_t MultiWaitWaitableObjectCountUsedByFileSystemProxyServerCount = 1;

        const int32_t FileSystemProxyServerSessionCountMax =
            MultiWaitWaitableObjectCountMax
            - MultiWaitWaitableObjectCountUsedByServiceFramework
            - MultiWaitWaitableObjectCountUsedByFileSystemProxyServerCount;
        const int32_t FileSystemProxyServerPortCountMax = 3;

        enum PortIndex
        {
            PortIndex_FileSystemProxy = 0,
            PortIndex_ProgramRegistry,
            PortIndex_FileSystemProxyForLoader,
        };

        nn::os::SemaphoreType g_SemaphoreForFileSystemProxyForLoader;
        nn::os::SemaphoreType g_SemaphoreForProgramRegistry;

        struct FileSystemProxyServerOption
        {
            static const size_t PointerTransferBufferSize = 2048;

            // FileSystemProxyServerSessionCountMax 以下、FS の利用プロセス数以上であればよい。
            static const int    SubDomainCountMax         = 64; // TODO: カーネルのセッション数上限が解放されたらもう少し増やす

            // 実際に開かれる File, Directory, FileSystem の合計数より十分大きな数
            static const int    ObjectInSubDomainCountMax = 16384;

            static const bool CanDeferInvokeRequest = true;
        };
        NN_STATIC_ASSERT(std::is_pod<FileSystemProxyServerOption>::value);

        class FileSystemProxyServerManager : public nn::sf::HipcSimpleAllInOneServerManager<FileSystemProxyServerSessionCountMax, FileSystemProxyServerPortCountMax, FileSystemProxyServerOption>
        {
        private:
            virtual Result OnNeedsToAccept(int portIndex, PortForAllInOne* pPort) NN_NOEXCEPT NN_OVERRIDE
            {
                switch (portIndex)
                {
                case PortIndex_FileSystemProxy:
                    return AcceptImpl(pPort, detail::GetFileSystemProxyServiceObject());
                case PortIndex_FileSystemProxyForLoader:
                {
                    if (nn::os::TryAcquireSemaphore(&g_SemaphoreForFileSystemProxyForLoader))
                    {
                        bool isSuccess = false;
                        NN_UTIL_SCOPE_EXIT
                        {
                            if (!isSuccess)
                            {
                                nn::os::ReleaseSemaphore(&g_SemaphoreForFileSystemProxyForLoader);
                            }
                        };
                        NN_RESULT_DO(AcceptImpl(pPort, detail::GetFileSystemProxyForLoaderServiceObject()));
                        isSuccess = true;
                        NN_RESULT_SUCCESS;
                    }
                    else
                    {
                        return AcceptImpl(pPort, detail::GetInvalidFileSystemProxyForLoaderServiceObject());
                    }
                }
                case PortIndex_ProgramRegistry:
                {
                    if (nn::os::TryAcquireSemaphore(&g_SemaphoreForProgramRegistry))
                    {
                        bool isSuccess = false;
                        NN_UTIL_SCOPE_EXIT
                        {
                            if (!isSuccess)
                            {
                                nn::os::ReleaseSemaphore(&g_SemaphoreForProgramRegistry);
                            }
                        };
                        NN_RESULT_DO(AcceptImpl(pPort, detail::GetProgramRegistryServiceObject()));
                        isSuccess = true;
                        NN_RESULT_SUCCESS;
                    }
                    else
                    {
                        return AcceptImpl(pPort, detail::GetInvalidProgramRegistryServiceObject());
                    }
                }
                default:
                    NN_UNEXPECTED_DEFAULT;
                }
            }
        };

        std::aligned_storage<sizeof(FileSystemProxyServerManager), NN_ALIGNOF(FileSystemProxyServerManager)>::type g_FileSystemProxyServerManagerStorage;
        FileSystemProxyServerManager* g_FileSystemProxyServerManager;

        nn::psc::PmModule g_PmModule;

        nn::os::BarrierType g_ServerLoopBarrier;
        nn::os::EventType g_ResumeWait;
        bool g_IsSuspended = false;

        DeferredProcessManager<FileSystemProxyServerManager, NotifyProcessDeferred> g_DeferredProcessManager;
        FileSystemProxyServerSessionResourceManager g_SessionResourceManager(g_DeferredProcessManager.GetDeferredProcessQueueForPriority());
        RequestHook g_RequestHook(&g_SessionResourceManager);

        // IPC受付を再開
        void ResumeFileSystemProxyServer() NN_NOEXCEPT
        {
            if (!g_IsSuspended)
            {
                // Resume 済みならば何もしない
                return;
            }

            g_IsSuspended = false;

            g_FileSystemProxyServerManager->Start();

            // IPC受付スレッドを起床させる
            nn::os::SignalEvent(&g_ResumeWait);

            // IPC受付の再開を待つ
            nn::os::AwaitBarrier(&g_ServerLoopBarrier);
        }

        // IPC受付を停止
        void SuspendFileSystemProxyServer() NN_NOEXCEPT
        {
            if (g_IsSuspended)
            {
                // Suspend 済ならば何もしない
                return;
            }

            g_IsSuspended = true;

            // IPC受付スレッドを起床できなくする
            nn::os::ClearEvent(&g_ResumeWait);

            g_FileSystemProxyServerManager->RequestStop();

            // IPC受付の停止を待つ
            nn::os::AwaitBarrier(&g_ServerLoopBarrier);
        }
    }

    void InitializeFileSystemProxyServer(int countThreads) NN_NOEXCEPT
    {
        // この関数でエラーが発生した場合は ABORT します。(失敗したらライブラリ自体が機能しなくなるため)

        nn::sf::InitializeForHipcServerSessionManagerHandler();

        // gc 向けに settings が終了したことを知らせるためのイベントを初期化
        detail::InitializeGameCardManager();

        // デバイスアドレス空間にマップしたバッファでバッファプールを初期化
        detail::InitializeDeviceBuffer();
        NN_ABORT_UNLESS_RESULT_SUCCESS(fssystem::InitializeBufferPool(
            reinterpret_cast<char*>(detail::GetDeviceBuffer()),
            detail::DeviceBufferSize));

        // メインスレッドとの待ち合わせに使用するため、待ち合わせ数は ( IPC受付スレッド数 + 1 )
        nn::os::InitializeBarrier(&g_ServerLoopBarrier, countThreads + 1);

        nn::os::InitializeEvent(&g_ResumeWait, false, nn::os::EventClearMode_ManualClear);

        g_IsSuspended = false;

        nn::os::InitializeSemaphore(&g_SemaphoreForFileSystemProxyForLoader, 1, 1);
        nn::os::InitializeSemaphore(&g_SemaphoreForProgramRegistry, 1, 1);

        g_DeferredProcessManager.Initialize();

        NN_SDK_ASSERT(!g_FileSystemProxyServerManager);

        g_FileSystemProxyServerManager = new (&g_FileSystemProxyServerManagerStorage) FileSystemProxyServerManager;
        g_FileSystemProxyServerManager->SetManagerHandler(&g_RequestHook);

        int32_t sessionCountMax = FileSystemProxyServerSessionCountMax;
        int32_t sessionCountForFileSystemProxyForLoader = 1;
        int32_t sessionCountForProgramRegistry = 1;

        NN_ABORT_UNLESS_RESULT_SUCCESS(g_FileSystemProxyServerManager->InitializePort(PortIndex_FileSystemProxy, sessionCountMax - sessionCountForFileSystemProxyForLoader - sessionCountForProgramRegistry, nn::fs::detail::FileSystemProxyServiceName));
        NN_ABORT_UNLESS_RESULT_SUCCESS(g_FileSystemProxyServerManager->InitializePort(PortIndex_FileSystemProxyForLoader, sessionCountForFileSystemProxyForLoader, nn::fs::detail::FileSystemProxyForLoaderServiceName));
        NN_ABORT_UNLESS_RESULT_SUCCESS(g_FileSystemProxyServerManager->InitializePort(PortIndex_ProgramRegistry, sessionCountForProgramRegistry, nn::fs::detail::ProgramRegistryServiceName));

        g_FileSystemProxyServerManager->Start();
    }

    void LoopFileSystemProxyServer() NN_NOEXCEPT
    {
        NN_SDK_ASSERT(g_FileSystemProxyServerManager);

        RequestHookContext requestHookContext;
        {
            auto* pContext = fssystem::GetServiceContext();
            if( pContext != nullptr )
            {
                pContext->SetRequestHookContext(&requestHookContext);
            }
        }

        while (NN_STATIC_CONDITION(true))
        {
            while (auto p = g_FileSystemProxyServerManager->Wait())
            {
                fssystem::GetServiceContext()->ResetDeferredProcessContext(false);

                switch (nn::os::GetMultiWaitHolderUserData(p))
                {
                case FileSystemProxyServerManager::InvokeTag:
                {
                    auto result = g_FileSystemProxyServerManager->ProcessInvokeRequestWithDefer(p);
                    if (nn::sf::ResultProcessDeferred::Includes(result))
                    {
                        g_DeferredProcessManager.QueueDeferredProcess(g_FileSystemProxyServerManager, p);
                        break;
                    }
                    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
                    break;
                }
                case FileSystemProxyServerManager::AcceptTag:
                {
                    NN_ABORT_UNLESS_RESULT_SUCCESS(g_FileSystemProxyServerManager->ProcessAcceptRequest(p));
                    break;
                }
                case g_DeferredProcessManager.InvokeTag:
                {
                    g_DeferredProcessManager.ClearInvokeDeferredProcessEvent();
                    break;
                }
                default:
                    NN_UNEXPECTED_DEFAULT;
                }

                // 延期されたオブジェクトのうち実行可能なものを実行する
                // (実行できないものはキューに残す)
                g_DeferredProcessManager.InvokeDeferredProcess(g_FileSystemProxyServerManager, &g_SessionResourceManager);
            }

            // スリープ準備完了
            nn::os::AwaitBarrier(&g_ServerLoopBarrier);

            // 起床指示を待つ
            nn::os::WaitEvent(&g_ResumeWait);

            // 起床完了
            nn::os::AwaitBarrier(&g_ServerLoopBarrier);
        }
    }

    void WaitSettingsAndPresetGcInternalKeys() NN_NOEXCEPT
    {
        static const size_t GcCalibKeySize = 256;
        static const size_t GcCalibIvSize = 16;

        bool isSettingsReadSuccess = false;
        nn::settings::factory::GameCardKey gcKey;
        nn::settings::factory::GameCardCertificate gcCert;

        do {
            // この処理は settings サービスが利用できるようになるまで待たされる
            nn::Result result = nn::settings::factory::GetGameCardKey(&gcKey);
            if(result.IsFailure())
            {
                NN_SDK_LOG("[fs] Error: GetGameCardKey Failure (Module:%d, Description:%d)\n", result.GetModule(), result.GetDescription());
                break;
            }
            result = nn::settings::factory::GetGameCardCertificate(&gcCert);
            if(result.IsFailure())
            {
                NN_SDK_LOG("[fs] Error: GetGameCardCertificate Failure (Module:%d, Description:%d)\n", result.GetModule(), result.GetDescription());
                break;
            }
            if(gcKey.size < (GcCalibIvSize + GcCalibKeySize))
            {
                NN_SDK_LOG("[fs] Error: GetGameCardKey: too small key size\n");
                break;
            }
            isSettingsReadSuccess = true;

        } while (NN_STATIC_CONDITION(false));

        if(isSettingsReadSuccess)
        {
            detail::PresetInternalKeys(gcKey.data, gcKey.size, gcCert.data, sizeof(gcCert.data));
        }
        else
        {
            detail::PresetInternalKeys(nullptr, 0, nullptr, 0);
        }
    }

    void WaitPcvAndSwitchToPcvClockResetControl()
    {
        #if (defined(NN_BUILD_CONFIG_HARDWARE_NX))

            // この処理は pcv の調停サービスが利用できるようになるまで待たされる
            nn::pcv::InitializeForArbitration();

            // IPC受付を停止
            SuspendFileSystemProxyServer();

            // MMC、SD カード、ゲームカードともスリープ制御を利用して停止
            detail::PutGameCardToSleep();
            nn::pcv::ReleaseControl(nn::pcv::Module_Sdmmc2);

            detail::PutSdCardToSleep();
            nn::pcv::ReleaseControl(nn::pcv::Module_Sdmmc1);

            detail::PutMmcToSleep();
            nn::pcv::ReleaseControl(nn::pcv::Module_Sdmmc4);

            // 調停が完了し pcv が利用できるようになるまで待たされる
            nn::pcv::Initialize();

            // pcv 経由のクロック制御に切り替え
            detail::SwitchToPcvClockResetControl();

            // MMC、SD カード、ゲームカードとも復帰
            detail::AwakenMmc();
            detail::AwakenSdCard();
            detail::AwakenGameCard();

            // IPC受付を再開
            ResumeFileSystemProxyServer();
        #endif
    }

    void LoopPmEventServer(void(*pCallbackOnWake)()) NN_NOEXCEPT
    {
        // この処理は psc サービスが利用できるようになるまで待たされる。
        // イベントを逃さないよう、早めに実行する。
        nn::psc::PmModuleId dependencyList[] = {
            nn::psc::PmModuleId_Pinmux,
            nn::psc::PmModuleId_Gpio,
            nn::psc::PmModuleId_PcvClock,
            nn::psc::PmModuleId_PcvVoltage,
            nn::psc::PmModuleId_TmHostIo
        };
        NN_ABORT_UNLESS_RESULT_SUCCESS(g_PmModule.Initialize(nn::psc::PmModuleId_Fs,
            dependencyList, sizeof(dependencyList) / sizeof(dependencyList[0]), nn::os::EventClearMode_ManualClear));

        nn::os::SystemEventType* pPmSystemEvent = g_PmModule.GetEventPointer()->GetBase();

        // sdmmc 向け pcv 対応
        WaitPcvAndSwitchToPcvClockResetControl();

        // この時点で SD カードが利用可能
        GetFileSystemProxyCoreImpl()->SetSdCardPortReady();

        while (NN_STATIC_CONDITION(true))
        {
            nn::os::WaitSystemEvent(pPmSystemEvent);
            nn::os::ClearSystemEvent(pPmSystemEvent);

            nn::psc::PmState pmState;
            nn::psc::PmFlagSet pmFlagSet;
            NN_ABORT_UNLESS_RESULT_SUCCESS(g_PmModule.GetRequest(&pmState, &pmFlagSet));

            switch (pmState)
            {
            case nn::psc::PmState_MinimumAwake:
                pCallbackOnWake();

                // MMC、SD カード、ゲームカードとも復帰
                detail::AwakenMmc();
                detail::AwakenSdCard();
                detail::AwakenGameCard();

                // IPC受付を再開
                ResumeFileSystemProxyServer();
                break;

            case nn::psc::PmState_SleepReady:
                // IPC受付を停止
                SuspendFileSystemProxyServer();

                detail::AccessLogSdCardWriter::Flush();
                NN_FS_SCOPED_TRACE_FLUSH;

                FlushFatCache();

                // MMC、SD カード、ゲームカードともスリープ
                detail::PutGameCardToSleep();
                detail::PutSdCardToSleep();
                detail::PutMmcToSleep();
                break;

            case nn::psc::PmState_ShutdownReady:
                // IPC受付を停止
                SuspendFileSystemProxyServer();

                detail::AccessLogSdCardWriter::Finalize();
                NN_FS_SCOPED_TRACE_FLUSH;

                FlushFatCache();

                // MMC、SD カード、ゲームカードとも停止（再開不可）
                detail::ShutdownGameCard();
                detail::ShutdownSdCard();
                detail::ShutdownMmc();
                break;

            default:
                break;
            }

            NN_ABORT_UNLESS_RESULT_SUCCESS(g_PmModule.Acknowledge(pmState, nn::ResultSuccess()));
        }

        // 現状のところ、ソフト上の終了処理は行わない
    }

}}
