﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_SdkLog.h>
#include <nn/fs/fs_ResultPrivate.h>
#include <nn/fssrv/fssrv_DeferredProcessQueue.h>
#include <nn/os/os_MultipleWaitTypes.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_Optional.h>

namespace nn { namespace fssrv {

    DeferredProcessQueueForDeviceError::DeferredProcessQueueForDeviceError() NN_NOEXCEPT : m_Mutex(false)
    {
    }

    DeferredProcessQueueForDeviceError::~DeferredProcessQueueForDeviceError() NN_NOEXCEPT
    {
        Finalize();
    }

    void DeferredProcessQueueForDeviceError::Finalize() NN_NOEXCEPT
    {
        while( !m_List.empty() )
        {
            auto& entry = m_List.front();
            m_List.pop_front();
            delete &entry;
        }
    }

    Result DeferredProcessQueueForDeviceError::Push(const DeferredProcessEntryForDeviceError& entry) NN_NOEXCEPT
    {
        auto pNewEntry = new DeferredProcessEntryForDeviceError();
        NN_RESULT_THROW_UNLESS(pNewEntry, nn::fs::ResultAllocationMemoryFailedNew());
        {
            pNewEntry->pProcessHolder = entry.pProcessHolder;
            pNewEntry->processId = entry.processId;
        }

        std::lock_guard<os::Mutex> scopedLock(m_Mutex);
        m_List.push_back(*pNewEntry);

        NN_RESULT_SUCCESS;
    }

    void DeferredProcessQueueForDeviceError::Pop() NN_NOEXCEPT
    {
        std::lock_guard<os::Mutex> scopedLock(m_Mutex);
        auto& entry = m_List.front();
        delete &entry;
        m_List.pop_front();
    }

    void DeferredProcessQueueForDeviceError::Pop(Bit64 processId) NN_NOEXCEPT
    {
        std::lock_guard<os::Mutex> scopedLock(m_Mutex);
        for (auto& entry : m_List)
        {
            if (entry.processId == processId)
            {
                m_List.erase(m_List.iterator_to(entry));
                delete &entry;
                return;
            }
        }
    }

    const DeferredProcessEntryForDeviceError& DeferredProcessQueueForDeviceError::Front() NN_NOEXCEPT
    {
        std::lock_guard<os::Mutex> scopedLock(m_Mutex);
        return m_List.front();
    }

    const DeferredProcessEntryForDeviceError& DeferredProcessQueueForDeviceError::Front(Bit64 processId) NN_NOEXCEPT
    {
        std::lock_guard<os::Mutex> scopedLock(m_Mutex);
        for (auto& entry : m_List)
        {
            if (entry.processId == processId)
            {
                return entry;
            }
        }
        NN_ABORT("Queue must be not empty.");
    }

    bool DeferredProcessQueueForDeviceError::IsEmpty() NN_NOEXCEPT
    {
        std::lock_guard<os::Mutex> scopedLock(m_Mutex);
        return m_List.empty();
    }

    bool DeferredProcessQueueForDeviceError::IsEmpty(Bit64 processId) NN_NOEXCEPT
    {
        std::lock_guard<os::Mutex> scopedLock(m_Mutex);
        for (auto& entry : m_List)
        {
            if (entry.processId == processId)
            {
                return false;
            }
        }
        return true;
    }

    size_t DeferredProcessQueueForDeviceError::Count() NN_NOEXCEPT
    {
        std::lock_guard<os::Mutex> scopedLock(m_Mutex);
        return static_cast<size_t>(m_List.size());
    }

    size_t DeferredProcessQueueForDeviceError::Count(Bit64 processId) NN_NOEXCEPT
    {
        std::lock_guard<os::Mutex> scopedLock(m_Mutex);
        size_t count = 0;
        for (auto& entry : m_List)
        {
            if (entry.processId == processId)
            {
                count++;
            }
        }
        return count;
    }

    NN_DEFINE_STATIC_CONSTANT(const size_t DeferredProcessQueueForPriority::WaitConditionCount);

    DeferredProcessQueueForPriority::DeferredProcessQueueForPriority() NN_NOEXCEPT
        : m_Mutex(true)
    {
    }

    DeferredProcessQueueForPriority::~DeferredProcessQueueForPriority() NN_NOEXCEPT
    {
        Finalize();
    }

    void DeferredProcessQueueForPriority::Finalize() NN_NOEXCEPT
    {
        while( !m_List.empty() )
        {
            auto& entry = m_List.front();
            m_List.pop_front();
            delete &entry;
        }
    }

    Result DeferredProcessQueueForPriority::Push(os::MultiWaitHolderType* pProcessHolder, FileSystemProxyServerSessionType sessionType) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pProcessHolder);
        NN_SDK_REQUIRES(
            sessionType == FileSystemProxyServerSessionType::Realtime
            || sessionType == FileSystemProxyServerSessionType::Other
            || sessionType == FileSystemProxyServerSessionType::Background
        );

        auto pNewEntry = new DeferredProcessEntryForPriority(pProcessHolder, sessionType);
        NN_RESULT_THROW_UNLESS(pNewEntry, nn::fs::ResultAllocationMemoryFailedNew());

        {
            std::lock_guard<os::Mutex> scopedLock(m_Mutex);
            m_List.push_back(*pNewEntry);
        }

        NN_RESULT_SUCCESS;
    }

    bool DeferredProcessQueueForPriority::Pop(os::MultiWaitHolderType** outValue, FileSystemProxyServerSessionResourceManager* pSessionResourceManager) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(outValue);
        NN_SDK_REQUIRES_NOT_NULL(pSessionResourceManager);

        std::lock_guard<os::Mutex> scopedLock(m_Mutex);
        for (auto& entry : m_List)
        {
            if (pSessionResourceManager->Acquire(entry.GetSessionType()))
            {
                *outValue = entry.GetProcessHolder();
                m_List.erase(m_List.iterator_to(entry));
                delete &entry;

                m_ConditionalVariable.Signal();

                return true;
            }
        }
        return false;
    }

    bool DeferredProcessQueueForPriority::HasAny(FileSystemProxyServerSessionType sessionType) NN_NOEXCEPT
    {
        std::lock_guard<os::Mutex> scopedLock(m_Mutex);
        for( auto& entry : m_List )
        {
            if( entry.GetSessionType() == sessionType )
            {
                return true;
            }
        }
        return false;
    }

    size_t DeferredProcessQueueForPriority::Count() NN_NOEXCEPT
    {
        std::lock_guard<os::Mutex> scopedLock(m_Mutex);
        return static_cast<size_t>(m_List.size());
    }

    void DeferredProcessQueueForPriority::Wait() NN_NOEXCEPT
    {
        std::lock_guard<os::Mutex> scopedLock(m_Mutex);
        while( static_cast<size_t>(m_List.size()) > WaitConditionCount )
        {
            m_ConditionalVariable.Wait(m_Mutex);
        }
    }

}}
