﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <memory>

#include <nn/nn_Abort.h>
#include <nn/nn_Allocator.h>
#include <nn/fs/fs_Result.h>
#include <nn/fssystem/fs_AllocatorUtility.h>
#include <nn/fs/fs_IStorage.h>
#include <nn/result/result_HandlingUtility.h>

#if defined(NN_BUILD_CONFIG_HARDWARE_JETSONTK1) || defined(NN_BUILD_CONFIG_HARDWARE_JETSONTK2) || defined(NN_BUILD_CONFIG_HARDWARE_NX) || defined(NN_BUILD_CONFIG_HARDWARE_JETSONTX2)
#include <nn/fat/fat_FatFileSystem.h>
#include <nn/fssystem/fs_ConcatenationFileSystem.h>
#endif

#include <nn/fssystem/fs_WriteThroughCacheStorage.h>
#include <nn/fssrv/fscreator/fssrv_FatFileSystemCreator.h>

namespace nn { namespace fssrv { namespace fscreator {

    namespace {
#if defined(NN_BUILD_CONFIG_HARDWARE_JETSONTK1) || defined(NN_BUILD_CONFIG_HARDWARE_JETSONTK2) || defined(NN_BUILD_CONFIG_HARDWARE_NX) || defined(NN_BUILD_CONFIG_HARDWARE_JETSONTX2)
        class FatFileSystemWithBuffer : public nn::fat::FatFileSystem
        {
        public:
            explicit FatFileSystemWithBuffer(nn::MemoryResource* pAllocator) NN_NOEXCEPT
              : m_CacheBuffer(nullptr)
            {
                m_Allocator = pAllocator;
            }
            virtual ~FatFileSystemWithBuffer() NN_NOEXCEPT NN_OVERRIDE
            {
                FatFileSystem::Finalize();
                m_WriteThroughCacheStorage.Finalize();
                if (m_CacheBuffer != nullptr)
                {
                    m_Allocator->deallocate(m_CacheBuffer, m_CacheBufferSize);
                }
            }
            Result Initialize(std::shared_ptr<fs::IStorage> storage, nn::fat::FatAttribute* pFatAttribute, std::unique_ptr<nn::fat::FatErrorInfoSetter>&& pFatErrorInfoSetter, nn::Result resultInvalidFatFormat, nn::Result resultUsableSpaceNotEnough) NN_NOEXCEPT
            {
                size_t cacheBufferSize = nn::fat::FatFileSystem::GetCacheBufferSize();
                m_CacheBuffer = m_Allocator->allocate(cacheBufferSize);
                NN_RESULT_THROW_UNLESS(m_CacheBuffer, nn::fs::ResultAllocationMemoryFailedInFatFileSystemCreatorA());
                m_CacheBufferSize = cacheBufferSize;

                // ライトスルーキャッシュを挿入する
                {
                    m_BaseStorage = std::move(storage);
                    int64_t sizeStorage;
                    NN_RESULT_DO(m_BaseStorage->GetSize(&sizeStorage));
                    NN_RESULT_DO(
                        m_WriteThroughCacheStorage.Initialize(
                            m_Allocator,
                            nn::fs::SubStorage(m_BaseStorage.get(), 0, sizeStorage),
                            BufferCacheSize,
                            BufferCacheCount
                        )
                    );

                    NN_RESULT_DO(
                        FatFileSystem::Initialize(
                            &m_WriteThroughCacheStorage,
                            m_CacheBuffer,
                            cacheBufferSize,
                            pFatAttribute,
                            std::move(pFatErrorInfoSetter),
                            resultInvalidFatFormat,
                            resultUsableSpaceNotEnough
                        )
                    );
                }

                NN_RESULT_SUCCESS;
            }
        private:
            static const int BufferCacheSize = 16 * 1024;
            static const int BufferCacheCount = 32;

            nn::MemoryResource* m_Allocator;
            void* m_CacheBuffer;
            size_t m_CacheBufferSize;

            std::shared_ptr<fs::IStorage> m_BaseStorage;
            nn::fssystem::WriteThroughCacheStorage m_WriteThroughCacheStorage;
        };
#endif
    }

    FatFileSystemCreator::FatFileSystemCreator(nn::MemoryResource* pAllocator) NN_NOEXCEPT
    {
        m_Allocator = pAllocator;
#if defined(NN_BUILD_CONFIG_HARDWARE_JETSONTK1) || defined(NN_BUILD_CONFIG_HARDWARE_JETSONTK2) || defined(NN_BUILD_CONFIG_HARDWARE_NX) || defined(NN_BUILD_CONFIG_HARDWARE_JETSONTX2)
        nn::fat::FatFileSystem::SetAllocatorForFat(m_Allocator);
#endif
    }

    Result FatFileSystemCreator::Create(std::shared_ptr<nn::fs::fsa::IFileSystem>* outValue, std::shared_ptr<fs::IStorage> storage, nn::fat::FatAttribute* pFatAttribute, std::unique_ptr<nn::fat::FatErrorInfoSetter>&& pFatErrorInfoSetter, nn::Result resultInvalidFatFormat, nn::Result resultUsableSpaceNotEnough) NN_NOEXCEPT
    {
#if defined(NN_BUILD_CONFIG_HARDWARE_JETSONTK1) || defined(NN_BUILD_CONFIG_HARDWARE_JETSONTK2) || defined(NN_BUILD_CONFIG_HARDWARE_NX) || defined(NN_BUILD_CONFIG_HARDWARE_JETSONTX2)
        std::unique_ptr<FatFileSystemWithBuffer> pFatFs(new FatFileSystemWithBuffer(m_Allocator));
        NN_RESULT_THROW_UNLESS(pFatFs != nullptr, nn::fs::ResultAllocationMemoryFailedInFatFileSystemCreatorB());
        NN_RESULT_DO(pFatFs->Initialize(std::move(storage), pFatAttribute, std::move(pFatErrorInfoSetter), resultInvalidFatFormat, resultUsableSpaceNotEnough));
        NN_RESULT_DO(pFatFs->Mount());

        std::shared_ptr<nn::fssystem::ConcatenationFileSystem> pFs = fssystem::AllocateShared<fssystem::ConcatenationFileSystem>(std::move(pFatFs));
        NN_RESULT_THROW_UNLESS(pFs != nullptr, nn::fs::ResultAllocationMemoryFailedInFatFileSystemCreatorC());

        *outValue = std::move(pFs);
        NN_RESULT_SUCCESS;
#else
        NN_UNUSED(outValue);
        NN_UNUSED(storage);
        NN_UNUSED(pFatAttribute);
        NN_UNUSED(pFatErrorInfoSetter);
        NN_UNUSED(resultInvalidFatFormat);
        NN_UNUSED(resultUsableSpaceNotEnough);
        return nn::fs::ResultNotImplemented();
#endif
    }

    Result FatFileSystemCreator::Format(std::shared_ptr<fs::IStorage> storage, nn::fat::FatAttribute* pFatAttribute, nn::fat::FatFormatParam* pFatFormatParam, std::unique_ptr<nn::fat::FatErrorInfoSetter>&& pFatErrorInfoSetter, nn::Result resultInvalidFatFormat, nn::Result resultUsableSpaceNotEnough) NN_NOEXCEPT
    {
#if (defined(NN_BUILD_CONFIG_HARDWARE_JETSONTK1) || defined(NN_BUILD_CONFIG_HARDWARE_JETSONTK2) || defined(NN_BUILD_CONFIG_HARDWARE_NX)) || defined(NN_BUILD_CONFIG_HARDWARE_JETSONTX2)
        std::unique_ptr<FatFileSystemWithBuffer> pFatFs(new FatFileSystemWithBuffer(m_Allocator));
        NN_RESULT_THROW_UNLESS(pFatFs != nullptr, nn::fs::ResultAllocationMemoryFailedInFatFileSystemCreatorB());
        NN_RESULT_DO(pFatFs->Initialize(std::move(storage), pFatAttribute, std::move(pFatErrorInfoSetter), resultInvalidFatFormat, resultUsableSpaceNotEnough));
        NN_RESULT_DO(pFatFs->Format(pFatFormatParam));
        NN_RESULT_SUCCESS;
#else
        NN_UNUSED(storage);
        NN_UNUSED(pFatAttribute);
        NN_UNUSED(pFatFormatParam);
        NN_UNUSED(pFatErrorInfoSetter);
        NN_UNUSED(resultInvalidFatFormat);
        NN_UNUSED(resultUsableSpaceNotEnough);
        return nn::fs::ResultNotImplemented();
#endif
    }

}}}
