﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <memory>
#include <mutex>

#include <nn/nn_Common.h>
#include <nn/nn_SdkLog.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/fs/fs_Bis.h>
#include <nn/fs/fs_Result.h>
#include <nn/fs/fsa/fs_IFileSystem.h>
#include <nn/fssrv/fscreator/fssrv_BuiltInStorageFileSystemCreatorHostFs.h>
#include <nn/fssrv/detail/fssrv_BisPartitionRootPathForHost.h>
#include "../detail/fssrv_Utility.h"

#include <nn/fssystem/fs_AllocatorUtility.h>
#include <nn/fssystem/fs_Utility.h>
#include <nn/fssystem/fs_SubdirectoryFileSystem.h>


namespace nn { namespace fssrv { namespace fscreator {

bool BuiltInStorageFileSystemCreatorHostFs::IsSignedSystemPartitionOnSdCardValid() NN_NOEXCEPT
{
    return false;
}

void BuiltInStorageFileSystemCreatorHostFs::SetSdCardPortReady() NN_NOEXCEPT
{
}


void BuiltInStorageFileSystemCreatorHostFs::Initialize(fs::FileSystemProxyErrorInfo* pErrorInfo, os::Mutex* pErrorInfoMutex) NN_NOEXCEPT
{
    NN_UNUSED(pErrorInfo);
    NN_UNUSED(pErrorInfoMutex);
}

Result BuiltInStorageFileSystemCreatorHostFs::Create(std::shared_ptr<fs::fsa::IFileSystem>* pOutFileSystem, const char* rootPath, nn::fs::BisPartitionId id) NN_NOEXCEPT
{
    std::shared_ptr<nn::fs::fsa::IFileSystem> pFileSystem;
    NN_FSP_REQUIRES(detail::BisPartitionRootPathForHost::IsValid(id), nn::fs::ResultInvalidArgument());

    NN_UNUSED(rootPath);
    if (id == nn::fs::BisPartitionId::SystemProperPartition)
    {
        // 実機ノーマル起動時に合わせて InvalidArgument
        return fs::ResultInvalidArgument();
    }
    else if (m_BisRootPath.IsEmulated(id))
    {
        NN_RESULT_DO(m_pHostFsCreator->Create(pOutFileSystem, m_BisRootPath.Get(id)));
    }
    else
    {
        std::shared_ptr<nn::fs::fsa::IFileSystem> pBaseFileSystem;
        NN_RESULT_DO(m_pHostFsCreator->Create(&pBaseFileSystem, detail::GetExecutionDirectoryPath()));
        const char* BisRootPath = nullptr;
        switch (id)
        {
            case nn::fs::BisPartitionId::User:
            {
                BisRootPath = "/bis/user";
                break;
            }
            case nn::fs::BisPartitionId::System:
            {
                BisRootPath = "/bis/system";
                break;
            }
            case nn::fs::BisPartitionId::SafeMode:
            {
                BisRootPath = "/bis/safe";
                break;
            }
            case nn::fs::BisPartitionId::CalibrationFile:
            {
                BisRootPath = "/bis/cal";
                break;
            }
            default: BisRootPath = "";
        }
        NN_RESULT_DO(detail::WrapSubDirectory(pOutFileSystem, pBaseFileSystem, BisRootPath));
    }
    NN_RESULT_SUCCESS;
}

Result BuiltInStorageFileSystemCreatorHostFs::SetBisRoot(nn::fs::BisPartitionId id, const char* rootPath) NN_NOEXCEPT
{
    return m_BisRootPath.Set(id, rootPath);
}


}}}
