﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <memory>
#include <mutex>

#include <nn/nn_Common.h>
#include <nn/nn_SdkLog.h>
#include <nn/dd.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/fs/fs_Bis.h>
#include <nn/fs/fs_Result.h>
#include <nn/fs/fsa/fs_IFileSystem.h>
#include <nn/fs/fsa/fs_IFile.h>
#include <nn/fs/fs_FileStorage.h>
#include <nn/fssrv/fscreator/fssrv_BuiltInStorageFileSystemCreator.h>
#include <nn/fssrv/detail/fssrv_BisPartitionRootPathForHost.h>
#include "../detail/fssrv_FatFileSystemCacheManager.h"
#include "../detail/fssrv_Utility.h"

#include <nn/fssystem/fs_AllocatorUtility.h>
#include <nn/fssystem/fs_Utility.h>
#include <nn/fssystem/fs_AsynchronousAccess.h>

#include <nn/fssystem/fs_SubdirectoryFileSystem.h>


namespace {

const char* const SignedSystemPartitionOnSdCardFileName = "/recovery";

}

namespace nn { namespace fssrv { namespace fscreator {

BuiltInStorageFileSystemCreator::BuiltInStorageFileSystemCreator(
    IBuiltInStorageCreator* pBisCreator,
    IMemoryStorageCreator* pMemoryStorageCreator,
    IFatFileSystemCreator* pFatFsCreator,
    ISdCardProxyFileSystemCreator* pSdCardProxyFileSystemCreator,
    detail::FatFileSystemCacheManager* pFatFileSystemCacheManager
) NN_NOEXCEPT
    : m_pBisCreator(pBisCreator)
    , m_pMemoryStorageCreator(pMemoryStorageCreator)
    , m_pFatFsCreator(pFatFsCreator)
    , m_pSdCardProxyFileSystemCreator(pSdCardProxyFileSystemCreator)
    , m_pFatFileSystemCacheManager(pFatFileSystemCacheManager)
{
}

bool BuiltInStorageFileSystemCreator::IsSignedSystemPartitionOnSdCardValid() NN_NOEXCEPT
{
    return m_IsSignedSystemPartitionOnSdCardValid;
}

void BuiltInStorageFileSystemCreator::SetSdCardPortReady() NN_NOEXCEPT
{
    m_IsSdCardPortReady = true;
}


void BuiltInStorageFileSystemCreator::Initialize(fs::FileSystemProxyErrorInfo* pErrorInfo, os::Mutex* pErrorInfoMutex) NN_NOEXCEPT
{
    m_pErrorInfo = pErrorInfo;
    m_pErrorInfoMutex = pErrorInfoMutex;
}



Result BuiltInStorageFileSystemCreator::CreateFatFileSystemOnMemory(std::shared_ptr<nn::fs::fsa::IFileSystem>* pOutValue, fscreator::IMemoryStorageCreator::MemoryStorageId id) NN_NOEXCEPT
{
    char* bufferUnused;
    std::shared_ptr<fs::IStorage> memorySystemPartition;
    NN_RESULT_DO(m_pMemoryStorageCreator->Create(&memorySystemPartition, &bufferUnused, id));

    nn::fat::FatAttribute fatAttrs = { false, false, false };
    nn::fat::FatFormatParam fatFormatParam = { false, false, fs::ResultInvalidFatFormat(), nullptr, 0 };
    {
        std::unique_ptr<fat::FatErrorInfoSetter> dummySetter(new fat::FatErrorInfoSetter());
        NN_RESULT_THROW_UNLESS(dummySetter != nullptr, nn::fs::ResultAllocationMemoryFailedNew());
        NN_RESULT_DO(m_pFatFsCreator->Format(memorySystemPartition, &fatAttrs, &fatFormatParam, std::move(dummySetter), fs::ResultInvalidFatFormat(), fs::ResultUsableSpaceNotEnough()));
    }

    std::shared_ptr<nn::fs::fsa::IFileSystem> fatFs;
    {
        std::unique_ptr<fat::FatErrorInfoSetter> dummySetter(new fat::FatErrorInfoSetter());
        NN_RESULT_THROW_UNLESS(dummySetter != nullptr, nn::fs::ResultAllocationMemoryFailedNew());
        NN_RESULT_DO(m_pFatFsCreator->Create(&fatFs, std::move(memorySystemPartition), &fatAttrs, std::move(dummySetter), fs::ResultInvalidFatFormat(), fs::ResultUsableSpaceNotEnough()));
    }

    *pOutValue = std::move(fatFs);
    NN_RESULT_SUCCESS;
};

Result BuiltInStorageFileSystemCreator::CreateFatFileSystem(std::shared_ptr<nn::fs::fsa::IFileSystem>* pOutValue, std::shared_ptr<nn::fs::IStorage> pStorage, fs::BisPartitionId id, Result resultForInvalidFatFormat, Result resultUsableSpaceNotEnough) NN_NOEXCEPT
{
    bool enableFatSafe = (
        id == fs::BisPartitionId::User ||
        id == fs::BisPartitionId::System ||
        id == fs::BisPartitionId::SafeMode ||
        id == fs::BisPartitionId::SystemProperPartition
        );

    nn::fat::FatAttribute attrs = { enableFatSafe, false, false };
    std::unique_ptr<nn::fat::FatErrorInfoSetter> pFatErrorInfoSetter(new nn::fat::FatErrorInfoSetter(&m_pErrorInfo->fatFsError, m_pErrorInfoMutex, detail::GetFatDriveId(id)));
    NN_RESULT_THROW_UNLESS(pFatErrorInfoSetter != nullptr, nn::fs::ResultAllocationMemoryFailedNew());
    std::shared_ptr<nn::fs::fsa::IFileSystem> fatFs;
    NN_RESULT_DO(m_pFatFsCreator->Create(&fatFs, std::move(pStorage), &attrs, std::move(pFatErrorInfoSetter), resultForInvalidFatFormat, resultUsableSpaceNotEnough));

    *pOutValue = std::move(fatFs);
    NN_RESULT_SUCCESS;
}

Result BuiltInStorageFileSystemCreator::CreateAndLoadFatFileSystemOnMemory(std::shared_ptr<nn::fs::fsa::IFileSystem>* pOutValue, std::shared_ptr<nn::fs::IStorage> pStorage, fs::BisPartitionId bisId, fscreator::IMemoryStorageCreator::MemoryStorageId memoryStorageId, Result resultForInvalidFatFormat, Result resultUsableSpaceNotEnough) NN_NOEXCEPT
{
    std::shared_ptr<nn::fs::fsa::IFileSystem> fatFsSrc;
    NN_RESULT_DO(CreateFatFileSystem(&fatFsSrc, std::move(pStorage), bisId, resultForInvalidFatFormat, resultUsableSpaceNotEnough));

    std::shared_ptr<nn::fs::fsa::IFileSystem> fatFsOnMemory;
    NN_RESULT_DO(CreateFatFileSystemOnMemory(&fatFsOnMemory, memoryStorageId));
    fssystem::PooledBuffer pooledBuffer(512 * 1024, 1024);
    NN_RESULT_DO(fssystem::CopyDirectoryRecursive(fatFsOnMemory.get(), fatFsSrc.get(), "/", "/", pooledBuffer.GetBuffer(), pooledBuffer.GetSize()));

    *pOutValue = std::move(fatFsOnMemory);
    NN_RESULT_SUCCESS;
}


Result BuiltInStorageFileSystemCreator::Create(std::shared_ptr<fs::fsa::IFileSystem>* pOutFileSystem, const char* rootPath, nn::fs::BisPartitionId id) NN_NOEXCEPT
{
    std::shared_ptr<nn::fs::fsa::IFileSystem> pFileSystem;
    NN_FSP_REQUIRES(detail::BisPartitionRootPathForHost::IsValid(id), nn::fs::ResultInvalidArgument());

    NN_FSP_REQUIRES(rootPath != nullptr, nn::fs::ResultInvalidPath());
    detail::FatDriveId fatDriveId = detail::GetFatDriveId(id);

    {
        auto scopedLock = m_pFatFileSystemCacheManager->GetScopedLock();
        pFileSystem = m_pFatFileSystemCacheManager->GetCache(fatDriveId);
        if (pFileSystem == nullptr)
        {
            NN_RESULT_DO(CreateCore(&pFileSystem, id));
            m_pFatFileSystemCacheManager->SetCache(pFileSystem, fatDriveId);
        }
    }

    if (rootPath[0] != '\0')
    {
        std::shared_ptr<nn::fs::fsa::IFileSystem> subDirFs;
        NN_RESULT_DO(detail::CreateSubDirectoryFileSystem(&subDirFs, std::move(pFileSystem), rootPath, false));
        pFileSystem = std::move(subDirFs);
    }

    *pOutFileSystem = std::move(pFileSystem);
    NN_RESULT_SUCCESS;
}

Result BuiltInStorageFileSystemCreator::SetBisRoot(nn::fs::BisPartitionId id, const char* rootPath) NN_NOEXCEPT
{
    NN_UNUSED(id);
    NN_UNUSED(rootPath);
    return fs::ResultNotImplemented();
}


// 通常用, safe パーティションリダイレクト用
// fat on bis/*
Result BuiltInStorageFileSystemCreator::CreateCore(std::shared_ptr<nn::fs::fsa::IFileSystem>* pOutValue, nn::fs::BisPartitionId id) NN_NOEXCEPT
{
    std::shared_ptr<fs::IStorage> builtInStorage;
    NN_RESULT_DO(m_pBisCreator->Create(&builtInStorage, id));

    std::shared_ptr<nn::fs::fsa::IFileSystem> bisfs;
    NN_RESULT_DO(CreateFatFileSystem(&bisfs, std::move(builtInStorage), id, detail::GetResultForInvalidFatFormat(id), detail::GetResultForUsableSpaceNotEnough(id)));

    *pOutValue = std::move(bisfs);
    NN_RESULT_SUCCESS;
}


// Initialize, 修理用
// User: 空 fat on memory
// System: ssp on bis/system(bis/safe)
// 他: fat on bis/*
Result BuiltInStorageFileSystemCreatorSspOnBis::CreateCore(std::shared_ptr<nn::fs::fsa::IFileSystem>* pOutValue, nn::fs::BisPartitionId id) NN_NOEXCEPT
{
    switch (id)
    {
        case fs::BisPartitionId::User:
        {
            return CreateFatFileSystemOnMemory(pOutValue, fscreator::IMemoryStorageCreator::MemoryStorageId::UserPartition);
        }
        case fs::BisPartitionId::System:
        {
            std::shared_ptr<fs::IStorage> builtInStorage;
            NN_RESULT_DO(m_pBisCreator->Create(&builtInStorage, id));

            std::shared_ptr<nn::fs::fsa::IFileSystem> bisfsOnMemory;
            NN_RESULT_DO(CreateAndLoadFatFileSystemOnMemory(&bisfsOnMemory, std::move(builtInStorage), id, fscreator::IMemoryStorageCreator::MemoryStorageId::SystemPartition, detail::GetResultForInvalidFatFormat(id), detail::GetResultForUsableSpaceNotEnough(id)));

            *pOutValue = std::move(bisfsOnMemory);
            NN_RESULT_SUCCESS;
        }
        default:
        {
            return BuiltInStorageFileSystemCreator::CreateCore(pOutValue, id);
        }
    }
}


// 製品セーフモード用
// User: 空 fat on memory
// System: ssp on SD:/recovery または fat on bis/system
// 他: fat on bis/*
Result BuiltInStorageFileSystemCreatorSspOnSd::CreateCore(std::shared_ptr<nn::fs::fsa::IFileSystem>* pOutValue, nn::fs::BisPartitionId id) NN_NOEXCEPT
{
    switch (id)
    {
        case fs::BisPartitionId::User:
        {
            return CreateFatFileSystemOnMemory(pOutValue, fscreator::IMemoryStorageCreator::MemoryStorageId::UserPartition);
        }
        case fs::BisPartitionId::System:
        {
            std::shared_ptr<nn::fs::fsa::IFileSystem> pFileSystem;

            // システムパーティションの（再）マウント時には SignedSystemPartition 検証済みフラグを落とす
            m_IsSignedSystemPartitionOnSdCardValid = false;

            if (!m_IsSdCardPortReady)
            {
                return fs::ResultSystemPartitionNotReady();
            }

            // セーフモード時には SD 上の特定のファイル名を SignedSystemPartition としてマウント
            auto result = [&]() -> Result {
                std::shared_ptr<nn::fs::fsa::IFileSystem> sdFatFileSystem;
                std::unique_ptr<fs::fsa::IFile> signedSystemPartitionFile;

                NN_RESULT_DO(m_pSdCardProxyFileSystemCreator->Create(&sdFatFileSystem));
                NN_RESULT_DO(sdFatFileSystem->OpenFile(&signedSystemPartitionFile, SignedSystemPartitionOnSdCardFileName, fs::OpenMode_Read));

                std::unique_ptr<fs::IStorage> signedSystemPartitionFileStorage(new fs::FileStorage(signedSystemPartitionFile.get()));
                NN_RESULT_THROW_UNLESS(signedSystemPartitionFileStorage != nullptr, nn::fs::ResultAllocationMemoryFailedNew());

                std::unique_ptr<fs::IStorage> verifiedSignedSystemPartitionStorage;
                NN_RESULT_DO(m_pBisCreator->LoadAndVerifySignedSystemPartition(&verifiedSignedSystemPartitionStorage, std::move(signedSystemPartitionFileStorage)));

                std::shared_ptr<nn::fs::fsa::IFileSystem> bisfsOnMemory;
                NN_RESULT_DO(CreateAndLoadFatFileSystemOnMemory(&bisfsOnMemory, std::move(verifiedSignedSystemPartitionStorage), id, fscreator::IMemoryStorageCreator::MemoryStorageId::SystemPartition, fs::ResultInvalidFatFormatSd(), fs::ResultUsableSpaceNotEnoughSdCard()));

                pFileSystem = std::move(bisfsOnMemory);
                m_IsSignedSystemPartitionOnSdCardValid = true;
                NN_RESULT_SUCCESS;
            }();
            if (result.IsFailure())
            {
                // SD 上の ssp が開けない場合は SystemProper を開き、System としても登録する
                std::shared_ptr<nn::fs::fsa::IFileSystem> bisFs;
                NN_RESULT_DO(Create(&bisFs, "", nn::fs::BisPartitionId::SystemProperPartition));
                pFileSystem = std::move(bisFs);
            }

            *pOutValue = std::move(pFileSystem);
            NN_RESULT_SUCCESS;
        }
        default:
        {
            return BuiltInStorageFileSystemCreator::CreateCore(pOutValue, id);
        }
    }

}


// 評価/generic 用
// User: 空 fat on memory
// System: fatimg on memory
// 他: 非対応
Result BuiltInStorageFileSystemCreatorSystemPartitionOnMemoryBase::CreateCore(std::shared_ptr<nn::fs::fsa::IFileSystem>* pOutValue, nn::fs::BisPartitionId id) NN_NOEXCEPT
{
    switch (id)
    {
        case fs::BisPartitionId::User:
        {
            return CreateFatFileSystemOnMemory(pOutValue, fscreator::IMemoryStorageCreator::MemoryStorageId::UserPartition);
        }
        case fs::BisPartitionId::System:
        {
            auto pSystemPartitionOnMemory = reinterpret_cast<char*>(nn::dd::QueryIoMappingAddress(m_SystemPartitionOnMemoryPhysicalAddress, m_SystemPartitionOnMemorySize));
            std::unique_ptr<fs::IStorage> systemPartitionOnMemoryStorage(new fs::MemoryStorage(pSystemPartitionOnMemory, m_SystemPartitionOnMemorySize));
            NN_RESULT_THROW_UNLESS(systemPartitionOnMemoryStorage != nullptr, fs::ResultAllocationMemoryFailedNew());

            std::shared_ptr<nn::fs::fsa::IFileSystem> bisfsOnMemory;
            NN_RESULT_DO(CreateAndLoadFatFileSystemOnMemory(&bisfsOnMemory, std::move(systemPartitionOnMemoryStorage), id, fscreator::IMemoryStorageCreator::MemoryStorageId::SystemPartition, fs::ResultInvalidFatFormat(), fs::ResultUsableSpaceNotEnough()));

            *pOutValue = std::move(bisfsOnMemory);
            NN_RESULT_SUCCESS;
        }
        default:
            return fs::ResultInvalidArgument();
    }
}



}}}
