﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <memory>
#include <mutex>
#include <nn/fssrv/sf/fssrv_IStorage.h>
#include <nn/fssystem/fs_AsynchronousAccess.h>
#include <nn/fssystem/fs_Utility.h>
#include <nn/fs/fs_QueryRange.h>
#include <nn/os/os_ReaderWriterLock.h>
#include <nn/util/util_Optional.h>

#include "fssrv_SharedLock.h"
#include <nn/fssrv/detail/fssrv_IFileSystemProxyForAdapter.h>

namespace nn { namespace fs {
    class IStorage;
}}

namespace nn { namespace fssystem {
    struct Hash;
    typedef Hash NcaDigest;
}}

namespace nn { namespace fssrv {

    class FileSystemProxyImpl;

namespace detail {

    class StorageInterfaceAdapter
    {
    private:
        class DataStorageContext
        {
            NN_DISALLOW_COPY(DataStorageContext);
        public:
            DataStorageContext() NN_NOEXCEPT
                : m_Digest()
                , m_ProgramIdValue()
                , m_StorageId()
                , m_IsValid(false)
            {
            }
            DataStorageContext(std::unique_ptr<fssystem::NcaDigest>&& digest, nn::Bit64 programIdValue, nn::ncm::StorageId storageId) NN_NOEXCEPT
                : m_Digest(std::move(digest))
                , m_ProgramIdValue(programIdValue)
                , m_StorageId(storageId)
                , m_IsValid(true)
            {
            }
            fssystem::NcaDigest* GetDigest() const NN_NOEXCEPT
            {
                return m_Digest.get();
            }
            nn::Bit64 GetProgramIdValue() const NN_NOEXCEPT
            {
                return m_ProgramIdValue;
            }
            nn::ncm::StorageId GetStorageId() const NN_NOEXCEPT
            {
                return m_StorageId;
            }
            bool IsValid() const NN_NOEXCEPT
            {
                return m_IsValid;
            }
        private:
            const std::unique_ptr<fssystem::NcaDigest> m_Digest;
            const nn::Bit64 m_ProgramIdValue;
            const nn::ncm::StorageId m_StorageId;
            const bool m_IsValid;
        };

        NN_DISALLOW_COPY(StorageInterfaceAdapter);
    public:
        explicit StorageInterfaceAdapter(std::shared_ptr<fs::IStorage>&& storageInterface) NN_NOEXCEPT;
        StorageInterfaceAdapter(std::shared_ptr<fs::IStorage>&& storageInterface, FileSystemProxyImpl* pFileSystemProxy, std::unique_lock<fssystem::SemaphoreAdaptor>&& openCountSemaphore) NN_NOEXCEPT;

        // storage リマウントによるキャッシュ無効化リトライ用
        StorageInterfaceAdapter(std::shared_ptr<fs::IStorage>&& storageInterface, FileSystemProxyImpl* pFileSystemProxy, std::unique_lock<fssystem::SemaphoreAdaptor>&& openCountSemaphore, std::unique_ptr<fssystem::NcaDigest>&& digest, nn::Bit64 programIdValue, nn::ncm::StorageId storageId) NN_NOEXCEPT;

        // キャッシュ無効化リトライ用
        StorageInterfaceAdapter(std::shared_ptr<fs::IStorage>&& storageInterface, FileSystemProxyImpl* pFileSystemProxy, std::unique_lock<fssystem::SemaphoreAdaptor>&& openCountSemaphore, bool enableDeepRetry) NN_NOEXCEPT;

        StorageInterfaceAdapter(std::shared_ptr<fs::IStorage>&& storageInterface, FileSystemProxyImpl* pFileSystemProxy) NN_NOEXCEPT;
        ~StorageInterfaceAdapter() NN_NOEXCEPT;
        Result Read(std::int64_t offset, const nn::sf::OutBuffer& buffer, std::int64_t size) NN_NOEXCEPT;
        Result Write(std::int64_t offset, const nn::sf::InBuffer& buffer, std::int64_t size) NN_NOEXCEPT;
        Result Flush() NN_NOEXCEPT;
        Result SetSize(std::int64_t size) NN_NOEXCEPT;
        Result GetSize(nn::sf::Out<std::int64_t> outValue) NN_NOEXCEPT;
        Result OperateRange(
            nn::sf::Out<nn::fs::StorageQueryRangeInfo> outValue,
            std::int32_t operationId,
            int64_t offset,
            int64_t size) NN_NOEXCEPT;
    private:
        util::optional<SharedLock<os::ReaderWriterLock>> AcquireReaderLockForCacheInvalidation() NN_NOEXCEPT;
        Result InvalidateCacheOnStorage(bool doRemountStorage) NN_NOEXCEPT;

    private:
        nn::sf::SharedPointer<detail::IFileSystemProxyForAdapter> m_pFileSystemProxyImpl;
        nn::fssystem::AsynchronousAccessStorage m_Storage;
        std::unique_lock<fssystem::SemaphoreAdaptor> m_OpenCountSemaphore;
        nn::os::ReaderWriterLock m_InvalidateCacheLock;
        const DataStorageContext m_DataStorageContext;
        bool m_DeepRetryEnabled = false;
    };
}}}
