﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>
#include <cstring>
#include <mutex>

#include <nn/nn_Abort.h>
#include <nn/nn_SdkLog.h>
#include <nn/nn_Common.h>
#include <nn/crypto.h>
#include <nn/crypto/crypto_Compare.h>
#include <nn/fs/fs_Result.h>
#include <nn/fs/fs_ResultPrivate.h>
#include <nn/os/os_Event.h>
#include <nn/os/os_Mutex.h>
#include <nn/os/os_MemoryHeap.h>
#include <nn/os/os_SdkThreadCommon.h>
#include <nn/os/os_Thread.h>
#include <nn/os/os_TimerEvent.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/sdmmc/sdmmc_Common.h>
#include <nn/sdmmc/sdmmc_Mmc.h>
#include <nn/sdmmc/sdmmc_SdCard.h>
#if (defined(NN_DETAIL_FS_SMMU_FOR_SDMMC_ENABLE))
#include <nn/sdmmc/sdmmc_DeviceVirtualAddress.h>
#endif
#include <nn/dd.h>
#include <nn/util/util_BitUtil.h>
#include <nn/util/util_Optional.h>
#include <nn/util/util_ScopeExit.h>

#include <nn/fs/fs_IStorage.h>
#include <nn/fs/fs_QueryRange.h>
#include "fssrv_SdmmcStorageService.h"
#include "fssrv_SdmmcResultConverter.h"
#include "fssrv_DeviceBuffer.h"
#include "fssrv_Trace.h"
#include <nn/fs/detail/fs_Newable.h>
#include <nn/fs/fs_MmcPrivate.h>
#include <nn/fssrv/fssrv_SpeedEmulationStorage.h>
#include <nn/fssystem/fs_AsynchronousAccess.h>
#include <nn/fssystem/fs_AllocatorUtility.h>
#include <nn/nn_SystemThreadDefinition.h>

using namespace nn::fs::detail;
using namespace nn::fs;

#define NN_FS_SDMMC_TRACE_ENABLED 0 // NOLINT(preprocessor/const)

#if defined(NN_BUILD_CONFIG_HARDWARE_NX)
#define NN_FS_PATROL_READER_ENABLED
#endif

#if NN_FS_SDMMC_TRACE_ENABLED && NN_FS_SCOPED_TRACE_ENABLED

namespace nn { namespace fssrv { namespace detail {

namespace {

class MmcPartitionTrace
{
public:
    explicit MmcPartitionTrace(nn::sdmmc::MmcPartition id) NN_NOEXCEPT
    {
        g_Id = id;
    }
    ~MmcPartitionTrace() NN_NOEXCEPT
    {
        g_Id = nn::sdmmc::MmcPartition_UserData;
    }
public:
    static int GetId(nn::sdmmc::Port port) NN_NOEXCEPT
    {
        if( port == nn::sdmmc::Port_Mmc0 )
        {
            return g_Id;
        }
        return 0;
    }
private:
    static nn::sdmmc::MmcPartition g_Id;
};

nn::sdmmc::MmcPartition MmcPartitionTrace::g_Id = nn::sdmmc::MmcPartition_UserData;

}

}}}


#define NN_FS_SDMMC_TRACE(format, ...) \
    nn::fssrv::detail::ScopedTrace fsAccessLogScopedTrace("SdmmcStorage", \
    NN_CURRENT_FUNCTION_NAME, \
    [=](char* buffer, size_t bufferLength) NN_NOEXCEPT \
    { \
        return nn::fssrv::detail::ScopedTrace::Formatter(buffer, bufferLength).FormatLog(format " %d", __VA_ARGS__, ::nn::fssrv::detail::MmcPartitionTrace::GetId(this->GetPort())); \
    })

#define NN_FS_MMC_PARTITION_TRACE(id) ::nn::fssrv::detail::MmcPartitionTrace NN_MAKE_TEMPORARY_NAME(mmcPartitionTrace)(id)

#else

#define NN_FS_SDMMC_TRACE(...) (void)([&]() NN_NOEXCEPT { [](...) NN_NOEXCEPT {}( __VA_ARGS__); })
#define NN_FS_MMC_PARTITION_TRACE(...)

#endif

#define NN_DETAIL_FS_SDMMC_RESULT_DO(port, r)   NN_RESULT_DO(GetFsResult(port, r))

namespace nn { namespace fssrv { namespace detail {

namespace {

SdCardHandle g_SdCardHandle = InvalidSdCardHandle;

const int Log2SectorSize = 9;
static_assert((1 << Log2SectorSize) == nn::sdmmc::SectorSize, "nn::fssrv::detail::Log2SectorSize is invalid.");

uint32_t BytesToSectors(int64_t size) NN_NOEXCEPT
{
    return static_cast<uint32_t>(size >> Log2SectorSize);
}

/**
    @brief sdmmc ライブラリを介して sd と nand にアクセスするストレージレイヤです。

    @pre
        - アクセスのサイズ・オフセットは nn::sdmmc::SectorSize アライメントである。

    @details GetSize() で取得できる値は暫定の固定値です。
 */

class SdmmcStorage : public fs::IStorage
{
public:
    explicit SdmmcStorage(nn::sdmmc::Port port) NN_NOEXCEPT
        : m_Port(port)
    {
    }

    virtual Result Read(int64_t offset, void* buffer, size_t size) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_SDK_REQUIRES(nn::util::is_aligned(offset, nn::sdmmc::SectorSize));
        NN_SDK_REQUIRES(nn::util::is_aligned(size,   nn::sdmmc::SectorSize));

        if( size == 0 )
        {
            NN_RESULT_SUCCESS;
        }
        else if( buffer == nullptr )
        {
            return nn::fs::ResultNullptrArgument();
        }
        else if( IsBufferCapable(buffer) )
        {
            NN_FS_SDMMC_TRACE("R %d %8u %2u", m_Port, BytesToSectors(offset), BytesToSectors(size));
            NN_DETAIL_FS_SDMMC_RESULT_DO(m_Port,
                nn::sdmmc::Read(
                    buffer,
                    size,
                    m_Port,
                    BytesToSectors(offset),
                    BytesToSectors(size)));

            NN_RESULT_SUCCESS;
        }
        else
        {
            const fssystem::PooledBuffer pooledBuffer(size, nn::sdmmc::SectorSize);
            NN_SDK_ASSERT(IsBufferCapable(pooledBuffer.GetBuffer()));

            size_t read = 0;
            while( read != size )
            {
                size_t readSize = std::min(size - read, pooledBuffer.GetSize());
                NN_FS_SDMMC_TRACE("R %d %8u %2u", m_Port, BytesToSectors(offset + read), BytesToSectors(readSize));
                NN_DETAIL_FS_SDMMC_RESULT_DO(m_Port,
                    nn::sdmmc::Read(
                        pooledBuffer.GetBuffer(),
                        readSize,
                        m_Port,
                        BytesToSectors(offset + read),
                        BytesToSectors(readSize)));
                std::memcpy(reinterpret_cast<char*>(buffer) + read, pooledBuffer.GetBuffer(), readSize);
                read += readSize;
            }

            NN_RESULT_SUCCESS;
        }
    }

    virtual Result Write(int64_t offset, const void* buffer, size_t size) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_SDK_REQUIRES(nn::util::is_aligned(offset, nn::sdmmc::SectorSize));
        NN_SDK_REQUIRES(nn::util::is_aligned(size,   nn::sdmmc::SectorSize));

        static const size_t Alignment = 16 * 1024;

        if( size == 0 )
        {
            NN_RESULT_SUCCESS;
        }
        else if( buffer == nullptr )
        {
            return nn::fs::ResultNullptrArgument();
        }
        else if( IsBufferCapable(buffer) )
        {
            const auto alignedUpOffset = nn::util::align_up(offset, Alignment);
            const auto unalignedSize = static_cast<size_t>(alignedUpOffset - offset);
            size_t restSize = size;
            if( 0 < unalignedSize && 0 < size )
            {
                const auto paddingSize = Alignment - unalignedSize;
                size_t writeSize = std::min(Alignment, paddingSize + size);
                const fssystem::PooledBuffer pooledBuffer(writeSize, Alignment);
                NN_SDK_ASSERT(IsBufferCapable(pooledBuffer.GetBuffer()));

                size_t headNewDataSize = writeSize - paddingSize;
                std::memcpy(
                    pooledBuffer.GetBuffer() + paddingSize,
                    buffer,
                    headNewDataSize);
                {
                    NN_FS_SDMMC_TRACE("R %d %8u %2u", m_Port, BytesToSectors(alignedUpOffset - Alignment), BytesToSectors(paddingSize));
                    NN_DETAIL_FS_SDMMC_RESULT_DO(m_Port,
                        nn::sdmmc::Read(
                            pooledBuffer.GetBuffer(),
                            paddingSize,
                            m_Port,
                            BytesToSectors(alignedUpOffset - Alignment),
                            BytesToSectors(paddingSize)));
                }
                {
                    NN_FS_SDMMC_TRACE("W %d %8u %2u", m_Port, BytesToSectors(alignedUpOffset - Alignment), BytesToSectors(writeSize));
                    NN_DETAIL_FS_SDMMC_RESULT_DO(m_Port,
                        nn::sdmmc::Write(
                            m_Port,
                            BytesToSectors(alignedUpOffset - Alignment),
                            BytesToSectors(writeSize),
                            pooledBuffer.GetBuffer(),
                            writeSize));
                }
                restSize -= headNewDataSize;
            }
            if (restSize > 0)
            {
                NN_FS_SDMMC_TRACE("W %d %8u %2u", m_Port, BytesToSectors(alignedUpOffset), BytesToSectors(restSize));
                NN_DETAIL_FS_SDMMC_RESULT_DO(m_Port,
                    nn::sdmmc::Write(
                        m_Port,
                        BytesToSectors(alignedUpOffset),
                        BytesToSectors(restSize),
                        reinterpret_cast<const int8_t*>(buffer) + unalignedSize,
                        restSize));
            }

            NN_RESULT_SUCCESS;
        }
        else
        {
            const auto alignedOffset = nn::util::align_down(offset, Alignment);
            auto paddingSize = static_cast<size_t>(0 < size ? offset - alignedOffset : 0);
            const auto alignedSize = size + paddingSize;
            const fssystem::PooledBuffer pooledBuffer(alignedSize , Alignment);
            NN_SDK_ASSERT(IsBufferCapable(pooledBuffer.GetBuffer()));
            auto writeBuffer = reinterpret_cast<const char*>(buffer);

            size_t wrote = 0;
            while( wrote != alignedSize )
            {
                size_t writeSize = std::min(alignedSize - wrote, pooledBuffer.GetSize());
                std::memcpy(
                    pooledBuffer.GetBuffer() + paddingSize,
                    writeBuffer,
                    writeSize - paddingSize);
                if( 0 < paddingSize )
                {
                    NN_FS_SDMMC_TRACE("R %d %8u %2u", m_Port, BytesToSectors(alignedOffset), BytesToSectors(paddingSize));
                    NN_DETAIL_FS_SDMMC_RESULT_DO(m_Port,
                        nn::sdmmc::Read(
                            pooledBuffer.GetBuffer(),
                            paddingSize,
                            m_Port,
                            BytesToSectors(alignedOffset),
                            BytesToSectors(paddingSize)));
                }
                NN_FS_SDMMC_TRACE("W %d %8u %2u", m_Port, BytesToSectors(alignedOffset + wrote), BytesToSectors(writeSize));
                NN_DETAIL_FS_SDMMC_RESULT_DO(m_Port,
                    nn::sdmmc::Write(
                        m_Port,
                        BytesToSectors(alignedOffset + wrote),
                        BytesToSectors(writeSize),
                        pooledBuffer.GetBuffer(),
                        writeSize));
                writeBuffer += writeSize - paddingSize;
                wrote += writeSize;
                paddingSize = 0;
            }

            NN_RESULT_SUCCESS;
        }
    } // NOLINT(impl/function_size)

    virtual Result Flush() NN_NOEXCEPT NN_OVERRIDE
    {
        NN_RESULT_SUCCESS;
    }

    virtual Result GetSize(int64_t* outValue) NN_NOEXCEPT NN_OVERRIDE
    {
        uint32_t numSectors = 0;
        NN_DETAIL_FS_SDMMC_RESULT_DO(m_Port, nn::sdmmc::GetDeviceMemoryCapacity(&numSectors, m_Port));
        NN_RESULT_THROW_UNLESS(outValue != nullptr, ResultNullptrArgument());
        *outValue = static_cast<int64_t>(numSectors) * nn::sdmmc::SectorSize;
        NN_RESULT_SUCCESS;
    }

    virtual Result OperateRange(
        void* outBuffer,
        size_t outBufferSize,
        OperationId operationId,
        int64_t,
        int64_t,
        const void*,
        size_t) NN_NOEXCEPT NN_OVERRIDE
    {
        switch( operationId )
        {
        case OperationId::Invalidate:
            NN_RESULT_SUCCESS;
        case OperationId::QueryRange:
            NN_RESULT_THROW_UNLESS(outBuffer != nullptr, nn::fs::ResultNullptrArgument());
            NN_RESULT_THROW_UNLESS(outBufferSize == sizeof(nn::fs::QueryRangeInfo), nn::fs::ResultInvalidSize());

            reinterpret_cast<nn::fs::QueryRangeInfo*>(outBuffer)->Clear();
            NN_RESULT_SUCCESS;
        default:
            return nn::fs::ResultUnsupportedOperation();
        }
    }

    nn::sdmmc::Port GetPort()
    {
        return m_Port;
    }

private:
    nn::sdmmc::Port m_Port;
};

class MmcUserDataPartitionStorage : public IStorage, public nn::fs::detail::Newable
{
public:
    MmcUserDataPartitionStorage(std::shared_ptr<IStorage> pSdmmcStorage, nn::os::Mutex* accessMutex) NN_NOEXCEPT
      : m_Impl(pSdmmcStorage),
        m_pAccessMutex(accessMutex)
    {
    }

    virtual Result Read(int64_t offset, void* buffer, size_t size) NN_NOEXCEPT NN_OVERRIDE
    {
        std::lock_guard<os::Mutex> scopedLock(*m_pAccessMutex);
        return m_Impl->Read(offset, buffer, size);
    }

    virtual Result Write(int64_t offset, const void* buffer, size_t size) NN_NOEXCEPT NN_OVERRIDE
    {
        std::lock_guard<os::Mutex> scopedLock(*m_pAccessMutex);
        return m_Impl->Write(offset, buffer, size);
    }

    virtual Result Flush() NN_NOEXCEPT NN_OVERRIDE
    {
        return m_Impl->Flush();
    }

    virtual Result GetSize(int64_t* outValue) NN_NOEXCEPT NN_OVERRIDE
    {
        std::lock_guard<os::Mutex> scopedLock(*m_pAccessMutex);
        return m_Impl->GetSize(outValue);
    }

    virtual Result OperateRange(
        void* outBuffer,
        size_t outBufferSize,
        OperationId operationId,
        int64_t offset,
        int64_t size,
        const void* inBuffer,
        size_t inBufferSize) NN_NOEXCEPT NN_OVERRIDE
    {
        return m_Impl->OperateRange(
            outBuffer,
            outBufferSize,
            operationId,
            offset,
            size,
            inBuffer,
            inBufferSize);
    }

private:
    std::shared_ptr<IStorage>   m_Impl;
    nn::os::Mutex*              m_pAccessMutex;
};

class MmcBootPartitionStorage : public IStorage, public nn::fs::detail::Newable
{
public:
    MmcBootPartitionStorage(SdmmcStorage* pSdmmcStorage, MmcPartition id, nn::os::Mutex* accessMutex) NN_NOEXCEPT
      : m_Impl(pSdmmcStorage),
        m_pAccessMutex(accessMutex),
        m_Partition(GetPartition(id))
    {
    }

    virtual Result Read(int64_t offset, void* buffer, size_t size) NN_NOEXCEPT NN_OVERRIDE
    {
        std::lock_guard<os::Mutex> scopedLock(*m_pAccessMutex);
        const nn::sdmmc::Port Port = m_Impl->GetPort();

        // Boot Partition の切り替えに失敗した場合は ABORT する
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::sdmmc::SelectMmcPartition(Port, m_Partition));
        NN_UTIL_SCOPE_EXIT
        {
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::sdmmc::SelectMmcPartition(Port, nn::sdmmc::MmcPartition::MmcPartition_UserData));
        };
        NN_FS_MMC_PARTITION_TRACE(m_Partition);
        return m_Impl->Read(offset, buffer, size);
    }

    virtual Result Write(int64_t offset, const void* buffer, size_t size) NN_NOEXCEPT NN_OVERRIDE
    {
        std::lock_guard<os::Mutex> scopedLock(*m_pAccessMutex);
        const nn::sdmmc::Port Port = m_Impl->GetPort();

        // Boot Partition の切り替えに失敗した場合は ABORT する
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::sdmmc::SelectMmcPartition(Port, m_Partition));
        NN_UTIL_SCOPE_EXIT
        {
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::sdmmc::SelectMmcPartition(Port, nn::sdmmc::MmcPartition::MmcPartition_UserData));
        };
        NN_FS_MMC_PARTITION_TRACE(m_Partition);
        return m_Impl->Write(offset, buffer, size);
    }

    virtual Result Flush() NN_NOEXCEPT NN_OVERRIDE
    {
        return m_Impl->Flush();
    }

    virtual Result GetSize(int64_t* outValue) NN_NOEXCEPT NN_OVERRIDE
    {
        std::lock_guard<os::Mutex> scopedLock(*m_pAccessMutex);
        const nn::sdmmc::Port Port = m_Impl->GetPort();

        // Boot Partition の切り替えに失敗した場合は ABORT する
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::sdmmc::SelectMmcPartition(Port, m_Partition));
        NN_UTIL_SCOPE_EXIT
        {
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::sdmmc::SelectMmcPartition(Port, nn::sdmmc::MmcPartition::MmcPartition_UserData));
        };
        uint32_t numSectors = 0;
        NN_DETAIL_FS_SDMMC_RESULT_DO(Port, nn::sdmmc::GetMmcBootPartitionCapacity(&numSectors, Port));
        NN_RESULT_THROW_UNLESS(outValue != nullptr, ResultNullptrArgument());
        *outValue = static_cast<int64_t>(numSectors) * nn::sdmmc::SectorSize;
        NN_RESULT_SUCCESS;
    }

    virtual Result OperateRange(
        void* outBuffer,
        size_t outBufferSize,
        OperationId operationId,
        int64_t offset,
        int64_t size,
        const void* inBuffer,
        size_t inBufferSize) NN_NOEXCEPT NN_OVERRIDE
    {
        return m_Impl->OperateRange(
            outBuffer,
            outBufferSize,
            operationId,
            offset,
            size,
            inBuffer,
            inBufferSize);
    }

private:
    nn::sdmmc::MmcPartition GetPartition(MmcPartition id)
    {
        switch(id)
        {
        case MmcPartition::BootPartition1:
            return nn::sdmmc::MmcPartition::MmcPartition_BootPartition1;
        case MmcPartition::BootPartition2:
            return nn::sdmmc::MmcPartition::MmcPartition_BootPartition2;
        default:
            NN_UNEXPECTED_DEFAULT;
        }
    }

    SdmmcStorage*            m_Impl;
    nn::os::Mutex*           m_pAccessMutex;
    nn::sdmmc::MmcPartition m_Partition;
};

nn::util::optional<SdmmcStorage> g_MmcStorage;
nn::os::Mutex                    g_MmcStorageMutex(false);
bool                             g_MmcStorageActive = false;

nn::util::optional<SdmmcStorage> g_SdStorage;
nn::os::Mutex                    g_SdStorageMutex(false);

nn::os::Mutex                    g_MmcPartitionStorageMutex(false);

#if defined(NN_BUILD_CONFIG_HARDWARE_JETSONTK1) || defined(NN_BUILD_CONFIG_HARDWARE_JETSONTK2) || defined(NN_BUILD_CONFIG_HARDWARE_NX) || defined(NN_BUILD_CONFIG_HARDWARE_JETSONTX2)
fssystem::PooledBuffer g_WorkBufferForMmc;
fssystem::PooledBuffer g_WorkBufferForSdCard;
#endif

void InitializeMmc() NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> scopedLock(g_MmcStorageMutex);
    if( !g_MmcStorage )
    {
        #if defined(NN_BUILD_CONFIG_HARDWARE_JETSONTK1) || defined(NN_BUILD_CONFIG_HARDWARE_JETSONTK2) || defined(NN_BUILD_CONFIG_HARDWARE_NX) || defined(NN_BUILD_CONFIG_HARDWARE_JETSONTX2)
            const nn::sdmmc::Port Port = nn::sdmmc::Port_Mmc0;

            nn::sdmmc::Initialize(Port);

            #if (defined(NN_DETAIL_FS_SMMU_FOR_SDMMC_ENABLE))
                nn::sdmmc::RegisterDeviceVirtualAddress(
                    Port,
                    GetDeviceBuffer(),
                    DeviceBufferSize,
                    GetDeviceVirtualAddress());
            #endif

            g_WorkBufferForMmc.Allocate(
                nn::sdmmc::MmcWorkBufferSize,
                nn::sdmmc::MmcWorkBufferSize);
            NN_SDK_ASSERT(IsBufferCapable(g_WorkBufferForMmc.GetBuffer()));
            nn::sdmmc::SetMmcWorkBuffer(
                Port,
                g_WorkBufferForMmc.GetBuffer(),
                g_WorkBufferForMmc.GetSize());

            g_MmcStorage.emplace(Port);
        #endif
    }
}

void InitializeSd() NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> scopedLock(g_SdStorageMutex);
    if( !g_SdStorage )
    {
        #if defined(NN_BUILD_CONFIG_HARDWARE_JETSONTK1) || defined(NN_BUILD_CONFIG_HARDWARE_JETSONTK2) || defined(NN_BUILD_CONFIG_HARDWARE_NX) || defined(NN_BUILD_CONFIG_HARDWARE_JETSONTX2)
            const nn::sdmmc::Port Port = nn::sdmmc::Port_SdCard0;

            nn::sdmmc::Initialize(Port);

            #if (defined(NN_DETAIL_FS_SMMU_FOR_SDMMC_ENABLE))
                nn::sdmmc::RegisterDeviceVirtualAddress(
                    Port,
                    GetDeviceBuffer(),
                    DeviceBufferSize,
                    GetDeviceVirtualAddress());
            #endif

            g_WorkBufferForSdCard.Allocate(
                nn::sdmmc::SdCardWorkBufferSize,
                nn::sdmmc::SdCardWorkBufferSize);
            NN_SDK_ASSERT(IsBufferCapable(g_WorkBufferForSdCard.GetBuffer()));
            nn::sdmmc::SetSdCardWorkBuffer(
                Port,
                g_WorkBufferForSdCard.GetBuffer(),
                g_WorkBufferForSdCard.GetSize());

            g_SdStorage.emplace(Port);
        #endif
    }
}

#if defined(NN_FS_PATROL_READER_ENABLED)
// http://spdlybra.nintendo.co.jp/confluence/pages/viewpage.action?pageId=172915113

const size_t PatrolReaderStackSize = 4 * 1024;
NN_OS_ALIGNAS_GUARDED_STACK char g_PatrolReaderStack[PatrolReaderStackSize];
const char PartolMacKey[] = "U{W5>1Kq#Gt`f6r86o`9|*||hTy9U2C";

class PatrolReader : public nn::fs::detail::Newable
{
public:
    explicit PatrolReader(nn::os::Mutex& accessMutex) NN_NOEXCEPT
      : m_IsStateLoaded(false),
        m_AccessMutex(accessMutex),
        m_IsInitialized(false),
        m_ThreadState(PatrolReaderThreadState_Stop),
        m_PatrolReadAllocateBufferSuccessCount(0),
        m_PatrolReadAllocateBufferFailureCount(0)
    {
    }

    ~PatrolReader() NN_NOEXCEPT
    {
        Finalize();
    }

    void Initialize() NN_NOEXCEPT
    {
        if (!m_IsInitialized)
        {
            m_IsInitialized = true;
            m_ThreadState = PatrolReader::PatrolReaderThreadState_Stop;
            nn::os::InitializeEvent(&m_EventSignal, false, nn::os::EventClearMode_AutoClear);
            nn::os::InitializeEvent(&m_CompleteEvent, false, nn::os::EventClearMode_AutoClear);
        }
    }

    void Finalize() NN_NOEXCEPT
    {
        if (m_IsInitialized)
        {
            if (m_ThreadState == PatrolReaderThreadState_Sleep)
            {
                Resume();
            }
            if (m_ThreadState == PatrolReaderThreadState_Active)
            {
                Stop();
            }
            nn::os::FinalizeEvent(&m_EventSignal);
            nn::os::FinalizeEvent(&m_CompleteEvent);
            m_IsInitialized = false;
        }
    }

    Result GetPatrolCount(uint32_t* pOutPatrolCount)
    {
        // まだロードしていない場合は取得を断念する
        NN_RESULT_THROW_UNLESS(m_IsStateLoaded, ResultHasNotGottenPatrolCount());

        NN_RESULT_THROW_UNLESS(pOutPatrolCount != nullptr, ResultNullptrArgument());
        *pOutPatrolCount = m_CountPatrol;
        NN_RESULT_SUCCESS;
    }

    bool LoadState() NN_NOEXCEPT
    {
        fssystem::PooledBuffer workBuffer(StateSize, 1);
        if (workBuffer.GetSize() < StateSize)
        {
            NN_SDK_LOG("[fs] Error: PatrolReader::LoadState() cannot allocate buffer\n");
            return false;
        }
        NN_SDK_ASSERT(IsBufferCapable(workBuffer.GetBuffer()));

        std::lock_guard<nn::os::Mutex> scopedLock(m_AccessMutex);
        const nn::sdmmc::Port Port = nn::sdmmc::Port_Mmc0;

        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::sdmmc::SelectMmcPartition(Port, StatePartition));
        NN_UTIL_SCOPE_EXIT
        {
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::sdmmc::SelectMmcPartition(Port, nn::sdmmc::MmcPartition::MmcPartition_UserData));
        };

        nn::Result result = nn::sdmmc::GetDeviceMemoryCapacity(&m_MaxSector, Port);
        if (result.IsFailure())
        {
            NN_SDK_LOG("[fs] Error: nn::sdmmc::GetDeviceMemoryCapacity() is failed (Module:%d, Description:%d) at PatrolReader\n", result.GetModule(), result.GetDescription());

            // 容量が取得できない状態のためリトライ
            return false;
        }

        char* pBuffer = workBuffer.GetBuffer();
        result = nn::sdmmc::Read(pBuffer,
                                 StateSize,
                                 Port,
                                 BytesToSectors(StateOffset),
                                 BytesToSectors(StateSize));
        if (result.IsFailure())
        {
            NN_SDK_LOG("[fs] Error: nn::sdmmc::Read() is failed (Module:%d, Description:%d) at PatrolReader\n", result.GetModule(), result.GetDescription());

            // Boot Partition 1 がリードできない状態のためリトライ
            return false;
        }

        m_NextSector = 0;
        m_CountPatrol = 0;

        char mac[PatrolMacSize];
        nn::crypto::GenerateHmacSha256Mac(
            mac, sizeof(mac),
            pBuffer + PatrolMacSize, StateSize - PatrolMacSize,
            PartolMacKey, sizeof(PartolMacKey)
        );

        if ( nn::crypto::IsSameBytes(mac, pBuffer, sizeof(mac)) )
        {
            const PatrolHeader* pHeader = reinterpret_cast<const PatrolHeader*>(pBuffer + PatrolMacSize);

            if ((pHeader->nextSector + BytesToSectors(PatrolReadSize)) <= m_MaxSector)
            {
                m_NextSector = pHeader->nextSector;
                m_CountPatrol = pHeader->patrolCount;
            }
        }

        return true;
    }

    void SaveState() NN_NOEXCEPT
    {
        fssystem::PooledBuffer workBuffer(StateSize, 1);
        if (workBuffer.GetSize() < StateSize)
        {
            NN_SDK_LOG("[fs] Warning: PatrolReader::SaveState() cannot allocate buffer\n");
            return;
        }
        NN_SDK_ASSERT(IsBufferCapable(workBuffer.GetBuffer()));

        std::lock_guard<nn::os::Mutex> scopedLock(m_AccessMutex);
        const nn::sdmmc::Port Port = nn::sdmmc::Port_Mmc0;

        char* pBuffer = workBuffer.GetBuffer();
        memset(pBuffer, 0, StateSize);

        PatrolHeader* pHeader = reinterpret_cast<PatrolHeader*>(pBuffer + PatrolMacSize);
        pHeader->nextSector = m_NextSector;
        pHeader->patrolCount = m_CountPatrol;

        nn::crypto::GenerateHmacSha256Mac(
            pBuffer, PatrolMacSize,
            pBuffer + PatrolMacSize, StateSize - PatrolMacSize,
            PartolMacKey, sizeof(PartolMacKey)
        );

        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::sdmmc::SelectMmcPartition(Port, StatePartition));
        NN_UTIL_SCOPE_EXIT
        {
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::sdmmc::SelectMmcPartition(Port, nn::sdmmc::MmcPartition::MmcPartition_UserData));
        };

        nn::Result result = nn::sdmmc::Write(Port,
                         BytesToSectors(StateOffset),
                         BytesToSectors(StateSize),
                         pBuffer, StateSize);
        if (result.IsFailure())
        {
            NN_SDK_LOG("[fs] Error: nn::sdmmc::Write() is failed (Module:%d, Description:%d) at PatrolReader\n", result.GetModule(), result.GetDescription());
        }
    }

    bool DoPatrolRead() NN_NOEXCEPT
    {
        fssystem::PooledBuffer workBuffer(PatrolReadSize, 1);
        std::lock_guard<nn::os::Mutex> scopedLock(m_AccessMutex);
        if (workBuffer.GetSize() < PatrolReadSize)
        {
            if (m_PatrolReadAllocateBufferFailureCount < UINT64_MAX)
            {
                ++m_PatrolReadAllocateBufferFailureCount;
            }
            NN_SDK_LOG("[fs] Warning: PatrolReader::DoPatrolRead() cannot allocate buffer\n");
            return false;
        }
        if (m_PatrolReadAllocateBufferSuccessCount < UINT64_MAX)
        {
            ++m_PatrolReadAllocateBufferSuccessCount;
        }

        NN_SDK_ASSERT(IsBufferCapable(workBuffer.GetBuffer()));

        const nn::sdmmc::Port Port = nn::sdmmc::Port_Mmc0;

        uint32_t* pBuffer = reinterpret_cast<uint32_t*>(workBuffer.GetBuffer());
        nn::Result result = nn::sdmmc::Read(pBuffer,
                        PatrolReadSize,
                        Port,
                        m_NextSector,
                        BytesToSectors(PatrolReadSize));
        if (result.IsFailure())
        {
            NN_SDK_LOG("[fs] Error: nn::sdmmc::Read() is failed (Module:%d, Description:%d) at PatrolReader\n", result.GetModule(), result.GetDescription());
            // ECC エラー等を検出しても無視して進める
        }

        m_NextSector += BytesToSectors(PatrolReadSize);
        if ((m_NextSector + BytesToSectors(PatrolReadSize)) > m_MaxSector)
        {
            m_NextSector = 0;
            m_CountPatrol++;
        }
        return true;
    }

    void Start() NN_NOEXCEPT
    {
        NN_SDK_ASSERT(m_ThreadState == PatrolReaderThreadState_Active ||
            m_ThreadState == PatrolReaderThreadState_Stop);
        if (m_ThreadState == PatrolReaderThreadState_Stop)
        {
            // イベント初期化
            Initialize();

            // スレッド起動
            m_ThreadState = PatrolReader::PatrolReaderThreadState_Active;
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::os::CreateThread(
                &m_PatrolReaderThread,
                &PatrolReader::PatrolReaderThreadEntry,
                this,
                g_PatrolReaderStack,
                PatrolReaderStackSize,
                NN_SYSTEM_THREAD_PRIORITY(fs, PatrolReader)));
            nn::os::SetThreadNamePointer(&m_PatrolReaderThread, NN_SYSTEM_THREAD_NAME(fs, PatrolReader));
            nn::os::StartThread(&m_PatrolReaderThread);
        }
    }

    void Stop() NN_NOEXCEPT
    {
        NN_SDK_ASSERT(m_ThreadState == PatrolReaderThreadState_Active ||
            m_ThreadState == PatrolReaderThreadState_Stop ||
            m_ThreadState == PatrolReaderThreadState_Sleep);
        if (m_ThreadState != PatrolReaderThreadState_Stop)
        {
            // スレッド終了シグナル
            m_ThreadState = PatrolReader::PatrolReaderThreadState_Stop;
            nn::os::SignalEvent(&m_EventSignal);

            // スレッド終了待ち
            nn::os::WaitThread(&m_PatrolReaderThread);
            nn::os::DestroyThread(&m_PatrolReaderThread);
        }
    }

    void Sleep() NN_NOEXCEPT
    {
        NN_SDK_ASSERT(m_ThreadState == PatrolReaderThreadState_Active ||
            m_ThreadState == PatrolReaderThreadState_Sleep);
        if (m_ThreadState == PatrolReaderThreadState_Active)
        {
            // 休止シグナル
            m_ThreadState = PatrolReaderThreadState_Sleep;
            nn::os::SignalEvent(&m_EventSignal);

            // 休止待ち
            nn::os::WaitEvent(&m_CompleteEvent);
        }
    }

    void Resume() NN_NOEXCEPT
    {
        NN_SDK_ASSERT(m_ThreadState == PatrolReaderThreadState_Active ||
            m_ThreadState == PatrolReaderThreadState_Sleep);
        if (m_ThreadState == PatrolReaderThreadState_Sleep)
        {
            // 再開シグナル
            m_ThreadState = PatrolReaderThreadState_Active;
            nn::os::SignalEvent(&m_EventSignal);

            // 再開待ち
            nn::os::WaitEvent(&m_CompleteEvent);
        }
    }

    static void PatrolReaderThreadEntry(void* p) NN_NOEXCEPT
    {
        reinterpret_cast<PatrolReader*>(p)->PatrolReaderThread();
    }

    void PatrolReaderThread() NN_NOEXCEPT
    {
        nn::fssystem::ServiceContext context;
        nn::fssystem::RegisterServiceContext(&context);

        nn::os::TimerEvent timerEvent(nn::os::EventClearMode_AutoClear);

        // 起動ウェイトと MMC Boot Partition 1 書き出しタイマーセット
        timerEvent.StartPeriodic(nn::TimeSpan::FromSeconds(PatrolReadStartupWait),
                                 nn::TimeSpan::FromSeconds(StateSaveInterval));

        auto nextPatrolReadInterval = PatrolReadIntervalNormal;

        while (NN_STATIC_CONDITION(true))
        {
            bool event = nn::os::TimedWaitEvent(&m_EventSignal,
                                              nn::TimeSpan::FromSeconds(nextPatrolReadInterval));
            if (event)
            {
                if (m_ThreadState == PatrolReaderThreadState_Stop)
                {
                    break;
                }
                else if (m_ThreadState == PatrolReaderThreadState_Sleep)
                {
                    // スレッド休止完了シグナル
                    nn::os::SignalEvent(&m_CompleteEvent);

                    // レジューム待ち
                    nn::os::WaitEvent(&m_EventSignal);
                    if (m_ThreadState == PatrolReaderThreadState_Stop)
                    {
                        break;
                    }
                    else if (m_ThreadState == PatrolReaderThreadState_Active)
                    {
                        // レジューム完了シグナル
                        nn::os::SignalEvent(&m_CompleteEvent);
                        continue;
                    }
                    else
                    {
                        // ここに来るのはおかしい
                        NN_SDK_ASSERT(false);
                    }
                }
            }

            if (!m_IsStateLoaded)
            {
                if (!timerEvent.TryWait())
                {
                    // PatrolReadStartupWait 秒経過するまでは読み出しに行かない
                    continue;
                }

                // 巡回読み出し開始場所の読み込み
                if (!LoadState())
                {
                    // 失敗時は先に進まない
                    continue;
                }

                m_IsStateLoaded = true;
            }

            // 巡回処理
            if (DoPatrolRead())
            {
                nextPatrolReadInterval = PatrolReadIntervalNormal;
            }
            else
            {
                // 失敗したら通常より早くリトライする
                nextPatrolReadInterval = PatrolReadIntervalRetry;
            }

            // 巡回読み出し開始場所の保存
            if (timerEvent.TryWait())
            {
                SaveState();
            }
        }

        // 巡回読み出し開始場所の保存
        if (m_IsStateLoaded)
        {
            SaveState();
        }
    }

    void GetAndClearPatrolReadAllocateBufferCount(uint64_t* outSuccessCount, uint64_t* outFailureCount) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(outSuccessCount);
        NN_SDK_REQUIRES_NOT_NULL(outFailureCount);

        std::lock_guard<nn::os::Mutex> scopedLock(m_AccessMutex);
        *outSuccessCount = m_PatrolReadAllocateBufferSuccessCount;
        *outFailureCount = m_PatrolReadAllocateBufferFailureCount;
        m_PatrolReadAllocateBufferSuccessCount = 0;
        m_PatrolReadAllocateBufferFailureCount = 0;
    }

    enum PatrolReaderThreadState {
        PatrolReaderThreadState_Stop = 0,
        PatrolReaderThreadState_Active,
        PatrolReaderThreadState_Sleep
    };

private:
    struct PatrolHeader
    {
        uint32_t nextSector;
        uint32_t patrolCount;
    };
    NN_STATIC_ASSERT(std::is_pod<PatrolHeader>::value);

    const size_t PatrolMacSize = 32;
    const nn::sdmmc::MmcPartition StatePartition = nn::sdmmc::MmcPartition::MmcPartition_BootPartition1;
    const int StateOffset = 0x00184000;
    const size_t StateSize = 512;
    const uint32_t StateSaveInterval = 2 * 60 * 60;
    const uint32_t PatrolReadSize = 512 * 1024; // 採用する MMC の User Area のサイズを割り切れる値にすること
    const uint32_t PatrolReadIntervalNormal = 6; // 通常の読み出し間隔（秒）
    const uint32_t PatrolReadIntervalRetry = 1;  // 失敗時のリトライ間隔（秒）
    const uint32_t PatrolReadStartupWait = 18;

    uint32_t m_MaxSector;
    uint32_t m_NextSector;
    uint32_t m_CountPatrol;
    bool m_IsStateLoaded;

    nn::os::Mutex& m_AccessMutex;
    bool m_IsInitialized;
    std::atomic<PatrolReaderThreadState> m_ThreadState;
    nn::os::ThreadType m_PatrolReaderThread;

    nn::os::EventType m_EventSignal;
    nn::os::EventType m_CompleteEvent;

    uint64_t m_PatrolReadAllocateBufferSuccessCount;
    uint64_t m_PatrolReadAllocateBufferFailureCount;

};

PatrolReader g_PatrolReader(g_MmcPartitionStorageMutex);

#endif

Result GetAndClearSdmmcStorageErrorInfo(nn::fs::StorageErrorInfo* pOutStorageErrorInfo, size_t* pOutLogSize, char* pOutLogBuffer, size_t logBufferSize, nn::sdmmc::Port port) NN_NOEXCEPT
{
    nn::sdmmc::ErrorInfo errorInfo;
    nn::sdmmc::GetAndClearErrorInfo(&errorInfo, pOutLogSize, pOutLogBuffer, logBufferSize, port);

    NN_RESULT_THROW_UNLESS(pOutStorageErrorInfo != nullptr, ResultNullptrArgument());
    pOutStorageErrorInfo->numActivationFailures = errorInfo.numActivationFailures;
    pOutStorageErrorInfo->numActivationErrorCorrections = errorInfo.numActivationErrorCorrections;
    pOutStorageErrorInfo->numReadWriteFailures = errorInfo.numReadWriteFailures;
    pOutStorageErrorInfo->numReadWriteErrorCorrections = errorInfo.numReadWriteErrorCorrections;

    NN_RESULT_SUCCESS;
}

} // namespace


#if 0
Result FinalizeSdmmcStorageService() NN_NOEXCEPT
{
}
#endif

void SwitchToPcvClockResetControl()
{
    // 本関数は、MMC・SD カード・ゲームカードの制御を停止させてからコールする想定のため、排他制御は不要。
    // 以下の関数は nn::sdmmc::Initialize(xxx) 前にも実行できるため、初期化済みチェックは不要。
    nn::sdmmc::SwitchToPcvClockResetControl();
}

Result OpenSdStorage(IStorage** outValue) NN_NOEXCEPT
{
    InitializeSd();

    std::lock_guard<nn::os::Mutex> scopedLock(g_SdStorageMutex);
    NN_SDK_REQUIRES(g_SdStorage);

    const nn::sdmmc::Port Port = nn::sdmmc::Port_SdCard0;
    nn::sdmmc::Deactivate(Port); // TORIAEZU
    g_SdCardHandle++;
    NN_DETAIL_FS_SDMMC_RESULT_DO(Port, nn::sdmmc::Activate(Port));

    *outValue = new nn::fssrv::SpeedEmulationStorage(&g_SdStorage.value());
    NN_RESULT_THROW_UNLESS(*outValue != nullptr, nn::fs::ResultAllocationMemoryFailedNew());

    NN_RESULT_SUCCESS;
}

void CloseSdStorage() NN_NOEXCEPT
{
    InitializeSd();

    std::lock_guard<nn::os::Mutex> scopedLock(g_SdStorageMutex);

    nn::sdmmc::Deactivate(nn::sdmmc::Port_SdCard0);
    g_SdCardHandle++;   // g_SdCardHandle が重複しないようカウントアップのみ
}

Result OpenMmcStorage(IStorage** outValue, MmcPartition id) NN_NOEXCEPT
{
    InitializeMmc();

    std::lock_guard<nn::os::Mutex> scopedLockForPartition(g_MmcPartitionStorageMutex);
    std::lock_guard<nn::os::Mutex> scopedLock(g_MmcStorageMutex);
    NN_SDK_REQUIRES(g_MmcStorage);

    if (!g_MmcStorageActive)
    {
        const nn::sdmmc::Port Port = nn::sdmmc::Port_Mmc0;
        NN_DETAIL_FS_SDMMC_RESULT_DO(Port, nn::sdmmc::Activate(Port));

#if defined(NN_FS_PATROL_READER_ENABLED)
        // パトロールリードスレッド起動
        g_PatrolReader.Start();
#endif

        g_MmcStorageActive = true;
    }

    switch( id )
    {
    case MmcPartition::UserData:
    {
        auto emulationStorage = nn::fssystem::AllocateShared<nn::fssrv::SpeedEmulationStorage>(&g_MmcStorage.value());
        NN_RESULT_THROW_UNLESS(
            emulationStorage != nullptr,
            nn::fs::ResultAllocationMemoryFailedInSdmmcStorageServiceB());
        *outValue = new MmcUserDataPartitionStorage(emulationStorage, &g_MmcPartitionStorageMutex);
        break;
    }
    case MmcPartition::BootPartition1:
    case MmcPartition::BootPartition2:
        *outValue = new MmcBootPartitionStorage(&g_MmcStorage.value(), id, &g_MmcPartitionStorageMutex);
        break;
    default:
        return nn::fs::ResultInvalidArgument();
    }
    if( *outValue == nullptr )
    {
        return nn::fs::ResultAllocationMemoryFailedInSdmmcStorageServiceA();
    }
    NN_RESULT_SUCCESS;
}

bool IsSdCardRemovedAfterOpen() NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> scopedLock(g_SdStorageMutex);
    NN_SDK_REQUIRES(g_SdStorage);
    return nn::sdmmc::IsSdCardRemoved(nn::sdmmc::Port_SdCard0);
}

Result GetSdCardHandle(SdCardHandle* pOutValue) NN_NOEXCEPT
{
    InitializeSd();

    NN_RESULT_THROW_UNLESS(pOutValue != nullptr, ResultNullptrArgument());
    *pOutValue = g_SdCardHandle;
    NN_RESULT_SUCCESS;
}

bool IsSdCardValid(SdCardHandle handle) NN_NOEXCEPT
{
    InitializeSd();

    if (handle != g_SdCardHandle)
    {
        return false;
    }

    return !IsSdCardRemovedAfterOpen();
}

void PutSdCardToSleep() NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> scopedLock(g_SdStorageMutex);
    if (g_SdStorage)
    {
        nn::sdmmc::PutSdCardToSleep(nn::sdmmc::Port_SdCard0);
    }
}

void AwakenSdCard() NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> scopedLock(g_SdStorageMutex);
    if (g_SdStorage)
    {
        nn::sdmmc::AwakenSdCard(nn::sdmmc::Port_SdCard0);
    }
}

void ShutdownSdCard() NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> scopedLock(g_SdStorageMutex);
    if (g_SdStorage)
    {
        nn::sdmmc::Deactivate(nn::sdmmc::Port_SdCard0);
    }
}

void RegisterSdCardDetectionEventCallback(DeviceDetectionEventCallback callback, void* pParameter) NN_NOEXCEPT
{
    InitializeSd();
    nn::sdmmc::RegisterSdCardDetectionEventCallback(nn::sdmmc::Port_SdCard0, callback, pParameter);
}

void UnregisterSdCardDetectionEventCallback() NN_NOEXCEPT
{
    InitializeSd();
    nn::sdmmc::UnregisterSdCardDetectionEventCallback(nn::sdmmc::Port_SdCard0);
}

bool IsSdCardInserted() NN_NOEXCEPT
{
    InitializeSd();
    return nn::sdmmc::IsSdCardInserted(nn::sdmmc::Port_SdCard0);
}

Result GetSdCardSpeedMode(nn::fs::SdCardSpeedMode* pOutValue) NN_NOEXCEPT
{
    InitializeSd();

    nn::sdmmc::SpeedMode sdmmcSpeedMode;
    const nn::sdmmc::Port Port = nn::sdmmc::Port_SdCard0;
    NN_DETAIL_FS_SDMMC_RESULT_DO(Port, nn::sdmmc::GetDeviceSpeedMode(&sdmmcSpeedMode, Port));

    NN_RESULT_THROW_UNLESS(pOutValue != nullptr, ResultNullptrArgument());
    switch(sdmmcSpeedMode)
    {
    case nn::sdmmc::SpeedMode_SdCardIdentification:
        *pOutValue = SdCardSpeedMode_Identification;
        break;
    case nn::sdmmc::SpeedMode_SdCardDefaultSpeed:
        *pOutValue = SdCardSpeedMode_DefaultSpeed;
        break;
    case nn::sdmmc::SpeedMode_SdCardHighSpeed:
        *pOutValue = SdCardSpeedMode_HighSpeed;
        break;
    case nn::sdmmc::SpeedMode_SdCardSdr12:
        *pOutValue = SdCardSpeedMode_Sdr12;
        break;
    case nn::sdmmc::SpeedMode_SdCardSdr25:
        *pOutValue = SdCardSpeedMode_Sdr25;
        break;
    case nn::sdmmc::SpeedMode_SdCardSdr50:
        *pOutValue = SdCardSpeedMode_Sdr50;
        break;
    case nn::sdmmc::SpeedMode_SdCardSdr104:
        *pOutValue = SdCardSpeedMode_Sdr104;
        break;
    case nn::sdmmc::SpeedMode_SdCardDdr50:
        *pOutValue = SdCardSpeedMode_Ddr50;
        break;
    default:
        *pOutValue = SdCardSpeedMode_Unknown;
        break;
    }

    NN_RESULT_SUCCESS;
}

Result GetSdCardCid(void* pOutBuffer, size_t size) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(pOutBuffer != nullptr, ResultNullptrArgument());
    NN_RESULT_THROW_UNLESS(size >= nn::sdmmc::DeviceCidSize, ResultInvalidSize());

    InitializeSd();
    const nn::sdmmc::Port Port = nn::sdmmc::Port_SdCard0;
    NN_DETAIL_FS_SDMMC_RESULT_DO(Port, nn::sdmmc::GetDeviceCid(pOutBuffer, size, Port));
    NN_RESULT_SUCCESS;
}

Result GetSdCardUserAreaNumSectors(uint32_t* outValue) NN_NOEXCEPT
{
    InitializeSd();

    const nn::sdmmc::Port Port = nn::sdmmc::Port_SdCard0;
    NN_DETAIL_FS_SDMMC_RESULT_DO(Port, nn::sdmmc::GetDeviceMemoryCapacity(outValue, Port));

    NN_RESULT_SUCCESS;
}

Result GetSdCardUserAreaSize(int64_t* outValue) NN_NOEXCEPT
{
    uint32_t numSectors;
    NN_RESULT_DO(GetSdCardUserAreaNumSectors(&numSectors));

    NN_RESULT_THROW_UNLESS(outValue != nullptr, ResultNullptrArgument());
    *outValue = static_cast<int64_t>(numSectors) * nn::sdmmc::SectorSize;

    NN_RESULT_SUCCESS;
}

Result GetSdCardProtectedAreaNumSectors(uint32_t* outValue) NN_NOEXCEPT
{
    InitializeSd();

    const nn::sdmmc::Port Port = nn::sdmmc::Port_SdCard0;
    NN_DETAIL_FS_SDMMC_RESULT_DO(Port, nn::sdmmc::GetSdCardProtectedAreaCapacity(outValue, Port));

    NN_RESULT_SUCCESS;
}

Result GetSdCardProtectedAreaSize(int64_t* outValue) NN_NOEXCEPT
{
    uint32_t numSectors;
    NN_RESULT_DO(GetSdCardProtectedAreaNumSectors(&numSectors));

    NN_RESULT_THROW_UNLESS(outValue != nullptr, ResultNullptrArgument());
    *outValue = static_cast<int64_t>(numSectors) * nn::sdmmc::SectorSize;

    NN_RESULT_SUCCESS;
}

Result GetAndClearSdCardErrorInfo(nn::fs::StorageErrorInfo* pOutStorageErrorInfo, size_t* pOutLogSize, char* pOutLogBuffer, size_t logBufferSize) NN_NOEXCEPT
{
    InitializeSd();

    return GetAndClearSdmmcStorageErrorInfo(pOutStorageErrorInfo, pOutLogSize, pOutLogBuffer, logBufferSize, nn::sdmmc::Port_SdCard0);
}

void PutMmcToSleep() NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> scopedLock(g_MmcStorageMutex);
    if (g_MmcStorage)
    {
#if defined(NN_FS_PATROL_READER_ENABLED)
        // パトロールリードスレッド休止
        g_PatrolReader.Sleep();
#endif

        nn::sdmmc::PutMmcToSleep(nn::sdmmc::Port_Mmc0);
    }
}

void AwakenMmc() NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> scopedLock(g_MmcStorageMutex);
    if (g_MmcStorage)
    {
        nn::sdmmc::AwakenMmc(nn::sdmmc::Port_Mmc0);

#if defined(NN_FS_PATROL_READER_ENABLED)
        // パトロールリードスレッド再開
        g_PatrolReader.Resume();
#endif
    }
}

void ShutdownMmc() NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> scopedLock(g_MmcStorageMutex);
    if (g_MmcStorage)
    {
#if defined(NN_FS_PATROL_READER_ENABLED)
        // パトロールリードスレッド終了
        g_PatrolReader.Stop();
#endif

        nn::sdmmc::Deactivate(nn::sdmmc::Port_Mmc0);
    }
}

Result GetMmcSpeedMode(nn::fs::MmcSpeedMode* pOutValue) NN_NOEXCEPT
{
    InitializeMmc();
    nn::sdmmc::SpeedMode sdmmcSpeedMode;
    const nn::sdmmc::Port Port = nn::sdmmc::Port_Mmc0;
    NN_DETAIL_FS_SDMMC_RESULT_DO(Port, nn::sdmmc::GetDeviceSpeedMode(&sdmmcSpeedMode, nn::sdmmc::Port_Mmc0));

    NN_RESULT_THROW_UNLESS(pOutValue != nullptr, ResultNullptrArgument());
    switch(sdmmcSpeedMode)
    {
    case nn::sdmmc::SpeedMode_MmcIdentification:
        *pOutValue = nn::fs::MmcSpeedMode_Identification;
        break;
    case nn::sdmmc::SpeedMode_MmcLegacySpeed:
        *pOutValue = nn::fs::MmcSpeedMode_LegacySpeed;
        break;
    case nn::sdmmc::SpeedMode_MmcHighSpeed:
        *pOutValue = nn::fs::MmcSpeedMode_HighSpeed;
        break;
    case nn::sdmmc::SpeedMode_MmcHs200:
        *pOutValue = nn::fs::MmcSpeedMode_Hs200;
        break;
    case nn::sdmmc::SpeedMode_MmcHs400:
        *pOutValue = nn::fs::MmcSpeedMode_Hs400;
        break;
    default:
        *pOutValue = nn::fs::MmcSpeedMode_Unknown;
        break;
    }

    NN_RESULT_SUCCESS;
}

Result GetMmcCid(void* pOutBuffer, size_t size) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(pOutBuffer != nullptr, ResultNullptrArgument());
    NN_RESULT_THROW_UNLESS(size >= nn::sdmmc::DeviceCidSize, ResultInvalidSize());

    InitializeMmc();
    const nn::sdmmc::Port Port = nn::sdmmc::Port_Mmc0;
    NN_DETAIL_FS_SDMMC_RESULT_DO(Port, nn::sdmmc::GetDeviceCid(pOutBuffer, size, Port));
    NN_RESULT_SUCCESS;
}

Result EraseMmc(nn::fs::MmcPartition id) NN_NOEXCEPT
{
    InitializeMmc();

    nn::sdmmc::MmcPartition sdmmcPartition = nn::sdmmc::MmcPartition_Unknown;
    switch(id)
    {
    case nn::fs::MmcPartition::UserData:
        sdmmcPartition = nn::sdmmc::MmcPartition_UserData;
        break;
    case nn::fs::MmcPartition::BootPartition1:
        sdmmcPartition = nn::sdmmc::MmcPartition_BootPartition1;
        break;
    case nn::fs::MmcPartition::BootPartition2:
        sdmmcPartition = nn::sdmmc::MmcPartition_BootPartition2;
        break;
    default:
        return ResultInvalidArgument();
    }

    std::lock_guard<nn::os::Mutex> scopedLockForPartition(g_MmcPartitionStorageMutex);
    std::lock_guard<nn::os::Mutex> scopedLock(g_MmcStorageMutex);

    const nn::sdmmc::Port Port = nn::sdmmc::Port_Mmc0;

    // Boot Partition の切り替えに失敗した場合は ABORT する
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::sdmmc::SelectMmcPartition(Port, sdmmcPartition));
    NN_UTIL_SCOPE_EXIT
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::sdmmc::SelectMmcPartition(Port, nn::sdmmc::MmcPartition::MmcPartition_UserData));
    };

    NN_DETAIL_FS_SDMMC_RESULT_DO(Port, nn::sdmmc::EraseMmc(Port));
    NN_RESULT_SUCCESS;
}

Result GetMmcPartitionSize(int64_t* outValue, MmcPartition id) NN_NOEXCEPT
{
    InitializeMmc();

    std::lock_guard<nn::os::Mutex> scopedLockForPartition(g_MmcPartitionStorageMutex);
    std::lock_guard<nn::os::Mutex> scopedLock(g_MmcStorageMutex);

    uint32_t numSectors = 0;
    const nn::sdmmc::Port Port = nn::sdmmc::Port_Mmc0;
    switch(id)
    {
    case nn::fs::MmcPartition::UserData:
        NN_DETAIL_FS_SDMMC_RESULT_DO(Port, nn::sdmmc::GetDeviceMemoryCapacity(&numSectors, Port));
        break;
    case nn::fs::MmcPartition::BootPartition1:
        NN_FALL_THROUGH;    // Boot Partition 1, 2 のサイズは同じ
    case nn::fs::MmcPartition::BootPartition2:
        NN_DETAIL_FS_SDMMC_RESULT_DO(Port, nn::sdmmc::GetMmcBootPartitionCapacity(&numSectors, Port));
        break;
    default:
        return ResultInvalidArgument();
    }
    NN_RESULT_THROW_UNLESS(outValue != nullptr, ResultNullptrArgument());
    *outValue = static_cast<int64_t>(numSectors) * nn::sdmmc::SectorSize;

    NN_RESULT_SUCCESS;
}

Result GetMmcPatrolCount(uint32_t* outValue) NN_NOEXCEPT
{
#if defined(NN_FS_PATROL_READER_ENABLED)
    return g_PatrolReader.GetPatrolCount(outValue);
#else
    NN_RESULT_THROW_UNLESS(outValue != nullptr, ResultNullptrArgument());
    *outValue = 0;
    NN_RESULT_SUCCESS;
#endif
}

Result GetAndClearMmcErrorInfo(nn::fs::StorageErrorInfo* pOutStorageErrorInfo, size_t* pOutLogSize, char* pOutLogBuffer, size_t logBufferSize) NN_NOEXCEPT
{
    InitializeMmc();

    std::lock_guard<nn::os::Mutex> scopedLockForPartition(g_MmcPartitionStorageMutex);
    std::lock_guard<nn::os::Mutex> scopedLock(g_MmcStorageMutex);

    return GetAndClearSdmmcStorageErrorInfo(pOutStorageErrorInfo, pOutLogSize, pOutLogBuffer, logBufferSize, nn::sdmmc::Port_Mmc0);
}

Result GetMmcExtendedCsd(void* pOutBuffer, size_t size) NN_NOEXCEPT
{
    InitializeMmc();

    const fssystem::PooledBuffer pooledBuffer(
        nn::sdmmc::MmcExtendedCsdSize,
        nn::sdmmc::MmcExtendedCsdSize);
    NN_RESULT_THROW_UNLESS(pooledBuffer.GetSize() >= nn::sdmmc::MmcExtendedCsdSize, ResultAllocationMemoryFailedInSdmmcStorageServiceC());
    NN_SDK_ASSERT(IsBufferCapable(pooledBuffer.GetBuffer()));

    const nn::sdmmc::Port Port = nn::sdmmc::Port_Mmc0;
    NN_DETAIL_FS_SDMMC_RESULT_DO(Port, nn::sdmmc::GetMmcExtendedCsd(pooledBuffer.GetBuffer(), pooledBuffer.GetSize(), Port));

    NN_RESULT_THROW_UNLESS(pOutBuffer != nullptr, ResultNullptrArgument());
    NN_RESULT_THROW_UNLESS(size >= nn::sdmmc::MmcExtendedCsdSize, ResultInvalidSize());
    std::memcpy(pOutBuffer, pooledBuffer.GetBuffer(), nn::sdmmc::MmcExtendedCsdSize);

    NN_RESULT_SUCCESS;
}

Result SuspendMmcPatrol() NN_NOEXCEPT
{
#if defined(NN_FS_PATROL_READER_ENABLED)
    std::lock_guard<nn::os::Mutex> scopedLock(g_MmcStorageMutex);
    if (g_MmcStorage)
    {
        // パトロールリードスレッド休止
        g_PatrolReader.Sleep();
    }
#endif

    NN_RESULT_SUCCESS;
}

Result ResumeMmcPatrol() NN_NOEXCEPT
{
#if defined(NN_FS_PATROL_READER_ENABLED)
    std::lock_guard<nn::os::Mutex> scopedLock(g_MmcStorageMutex);
    if (g_MmcStorage)
    {
        // パトロールリードスレッド再開
        g_PatrolReader.Resume();
    }
#endif

    NN_RESULT_SUCCESS;
}

void GetAndClearPatrolReadAllocateBufferCount(uint64_t* outSuccessCount, uint64_t* outFailureCount) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outSuccessCount);
    NN_SDK_REQUIRES_NOT_NULL(outFailureCount);
#if defined(NN_FS_PATROL_READER_ENABLED)
    g_PatrolReader.GetAndClearPatrolReadAllocateBufferCount(outSuccessCount, outFailureCount);
#else
    *outSuccessCount = 0;
    *outFailureCount = 0;
#endif
}

Result GetSdmmcConnectionStatus(nn::fs::SdmmcSpeedMode* pOutSdmmcSpeedMode, nn::fs::SdmmcBusWidth* pOutSdmmcBusWidth, nn::fs::SdmmcPort sdmmcPort) NN_NOEXCEPT
{
    // 情報取得のみで制御ポートに作用は起こらないため、排他制御は省略する

    NN_SDK_REQUIRES_NOT_NULL(pOutSdmmcSpeedMode);
    NN_SDK_REQUIRES_NOT_NULL(pOutSdmmcBusWidth);

    nn::sdmmc::Port port;
    switch (sdmmcPort)
    {
    case nn::fs::SdmmcPort_Mmc0:
        port = nn::sdmmc::Port_Mmc0;
        break;
    case nn::fs::SdmmcPort_SdCard0:
        port = nn::sdmmc::Port_SdCard0;
        break;
    case nn::fs::SdmmcPort_GcAsic0:
        port = nn::sdmmc::Port_GcAsic0;
        break;
    default:
        return nn::fs::ResultInvalidArgument();
    }

    nn::sdmmc::SpeedMode speedMode;
    nn::sdmmc::BusWidth busWidth;
    NN_RESULT_DO(nn::sdmmc::CheckConnection(&speedMode, &busWidth, port));

    switch (speedMode)
    {
    case nn::sdmmc::SpeedMode_MmcIdentification:
        *pOutSdmmcSpeedMode= nn::fs::SdmmcSpeedMode_MmcIdentification;
        break;
    case nn::sdmmc::SpeedMode_MmcLegacySpeed:
        *pOutSdmmcSpeedMode= nn::fs::SdmmcSpeedMode_MmcLegacySpeed;
        break;
    case nn::sdmmc::SpeedMode_MmcHighSpeed:
        *pOutSdmmcSpeedMode= nn::fs::SdmmcSpeedMode_MmcHighSpeed;
        break;
    case nn::sdmmc::SpeedMode_MmcHs200:
        *pOutSdmmcSpeedMode= nn::fs::SdmmcSpeedMode_MmcHs200;
        break;
    case nn::sdmmc::SpeedMode_MmcHs400:
        *pOutSdmmcSpeedMode= nn::fs::SdmmcSpeedMode_MmcHs400;
        break;
    case nn::sdmmc::SpeedMode_SdCardIdentification:
        *pOutSdmmcSpeedMode= nn::fs::SdmmcSpeedMode_SdCardIdentification;
        break;
    case nn::sdmmc::SpeedMode_SdCardDefaultSpeed:
        *pOutSdmmcSpeedMode= nn::fs::SdmmcSpeedMode_SdCardDefaultSpeed;
        break;
    case nn::sdmmc::SpeedMode_SdCardHighSpeed:
        *pOutSdmmcSpeedMode= nn::fs::SdmmcSpeedMode_SdCardHighSpeed;
        break;
    case nn::sdmmc::SpeedMode_SdCardSdr12:
        *pOutSdmmcSpeedMode= nn::fs::SdmmcSpeedMode_SdCardSdr12;
        break;
    case nn::sdmmc::SpeedMode_SdCardSdr25:
        *pOutSdmmcSpeedMode= nn::fs::SdmmcSpeedMode_SdCardSdr25;
        break;
    case nn::sdmmc::SpeedMode_SdCardSdr50:
        *pOutSdmmcSpeedMode= nn::fs::SdmmcSpeedMode_SdCardSdr50;
        break;
    case nn::sdmmc::SpeedMode_SdCardSdr104:
        *pOutSdmmcSpeedMode= nn::fs::SdmmcSpeedMode_SdCardSdr104;
        break;
    case nn::sdmmc::SpeedMode_SdCardDdr50:
        *pOutSdmmcSpeedMode= nn::fs::SdmmcSpeedMode_SdCardDdr50;
        break;
    case nn::sdmmc::SpeedMode_GcAsicFpgaSpeed:
        *pOutSdmmcSpeedMode= nn::fs::SdmmcSpeedMode_GcAsicFpgaSpeed;
        break;
    case nn::sdmmc::SpeedMode_GcAsicSpeed:
        *pOutSdmmcSpeedMode= nn::fs::SdmmcSpeedMode_GcAsicSpeed;
        break;
    default:
        *pOutSdmmcSpeedMode= nn::fs::SdmmcSpeedMode_Unknown;
        break;
    }

    switch (busWidth)
    {
    case nn::sdmmc::BusWidth_1Bit:
        *pOutSdmmcBusWidth= nn::fs::SdmmcBusWidth_1Bit;
        break;
    case nn::sdmmc::BusWidth_4Bit:
        *pOutSdmmcBusWidth= nn::fs::SdmmcBusWidth_4Bit;
        break;
    case nn::sdmmc::BusWidth_8Bit:
        *pOutSdmmcBusWidth= nn::fs::SdmmcBusWidth_8Bit;
        break;
    default:
        *pOutSdmmcBusWidth= nn::fs::SdmmcBusWidth_Unknown;
        break;
    }

    NN_RESULT_SUCCESS;
}

void SuspendSdmmcControl() NN_NOEXCEPT
{
    if (g_SdStorage)
    {
        nn::sdmmc::Finalize(nn::sdmmc::Port_SdCard0);
    }
    if (g_MmcStorage)
    {
        SuspendMmcPatrol();
        nn::sdmmc::Finalize(nn::sdmmc::Port_Mmc0);
    }
}

void ResumeSdmmcControl() NN_NOEXCEPT
{
    // ローカルのデバッグ用途で、動作を保証できるものではない
    if (g_MmcStorage)
    {
        nn::sdmmc::Initialize(nn::sdmmc::Port_Mmc0);
        if (g_MmcStorageActive)
        {
            // MMC は抜き差しがないため、つじつまは合うはず
            (void)nn::sdmmc::Activate(nn::sdmmc::Port_Mmc0);
        }
    }
    if (g_SdStorage)
    {
        nn::sdmmc::Initialize(nn::sdmmc::Port_SdCard0);
        if (g_SdCardHandle != InvalidSdCardHandle)
        {
            // Suspend 中に抜き差しされた場合はつじつまが合わない
            (void)nn::sdmmc::Activate(nn::sdmmc::Port_SdCard0);
            ResumeMmcPatrol();
        }
    }
}

}}}
