﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <mutex>
#include <nn/nn_Common.h>
#include <nn/nn_SdkLog.h>
#include <nn/fs/fs_Result.h>
#include <nn/util/util_IntrusiveList.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/os/os_Mutex.h>
#include "fssrv_SaveDataTransferPorterManager.h"

namespace nn { namespace fssrv { namespace detail {


Prohibitee::~Prohibitee() NN_NOEXCEPT
{
    if (m_IsRegistered)
    {
        m_pPorterManager->UnregisterPorter(this);
    }
}

Prohibitee::Prohibitee(SaveDataPorterManager* pPorterManager) NN_NOEXCEPT
    : m_pPorterManager(pPorterManager)
{
}

Result Prohibitee::Initialize() NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(m_pPorterManager->RegisterPorter(this, GetApplicationId()), fs::ResultTargetLocked());

    m_IsRegistered = true;
    NN_RESULT_SUCCESS;
}

SaveDataPorterProhibiter::~SaveDataPorterProhibiter() NN_NOEXCEPT
{
    m_pPorterManager->UnregisterProhibiter(this);
}


bool SaveDataPorterManager::RegisterPorter(Prohibitee* pPorter, ncm::ApplicationId applicationId) NN_NOEXCEPT
{
    std::lock_guard<os::Mutex> scopedLock(m_Mutex);

    for (auto& prohibiter : m_ProhibiterList)
    {
        if (applicationId == prohibiter.GetApplicationId())
        {
            return false;
        }
    }
    m_PorterList.push_back(*pPorter);
    return true;
}

void SaveDataPorterManager::UnregisterPorter(Prohibitee* pPorter) NN_NOEXCEPT
{
    std::lock_guard<os::Mutex> scopedLock(m_Mutex);
    m_PorterList.erase(m_PorterList.iterator_to(*pPorter));
}

void SaveDataPorterManager::RegisterProhibiter(SaveDataPorterProhibiter* pProhibiter) NN_NOEXCEPT
{
    std::lock_guard<os::Mutex> scopedLock(m_Mutex);
    m_ProhibiterList.push_back(*pProhibiter);

    for (auto& porter : m_PorterList)
    {
        if (porter.GetApplicationId() == pProhibiter->GetApplicationId())
        {
            porter.Invalidate();
        }
    }

}

void SaveDataPorterManager::UnregisterProhibiter(SaveDataPorterProhibiter* pProhibiter) NN_NOEXCEPT
{
    std::lock_guard<os::Mutex> scopedLock(m_Mutex);
    m_ProhibiterList.erase(m_ProhibiterList.iterator_to(*pProhibiter));
}



}}}

