﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <array>

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>

#include <nn/fs.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/nn_SdkLog.h>
#include <nn/nn_SdkAssert.h>

#include <nn/fs/detail/fs_Newable.h>
#include <nn/fs/fsa/fs_IFile.h>
#include <nn/fs/fsa/fs_IFileSystem.h>
#include <nn/fs/fs_Utility.h>
#include <nn/fs/fs_SubStorage.h>
#include <nn/fs/fs_FileStorage.h>
#include <nn/fs/fs_MemoryStorage.h>
#include <nn/crypto/crypto_Compare.h>
#include <nn/crypto/crypto_Sha256Generator.h>
#include <nn/crypto/crypto_Aes128CmacGenerator.h>
#include <nn/sf/sf_ObjectFactory.h>
#include <nn/fs/fs_SaveDataTransferVersion2.h>

#include "fssrv_AllocatorForServiceFrameWork.h"
#include <nn/fs/fs_SaveDataManagement.h>
#include <nn/fs/fs_SaveDataPrivate.h>
#include "fssrv_SaveDataTransferVersion2.h"
#include "fssrv_SaveDataTransferStream.h"
#include "fssrv_SaveDataTransferPorterCore.h"
#include "fssrv_SaveDataTransferChunkIterator.h"
#include <nn/fssystem/fs_AllocatorUtility.h>
#include <nn/fssystem/fs_Utility.h>
#include <nn/fssystem/fs_ConcatenationFile.h>
#include <nn/sf/sf_ISharedObject.h>

namespace nn { namespace fssrv { namespace detail {

namespace {
    typedef nn::sf::ObjectFactory<AllocatorForServiceFrameWork::Policy> FileSystemFactory;

    Result CalculateStorageHash(InitialDataVersion2Detail::Hash* pOutValue, fs::IStorage* pStorage, char* buffer, size_t bufferSize)
    {
        int64_t restSize;
        NN_RESULT_DO(pStorage->GetSize(&restSize));

        crypto::Sha256Generator sha;
        sha.Initialize();

        int64_t offset = 0;
        while (restSize > 0)
        {
            auto size = static_cast<size_t>(std::min(static_cast<int64_t>(bufferSize), restSize));
            NN_RESULT_DO(pStorage->Read(offset, buffer, size));
            sha.Update(buffer, size);

            restSize -= size;
            offset += size;
        }
        sha.GetHash(pOutValue->value, sizeof(pOutValue->value));

        NN_RESULT_SUCCESS;
    }
}

// for export
SaveDataPorterCore::SaveDataPorterCore(std::shared_ptr<fssystem::ProxyStorage> saveDataConcatenatedStorage, const fs::SaveDataExtraData& extraData, std::unique_ptr<InitialDataVersion2Detail::Content>&& initialDataTheirs) NN_NOEXCEPT
    : m_ConcatenatedStorage(std::move(saveDataConcatenatedStorage))
    , m_InitialDataTheirs(std::move(initialDataTheirs))
{
    memset(&m_InitialDataOurs, 0, sizeof(m_InitialDataOurs));
    m_InitialDataOurs.singleSaveData.extraData = extraData;
}

// for import
SaveDataPorterCore::SaveDataPorterCore(std::shared_ptr<fssystem::ProxyStorage> saveDataConcatenatedStorage, std::shared_ptr<fs::fsa::IFileSystem> fileSystem, std::unique_ptr<InitialDataVersion2Detail::Content>&& initialDataTheirs) NN_NOEXCEPT
    : m_ConcatenatedStorage(std::move(saveDataConcatenatedStorage))
    , m_FileSystem(fileSystem)
    , m_InitialDataTheirs(std::move(initialDataTheirs))
{
    memset(&m_InitialDataOurs, 0, sizeof(m_InitialDataOurs));
}

Result SaveDataPorterCore::CreateForExport(std::shared_ptr<SaveDataPorterCore>* pOutValue, std::shared_ptr<fs::IStorage> saveDataConcatenatedStorage, const fs::SaveDataExtraData& extraData, std::unique_ptr<InitialDataVersion2Detail::Content>&& initialDataTheirs) NN_NOEXCEPT
{
    auto proxyStorage = fssystem::AllocateShared<fssystem::ProxyStorage>(std::move(saveDataConcatenatedStorage), fs::ResultSaveDataPorterCoreInvalidated());
    NN_RESULT_THROW_UNLESS(proxyStorage != nullptr, fs::ResultAllocationMemoryFailedAllocateShared());

    auto porter = fssystem::AllocateShared<SaveDataPorterCore>(std::move(proxyStorage), extraData, std::move(initialDataTheirs));
    NN_RESULT_THROW_UNLESS(porter != nullptr, fs::ResultAllocationMemoryFailedAllocateShared());

    *pOutValue = std::move(porter);
    NN_RESULT_SUCCESS;
}

Result SaveDataPorterCore::CreateForImport(std::shared_ptr<SaveDataPorterCore>* pOutValue, std::shared_ptr<fs::IStorage> saveDataConcatenatedStorage, std::shared_ptr<fs::fsa::IFileSystem> fileSystem, std::unique_ptr<InitialDataVersion2Detail::Content>&& initialDataTheirs) NN_NOEXCEPT
{
    auto proxyStorage = fssystem::AllocateShared<fssystem::ProxyStorage>(std::move(saveDataConcatenatedStorage), fs::ResultSaveDataPorterCoreInvalidated());
    NN_RESULT_THROW_UNLESS(proxyStorage != nullptr, fs::ResultAllocationMemoryFailedAllocateShared());

    auto porter = fssystem::AllocateShared<SaveDataPorterCore>(std::move(proxyStorage), std::move(fileSystem), std::move(initialDataTheirs));
    NN_RESULT_THROW_UNLESS(porter != nullptr, fs::ResultAllocationMemoryFailedAllocateShared());

    NN_RESULT_DO(porter->SetDivisionCount(porter->m_InitialDataTheirs->singleSaveData.divisionCount));

    *pOutValue = std::move(porter);
    NN_RESULT_SUCCESS;
}

Result SaveDataPorterCore::SetDivisionCount(int count) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(count > 0, fs::ResultPreconditionViolation());
    NN_RESULT_THROW_UNLESS(count <= InitialDataVersion2Detail::ChunkDigestCountMax, fs::ResultPreconditionViolation());

    // TBD: align_up 不要？
    int64_t concatenatedSize;
    NN_RESULT_DO(m_ConcatenatedStorage->GetSize(&concatenatedSize));
    m_DivisionSize = util::align_up((concatenatedSize + count - 1) / count, AlignSize);
    m_DivisionCount = static_cast<int>((concatenatedSize + (m_DivisionSize - 1)) / m_DivisionSize);

    NN_RESULT_DO(ConstructInitialData());

    NN_RESULT_SUCCESS;
}

Result SaveDataPorterCore::OpenSaveDataDiffChunkIteratorAll(nn::sf::Out<nn::sf::SharedPointer<nn::fssrv::sf::ISaveDataChunkIterator>> outValue, bool includeInitialData, int divisionCount) NN_NOEXCEPT
{
    auto iterator = FileSystemFactory::CreateSharedEmplaced<nn::fssrv::sf::ISaveDataChunkIterator, SaveDataChunkIteratorAll>(includeInitialData, divisionCount);
    NN_RESULT_THROW_UNLESS(iterator != nullptr, nn::fs::ResultAllocationMemoryFailed());
    outValue.Set(std::move(iterator));

    NN_RESULT_SUCCESS;
}

Result SaveDataPorterCore::OpenSaveDataDiffChunkIterator(nn::sf::Out<nn::sf::SharedPointer<nn::fssrv::sf::ISaveDataChunkIterator>> outValue, bool includeInitialData) NN_NOEXCEPT
{
    NN_SDK_ASSERT(m_Initialized);
    NN_RESULT_THROW_UNLESS(m_DivisionCount > 0, fs::ResultPreconditionViolation());

    if (m_InitialDataTheirs == nullptr)
    {
        return OpenSaveDataDiffChunkIteratorAll(outValue, includeInitialData, m_DivisionCount);
    }
    else
    {
        auto iterator = FileSystemFactory::CreateSharedEmplaced<nn::fssrv::sf::ISaveDataChunkIterator, SaveDataChunkIteratorDiff>(*m_InitialDataTheirs, m_InitialDataOurs, includeInitialData, m_DivisionCount);;
        NN_RESULT_THROW_UNLESS(iterator != nullptr, nn::fs::ResultAllocationMemoryFailed());
        outValue.Set(std::move(iterator));

        NN_RESULT_SUCCESS;
    }
}

Result SaveDataPorterCore::OpenSaveDataChunkStorage(std::shared_ptr<fs::IStorage>* outValue, fs::SaveDataChunkId id) NN_NOEXCEPT
{
    NN_SDK_ASSERT(m_Initialized);
    NN_RESULT_THROW_UNLESS(m_DivisionCount > 0, fs::ResultPreconditionViolation());

    if (id == fs::SaveDataChunkIdForInitialData)
    {
        // TODO: initialData に必要なデータが揃っていない(e.g. 必要な chunk の export が未実施)場合にエラーにする

        // TORIAEZU: initialdata をこの時点で作る
        m_InitialDataOurs.singleSaveData.divisionCount = m_DivisionCount;

        auto initialDataStorage = fssystem::AllocateShared<fs::MemoryStorage>(&m_InitialDataOurs, sizeof(m_InitialDataOurs));
        NN_RESULT_THROW_UNLESS(initialDataStorage != nullptr, fs::ResultAllocationMemoryFailedNew());

        *outValue = std::move(initialDataStorage);
        NN_RESULT_SUCCESS;
    }
    else
    {
        std::shared_ptr<fs::IStorage> subStorage;
        NN_RESULT_DO(OpenSaveDataSubStorage(&subStorage, GetIndexById(id)));

        *outValue = std::move(subStorage);
        NN_RESULT_SUCCESS;
    }

}

Result SaveDataPorterCore::ReportMac(fs::SaveDataChunkId id, char* pMac, size_t macSize) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(macSize == InitialDataVersion2Detail::MacSize, fs::ResultInvalidArgument());
    NN_RESULT_THROW_UNLESS(id != fs::SaveDataChunkIdForInitialData, fs::ResultInvalidArgument());

    memcpy(m_InitialDataOurs.singleSaveData.chunkMac[GetIndexById(id)], pMac, macSize);

    // TODO: reported か管理

    NN_RESULT_SUCCESS;
}

Result SaveDataPorterCore::ReportSize(fs::SaveDataChunkId id, int64_t size) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(id != fs::SaveDataChunkIdForInitialData, fs::ResultInvalidArgument());

    m_InitialDataOurs.singleSaveData.chunkSize[GetIndexById(id)] = size;

    // TODO: reported か管理

    NN_RESULT_SUCCESS;
}

Result SaveDataPorterCore::FinalizeImport() NN_NOEXCEPT
{
    // todo: ダイジェスト、サイズ検証

    return m_FileSystem->Commit();
}

Result SaveDataPorterCore::FinalizeExport(detail::InitialDataVersion2Detail::Content* pOutInitialData) NN_NOEXCEPT
{
    // TODO: 必要なチャンクについて完了していることの検証

    *pOutInitialData = m_InitialDataOurs;
    NN_RESULT_SUCCESS;
}

int64_t SaveDataPorterCore::GetSize(fs::SaveDataChunkId id) NN_NOEXCEPT
{
    return m_InitialDataTheirs->singleSaveData.chunkSize[GetIndexById(id)];
}

void SaveDataPorterCore::Invalidate() NN_NOEXCEPT
{
    m_ConcatenatedStorage->Invalidate();
    m_FileSystem.reset();
}

Result SaveDataPorterCore::OpenSaveDataSubStorage(std::shared_ptr<fs::IStorage>* outValue, int index) NN_NOEXCEPT
{
    int64_t concatenatedSize;
    NN_RESULT_DO(m_ConcatenatedStorage->GetSize(&concatenatedSize));

    int64_t offset = m_DivisionSize * index;
    int64_t size = std::min(m_DivisionSize * (index + 1), concatenatedSize) - offset;

    auto subStorage = fssystem::AllocateShared<fs::SubStorage>(m_ConcatenatedStorage.get(), offset, size);
    NN_RESULT_THROW_UNLESS(subStorage != nullptr, fs::ResultAllocationMemoryFailedNew());

    *outValue = std::move(subStorage);
    NN_RESULT_SUCCESS;
}

namespace {
    void InheritPreviousInitialData(InitialDataVersion2Detail::Content* pInitialDataOurs, const InitialDataVersion2Detail::Content& initialDataTheirs)
    {
        // TODO: 差分箇所は必ず改めて report されることを管理
        memcpy(pInitialDataOurs->singleSaveData.chunkMac, initialDataTheirs.singleSaveData.chunkMac, sizeof(pInitialDataOurs->singleSaveData.chunkMac));
        memcpy(pInitialDataOurs->singleSaveData.chunkSize, initialDataTheirs.singleSaveData.chunkSize, sizeof(pInitialDataOurs->singleSaveData.chunkSize));
    }
}

Result SaveDataPorterCore::ConstructInitialData() NN_NOEXCEPT
{
    if (m_InitialDataTheirs != nullptr)
    {
        InheritPreviousInitialData(&m_InitialDataOurs, *m_InitialDataTheirs);
    }

    for (int i = 0; i < m_DivisionCount; i++)
    {
        std::shared_ptr<fs::IStorage> subStorage;
        NN_RESULT_DO(OpenSaveDataSubStorage(&subStorage, i));

        // TODO: ワークバッファの変更
        const size_t WorkBufferSize = 128 * 1024;
        struct WorkBuffer : public fs::detail::Newable
        {
        public:
            char buffer[WorkBufferSize];
        };
        std::unique_ptr<WorkBuffer> workBuffer(new WorkBuffer());
        NN_RESULT_THROW_UNLESS(workBuffer != nullptr, fs::ResultAllocationMemoryFailedNew());

        // TODO: 計算済みの階層化ハッシュを利用した最適化
        NN_RESULT_DO(CalculateStorageHash(&m_InitialDataOurs.singleSaveData.chunkDigest[i], subStorage.get(), workBuffer->buffer, WorkBufferSize));
    }

    // TODO: 他の initial data member

    m_Initialized = true;
    NN_RESULT_SUCCESS;
}

Result SaveDataPorterCore::GetInitialDataAadTheirs(fs::detail::InitialDataAad* outValue) NN_NOEXCEPT
{
    memcpy(outValue->data, &m_InitialDataTheirs->singleSaveData.aad.data, fs::detail::InitialDataAad::Size);
    NN_RESULT_SUCCESS;
}

Result SaveDataPorterCore::SetInitialDataAad(const fs::detail::InitialDataAad& aad) NN_NOEXCEPT
{
    memcpy(&m_InitialDataOurs.singleSaveData.aad.data, aad.data, fs::detail::InitialDataAad::Size);
    NN_RESULT_SUCCESS;
}

}}}
