﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>

#include <nn/result/result_HandlingUtility.h>
#include <nn/nn_SdkLog.h>
#include <nn/nn_SdkAssert.h>

#include <nn/fs/detail/fs_Newable.h>
#include "fssrv_SaveDataTransferVersion2.h"
#include <nn/sf/sf_ISharedObject.h>


namespace nn { namespace fssrv { namespace detail {

const uint16_t IdBaseForDivisionIndex = 0x1000;

inline int GetIndexById(fs::SaveDataChunkId id)
{
    NN_SDK_ASSERT(id >= IdBaseForDivisionIndex);
    return id - IdBaseForDivisionIndex;
}

class SaveDataChunkIteratorDiff : public fs::detail::Newable
{
public:
    SaveDataChunkIteratorDiff(const InitialDataVersion2Detail::Content& initialDataTheirs, const InitialDataVersion2Detail::Content& initialDataOurs, bool includeInitialData, int divisionCount) NN_NOEXCEPT
        : m_Id(IdBaseForDivisionIndex - 1)
        , m_IndexDivisionEnd(IdBaseForDivisionIndex + divisionCount)
        , m_IncludeInitialData(includeInitialData)
    {
        // TODO: diff が全くない場合に SaveDataChunkIdForInitialData を列挙しない @export
        memcpy(&m_InitialDataTheirs, &initialDataTheirs, sizeof(m_InitialDataTheirs));
        memcpy(&m_InitialDataOurs, &initialDataOurs, sizeof(m_InitialDataOurs));

        Next();
    }

    Result GetId(nn::sf::Out<uint32_t> outValue) const NN_NOEXCEPT
    {
        NN_RESULT_THROW_UNLESS(!IsEnd(), fs::ResultPreconditionViolation());
        outValue.Set(m_Id);
        NN_RESULT_SUCCESS;
    }

    Result Next() NN_NOEXCEPT
    {
        while (!IsEnd())
        {
            m_Id++;

            if (m_Id == m_IndexDivisionEnd && m_IncludeInitialData)
            {
                m_Id = fs::SaveDataChunkIdForInitialData;
                NN_RESULT_SUCCESS;
            }

            if (IsEnd())
            {
                break;
            }

            auto index = GetIndexById(static_cast<fs::SaveDataChunkId>(m_Id));

            if (!crypto::IsSameBytes(m_InitialDataTheirs.singleSaveData.chunkDigest[index].value, m_InitialDataOurs.singleSaveData.chunkDigest[index].value, InitialDataVersion2Detail::Hash::Size))
            {
                break;
            }
        }

        NN_RESULT_SUCCESS;
    }

    Result IsEnd(nn::sf::Out<bool> outValue)
    {
        outValue.Set(IsEnd());
        NN_RESULT_SUCCESS;
    }

private:

    bool IsEnd() const NN_NOEXCEPT
    {
        if (m_IncludeInitialData)
        {
            return m_Id == fs::SaveDataChunkIdForInitialData + 1;
        }
        else
        {
            return m_Id == m_IndexDivisionEnd;
        }
    }

    int m_Id;
    const int m_IndexDivisionEnd;
    const bool m_IncludeInitialData;

    // サイズ要最適化
    InitialDataVersion2Detail::Content m_InitialDataTheirs;
    InitialDataVersion2Detail::Content m_InitialDataOurs;
};

class SaveDataChunkIteratorAll : public fs::detail::Newable
{
public:
    SaveDataChunkIteratorAll(bool includeInitialData, int divisionCount) NN_NOEXCEPT
        : m_Id(IdBaseForDivisionIndex)
        , m_IndexDivisionEnd(IdBaseForDivisionIndex + divisionCount)
        , m_IncludeInitialData(includeInitialData)
    {
    }

    ~SaveDataChunkIteratorAll() NN_NOEXCEPT {}

    Result GetId(nn::sf::Out<uint32_t> outValue) const NN_NOEXCEPT
    {
        NN_RESULT_THROW_UNLESS(!IsEnd(), fs::ResultPreconditionViolation());
        outValue.Set(m_Id);
        NN_RESULT_SUCCESS;
    }

    Result Next() NN_NOEXCEPT
    {
        if (IsEnd())
        {
            // do nothing
        }
        else if (m_Id + 1 == m_IndexDivisionEnd && m_IncludeInitialData)
        {
            m_Id = fs::SaveDataChunkIdForInitialData;
        }
        else
        {
            m_Id++;
        }
        NN_RESULT_SUCCESS;
    }

    Result IsEnd(nn::sf::Out<bool> outValue)
    {
        outValue.Set(IsEnd());
        NN_RESULT_SUCCESS;
    }

private:
    bool IsEnd() const NN_NOEXCEPT
    {
        if (m_IncludeInitialData)
        {
            return m_Id == fs::SaveDataChunkIdForInitialData + 1;
        }
        else
        {
            return m_Id == m_IndexDivisionEnd;
        }
    }

    int m_Id;
    const int m_IndexDivisionEnd;
    const bool m_IncludeInitialData;
};

}}}
