﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <memory>
#include <mutex>
#include <nn/fs/detail/fs_Newable.h>
#include <nn/fssystem/fs_ISaveDataExtraDataAccessor.h>
#include <nn/nn_Macro.h>
#include <nn/os/os_Mutex.h>
#include <nn/util/util_IntrusiveList.h>

namespace nn { namespace fssrv { namespace detail {

    class SaveDataExtraDataAccessorCacheManager : nn::fssystem::ISaveDataExtraDataAccessorCacheObserver
    {
        NN_DISALLOW_COPY(SaveDataExtraDataAccessorCacheManager);
        NN_DISALLOW_MOVE(SaveDataExtraDataAccessorCacheManager);

    public:
        class Cache : public nn::util::IntrusiveListBaseNode<Cache>, public nn::fs::detail::Newable
        {
        public:
            Cache(std::shared_ptr<nn::fssystem::ISaveDataExtraDataAccessor> accessor, nn::fs::SaveDataSpaceId spaceId, nn::fs::SaveDataId saveDataId)
                : m_pAccessor(accessor.get())
                , m_Accessor(accessor)
                , m_SpaceId(spaceId)
                , m_SaveDataId(saveDataId)
            {
            }
            bool Contains(nn::fs::SaveDataSpaceId spaceId, nn::fs::SaveDataId saveDataId) NN_NOEXCEPT
            {
                return m_SpaceId == spaceId && m_SaveDataId == saveDataId;
            }
            bool Contains(nn::fs::SaveDataSpaceId spaceId, nn::fs::SaveDataId saveDataId, void* accessor) NN_NOEXCEPT
            {
                return m_SpaceId == spaceId && m_SaveDataId == saveDataId && m_pAccessor == accessor;
            }
            std::shared_ptr<nn::fssystem::ISaveDataExtraDataAccessor> Lock() NN_NOEXCEPT
            {
                return m_Accessor.lock();
            }

        private:
            // キャッシュされたアクセサーのデストラクタから Unregister される時
            // weak_ptr はすでに無効になっている為、自身のアドレスを覚えておく
            const void* m_pAccessor;
            const std::weak_ptr<nn::fssystem::ISaveDataExtraDataAccessor> m_Accessor;
            const nn::fs::SaveDataSpaceId m_SpaceId;
            const nn::fs::SaveDataId      m_SaveDataId;
        };

        SaveDataExtraDataAccessorCacheManager() : m_Mutex(true)
        {
        }

        ~SaveDataExtraDataAccessorCacheManager()
        {
            std::lock_guard<nn::os::Mutex> scopedLock(m_Mutex);
            while (!m_CacheList.empty())
            {
                auto pCache = &(*m_CacheList.rbegin());
                m_CacheList.erase(m_CacheList.iterator_to(*pCache));
                delete pCache;
            }
        }

        Result Register(std::shared_ptr<nn::fssystem::ISaveDataExtraDataAccessor> accessor, nn::fs::SaveDataSpaceId spaceId, nn::fs::SaveDataId saveDataId) NN_NOEXCEPT
        {
            Cache* pCache = new Cache(std::move(accessor), spaceId, saveDataId);
            NN_RESULT_THROW_UNLESS(pCache, nn::fs::ResultAllocationMemoryFailedInSaveDataExtraDataAccessorCacheMnagerA());
            {
                std::lock_guard<nn::os::Mutex> scopedLock(m_Mutex);
                m_CacheList.push_back(*pCache);
            }
            NN_RESULT_SUCCESS;
        }

        Result GetCache(std::shared_ptr<nn::fssystem::ISaveDataExtraDataAccessor>* outValue, nn::fs::SaveDataSpaceId spaceId, nn::fs::SaveDataId saveDataId) NN_NOEXCEPT
        {
            std::lock_guard<nn::os::Mutex> scopedLock(m_Mutex);
            for (auto& cache : m_CacheList)
            {
                if (cache.Contains(spaceId, saveDataId))
                {
                    *outValue = cache.Lock();
                    if (!*outValue)
                    {
                        // 解放済みの場合エラーにする
                        return nn::fs::ResultTargetNotFound();
                    }
                    NN_RESULT_SUCCESS;
                }
            }
            return nn::fs::ResultTargetNotFound();
        }

        std::unique_lock<os::Mutex> GetScopedLock()
        {
            return std::unique_lock<os::Mutex>(m_Mutex);
        }

    public:
        virtual void Unregister(nn::fs::SaveDataSpaceId spaceId, nn::fs::SaveDataId saveDataId, nn::fssystem::ISaveDataExtraDataAccessor* accessor) NN_NOEXCEPT NN_OVERRIDE
        {
            std::lock_guard<nn::os::Mutex> scopedLock(m_Mutex);
            for (auto& cache : m_CacheList)
            {
                if (cache.Contains(spaceId, saveDataId, accessor))
                {
                    m_CacheList.erase(m_CacheList.iterator_to(cache));
                    delete &cache;
                    return;
                }
            }
        }

    private:
        nn::util::IntrusiveList<Cache, util::IntrusiveListBaseNodeTraits<Cache>> m_CacheList;
        nn::os::Mutex m_Mutex;
    };

}}}
