﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/fssystem/save/fs_JournalIntegritySaveDataFileSystemDriver.h>
#include "fssrv_SaveDataExtender.h"

namespace nn { namespace fssrv { namespace detail {

    namespace {
        const uint32_t Signature = NN_UTIL_CREATE_SIGNATURE_4('E', 'X', 'T', 'C');
        const uint32_t Version = 0x00010000;
    }

    int64_t SaveDataExtender::QueryContextSize() NN_NOEXCEPT
    {
        return sizeof(Context);
    }

    SaveDataExtender::SaveDataExtender() NN_NOEXCEPT
        : m_pContextStorage(nullptr)
    {
    }

    nn::Result SaveDataExtender::InitializeContext(
        const fssystem::save::JournalIntegritySaveDataParameters& extendedSaveDataParameters) NN_NOEXCEPT
    {
        m_Context.signature = Signature;
        m_Context.version = Version;
        m_Context.state = State::Extending;
        m_Context.availableBlockCount = extendedSaveDataParameters.countDataBlock;
        m_Context.journalBlockCount = extendedSaveDataParameters.countJournalBlock;
        m_Context.blockSize = extendedSaveDataParameters.blockSize;

        return nn::fssystem::save::JournalIntegritySaveDataFileSystemDriver::QueryTotalSize(
            &m_Context.extendedSaveDataSize,
            static_cast<size_t>(m_Context.blockSize),
            extendedSaveDataParameters.countExpandMax,
            extendedSaveDataParameters.paramDuplex,
            extendedSaveDataParameters.paramIntegrity,
            static_cast<uint32_t>(m_Context.availableBlockCount),
            static_cast<uint32_t>(m_Context.journalBlockCount),
            extendedSaveDataParameters.version);
    }

    nn::Result SaveDataExtender::WriteContext(nn::fs::IStorage* pContextStorage) const NN_NOEXCEPT
    {
        NN_RESULT_DO(pContextStorage->Write(0, &m_Context, sizeof(Context)));
        NN_RESULT_DO(pContextStorage->Flush());
        NN_RESULT_SUCCESS;
    }

    nn::Result SaveDataExtender::ReadContext(nn::fs::IStorage* pContextStorage) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(m_pContextStorage == nullptr);
        NN_SDK_REQUIRES_NOT_NULL(pContextStorage);

        NN_RESULT_DO(pContextStorage->Read(0, &m_Context, sizeof(Context)));

        {
            NN_RESULT_THROW_UNLESS(
                m_Context.signature == Signature,
                nn::fs::ResultIncorrectSaveDataExtensionContextMagicCode());

            NN_RESULT_THROW_UNLESS(
                m_Context.version <= Version,
                nn::fs::ResultUnsupportedSaveDataVersion());

            switch( m_Context.state )
            {
            case State::Extending:
            case State::Committing:
            case State::Finalizing:
                break;
            default:
                return nn::fs::ResultInvalidSaveDataExtensionContextState();
            }

            NN_RESULT_THROW_UNLESS(
                0 < m_Context.blockSize,
                nn::fs::ResultInvalidSaveDataExtensionContextParameter());
            NN_RESULT_THROW_UNLESS(
                0 < m_Context.extendedSaveDataSize,
                nn::fs::ResultInvalidSaveDataExtensionContextParameter());
        }

        m_pContextStorage = pContextStorage;

        NN_RESULT_SUCCESS;
    }

    int64_t SaveDataExtender::GetLogSize() const NN_NOEXCEPT
    {
        return nn::fssystem::save::JournalIntegritySaveDataFileSystemDriver::QueryExpandLogSize(
            static_cast<size_t>(m_Context.blockSize),
            static_cast<uint32_t>(m_Context.availableBlockCount),
            static_cast<uint32_t>(m_Context.journalBlockCount));
    }

    int64_t SaveDataExtender::GetExtendedSaveDataSize() const NN_NOEXCEPT
    {
        return m_Context.extendedSaveDataSize;
    }

    nn::Result SaveDataExtender::Extend(
        nn::fs::SubStorage saveDataStorage,
        nn::fs::SubStorage logStorage,
        nn::fssystem::IBufferManager* pBufferManager,
        nn::fssystem::IMacGenerator* pMacGenerator,
        uint32_t minimumVersion) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(m_pContextStorage);

        if( m_Context.state == State::Extending )
        {
            NN_RESULT_DO(nn::fssystem::save::JournalIntegritySaveDataFileSystemDriver::OperateExpand(
                saveDataStorage,
                logStorage,
                static_cast<size_t>(m_Context.blockSize),
                static_cast<uint32_t>(m_Context.availableBlockCount),
                static_cast<uint32_t>(m_Context.journalBlockCount),
                pBufferManager,
                pMacGenerator,
                minimumVersion));

            m_Context.state = State::Committing;
            NN_RESULT_DO(WriteContext(m_pContextStorage));
        }

        if( m_Context.state == State::Committing )
        {
            NN_RESULT_DO(nn::fssystem::save::JournalIntegritySaveDataFileSystemDriver::CommitExpand(
                saveDataStorage,
                logStorage,
                static_cast<size_t>(m_Context.blockSize),
                pBufferManager));

            m_Context.state = State::Finalizing;
            NN_RESULT_DO(WriteContext(m_pContextStorage));
        }

        NN_RESULT_THROW_UNLESS(m_Context.state == State::Finalizing, nn::fs::ResultBadState());
        NN_RESULT_SUCCESS;
    }

}}}
