﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>
#include <cstring>
#include <mutex>

#include <nn/nn_Abort.h>
#include <nn/nn_SdkLog.h>
#include <nn/nn_Common.h>
#include <nn/fs/fs_Result.h>
#include <nn/fs/fs_ResultPrivate.h>
#include <nn/os.h>
#include <nn/os/os_Mutex.h>
#include <nn/os/os_MemoryHeap.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/dd.h>
#include <nn/gc/gc.h>
#include <nn/gc/writer/gc_Writer.h>
#include <nn/gc/detail/gc_Define.h>
#include <nn/gc/detail/gc_Types.h>
#include <nn/fssystem/fs_AsynchronousAccess.h>
#include "fssrv_GameCardManager.h"
#include "fssrv_DeviceBuffer.h"
#include "fssrv_Trace.h"

#define NN_FS_GAMECARD_TRACE_ENABLED 0 // NOLINT(preprocessor/const)

#if NN_FS_GAMECARD_TRACE_ENABLED && NN_FS_SCOPED_TRACE_ENABLED

#define NN_FS_GAMECARD_TRACE(...) \
    nn::fssrv::detail::ScopedTrace fsAccessLogScopedTrace(nullptr, \
    NN_CURRENT_FUNCTION_NAME, \
    [=](char* buffer, size_t bufferLength) NN_NOEXCEPT \
    { \
        return nn::fssrv::detail::ScopedTrace::Formatter(buffer, bufferLength).FormatLog(__VA_ARGS__); \
    })

#else

#define NN_FS_GAMECARD_TRACE(...) (void)([&]() NN_NOEXCEPT { [](...) NN_NOEXCEPT {}( __VA_ARGS__); })

#endif

using namespace nn::fs;
using nn::gc::GcPageSize;

namespace nn { namespace fssrv { namespace detail {


    namespace{

        enum class CardState
        {
            Deactivated,
            Normal,
            Secure,
            Writable,
        };

        nn::os::EventType g_SettingsReadyEvent;

        const int Log2PageSize = 9;
        static_assert((1 << Log2PageSize) == GcPageSize, "nn::fssrv::fscreator::Log2PageSize is invalid.");

        os::Mutex g_GcDeviceDriverMutex(false);

        fssystem::PooledBuffer g_WorkBuffer;

        bool      g_IsInitialized = false;
        CardState g_State = CardState::Deactivated;
        uint32_t  g_CurrentHandle = 1;

        uint32_t BytesToPages(int64_t size) NN_NOEXCEPT
        {
            return static_cast<uint32_t>(size >> Log2PageSize);
        }

        void DeactivateAndChangeState() NN_NOEXCEPT
        {
            gc::Deactivate();
            ++g_CurrentHandle;
            g_State = CardState::Deactivated;
        }

        void CheckGameCardAndDeactivate() NN_NOEXCEPT
        {
            if(g_State != CardState::Deactivated && nn::gc::IsCardActivationValid() == false)
            {
                DeactivateAndChangeState();
            }
        }

        Result HandleGameCardAccessResult(Result result) NN_NOEXCEPT
        {
            if(result.IsFailure())
            {
                DeactivateAndChangeState();
            }

            return result;
        }

        Result ActivateGameCard() NN_NOEXCEPT
        {
            Result result = nn::gc::Activate();
            NN_RESULT_DO(HandleGameCardAccessResult(result));
            NN_RESULT_SUCCESS;
        }

        Result ActivateGameCardForWriter() NN_NOEXCEPT
        {
            Result result = gc::writer::ActivateForWriter();
            return HandleGameCardAccessResult(result);
        }

        Result SetGameCardToSecureMode() NN_NOEXCEPT
        {
            Result result = nn::gc::SetCardToSecureMode();
            return HandleGameCardAccessResult(result);
        }
    }


    void InitializeGameCardManager() NN_NOEXCEPT
    {
        nn::os::InitializeEvent(&g_SettingsReadyEvent, false, nn::os::EventClearMode_ManualClear);
    }

    void PresetInternalKeys(const void* encryptedKeyBuffer, const size_t encryptedKeyBufferSize,
                            const void* socCertBuffer, const size_t socCertBufferSize) NN_NOEXCEPT
    {
        if( encryptedKeyBuffer == nullptr || encryptedKeyBufferSize == 0 ||
            socCertBuffer == nullptr || socCertBufferSize == 0)
        {
            NN_SDK_LOG("[fs] Warning: skipped nn::gc::PresetInternalKeys\n");
        }
        else
        {
            // gc::Initialize() 前に実行する関数のため、g_GcDeviceDriverMutex による排他制御は不要
            nn::gc::PresetInternalKeys(encryptedKeyBuffer, encryptedKeyBufferSize, socCertBuffer, socCertBufferSize);
        }
        nn::os::SignalEvent(&g_SettingsReadyEvent);
    }

    void InitializeGcLibrary() NN_NOEXCEPT
    {
        std::lock_guard<os::Mutex> scopedLock(g_GcDeviceDriverMutex);
        if( !g_IsInitialized )
        {
            nn::os::WaitEvent(&g_SettingsReadyEvent);

#if defined(NN_BUILD_CONFIG_HARDWARE_JETSONTK1) || defined(NN_BUILD_CONFIG_HARDWARE_JETSONTK2) || defined(NN_BUILD_CONFIG_HARDWARE_NX) || defined(NN_BUILD_CONFIG_HARDWARE_JETSONTX2)
            AttachDeviceBufferToGameCardDriver();
            g_WorkBuffer.Allocate(gc::GcWorkBufferSize, gc::GcWorkBufferSize);
            NN_SDK_ASSERT(IsBufferCapable(g_WorkBuffer.GetBuffer()));

            const auto bufferOffset
                = reinterpret_cast<uintptr_t>(g_WorkBuffer.GetBuffer()) - GetDeviceBuffer();
            const auto workBufferDeviceAddress = GetDeviceVirtualAddress() + bufferOffset;
            gc::Initialize(g_WorkBuffer.GetBuffer(), g_WorkBuffer.GetSize(), workBufferDeviceAddress);

#if (defined(NN_DETAIL_FS_SMMU_FOR_SDMMC_ENABLE))
            gc::RegisterDeviceVirtualAddress(GetDeviceBuffer(), DeviceBufferSize, GetDeviceVirtualAddress());
#endif

#endif
            g_IsInitialized = true;
        }
    }


    void FinalizeGcLibrary() NN_NOEXCEPT
    {
        std::lock_guard<os::Mutex> scopedLock(g_GcDeviceDriverMutex);
        if( g_IsInitialized )
        {
#if defined(NN_BUILD_CONFIG_HARDWARE_JETSONTK1) || defined(NN_BUILD_CONFIG_HARDWARE_JETSONTK2) || defined(NN_BUILD_CONFIG_HARDWARE_NX) || defined(NN_BUILD_CONFIG_HARDWARE_JETSONTX2)

#if (defined(NN_DETAIL_FS_SMMU_FOR_SDMMC_ENABLE))
            nn::gc::UnregisterDetectionEventCallback();
#endif

            nn::gc::Finalize();
            g_WorkBuffer.Deallocate();
            DetachDeviceBufferFromGameCardDriver();
#endif

            g_IsInitialized = false;
        }
    }

    void PutGameCardToSleep() NN_NOEXCEPT
    {
        std::lock_guard<os::Mutex> scopedLock(g_GcDeviceDriverMutex);
        if( g_IsInitialized )
        {
            nn::gc::PutToSleep();
        }
    }

    void AwakenGameCard() NN_NOEXCEPT
    {
        std::lock_guard<os::Mutex> scopedLock(g_GcDeviceDriverMutex);
        if( g_IsInitialized )
        {
            nn::gc::Awaken();
        }
    }

    void ShutdownGameCard() NN_NOEXCEPT
    {
        std::lock_guard<os::Mutex> scopedLock(g_GcDeviceDriverMutex);
        if( g_IsInitialized )
        {
            nn::gc::PowerOffGameCard();
        }
    }

    Result GetGameCardHandle(nn::fs::GameCardHandle *pOutValue) NN_NOEXCEPT
    {
        InitializeGcLibrary();

        std::lock_guard<os::Mutex> scopedLock(g_GcDeviceDriverMutex);

        // TORIAEZU: 挿抜で無効になっていたら再 activate
        if( g_State == CardState::Normal || g_State == CardState::Secure )
        {
            CheckGameCardAndDeactivate();
        }

        switch(g_State)
        {
        case CardState::Deactivated:
            NN_RESULT_DO(ActivateGameCard());
            g_State = CardState::Normal;
            break;
        case CardState::Normal:
        case CardState::Secure: // TORIAEZU: ステート遷移しない
            break;
        case CardState::Writable: // TORIAEZU: 強制ステート遷移
            DeactivateAndChangeState();
            nn::gc::writer::ChangeMode(nn::gc::writer::AsicMode::AsicMode_ReadMode);
            NN_RESULT_DO(ActivateGameCard());
            g_State = CardState::Normal;
            break;
        default:
            NN_UNEXPECTED_DEFAULT;
        }

        *pOutValue = g_CurrentHandle;

        NN_RESULT_SUCCESS;
    }

    // @pre InitializeGcLibrary() 済
    Result ReadGameCard(GameCardHandle handle, int64_t offset, void* buffer, size_t size) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(nn::util::is_aligned(offset, GcPageSize));
        NN_SDK_REQUIRES(nn::util::is_aligned(size,   GcPageSize));

        NN_RESULT_THROW_UNLESS( handle == g_CurrentHandle, fs::ResultGameCardFsCheckHandleInReadFailure() );

        NN_RESULT_DO(GetGameCardEventSimulator().CheckSimulatedAccessFailureEvent(fs::SimulatingDeviceTargetOperation_Read));

        if( size == 0 )
        {
            NN_RESULT_SUCCESS;
        }
        else if( buffer == nullptr )
        {
            return nn::fs::ResultNullptrArgument();
        }
        else if( IsBufferCapable(buffer) )
        {
            NN_FS_GAMECARD_TRACE("R %8u %2u", BytesToPages(offset), BytesToPages(size));
            return gc::Read(buffer, size, BytesToPages(offset), BytesToPages(size));
        }
        else
        {
            fssystem::PooledBuffer pooledBuffer(size, gc::GcPageSize);
            NN_SDK_ASSERT(IsBufferCapable(pooledBuffer.GetBuffer()));

            size_t read = 0;
            while( read != size )
            {
                size_t readSize = std::min(size - read, pooledBuffer.GetSize());
                NN_FS_GAMECARD_TRACE("R %8u %2u", BytesToPages(offset + read), BytesToPages(readSize));
                NN_RESULT_DO(nn::gc::Read(pooledBuffer.GetBuffer(), readSize, BytesToPages(offset + read), BytesToPages(readSize)));
                std::memcpy(reinterpret_cast<char*>(buffer) + read, pooledBuffer.GetBuffer(), readSize);
                read += readSize;
            }
            NN_RESULT_SUCCESS;
        }
    }

    // @pre InitializeGcLibrary() 済
    Result WriteGameCard(GameCardHandle handle, int64_t offset, const void* buffer, size_t size) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(nn::util::is_aligned(offset, GcPageSize));
        NN_SDK_REQUIRES(nn::util::is_aligned(size,   GcPageSize));

        NN_RESULT_THROW_UNLESS( handle == g_CurrentHandle, fs::ResultGameCardFsCheckHandleInWriteFailure() );

        NN_RESULT_DO(GetGameCardEventSimulator().CheckSimulatedAccessFailureEvent(fs::SimulatingDeviceTargetOperation_Write));

        if( size == 0 )
        {
            NN_RESULT_SUCCESS;
        }
        else if( buffer == nullptr )
        {
            return nn::fs::ResultNullptrArgument();
        }
        else if( IsBufferCapable(buffer) )
        {
            return gc::writer::Write(const_cast<void*>(buffer), size, BytesToPages(offset), BytesToPages(size));
        }
        else
        {
            fssystem::PooledBuffer pooledBuffer;
            pooledBuffer.AllocateParticularlyLarge(size, gc::GcPageSize);
            NN_SDK_ASSERT(IsBufferCapable(pooledBuffer.GetBuffer()));

            size_t wrote = 0;
            while( wrote != size )
            {
                size_t writeSize = std::min(size - wrote, pooledBuffer.GetSize());
                std::memcpy(pooledBuffer.GetBuffer(), reinterpret_cast<const char*>(buffer) + wrote, writeSize);
                NN_RESULT_DO(gc::writer::Write(pooledBuffer.GetBuffer(), writeSize, BytesToPages(offset + wrote), BytesToPages(writeSize)));
                wrote += writeSize;
            }
            NN_RESULT_SUCCESS;
        }
    }

    void DeactivateGameCard() NN_NOEXCEPT
    {
        InitializeGcLibrary();
        std::lock_guard<os::Mutex> scopedLock(g_GcDeviceDriverMutex);
        DeactivateAndChangeState();
    }

    Result EnsureGameCardNormalMode(GameCardHandle* pOutValue) NN_NOEXCEPT
    {
        InitializeGcLibrary();

        std::lock_guard<os::Mutex> scopedLock(g_GcDeviceDriverMutex);

        // TORIAEZU: 挿抜で無効になっていたら再 activate
        if( g_State == CardState::Normal )
        {
            CheckGameCardAndDeactivate();
        }
        nn::Result result;

        switch(g_State)
        {
        case CardState::Deactivated:
            NN_RESULT_DO(ActivateGameCard());
            g_State = CardState::Normal;
            break;
        case CardState::Normal:
            *pOutValue = g_CurrentHandle;
            NN_RESULT_SUCCESS;
        case CardState::Secure:
            DeactivateAndChangeState();
            NN_RESULT_DO(ActivateGameCard());
            g_State = CardState::Normal;
            break;
        case CardState::Writable:
            DeactivateAndChangeState();
            nn::gc::writer::ChangeMode(nn::gc::writer::AsicMode::AsicMode_ReadMode);
            NN_RESULT_DO(ActivateGameCard());
            g_State = CardState::Normal;
            break;
        default:
            NN_UNEXPECTED_DEFAULT;
        }

        *pOutValue = g_CurrentHandle;

        NN_RESULT_SUCCESS;
    }

    Result EnsureGameCardSecureMode(GameCardHandle* pOutValue) NN_NOEXCEPT
    {
        InitializeGcLibrary();

        std::lock_guard<os::Mutex> scopedLock(g_GcDeviceDriverMutex);

        // TORIAEZU: 挿抜で無効になっていたら再 activate
        if( g_State == CardState::Secure )
        {
            CheckGameCardAndDeactivate();
        }

        switch(g_State)
        {
        case CardState::Deactivated:
            NN_RESULT_DO(ActivateGameCard());
            g_State = CardState::Normal;
            NN_RESULT_DO(SetGameCardToSecureMode());
            g_State = CardState::Secure;
            break;
        case CardState::Normal:
            NN_RESULT_DO(SetGameCardToSecureMode());
            g_State = CardState::Secure;
            break;
        case CardState::Secure:
            *pOutValue = g_CurrentHandle;
            NN_RESULT_SUCCESS;
        case CardState::Writable:
            DeactivateAndChangeState();
            nn::gc::writer::ChangeMode(nn::gc::writer::AsicMode::AsicMode_ReadMode);
            NN_RESULT_DO(ActivateGameCard());
            g_State = CardState::Normal;
            NN_RESULT_DO(SetGameCardToSecureMode());
            g_State = CardState::Secure;
            break;
        default:
            NN_UNEXPECTED_DEFAULT;
        }

        *pOutValue = g_CurrentHandle;

        NN_RESULT_SUCCESS;
    }

    Result EnsureGameCardWriteMode(GameCardHandle* pOutValue) NN_NOEXCEPT
    {
        InitializeGcLibrary();

        std::lock_guard<os::Mutex> scopedLock(g_GcDeviceDriverMutex);

        // TODO: 挿抜で無効になっていたら再 activate

        switch(g_State)
        {
        case CardState::Deactivated:
            nn::gc::writer::ChangeMode(nn::gc::writer::AsicMode::AsicMode_WriteMode);
            NN_RESULT_DO(ActivateGameCardForWriter());
            g_State = CardState::Writable;
            break;
        case CardState::Normal:
        case CardState::Secure:
            DeactivateAndChangeState();
            nn::gc::writer::ChangeMode(nn::gc::writer::AsicMode::AsicMode_WriteMode);
            NN_RESULT_DO(ActivateGameCardForWriter());
            g_State = CardState::Writable;
            break;
        case CardState::Writable:
            *pOutValue = g_CurrentHandle;
            NN_RESULT_SUCCESS;
        default:
            NN_UNEXPECTED_DEFAULT;
        }

        *pOutValue = g_CurrentHandle;

        NN_RESULT_SUCCESS;
    }

    bool IsGameCardInserted() NN_NOEXCEPT
    {
        InitializeGcLibrary();
        bool isInserted = nn::gc::IsCardInserted();
        return GetGameCardEventSimulator().FilterDetectionState(isInserted);
    }

    bool IsGameCardSecureMode() NN_NOEXCEPT
    {
        return g_State == CardState::Secure;
    }

    bool IsGameCardActivationValid(nn::fs::GameCardHandle handle) NN_NOEXCEPT
    {
        InitializeGcLibrary();
        return (handle == g_CurrentHandle) && nn::gc::IsCardActivationValid();
    }

    bool IsInvalidHandle(nn::fs::GameCardHandle handle) NN_NOEXCEPT
    {
        return (handle != g_CurrentHandle);
    }

    void RegisterGameCardDetectionEventCallback(DeviceDetectionEventCallback func, void* pContext) NN_NOEXCEPT
    {
        InitializeGcLibrary();
        gc::RegisterDetectionEventCallback(func, pContext);
    }

    void UnregisterGameCardDetectionEventCallback() NN_NOEXCEPT
    {
        InitializeGcLibrary();
        gc::UnregisterDetectionEventCallback();
    }

    Result GetInitializationResult() NN_NOEXCEPT
    {
        InitializeGcLibrary();

        // カード側へのアクセスが発生しないので、 HandleGameCardAccessResult は呼ばなくてよい
        return gc::GetInitializationResult();
    }

    Result GetGameCardStatus(gc::GameCardStatus* pOutValue) NN_NOEXCEPT
    {
        InitializeGcLibrary();

        Result result = gc::GetCardStatus(pOutValue);
        return HandleGameCardAccessResult(result);
    }

    Result EraseGameCard(nn::fs::GameCardSize size, uint64_t normalAreaSize) NN_NOEXCEPT
    {
        InitializeGcLibrary();
        static_assert(static_cast<int>(nn::fs::GameCardSize::Size1GB)  == gc::writer::MemorySize_1GB,  "GameCardSize is invalid.");
        static_assert(static_cast<int>(nn::fs::GameCardSize::Size16GB) == gc::writer::MemorySize_16GB, "GameCardSize is invalid.");
        Result result = gc::writer::EraseAndWriteParameter(static_cast<gc::writer::MemorySize>(size), static_cast<uint32_t>(normalAreaSize / gc::GcPageSize));
        return HandleGameCardAccessResult(result);
    }

    Result GetGameCardStatus(nn::gc::GameCardStatus* pOutValue, GameCardHandle handle) NN_NOEXCEPT
    {
        InitializeGcLibrary();
        NN_RESULT_THROW_UNLESS( handle == g_CurrentHandle, fs::ResultGameCardFsCheckHandleInGetStatusFailure() );
        gc::GameCardStatus gameCardStatus;

        Result result = gc::GetCardStatus(&gameCardStatus);
        NN_RESULT_DO(HandleGameCardAccessResult(result));

        *pOutValue = gameCardStatus;
        NN_RESULT_SUCCESS;
    }

    Result GetGameCardDeviceCertificate(char* pOutBuffer, size_t size, nn::fs::GameCardHandle handle) NN_NOEXCEPT
    {
        InitializeGcLibrary();
        NN_RESULT_THROW_UNLESS( handle == g_CurrentHandle, fs::ResultGameCardFsCheckHandleInGetDeviceCertFailure() );

        Result result = gc::GetCardDeviceCertificate(pOutBuffer, size);
        return HandleGameCardAccessResult(result);
    }

    Result GetGameCardAsicInfo(gc::RmaInformation* pOutValue,const char* pFirmwareBuffer, size_t size) NN_NOEXCEPT
    {
        InitializeGcLibrary();
        NN_SDK_REQUIRES(size == gc::detail::GcAsicFirmwareSize, "Size must be %d Byte\n", gc::detail::GcAsicFirmwareSize);
        nn::gc::RmaInformation rmaInfo;
        nn::gc::writer::SetUserAsicFirmwareBuffer(pFirmwareBuffer, size);
        nn::gc::writer::ChangeMode(nn::gc::writer::AsicMode_WriteMode);
        auto result = nn::gc::writer::GetRmaInformation(&rmaInfo);
        NN_RESULT_DO(HandleGameCardAccessResult(result));
        *pOutValue = rmaInfo;
        NN_RESULT_SUCCESS;
    }

    Result GetGameCardRawSize(int64_t* pOutValue) NN_NOEXCEPT
    {
        int64_t gameCardSize = 0;
        NN_RESULT_DO(HandleGameCardAccessResult(nn::gc::writer::GetCardAvailableRawSize(&gameCardSize)));
        *pOutValue = gameCardSize;
        NN_RESULT_SUCCESS;
    }

    Result GetGameCardIdSet(gc::GameCardIdSet* pOutValue) NN_NOEXCEPT
    {
        gc::GameCardIdSet gcIdSet;
        NN_RESULT_DO(HandleGameCardAccessResult(nn::gc::GetGameCardIdSet(&gcIdSet)));
        *pOutValue = gcIdSet;
        NN_RESULT_SUCCESS;
    }

    Result WriteToGameCardDirectly(size_t offset, char* pInOutBuffer, size_t size) NN_NOEXCEPT
    {
        if( size == 0 )
        {
            NN_RESULT_SUCCESS;
        }
        else if( pInOutBuffer == nullptr )
        {
            return nn::fs::ResultNullptrArgument();
        }

        InitializeGcLibrary();
#if (defined(NN_BUILD_CONFIG_HARDWARE_JETSONTK1) || defined(NN_BUILD_CONFIG_HARDWARE_JETSONTK2) || defined(NN_BUILD_CONFIG_HARDWARE_NX) || defined(NN_BUILD_CONFIG_HARDWARE_JETSONTX2)) && (defined(NN_DETAIL_FS_SMMU_FOR_SDMMC_ENABLE))
        uintptr_t bufferAddress = reinterpret_cast<uintptr_t>((pInOutBuffer));
        nn::dd::DeviceVirtualAddress deviceVirtualAddress = MapAdditionalDeviceAddress(reinterpret_cast<uint64_t>(pInOutBuffer), size);
        gc::RegisterDeviceVirtualAddress(bufferAddress, size, deviceVirtualAddress);
        auto result = gc::writer::Write(pInOutBuffer, size, BytesToPages(offset), BytesToPages(size));
        gc::UnregisterDeviceVirtualAddress(bufferAddress, size, deviceVirtualAddress);
        UnmapAdditionalDeviceAddress();
        return HandleGameCardAccessResult(result);
#else
        return HandleGameCardAccessResult(gc::writer::Write(pInOutBuffer, size, BytesToPages(offset), BytesToPages(size)));
#endif
    }

    void SetVerifyWriteEnalbleFlag(bool flag) NN_NOEXCEPT
    {
        nn::gc::writer::SetVerifyEnalbleFlag(flag);
    }

    Result GetGameCardImageHash(char* outBuffer, size_t outBufferSize) NN_NOEXCEPT
    {
        InitializeGcLibrary();
        Result result = gc::GetCardImageHash(outBuffer, outBufferSize);
        return HandleGameCardAccessResult(result);
    }

    Result GetGameCardImageHash(char* outBuffer, size_t outBufferSize, nn::fs::GameCardHandle handle) NN_NOEXCEPT
    {
        InitializeGcLibrary();
        NN_RESULT_THROW_UNLESS(handle == g_CurrentHandle, fs::ResultGameCardFsCheckHandleInGetCardImageHashFailure());

        Result result = gc::GetCardImageHash(outBuffer, outBufferSize);
        return HandleGameCardAccessResult(result);
    }

    Result GetGameCardDeviceIdForProdCard(char* outBuffer, size_t outBufferSize, const char* devHeaderBuffer, size_t devHeaderBufferSize) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(outBufferSize >= nn::gc::GcPageSize);
        NN_SDK_REQUIRES(devHeaderBufferSize >= nn::gc::GcPageSize);
        NN_UNUSED(outBufferSize);
        NN_UNUSED(devHeaderBufferSize);

        InitializeGcLibrary();
        const size_t writeSize = nn::gc::GcPageSize;
        fssystem::PooledBuffer pooledBuffer(writeSize, writeSize);
        NN_SDK_ASSERT(IsBufferCapable(pooledBuffer.GetBuffer()));

        NN_SDK_ASSERT_GREATER_EQUAL(pooledBuffer.GetSize(), writeSize);

        // Read : カードヘッダ読み込み -> tmp バッファへ
        nn::gc::writer::ChangeMode(nn::gc::writer::AsicMode::AsicMode_ReadMode);
        char tmpBuffer[nn::gc::GcPageSize];
        memset(tmpBuffer, 0x00, sizeof(tmpBuffer));
        NN_RESULT_DO(nn::gc::GetCardHeader(pooledBuffer.GetBuffer(), nn::gc::GcPageSize));
        memcpy(tmpBuffer, pooledBuffer.GetBuffer(), nn::gc::GcPageSize);

        // Write : 開発用カードヘッダ書き込み
        nn::gc::writer::ChangeMode(nn::gc::writer::AsicMode::AsicMode_WriteMode);
        NN_RESULT_DO(ActivateGameCardForWriter());
        std::memcpy(pooledBuffer.GetBuffer(), reinterpret_cast<const char*>(devHeaderBuffer), writeSize);
        NN_RESULT_DO(gc::writer::Write(pooledBuffer.GetBuffer(), writeSize, 8, 1));

        // Read : Activate 後 Device ID 読み
        nn::gc::writer::ChangeMode(nn::gc::writer::AsicMode::AsicMode_ReadMode);
        NN_RESULT_DO(nn::gc::Activate());
        // 証明書取得
        NN_FS_GAMECARD_TRACE("R %8u %2u", 0x38, 1);
        NN_RESULT_DO(nn::gc::Read(pooledBuffer.GetBuffer(), nn::gc::GcPageSize, 0x38, 1));
        char deviceCert[nn::gc::GcPageSize];
        memcpy(deviceCert, pooledBuffer.GetBuffer(), nn::gc::GcPageSize);

        // Write : カードヘッダ書き戻し
        nn::gc::writer::ChangeMode(nn::gc::writer::AsicMode::AsicMode_WriteMode);
        NN_RESULT_DO(ActivateGameCardForWriter());
        memcpy(pooledBuffer.GetBuffer(), tmpBuffer, nn::gc::GcPageSize);
        NN_RESULT_DO(gc::writer::Write(pooledBuffer.GetBuffer(), writeSize, 8, 1));
        memcpy(outBuffer, deviceCert, nn::gc::GcPageSize);

        NN_RESULT_SUCCESS;
    }

    Result EraseAndWriteParamDirectly(const char* inBuffer, size_t inBufferSize) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(inBufferSize >= sizeof(nn::gc::detail::DevCardParameter));
        NN_UNUSED(inBufferSize);
        InitializeGcLibrary();
        nn::gc::detail::DevCardParameter devCardParam;
        memcpy(&devCardParam, inBuffer, sizeof(nn::gc::detail::DevCardParameter));
        return nn::gc::writer::WriteDevCardParam(devCardParam);
    }

    Result ReadParamDirectly(char* outBuffer, size_t outBufferSize) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(outBufferSize >= sizeof(nn::gc::detail::DevCardParameter));
        NN_UNUSED(outBufferSize);
        InitializeGcLibrary();
        nn::gc::detail::DevCardParameter devCardParam;
        NN_RESULT_DO(nn::gc::writer::ReadDevCardParam(&devCardParam));
        memcpy(outBuffer, &devCardParam, sizeof(nn::gc::detail::DevCardParameter));
        NN_RESULT_SUCCESS;
    }

    Result ForceEraseGameCard() NN_NOEXCEPT
    {
        InitializeGcLibrary();
        nn::gc::writer::ChangeMode(nn::gc::writer::AsicMode::AsicMode_WriteMode);
        NN_RESULT_DO(nn::gc::writer::ForceErase());
        NN_RESULT_SUCCESS;
    }

    Result GetGameCardErrorInfo(nn::fs::GameCardErrorInfo* pOutValue) NN_NOEXCEPT
    {
        std::memset(pOutValue, 0, sizeof(nn::fs::GameCardErrorInfo));

        GameCardErrorReportInfo erinfo;
        NN_RESULT_DO(HandleGameCardAccessResult(nn::gc::GetErrorInfo(&erinfo)));
        {
            pOutValue->gameCardCrcErrorNum = erinfo.gameCardCrcErrorNum;
            pOutValue->asicCrcErrorNum     = erinfo.asicCrcErrorNum;
            pOutValue->refreshNum          = erinfo.refreshNum;
            pOutValue->timeoutRetryNum     = erinfo.timeoutRetryNum;
            pOutValue->retryLimitOutNum    = erinfo.retryLimitOutNum;
        }

        NN_RESULT_SUCCESS;
    }

    Result GetGameCardErrorReportInfo(nn::fs::GameCardErrorReportInfo* pOutValue) NN_NOEXCEPT
    {
        NN_RESULT_DO(HandleGameCardAccessResult(nn::gc::GetErrorInfo(pOutValue)));

        NN_RESULT_SUCCESS;
    }

    NN_STATIC_ASSERT(sizeof(nn::fs::GameCardErrorInfo) == 16);
    NN_STATIC_ASSERT(sizeof(nn::fs::GameCardErrorReportInfo) == 64);

    Result GetGameCardDeviceId(void* pOutValue, size_t size) NN_NOEXCEPT
    {
        NN_RESULT_DO(HandleGameCardAccessResult(nn::gc::GetCardDeviceId(pOutValue, size)));
        NN_RESULT_SUCCESS;
    }

}}}

