﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <memory>
#include <nn/nn_Common.h>
#include <nn/nn_Result.h>

#include <nn/fs/fs_ContentStorage.h>
#include <nn/fs/fs_GameCard.h>
#include <nn/fs/fs_ImageDirectory.h>
#include <nn/fs/fs_SaveDataTypes.h>
#include <nn/fs/fs_RightsId.h>
#include <nn/fs/fs_SdCardPrivate.h>
#include <nn/fs/fs_ErrorInfoPrivate.h>
#include <nn/fs/fs_CloudBackupWorkStorage.h>
#include <nn/fs/detail/fs_CommonMountName.h>
#include <nn/fs/detail/fs_FileSystemProxyTypes.h>
#include <nn/fssystem/fs_NcaHeader.h>
#include <nn/fssrv/fssrv_IFileSystemCreator.h>
#include <nn/fssrv/fssrv_FileSystemProxyServer.h>
#include <nn/spl/spl_Types.h>

#include "fssrv_ExternalKeyManager.h"
#include "fssrv_UpdatePartitionPath.h"

namespace nn { namespace fs { namespace fsa {
    class IFileSystem;
    class IFile;
}}}

namespace nn { namespace fssrv { namespace fscreator {
    struct FileSystemCreatorInterfaces;
}}}

namespace nn { namespace fssystem {
    class IBufferManager;
    class NcaReader;
    struct Hash;
    typedef Hash NcaDigest;
}}

namespace nn { namespace fssrv { namespace detail {

    const nn::Bit64 CheckThroughId = 0xFFFFFFFFFFFFFFFF;

    Result FlushFatCacheImpl() NN_NOEXCEPT;
    Result GetCurrentPosixTime(int64_t* pOutPosixTime, int32_t* pOutTimeDifference) NN_NOEXCEPT;
    void SetCurrentPosixTime(int64_t posixTime, int32_t timeDifference) NN_NOEXCEPT;

    class FileSystemProxyCoreImpl
    {
    public:
        FileSystemProxyCoreImpl(fscreator::FileSystemCreatorInterfaces* pFileSystemCreatorInterfaces, nn::fssystem::IBufferManager* pBufferManager, GenerateRandomFunction generateRandom) NN_NOEXCEPT;
        ~FileSystemProxyCoreImpl() NN_NOEXCEPT {}

        Result OpenFileSystem(std::shared_ptr<nn::fs::fsa::IFileSystem>* pOutFileSystem, const char* path, nn::fs::detail::FileSystemProxyType type, bool canReadPrivateDataContent, nn::Bit64 programId) NN_NOEXCEPT;

        Result OpenFileSystem(std::shared_ptr<nn::fs::fsa::IFileSystem>* pOutFileSystem, const char* path, nn::fs::detail::FileSystemProxyType type, nn::Bit64 programId) NN_NOEXCEPT
        {
            return OpenFileSystem(pOutFileSystem, path, type, false, programId);
        }

        Result OpenDataFileSystem(std::shared_ptr<nn::fs::fsa::IFileSystem>* pOutFileSystem, const char* path, nn::fs::detail::FileSystemProxyType type, nn::Bit64 programId) NN_NOEXCEPT;

        Result OpenDataStorage(std::shared_ptr<nn::fs::IStorage>* pOutStorage, fssystem::NcaDigest* pOutDigest, const char* path, nn::fs::detail::FileSystemProxyType type, nn::Bit64 programId, bool canReadPrivateDataContent) NN_NOEXCEPT;

        Result OpenDataStorage(std::shared_ptr<nn::fs::IStorage>* pOutStorage, fssystem::NcaDigest* pOutDigest, const char* path, nn::fs::detail::FileSystemProxyType type, nn::Bit64 programId) NN_NOEXCEPT
        {
            return OpenDataStorage(pOutStorage, pOutDigest, path, type, programId, false);
        }

        Result OpenStorageWithPatch(std::shared_ptr<nn::fs::IStorage>* pOutStorage, fssystem::NcaDigest* pOutDigest, const char* originalPath, const char* currentPath, nn::fs::detail::FileSystemProxyType type, nn::Bit64 programId) NN_NOEXCEPT;

        Result OpenFileSystemWithPatch(std::shared_ptr<nn::fs::fsa::IFileSystem>* pOutFileSystem, const char* originalPath, const char* currentPath, nn::fs::detail::FileSystemProxyType type, nn::Bit64 programId) NN_NOEXCEPT;

        Result IsSaveDataEntityExists(bool* outValue, nn::fs::SaveDataSpaceId saveDataSpaceId, nn::fs::SaveDataId saveDataId) NN_NOEXCEPT;

        Result OpenSaveDataFile(std::shared_ptr<nn::fs::fsa::IFile>* pOutFile, nn::fs::SaveDataSpaceId saveDataSpaceId, nn::fs::SaveDataId saveDataId, nn::fs::OpenMode mode) NN_NOEXCEPT;

        Result OpenSaveDataFileSystem(std::shared_ptr<nn::fs::fsa::IFileSystem>* pOutFileSystem, nn::fs::SaveDataSpaceId saveDataSpaceId, nn::fs::SaveDataId saveDataId, const char* saveDataRootPathBuffer, bool isReadOnly, nn::fs::SaveDataType saveDataType) NN_NOEXCEPT;

        Result OpenSaveDataDirectoryFileSystem(std::shared_ptr<nn::fs::fsa::IFileSystem>* pOutFileSystem, nn::fs::SaveDataSpaceId saveDataSpaceId, const char* saveDataRootPathBuffer, bool isOpen) NN_NOEXCEPT;

        Result OpenSaveDataMetaDirectoryFileSystem(std::shared_ptr<nn::fs::fsa::IFileSystem>* pOutFileSystem, nn::fs::SaveDataSpaceId saveDataSpaceId, nn::fs::SaveDataId saveDataId) NN_NOEXCEPT;

        Result OpenSaveDataInternalStorageFileSystem(std::shared_ptr<nn::fs::fsa::IFileSystem>* pOutFileSystem, nn::fs::SaveDataSpaceId saveDataSpaceId, nn::fs::SaveDataId saveDataId, const char* saveDataRootPathBuffer, bool isTemporaryFs) NN_NOEXCEPT;

        Result StartExtendSaveDataFileSystem(
            int64_t* outExtendedTotalSize,
            nn::fs::SaveDataId saveDataId,
            nn::fs::SaveDataSpaceId saveDataSpaceId,
            int64_t availableSize,
            int64_t journalSize,
            const char* pSaveDataRootPathBuffer) NN_NOEXCEPT;

        Result ResumeExtendSaveDataFileSystem(
            int64_t* outExtendedTotalSize,
            nn::fs::SaveDataId saveDataId,
            nn::fs::SaveDataSpaceId saveDataSpaceId,
            const char* pSaveDataRootPathBuffer) NN_NOEXCEPT;

        Result FinishExtendSaveDataFileSystem(
            nn::fs::SaveDataId saveDataId,
            nn::fs::SaveDataSpaceId saveDataSpaceId) NN_NOEXCEPT;

        void RevertExtendSaveDataFileSystem(
            nn::fs::SaveDataId saveDataId,
            nn::fs::SaveDataSpaceId saveDataSpaceId,
            int64_t beforeExtendSize,
            const char* pSaveDataRootPathBuffer) NN_NOEXCEPT;

        Result CreateSaveDataMeta(nn::fs::SaveDataId saveDataId, nn::fs::SaveDataSpaceId saveDataSpaceId, nn::fs::SaveDataMetaType metaType, size_t size) NN_NOEXCEPT;

        Result DeleteSaveDataMeta(nn::fs::SaveDataId saveDataId, nn::fs::SaveDataSpaceId saveDataSpaceId, nn::fs::SaveDataMetaType metaType) NN_NOEXCEPT;

        Result DeleteAllSaveDataMetas(nn::fs::SaveDataId saveDataId, nn::fs::SaveDataSpaceId saveDataSpaceId) NN_NOEXCEPT;

        Result OpenSaveDataMeta(std::unique_ptr<nn::fs::fsa::IFile>* pOutFile, nn::fs::SaveDataId saveDataId, nn::fs::SaveDataSpaceId saveDataSpaceId, nn::fs::SaveDataMetaType metaType) NN_NOEXCEPT;

        Result QuerySaveDataTotalSize(int64_t* outValue, int32_t blockSize, int64_t size, int64_t journalSize) NN_NOEXCEPT;

        Result CreateSaveDataFileSystem(nn::fs::SaveDataId saveDataId, const nn::fs::SaveDataAttribute& attribute, const nn::fs::SaveDataCreationInfo& creationInfo, const char* saveDataRootPathBuffer, const nn::fs::SaveDataHashSalt& hashSalt, bool skipFormat) NN_NOEXCEPT;

        Result DeleteSaveDataFileSystem(nn::fs::SaveDataSpaceId saveDataSpaceId, nn::fs::SaveDataId saveDataId, const char* saveDataRootPathBuffer) NN_NOEXCEPT;

        Result ReadSaveDataFileSystemExtraData(nn::fs::SaveDataExtraData* outValue, nn::fs::SaveDataSpaceId saveDataSpaceId, nn::fs::SaveDataId saveDataId, const char* saveDataRootPathBuffer, bool isTemporaryFs) NN_NOEXCEPT;

        Result WriteSaveDataFileSystemExtraData(nn::fs::SaveDataSpaceId saveDataSpaceId, nn::fs::SaveDataId saveDataId, const nn::fs::SaveDataExtraData& value, const char* saveDataRootPathBuffer, nn::fs::SaveDataType saveDataType, bool updateTimeStampAndHash) NN_NOEXCEPT;

        Result CorruptSaveDataFileSystem(nn::fs::SaveDataSpaceId saveDataSpaceId, nn::fs::SaveDataId saveDataId, const char* saveDataRootPathBuffer) NN_NOEXCEPT;

        Result RecoverSaveDataFileSystemMasterHeader(nn::fs::SaveDataSpaceId saveDataSpaceId, nn::fs::SaveDataId saveDataId) NN_NOEXCEPT;

        Result UpdateSaveDataFileSystemMac(nn::fs::SaveDataSpaceId saveDataSpaceId, nn::fs::SaveDataId saveDataId) NN_NOEXCEPT;

        Result CreatePaddingFile(std::int64_t paddingSize) NN_NOEXCEPT;

        Result DeleteAllPaddingFiles() NN_NOEXCEPT;

        // TODO: fscreator 化
        Result OpenImageDirectoryFileSystem(std::shared_ptr<nn::fs::fsa::IFileSystem>* pOutFileSystem, nn::fs::ImageDirectoryId storageId) NN_NOEXCEPT;

        Result OpenContentStorageFileSystem(std::shared_ptr<nn::fs::fsa::IFileSystem>* pOutFileSystem, nn::fs::ContentStorageId storageId) NN_NOEXCEPT;

        Result OpenCloudBackupWorkStorageFileSystem(std::shared_ptr<nn::fs::fsa::IFileSystem>* pOutFileSystem, nn::fs::CloudBackupWorkStorageId storageId) NN_NOEXCEPT;

        Result OpenBisFileSystem(std::shared_ptr<nn::fs::fsa::IFileSystem>* pOutFileSystem, const char* rootPath, nn::fs::BisPartitionId id) NN_NOEXCEPT;

        Result OpenBisStorage(std::shared_ptr<fs::IStorage>* pOutStorage, nn::fs::BisPartitionId id) NN_NOEXCEPT;

        Result InvalidateBisCache() NN_NOEXCEPT;

        Result OpenGameCardPartition(std::shared_ptr<fs::IStorage>* pOutStorage, nn::fs::GameCardHandle handle, nn::fs::GameCardPartitionRaw partition) NN_NOEXCEPT;

        Result OpenGameCardFileSystem(std::shared_ptr<nn::fs::fsa::IFileSystem>* pOutFileSystem, nn::fs::GameCardHandle handle, nn::fs::GameCardPartition partition) NN_NOEXCEPT;

        Result SetBisRootForHost(nn::fs::BisPartitionId id, const char* rootPath) NN_NOEXCEPT;

        //! 再挿抜に対応した SD カードファイルシステムを取得
        Result OpenSdCardProxyFileSystem(std::shared_ptr<nn::fs::fsa::IFileSystem>* pOutFileSystem) NN_NOEXCEPT;

        Result FormatSdCardProxyFileSystem() NN_NOEXCEPT;

        Result FormatSdCardDryRun() NN_NOEXCEPT;

        bool IsExFatSupported() NN_NOEXCEPT;

        Result OpenHostFileSystem(std::shared_ptr<nn::fs::fsa::IFileSystem>* pOutFileSystem, const char* rootPath) NN_NOEXCEPT;

        Result GetRightsId(fs::RightsId* pOutRightsId, uint8_t* pOutKeyGeneration, const char* path, nn::ncm::ProgramId programId) NN_NOEXCEPT;

        Result RegisterExternalKey(const fs::RightsId& rightsId, const nn::spl::AccessKey& accessKey) NN_NOEXCEPT;

        Result UnregisterAllExternalKey() NN_NOEXCEPT;

        Result SetSdCardEncryptionSeed(const fs::EncryptionSeed& seed) NN_NOEXCEPT;

        void GetAndClearErrorInfo(fs::FileSystemProxyErrorInfo* pOutValue) NN_NOEXCEPT;

        void IncrementRomFsRemountForDataCorruptionCount() NN_NOEXCEPT;
        void IncrementRomFsUnrecoverableDataCorruptionByRemountCount() NN_NOEXCEPT;
        void IncrementRomFsRecoveredByInvalidateCacheCount() NN_NOEXCEPT;

        Result RegisterUpdatePartition(nn::Bit64 programId, const char* path) NN_NOEXCEPT;

        Result OpenRegisteredUpdatePartition(std::shared_ptr<nn::fs::fsa::IFileSystem>* pOutFileSystem) NN_NOEXCEPT;

        void SetSdCardAccessibility(bool isAccessible) NN_NOEXCEPT;
        bool IsSdCardAccessible() NN_NOEXCEPT;
        void SetSdCardPortReady() NN_NOEXCEPT;
        bool IsSignedSystemPartitionOnSdCardValid() NN_NOEXCEPT;

    private:
        Result ParseMountName(const char** path, std::shared_ptr<nn::fs::fsa::IFileSystem>* pOutFileSystem, bool *isContinue, bool *isGameCard, nn::fs::GameCardHandle* outGcHandle, bool *isHostFs) NN_NOEXCEPT;

        Result CheckDirOrNcaOrNsp(const char** path, bool* isDir) NN_NOEXCEPT;

        Result ParseDir(const char** path, std::shared_ptr<nn::fs::fsa::IFileSystem>* pOutFileSystem, std::shared_ptr<nn::fs::fsa::IFileSystem> pBaseFileSystem, nn::fs::detail::FileSystemProxyType type) NN_NOEXCEPT;

        Result ParseNsp(const char** path, std::shared_ptr<nn::fs::fsa::IFileSystem>* pOutFileSystem, std::shared_ptr<nn::fs::fsa::IFileSystem> pBaseFileSystem) NN_NOEXCEPT;

        Result ParseNca(const char** path, std::shared_ptr<nn::fssystem::NcaReader>* pOutNcaReader, std::shared_ptr<nn::fs::fsa::IFileSystem> pBaseFileSystem, nn::Bit64 programId) NN_NOEXCEPT;

        Result ParseNca(std::shared_ptr<nn::fssystem::NcaReader>* pOutNcaReader, bool* pOutIsGameCard, const char* path, nn::Bit64 programId) NN_NOEXCEPT;

        Result ParseContentTypeForDirectory(std::shared_ptr<nn::fs::fsa::IFileSystem>* pOutFileSystem, std::shared_ptr<nn::fs::fsa::IFileSystem> pBaseFileSystem, nn::fs::detail::FileSystemProxyType type) NN_NOEXCEPT;

        Result OpenStorageByContentType(std::shared_ptr<nn::fs::IStorage>* pOutStorage, std::shared_ptr<nn::fssystem::NcaReader> pNcaReader, nn::fssystem::NcaFsHeader::FsType *outFsType, nn::fs::detail::FileSystemProxyType type, bool isGameCard, bool canReadPrivate) NN_NOEXCEPT;

        Result OpenStorageWithPatchByContentType(std::shared_ptr<nn::fs::IStorage>* pOutStorage, std::shared_ptr<nn::fssystem::NcaReader> pOriginalNcaReader, std::shared_ptr<nn::fssystem::NcaReader> pCurrentNcaReader, nn::fssystem::NcaFsHeader::FsType *outFsType, nn::fs::detail::FileSystemProxyType type) NN_NOEXCEPT;

        bool IsSaveEmulated(const char* saveDataRootPathBuffer) NN_NOEXCEPT;

        Result SetExternalKeyForRightsId(nn::fssystem::NcaReader* pNcaReader) NN_NOEXCEPT;

        // TODO: 以下 fscreator 化
        Result OpenSaveDataDirectoryFileSystemImpl(std::shared_ptr<nn::fs::fsa::IFileSystem>* pOutFileSystem, nn::fs::SaveDataSpaceId saveDataSpaceId, const char* rootDirectoryPath, bool isDirectoryCreate) NN_NOEXCEPT;

        Result OpenSaveDataDirectoryFileSystemImpl(std::shared_ptr<nn::fs::fsa::IFileSystem>* pOutFileSystem, nn::fs::SaveDataSpaceId saveDataSpaceId, const char* rootDirectoryPath) NN_NOEXCEPT;

        Result OpenSaveDataImageFile(
            std::unique_ptr<nn::fs::fsa::IFile>* pOutFile,
            nn::fs::SaveDataSpaceId saveDataSpaceId,
            nn::fs::SaveDataId saveDataId,
            const char* saveDataRootPathBuffer) NN_NOEXCEPT;

        Result OpenSaveDataExtensionContextFile(
            std::unique_ptr<nn::fs::fsa::IFile>* pOutFile,
            nn::fs::fsa::IFile* pSaveDataFile,
            nn::fs::SaveDataId saveDataId,
            nn::fs::SaveDataSpaceId saveDataSpaceId,
            int64_t availableSize,
            int64_t journalSize,
            bool isCreatable) NN_NOEXCEPT;

        Result ExtendSaveDataFileSystemCore(
            int64_t* outExtendedTotalSize,
            nn::fs::SaveDataId saveDataId,
            nn::fs::SaveDataSpaceId saveDataSpaceId,
            int64_t availableSize,
            int64_t journalSize,
            const char* pSaveDataRootPathBuffer,
            bool isStartable) NN_NOEXCEPT;

        Result FormatSdCardFileSystem(bool isUncache) NN_NOEXCEPT;

    private:
        fscreator::FileSystemCreatorInterfaces* const m_fsCreatorInterfaces;
        fssystem::IBufferManager* const m_pBufferManager;
        GenerateRandomFunction m_GenerateRandom;
        ExternalKeyManager m_ExternalKeyManager;
        fs::EncryptionSeed m_SdCardEncryptionSeed;

        fs::FileSystemProxyErrorInfo m_ErrorInfo;
        os::Mutex m_ErrorInfoMutex;

        UpdatePartitionPath m_UpdatePartitionPath;

        bool m_IsSdCardAccessibleFromNs;
    };

}}}
