﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdio>
#include <cstdlib>
#include <cstring>
#include <mutex>

#include <nn/os.h>
#include <nn/nn_Abort.h>
#include <nn/nn_SdkLog.h>
#include <nn/fs_Base.h>

#if defined(NN_BUILD_CONFIG_OS_WIN32)
    #include <nn/htc.h>
#endif

#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_FormatString.h>
#include <nn/util/util_ScopeExit.h>

#include <nn/crypto/crypto_Compare.h>

#include <nn/fat/fat_FatFileSystem.h>

#include <nn/fs/fs_ResultPrivate.h>
#include <nn/fs/fs_Bis.h>
#include <nn/fs/fs_Directory.h>
#include <nn/fs/fs_FileSystemPrivate.h>
#include <nn/fs/fs_SaveDataPrivate.h>
#include <nn/fs/fs_SubStorage.h>
#include <nn/fs/fs_RightsId.h>
#include <nn/fs/fs_ContentStorage.h>
#include <nn/fs/fs_ImageDirectory.h>
#include <nn/fs/fs_FileStorage.h>
#include <nn/fs/fs_ReadOnlyFileSystem.h>
#include <nn/fs/fs_CloudBackupWorkStorage.h>
#include <nn/fs/fsa/fs_IDirectory.h>
#include <nn/fs/fsa/fs_IFile.h>
#include <nn/fs/fsa/fs_IFileSystem.h>
#include <nn/fs/detail/fs_Log.h>
#include <nn/fs/detail/fs_ResultHandlingUtility.h>

#include <nn/fssrv/fssrv_IFileSystemCreator.h>
#include <nn/fssrv/fssrv_NcaCryptoConfiguration.h>

#include <nn/fssystem/fs_Assert.h>
#include <nn/fssystem/fs_AllocatorUtility.h>
#include <nn/fssystem/fs_ISaveDataExtraDataAccessor.h>
#include <nn/fssystem/fs_NcaFileSystemDriver.h>
#include <nn/fssystem/fs_SaveDataFileSystemCacheManager.h>
#include <nn/fssystem/fs_SaveDataFileSystemCacheRegister.h>
#include <nn/fssystem/save/fs_JournalIntegritySaveDataFileSystemDriver.h>
#include <nn/fssystem/save/fs_IntegritySaveDataFileSystemDriver.h>
#include <nn/fssystem/fs_Utility.h>

#include "fssrv_FatFileSystemCacheManager.h"
#include "fssrv_FileSystemProxyCoreImpl.h"
#include "fssrv_SaveDataExtraDataAccessorCacheManager.h"
#include "fssrv_SaveDataExtender.h"
#include "fssrv_SdmmcStorageService.h"
#include "fssrv_Utility.h"

using namespace nn::fs;
using namespace nn::fs::detail;
using namespace nn::fssystem;
using namespace nn::fssrv::detail;

namespace nn { namespace fssrv { namespace detail {

    namespace
    {
        // TORIAEZU: 暫定キャッシュ
        // SD カードの挿抜に非対応
        FatFileSystemCacheManager g_FatFileSystemCacheManager;

        fssystem::SaveDataFileSystemCacheManager g_SaveFileSystemCacheManager;

        SaveDataExtraDataAccessorCacheManager g_SaveDataExtraDataAccessorCacheManager;

        bool IsStaticSaveDataIdValueRange(nn::fs::SaveDataId id)
        {
            return ((id & 0x8000000000000000ULL) != 0);
        }

        const char* const PaddingDirectoryName = "/Padding";

        Result GetDeviceHandleByMountName(GameCardHandle* pOutValue, const char* path)
        {
            const int HandleValueStringLength = sizeof(GameCardHandle) * 2;
            NN_STATIC_ASSERT(HandleValueStringLength == 8);
            NN_RESULT_THROW_UNLESS(strnlen(path, HandleValueStringLength) >= HandleValueStringLength, ResultInvalidPath());

            char handleString[HandleValueStringLength + 1] = {0};
            memcpy(handleString, path, HandleValueStringLength);

            char* pEnd;
            // std::strtoul() は文字列の最大長を指定できないが、あらかじめ path の長さを確認しているので問題ない
            auto handleValue = std::strtoul(path, &pEnd, 16);

            NN_RESULT_THROW_UNLESS(pEnd - path == HandleValueStringLength, ResultInvalidPath());
            NN_RESULT_THROW_UNLESS(handleValue != ULONG_MAX, ResultInvalidPath());

            *pOutValue = static_cast<GameCardHandle>(handleValue & 0xFFFFFFFF);
            NN_RESULT_SUCCESS;
        }

        Result GetGameCardPartitionByMountName(GameCardPartition* pOutValue, const char* path)
        {
            if(      strncmp(path, GameCardFileSystemMountNameSuffixUpdate, GameCardFileSystemMountNameSuffixLength) == 0 )
            {
                *pOutValue = GameCardPartition::Update;
                NN_RESULT_SUCCESS;
            }
            else if( strncmp(path, GameCardFileSystemMountNameSuffixNormal, GameCardFileSystemMountNameSuffixLength) == 0 )
            {
                *pOutValue = GameCardPartition::Normal;
                NN_RESULT_SUCCESS;
            }
            else if( strncmp(path, GameCardFileSystemMountNameSuffixSecure, GameCardFileSystemMountNameSuffixLength) == 0 )
            {
                *pOutValue = GameCardPartition::Secure;
                NN_RESULT_SUCCESS;
            }
            else
            {
                return ResultInvalidPath();
            }
        }

        Result GetPartitionIndex(int* pOutValue, nn::fs::detail::FileSystemProxyType type) NN_NOEXCEPT
        {
            int partitionIndex;
            switch (type)
            {
            case FileSystemProxyType::FileSystemProxyType_Code:
                partitionIndex = 0;
                break;
            case FileSystemProxyType::FileSystemProxyType_Rom:
            case FileSystemProxyType::FileSystemProxyType_UpdatePartition:
                partitionIndex = 1;
                break;
            case FileSystemProxyType::FileSystemProxyType_Logo:
                partitionIndex = 2;
                break;
            case FileSystemProxyType::FileSystemProxyType_Control:
            case FileSystemProxyType::FileSystemProxyType_Manual:
            case FileSystemProxyType::FileSystemProxyType_Meta:
            case FileSystemProxyType::FileSystemProxyType_Data:
                partitionIndex = 0;
                break;
            default:
                return ResultInvalidArgument();
            }

            *pOutValue = partitionIndex;
            NN_RESULT_SUCCESS;
        }

        bool IsDeviceUniqueMac(SaveDataSpaceId spaceId)
        {
            return (spaceId == SaveDataSpaceId::User || spaceId == SaveDataSpaceId::Temporary || spaceId == SaveDataSpaceId::System || spaceId == SaveDataSpaceId::ProperSystem || spaceId == SaveDataSpaceId::SafeMode);
        }

        void GenerateNcaDigest(NcaDigest* pOutValue, const NcaReader* pReader1, const NcaReader* pReader2)
        {
            NN_SDK_ASSERT_NOT_NULL(pOutValue);
            NN_SDK_ASSERT_NOT_NULL(pReader1);

            crypto::Sha256Generator generator;
            generator.Initialize();

            NcaHeader ncaHeader;
            pReader1->GetRawData(&ncaHeader, sizeof(NcaHeader));
            generator.Update(ncaHeader.headerSign1, NcaHeader::HeaderSignSize);

            if (pReader2)
            {
                pReader2->GetRawData(&ncaHeader, sizeof(NcaHeader));
                generator.Update(ncaHeader.headerSign1, NcaHeader::HeaderSignSize);
            }

            generator.GetHash(pOutValue->value, NcaDigest::Size);
        }

        int64_t g_CurrentPosixTimeBase = 0;
        int32_t g_CurrentTimeDifference = 0;
        nn::os::Mutex g_PosixTimeMutex(false);

        int64_t GetSystemSeconds()
        {
            return nn::os::ConvertToTimeSpan(nn::os::GetSystemTick()).GetSeconds();
        }

        Result GetSaveDataCommitTimeStamp(int64_t* outValue) NN_NOEXCEPT
        {
            int32_t timeDiff = 0; // 使用しない
            return GetCurrentPosixTime(outValue, &timeDiff);
        }


    }

    FileSystemProxyCoreImpl::FileSystemProxyCoreImpl(fscreator::FileSystemCreatorInterfaces* pFileSystemCreatorInterfaces, nn::fssystem::IBufferManager* pBufferManager, GenerateRandomFunction generateRandom) NN_NOEXCEPT
        : m_fsCreatorInterfaces(pFileSystemCreatorInterfaces)
        , m_pBufferManager(pBufferManager)
        , m_GenerateRandom(generateRandom)
        , m_ErrorInfoMutex(false)
        , m_IsSdCardAccessibleFromNs(false)
    {
        memset(&m_SdCardEncryptionSeed, 0, sizeof(m_SdCardEncryptionSeed));
        memset(&m_ErrorInfo, 0, sizeof(m_ErrorInfo));

        m_fsCreatorInterfaces->bisfsCreator->Initialize(&m_ErrorInfo, &m_ErrorInfoMutex);
        m_fsCreatorInterfaces->sdProxyFsCreator->Initialize(&m_ErrorInfo, &m_ErrorInfoMutex);
    }

    Result FileSystemProxyCoreImpl::OpenFileSystem(std::shared_ptr<nn::fs::fsa::IFileSystem>* pOutFileSystem, const char* path, nn::fs::detail::FileSystemProxyType type, bool canReadPrivateDataContent, nn::Bit64 programId) NN_NOEXCEPT
    {
        const char *pPath = path;
        bool isContinue = true;
        bool isGameCard = false;
        GameCardHandle gcHandle = 0;
        bool isHostFs = false;

        std::shared_ptr<nn::fs::fsa::IFileSystem> pFileSystem;
        NN_RESULT_DO(ParseMountName(&pPath, &pFileSystem, &isContinue, &isGameCard, &gcHandle, &isHostFs));
        if (!isContinue)
        {
            // この関数では nsp もしくは nca を開きにくることしか無い想定
            return ResultInvalidArgument();
        }

        // ゲームカード上のロゴデータは nca を直接参照せず、事前に検証・キャッシュしたものを参照する
        if (isGameCard && type == FileSystemProxyType::FileSystemProxyType_Logo)
        {
            auto result = OpenGameCardFileSystem(pOutFileSystem, gcHandle, nn::fs::GameCardPartition::Logo);
            if (result.IsSuccess())
            {
                NN_RESULT_SUCCESS;
            }
            else if (fs::ResultPartitionNotFound::Includes(result))
            {
                // fall through
                // 5.0.0 NUP 以前に生産されたゲームカードには非暗号化ロゴ領域が存在しない
                // この場合は nca 上のロゴデータを参照する
            }
            else
            {
                return result;
            }
        }

        bool isDir = false;
        NN_RESULT_DO(CheckDirOrNcaOrNsp(&pPath, &isDir));

        if (isDir)
        {
            NN_RESULT_THROW_UNLESS(isHostFs, ResultPermissionDenied());
            return ParseDir(&pPath, pOutFileSystem, std::move(pFileSystem), type);
        }
        else
        {
            std::shared_ptr<nn::fs::fsa::IFileSystem> pNspFileSystem;
            if (ParseNsp(&pPath, &pNspFileSystem, pFileSystem).IsSuccess())
            {
                if (*pPath == '\0')
                {
                    NN_RESULT_THROW_UNLESS(type == FileSystemProxyType::FileSystemProxyType_Package, ResultInvalidArgument());
                    *pOutFileSystem = std::move(pNspFileSystem);
                    NN_RESULT_SUCCESS;
                }
                pFileSystem = std::move(pNspFileSystem);
            }

            std::shared_ptr<NcaReader> ncaReader;
            NN_RESULT_DO(ParseNca(&pPath, &ncaReader, std::move(pFileSystem), isHostFs ? CheckThroughId : programId));
            std::shared_ptr<fs::IStorage> storage;
            fssystem::NcaFsHeader::FsType fsType;
            NN_RESULT_DO(OpenStorageByContentType(&storage, std::move(ncaReader), &fsType, type, isGameCard, canReadPrivateDataContent));
            switch(fsType)
            {
            case fssystem::NcaFsHeader::FsType::RomFs:
                return m_fsCreatorInterfaces->romFsCreator->Create(pOutFileSystem, std::move(storage));
            case fssystem::NcaFsHeader::FsType::PartitionFs:
                return m_fsCreatorInterfaces->partitionFsCreator->Create(pOutFileSystem, std::move(storage));
            default:
                return nn::fs::ResultInvalidNcaFileSystemType();
            }
        }
    }

    Result FileSystemProxyCoreImpl::OpenDataFileSystem(std::shared_ptr<nn::fs::fsa::IFileSystem>* pOutFileSystem, const char* path, nn::fs::detail::FileSystemProxyType type, nn::Bit64 programId) NN_NOEXCEPT
    {
        const char *pPath = path;
        bool isContinue = true;
        bool isGameCard = false;
        GameCardHandle gcHandle = 0;
        bool isHostFs = false;

        // Rom（Raw 形式）のみ対応
        NN_RESULT_THROW_UNLESS(type == FileSystemProxyType_Rom, ResultPreconditionViolation());

        std::shared_ptr<nn::fs::fsa::IFileSystem> pFileSystem;
        NN_RESULT_DO(ParseMountName(&pPath, &pFileSystem, &isContinue, &isGameCard, &gcHandle, &isHostFs));
        if (!isContinue)
        {
            *pOutFileSystem = std::move(pFileSystem);
            NN_RESULT_SUCCESS;
        }

        bool isDir = false;
        NN_RESULT_DO(CheckDirOrNcaOrNsp(&pPath, &isDir));

        if (isDir)
        {
            NN_RESULT_THROW_UNLESS(isHostFs, ResultPermissionDenied());
            return ParseDir(&pPath, pOutFileSystem, std::move(pFileSystem), type);
        }
        else
        {
            std::shared_ptr<nn::fs::fsa::IFileSystem> pNspFileSystem;
            if (ParseNsp(&pPath, &pNspFileSystem, pFileSystem).IsSuccess())
            {
                if (*pPath == '\0')
                {
                    return nn::fs::ResultTargetNotFound();
                }
                pFileSystem = std::move(pNspFileSystem);
            }

            std::shared_ptr<NcaReader> ncaReader;
            NN_RESULT_DO(ParseNca(&pPath, &ncaReader, std::move(pFileSystem), programId));
            std::shared_ptr<fs::IStorage> storage;
            fssystem::NcaFsHeader::FsType fsType;
            NN_RESULT_DO(OpenStorageByContentType(&storage, std::move(ncaReader), &fsType, type, isGameCard, false));
            // RomFs でない場合は事前条件違反
            NN_RESULT_THROW_UNLESS(fsType ==  fssystem::NcaFsHeader::FsType::RomFs, ResultPreconditionViolation());
            return m_fsCreatorInterfaces->romFsCreator->Create(pOutFileSystem, std::move(storage));
        }
    }

    Result FileSystemProxyCoreImpl::OpenDataStorage(std::shared_ptr<nn::fs::IStorage>* pOutStorage, NcaDigest* pOutDigest, const char* path, nn::fs::detail::FileSystemProxyType type, nn::Bit64 programId, bool canReadPrivateDataContent) NN_NOEXCEPT
    {
        bool isGameCard = false;
        std::shared_ptr<NcaReader> ncaReader;
        NN_RESULT_DO(ParseNca(&ncaReader, &isGameCard, path, programId));

        if (pOutDigest != nullptr)
        {
            GenerateNcaDigest(pOutDigest, ncaReader.get(), nullptr);
        }

        fssystem::NcaFsHeader::FsType fsType;
        NN_RESULT_DO(OpenStorageByContentType(pOutStorage, std::move(ncaReader), &fsType, type, isGameCard, canReadPrivateDataContent));
        // RomFs でない場合は事前条件違反
        NN_RESULT_THROW_UNLESS(fsType ==  fssystem::NcaFsHeader::FsType::RomFs, ResultPreconditionViolation());
        NN_RESULT_SUCCESS;
    }

    Result FileSystemProxyCoreImpl::OpenStorageWithPatch(std::shared_ptr<nn::fs::IStorage>* pOutStorage, NcaDigest* pOutDigest, const char* originalPath, const char* currentPath, nn::fs::detail::FileSystemProxyType type, nn::Bit64 programId) NN_NOEXCEPT
    {
        bool isOriginalGameCard = false;
        std::shared_ptr<NcaReader> originalNcaReader;
        if (originalPath != nullptr)
        {
            NN_RESULT_DO(ParseNca(&originalNcaReader, &isOriginalGameCard, originalPath, programId));

            // GameCard 向け nca の場合、 GameCard 上の場合のみ開くことを許可する
            NN_RESULT_THROW_UNLESS(originalNcaReader->GetDistributionType() != fssystem::NcaHeader::DistributionType::GameCard || isOriginalGameCard, ResultPermissionDenied());
        }

        // GameCard 上の場合、SW AES で復号する
        if (isOriginalGameCard)
        {
            originalNcaReader->PrioritizeSwAes();
        }

        bool isCurrentGameCard = false;
        std::shared_ptr<NcaReader> currentNcaReader;
        NN_RESULT_DO(ParseNca(&currentNcaReader, &isCurrentGameCard, currentPath, programId));

        // GameCard 向け nca の場合、 GameCard 上の場合のみ開くことを許可する
        NN_RESULT_THROW_UNLESS(currentNcaReader->GetDistributionType() != fssystem::NcaHeader::DistributionType::GameCard || isCurrentGameCard, ResultPermissionDenied());

        if (isCurrentGameCard)
        {
            currentNcaReader->PrioritizeSwAes();
        }

        if (pOutDigest != nullptr)
        {
            GenerateNcaDigest(pOutDigest, originalNcaReader.get(), currentNcaReader.get());
        }

        fssystem::NcaFsHeader::FsType fsType;
        NN_RESULT_DO(OpenStorageWithPatchByContentType(
            pOutStorage,
            std::move(originalNcaReader),
            std::move(currentNcaReader),
            &fsType,
            type
        ));
        // RomFs でない場合は事前条件違反
        NN_RESULT_THROW_UNLESS(fsType ==  fssystem::NcaFsHeader::FsType::RomFs, ResultPreconditionViolation());
        NN_RESULT_SUCCESS;
    }

    Result FileSystemProxyCoreImpl::OpenFileSystemWithPatch(std::shared_ptr<nn::fs::fsa::IFileSystem>* pOutFileSystem, const char* originalPath, const char* currentPath, nn::fs::detail::FileSystemProxyType type, nn::Bit64 programId) NN_NOEXCEPT
    {
        std::shared_ptr<nn::fs::IStorage> storage;
        std::shared_ptr<nn::fs::fsa::IFileSystem> pFileSystem;

        NN_RESULT_DO(OpenStorageWithPatch(&storage, nullptr, originalPath, currentPath, type, programId));
        NN_RESULT_DO(m_fsCreatorInterfaces->romFsCreator->Create(&pFileSystem, std::move(storage)));

        *pOutFileSystem = std::move(pFileSystem);
        NN_RESULT_SUCCESS;
    }

    Result FileSystemProxyCoreImpl::IsSaveDataEntityExists(bool* outValue, nn::fs::SaveDataSpaceId saveDataSpaceId, nn::fs::SaveDataId saveDataId) NN_NOEXCEPT
    {
        std::shared_ptr<nn::fs::fsa::IFileSystem> pFileSystem;
        NN_RESULT_DO(OpenSaveDataDirectoryFileSystem(&pFileSystem, saveDataSpaceId, "", true));

        char saveImageName[1 + 16 + 1];
        util::SNPrintf(saveImageName, sizeof(saveImageName), "/%016llx", saveDataId);

        DirectoryEntryType type;
        auto result = pFileSystem->GetEntryType(&type, saveImageName);
        if (result.IsSuccess())
        {
            *outValue = true;
            NN_RESULT_SUCCESS;
        }
        else if (nn::fs::ResultPathNotFound::Includes(result))
        {
            *outValue = false;
            NN_RESULT_SUCCESS;
        }
        else
        {
            return result;
        }
    }

    Result FileSystemProxyCoreImpl::OpenSaveDataFile(std::shared_ptr<nn::fs::fsa::IFile>* pOutFile, nn::fs::SaveDataSpaceId saveDataSpaceId, nn::fs::SaveDataId saveDataId, nn::fs::OpenMode mode) NN_NOEXCEPT
    {
        std::shared_ptr<nn::fs::fsa::IFileSystem> pFileSystem;
        NN_RESULT_DO(OpenSaveDataDirectoryFileSystem(&pFileSystem, saveDataSpaceId, "", true));

        std::shared_ptr<nn::fs::fsa::IFile> saveDataFile;

        g_SaveFileSystemCacheManager.Unregister(saveDataSpaceId, saveDataId);

        NN_RESULT_DO(m_fsCreatorInterfaces->saveDataFsCreator->CreateRaw(&saveDataFile, std::move(pFileSystem), saveDataId, mode));

        *pOutFile = std::move(saveDataFile);
        NN_RESULT_SUCCESS;
    }

    Result FileSystemProxyCoreImpl::OpenSaveDataFileSystem(std::shared_ptr<nn::fs::fsa::IFileSystem>* pOutFileSystem, nn::fs::SaveDataSpaceId saveDataSpaceId, nn::fs::SaveDataId saveDataId, const char* saveDataRootPathBuffer, bool isReadOnly, nn::fs::SaveDataType saveDataType) NN_NOEXCEPT
    {
        std::shared_ptr<nn::fs::fsa::IFileSystem> pFileSystem;
        NN_RESULT_DO(OpenSaveDataDirectoryFileSystem(&pFileSystem, saveDataSpaceId, saveDataRootPathBuffer, true));

        bool isAllowdDirectorySaveData = IsSaveEmulated(saveDataRootPathBuffer) && saveDataSpaceId == SaveDataSpaceId::User;
        if(isAllowdDirectorySaveData)
        {
            char saveDirectoryName[1 + 16 + 1];
            util::SNPrintf(saveDirectoryName, sizeof(saveDirectoryName), "/%016llx", saveDataId);
            // ディレクトリ作成
            NN_RESULT_DO(EnsureDirectory(pFileSystem.get(), saveDirectoryName));
        }

        std::shared_ptr<nn::fs::fsa::IFileSystem> saveDataFs;

        std::shared_ptr<nn::fssystem::SaveDataFileSystem> cachedFs;
        if (g_SaveFileSystemCacheManager.GetCache(&cachedFs, saveDataSpaceId, saveDataId))
        {
            // 必要な fs がキャッシュされていたなら、それを saveDataFs に呼び戻す
            saveDataFs = fssystem::AllocateShared<fssystem::SaveDataFileSystemCacheRegister>(std::move(cachedFs), &g_SaveFileSystemCacheManager);
            NN_RESULT_THROW_UNLESS(saveDataFs, nn::fs::ResultAllocationMemoryFailedNew());
        }

        if (saveDataFs == nullptr)
        {
            auto scopedLock = g_SaveDataExtraDataAccessorCacheManager.GetScopedLock();
            std::shared_ptr<nn::fssystem::ISaveDataExtraDataAccessor> extraDataAccessor;
            NN_RESULT_DO(m_fsCreatorInterfaces->saveDataFsCreator->Create(&saveDataFs, &extraDataAccessor, &g_SaveFileSystemCacheManager, std::move(pFileSystem), saveDataId, isAllowdDirectorySaveData, IsDeviceUniqueMac(saveDataSpaceId), saveDataType, GetSaveDataCommitTimeStamp));

            // 開いているセーブデータの拡張データアクセサをキャッシュしておく
            if (extraDataAccessor)
            {
                extraDataAccessor->RegisterCacheObserver(reinterpret_cast<nn::fssystem::ISaveDataExtraDataAccessorCacheObserver*>(&g_SaveDataExtraDataAccessorCacheManager), saveDataSpaceId, saveDataId);
                NN_RESULT_DO(g_SaveDataExtraDataAccessorCacheManager.Register(std::move(extraDataAccessor), saveDataSpaceId, saveDataId));
            }
        }

        if (isReadOnly)
        {
            std::shared_ptr<ReadOnlyFileSystemShared> readOnlyFileSystem = fssystem::AllocateShared<ReadOnlyFileSystemShared>(std::move(saveDataFs));
            NN_RESULT_THROW_UNLESS(readOnlyFileSystem, nn::fs::ResultAllocationMemoryFailedInFileSystemProxyCoreImplF());
            *pOutFileSystem = std::move(readOnlyFileSystem);
        }
        else
        {
            *pOutFileSystem = std::move(saveDataFs);
        }

        NN_RESULT_SUCCESS;
    }

    Result FileSystemProxyCoreImpl::OpenSaveDataMetaDirectoryFileSystem(std::shared_ptr<nn::fs::fsa::IFileSystem>* pOutFileSystem, nn::fs::SaveDataSpaceId saveDataSpaceId, nn::fs::SaveDataId saveDataId) NN_NOEXCEPT
    {
        char saveDataMetaIdDirectoryName[1 + 8 + 1 + 16 + 1];
        util::SNPrintf(saveDataMetaIdDirectoryName, sizeof(saveDataMetaIdDirectoryName), "/saveMeta/%016llx", saveDataId);
        return OpenSaveDataDirectoryFileSystemImpl(pOutFileSystem, saveDataSpaceId, saveDataMetaIdDirectoryName);
    }

    Result FileSystemProxyCoreImpl::OpenSaveDataInternalStorageFileSystem(std::shared_ptr<nn::fs::fsa::IFileSystem>* pOutFileSystem, nn::fs::SaveDataSpaceId saveDataSpaceId, nn::fs::SaveDataId saveDataId, const char* saveDataRootPathBuffer, bool isTemporaryFs) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pOutFileSystem);
        NN_SDK_REQUIRES_NOT_NULL(saveDataRootPathBuffer);

        std::shared_ptr<nn::fs::fsa::IFileSystem> pFileSystem;
        NN_RESULT_DO(OpenSaveDataDirectoryFileSystem(&pFileSystem, saveDataSpaceId, saveDataRootPathBuffer, true));

        // SaveFileSystemCacheManager → SaveDataExtraDataAccessorCacheManager の順でロックする規則

        auto scopedLockForSaveFileSystemCacheManager = g_SaveFileSystemCacheManager.GetScopedLock();
        std::shared_ptr<nn::fssystem::SaveDataFileSystem> cachedFs;
        // キャッシュがあれば削除しておく
        if(g_SaveFileSystemCacheManager.GetCache(&cachedFs, saveDataSpaceId, saveDataId))
        {
            g_SaveFileSystemCacheManager.Unregister(saveDataSpaceId, saveDataId);
        }
        cachedFs.reset();

        auto scopedLockForSaveDataExtraDataAccessorCacheManager = g_SaveDataExtraDataAccessorCacheManager.GetScopedLock();
        std::shared_ptr<nn::fs::fsa::IFileSystem> pInternalStorage;
        std::shared_ptr<nn::fssystem::ISaveDataExtraDataAccessor> pExtraDataAccessor;
        NN_RESULT_DO(m_fsCreatorInterfaces->saveDataFsCreator->CreateInternalStorage(&pInternalStorage, &pExtraDataAccessor, std::move(cachedFs), std::move(pFileSystem), saveDataId, IsDeviceUniqueMac(saveDataSpaceId), isTemporaryFs, GetSaveDataCommitTimeStamp));

        // 開いているセーブデータの拡張データアクセサをキャッシュしておく
        if( pExtraDataAccessor )
        {
            pExtraDataAccessor->RegisterCacheObserver(reinterpret_cast<nn::fssystem::ISaveDataExtraDataAccessorCacheObserver*>(&g_SaveDataExtraDataAccessorCacheManager), saveDataSpaceId, saveDataId);
            NN_RESULT_DO(g_SaveDataExtraDataAccessorCacheManager.Register(std::move(pExtraDataAccessor), saveDataSpaceId, saveDataId));
        }

        *pOutFileSystem = std::move(pInternalStorage);

        NN_RESULT_SUCCESS;
    }

    Result FileSystemProxyCoreImpl::OpenSaveDataImageFile(
        std::unique_ptr<nn::fs::fsa::IFile>* pOutFile,
        nn::fs::SaveDataSpaceId saveDataSpaceId,
        nn::fs::SaveDataId saveDataId,
        const char* saveDataRootPathBuffer) NN_NOEXCEPT
    {
        std::shared_ptr<nn::fs::fsa::IFileSystem> pFileSystem;
        NN_RESULT_DO(OpenSaveDataDirectoryFileSystem(
            &pFileSystem,
            saveDataSpaceId,
            saveDataRootPathBuffer,
            true));

        char saveImageName[1 + 16 + 1] = {};
        util::SNPrintf(saveImageName, sizeof(saveImageName), "/%016llx", saveDataId);

        DirectoryEntryType type;
        NN_RESULT_TRY(pFileSystem->GetEntryType(&type, saveImageName))
            NN_RESULT_CATCH(ResultPathNotFound)
            {
                return ResultTargetNotFound();
            }
        NN_RESULT_END_TRY;

        if( type == nn::fs::DirectoryEntryType::DirectoryEntryType_File )
        {
            return pFileSystem->OpenFile(
                pOutFile,
                saveImageName,
                static_cast<nn::fs::OpenMode>(nn::fs::OpenMode_Read | nn::fs::OpenMode_Write));
        }
        else
        {
            return nn::fs::ResultIncompatiblePath();
        }
    }

    Result FileSystemProxyCoreImpl::OpenSaveDataExtensionContextFile(
        std::unique_ptr<nn::fs::fsa::IFile>* pOutFile,
        nn::fs::fsa::IFile* pSaveDataFile,
        nn::fs::SaveDataId saveDataId,
        nn::fs::SaveDataSpaceId saveDataSpaceId,
        int64_t availableSize,
        int64_t journalSize,
        bool isCreatable) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pOutFile);

        const auto result = OpenSaveDataMeta(
            pOutFile,
            saveDataId,
            saveDataSpaceId,
            SaveDataMetaType::ExtensionContext);

        NN_RESULT_TRY(result)
            NN_RESULT_CATCH(nn::fs::ResultPathNotFound)
            {
                if( !isCreatable )
                {
                    NN_RESULT_RETHROW;
                }

                // コンテキストが存在しない場合は新規作成する
                SaveDataExtender extender;
                nn::fs::FileStorage saveDataStorage(pSaveDataFile);

                fssystem::save::JournalIntegritySaveDataParameters parameters = {};
                auto& creator = m_fsCreatorInterfaces->saveDataFsCreator;
                NN_RESULT_DO(creator->ExtractSaveDataParameters(
                    &parameters,
                    &saveDataStorage,
                    IsDeviceUniqueMac(saveDataSpaceId)));
                parameters.countDataBlock
                    = static_cast<uint32_t>(availableSize / parameters.blockSize);
                parameters.countJournalBlock
                    = static_cast<uint32_t>(journalSize / parameters.blockSize);

                NN_RESULT_DO(extender.InitializeContext(parameters));

                const auto contextSize
                    = static_cast<uint32_t>(SaveDataExtender::QueryContextSize());
                const auto logSize
                    = static_cast<uint32_t>(extender.GetLogSize());
                NN_RESULT_DO(CreateSaveDataMeta(
                    saveDataId,
                    saveDataSpaceId,
                    SaveDataMetaType::ExtensionContext,
                    contextSize + logSize));

                NN_RESULT_DO(OpenSaveDataMeta(
                    pOutFile,
                    saveDataId,
                    saveDataSpaceId,
                    SaveDataMetaType::ExtensionContext));

                nn::fs::FileStorage contextStorage(pOutFile->get());
                extender.WriteContext(&contextStorage);
            }
        NN_RESULT_END_TRY

        NN_RESULT_SUCCESS;
    }

    Result FileSystemProxyCoreImpl::StartExtendSaveDataFileSystem(
        int64_t* outExtendedTotalSize,
        nn::fs::SaveDataId saveDataId,
        nn::fs::SaveDataSpaceId saveDataSpaceId,
        int64_t availableSize,
        int64_t journalSize,
        const char* pSaveDataRootPathBuffer) NN_NOEXCEPT
    {
        return ExtendSaveDataFileSystemCore(
            outExtendedTotalSize,
            saveDataId,
            saveDataSpaceId,
            availableSize,
            journalSize,
            pSaveDataRootPathBuffer,
            true);
    }

    Result FileSystemProxyCoreImpl::ResumeExtendSaveDataFileSystem(
        int64_t* outExtendedTotalSize,
        nn::fs::SaveDataId saveDataId,
        nn::fs::SaveDataSpaceId saveDataSpaceId,
        const char* pSaveDataRootPathBuffer) NN_NOEXCEPT
    {
        return ExtendSaveDataFileSystemCore(
            outExtendedTotalSize,
            saveDataId,
            saveDataSpaceId,
            0,
            0,
            pSaveDataRootPathBuffer,
            false);
    }

    Result FileSystemProxyCoreImpl::ExtendSaveDataFileSystemCore(
        int64_t* outExtendedTotalSize,
        nn::fs::SaveDataId saveDataId,
        nn::fs::SaveDataSpaceId saveDataSpaceId,
        int64_t availableSize,
        int64_t journalSize,
        const char* pSaveDataRootPathBuffer,
        bool isStartable) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(outExtendedTotalSize);

        g_SaveFileSystemCacheManager.Unregister(saveDataSpaceId, saveDataId);

        {
            std::unique_ptr<nn::fs::fsa::IFile> saveDataFile;
            std::unique_ptr<nn::fs::fsa::IFile> contextFile;

            // セーブデータイメージファイルを開く
            const auto result = OpenSaveDataImageFile(
                &saveDataFile,
                saveDataSpaceId,
                saveDataId,
                pSaveDataRootPathBuffer);
            NN_RESULT_TRY(result)
                NN_RESULT_CATCH(nn::fs::ResultIncompatiblePath)
            {
                // ディレクトリセーブの場合は何もせず成功扱い
                NN_RESULT_SUCCESS;
            }
            NN_RESULT_END_TRY

            // セーブデータ拡張コンテキストファイルを開く
            NN_RESULT_DO(OpenSaveDataExtensionContextFile(
                &contextFile,
                saveDataFile.get(),
                saveDataId,
                saveDataSpaceId,
                availableSize,
                journalSize,
                isStartable));

            // セーブデータを拡張する
            {
                const auto saveDataStorage = AllocateShared<nn::fs::FileStorage>(saveDataFile.get());
                nn::fs::FileStorage contextStorage(contextFile.get());

                // セーブデータ拡張コンテキストを読み込む
                SaveDataExtender extender;

                NN_RESULT_DO(extender.ReadContext(&contextStorage));

                if( isStartable )
                {
                    NN_RESULT_THROW_UNLESS(
                        extender.GetAvailableSize() == availableSize,
                        nn::fs::ResultDifferentSaveDataExtensionContextParameter());
                    NN_RESULT_THROW_UNLESS(
                        extender.GetJournalSize() == journalSize,
                        nn::fs::ResultDifferentSaveDataExtensionContextParameter());
                }

                // セーブデータファイルを拡張する
                *outExtendedTotalSize = extender.GetExtendedSaveDataSize();

                int64_t currentSize = 0;
                NN_RESULT_DO(saveDataStorage->GetSize(&currentSize));

                if( currentSize < *outExtendedTotalSize )
                {
                    NN_RESULT_DO(saveDataStorage->SetSize(*outExtendedTotalSize));
                }

                // セーブデータを拡張する
                nn::fs::SubStorage saveDataSubStorage(
                    saveDataStorage.get(),
                    0,
                    *outExtendedTotalSize);
                nn::fs::SubStorage logSubStorage(
                    &contextStorage,
                    SaveDataExtender::QueryContextSize(),
                    extender.GetLogSize());

                NN_RESULT_DO(m_fsCreatorInterfaces->saveDataFsCreator->ExtendSaveData(
                    &extender,
                    saveDataSubStorage,
                    logSubStorage,
                    IsDeviceUniqueMac(saveDataSpaceId)));

                // 属性を更新する
                std::shared_ptr<nn::fssystem::ISaveDataExtraDataAccessor> extraDataAccessor;
                nn::fs::SaveDataExtraData extraData = {};

                NN_RESULT_DO(m_fsCreatorInterfaces->saveDataFsCreator->CreateExtraDataAccessor(
                    &extraDataAccessor,
                    saveDataStorage,
                    IsDeviceUniqueMac(saveDataSpaceId),
                    false // 必ず SaveDataFs
                ));

                NN_RESULT_DO(extraDataAccessor->ReadExtraData(&extraData));

                // パラメータは引数で指定されない場合があるのでコンテキストから読み込んだ値を使用する
                extraData.availableSize = extender.GetAvailableSize();
                extraData.journalSize = extender.GetJournalSize();

                NN_RESULT_DO(extraDataAccessor->WriteExtraData(extraData));
                NN_RESULT_DO(extraDataAccessor->CommitExtraData(true));
            }
        }

        NN_RESULT_SUCCESS;
    }


    Result FileSystemProxyCoreImpl::FinishExtendSaveDataFileSystem(
            nn::fs::SaveDataId saveDataId,
            nn::fs::SaveDataSpaceId saveDataSpaceId) NN_NOEXCEPT
    {
        NN_RESULT_TRY(DeleteSaveDataMeta(saveDataId, saveDataSpaceId, SaveDataMetaType::ExtensionContext))
            NN_RESULT_CATCH(ResultPathNotFound)
            {
            }
        NN_RESULT_END_TRY;
        NN_RESULT_SUCCESS;
    }

    void FileSystemProxyCoreImpl::RevertExtendSaveDataFileSystem(
            nn::fs::SaveDataId saveDataId,
            nn::fs::SaveDataSpaceId saveDataSpaceId,
            int64_t beforeExtendSize,
            const char* pSaveDataRootPathBuffer) NN_NOEXCEPT
    {
        std::unique_ptr<nn::fs::fsa::IFile> saveDataFile;

        // セーブデータイメージファイルを開く
        const auto result = OpenSaveDataImageFile(
            &saveDataFile,
            saveDataSpaceId,
            saveDataId,
            pSaveDataRootPathBuffer);
        if( result.IsSuccess() )
        {
            (void)saveDataFile->SetSize(beforeExtendSize);
        }

        (void)FinishExtendSaveDataFileSystem(saveDataId, saveDataSpaceId);
    }

    Result FileSystemProxyCoreImpl::CreateSaveDataMeta(nn::fs::SaveDataId saveDataId, nn::fs::SaveDataSpaceId saveDataSpaceId, nn::fs::SaveDataMetaType metaType, size_t size) NN_NOEXCEPT
    {
        std::shared_ptr<nn::fs::fsa::IFileSystem> pFileSystem;
        NN_RESULT_DO(OpenSaveDataMetaDirectoryFileSystem(&pFileSystem, saveDataSpaceId, saveDataId));

        char saveDataMetaName[1 + 8 + 5 + 1];
        util::SNPrintf(saveDataMetaName, sizeof(saveDataMetaName), "/%08x.meta", metaType);
        NN_RESULT_DO(pFileSystem->CreateFile(saveDataMetaName, size));

        NN_RESULT_SUCCESS;
    }

    Result FileSystemProxyCoreImpl::DeleteSaveDataMeta(nn::fs::SaveDataId saveDataId, nn::fs::SaveDataSpaceId saveDataSpaceId, nn::fs::SaveDataMetaType metaType) NN_NOEXCEPT
    {
        std::shared_ptr<nn::fs::fsa::IFileSystem> pFileSystem;
        NN_RESULT_DO(OpenSaveDataMetaDirectoryFileSystem(&pFileSystem, saveDataSpaceId, saveDataId));

        char saveDataMetaName[1 + 8 + 5 + 1];
        util::SNPrintf(saveDataMetaName, sizeof(saveDataMetaName), "/%08x.meta", metaType);
        NN_RESULT_DO(pFileSystem->DeleteFile(saveDataMetaName));

        NN_RESULT_SUCCESS;
    }

    Result FileSystemProxyCoreImpl::DeleteAllSaveDataMetas(nn::fs::SaveDataId saveDataId, nn::fs::SaveDataSpaceId saveDataSpaceId) NN_NOEXCEPT
    {
        char saveDataMetaDirectoryName[1 + 8 + 1];
        util::SNPrintf(saveDataMetaDirectoryName, sizeof(saveDataMetaDirectoryName), "/saveMeta", saveDataId);
        std::shared_ptr<nn::fs::fsa::IFileSystem> pFileSystem;
        NN_RESULT_DO(OpenSaveDataDirectoryFileSystemImpl(&pFileSystem, saveDataSpaceId, saveDataMetaDirectoryName, false));

        char saveDataIdDirectoryName[1 + 16 + 1];
        util::SNPrintf(saveDataIdDirectoryName, sizeof(saveDataIdDirectoryName), "/%016llx", saveDataId);

        NN_RESULT_TRY(pFileSystem->DeleteDirectoryRecursively(saveDataIdDirectoryName))
            NN_RESULT_CATCH(ResultPathNotFound)
        {
            NN_RESULT_SUCCESS;
        }
        NN_RESULT_END_TRY;

        NN_RESULT_SUCCESS;
    }

    Result FileSystemProxyCoreImpl::OpenSaveDataMeta(std::unique_ptr<nn::fs::fsa::IFile>* pOutFile, nn::fs::SaveDataId saveDataId, nn::fs::SaveDataSpaceId saveDataSpaceId, nn::fs::SaveDataMetaType metaType) NN_NOEXCEPT
    {
        std::shared_ptr<nn::fs::fsa::IFileSystem> pFileSystem;
        NN_RESULT_DO(OpenSaveDataMetaDirectoryFileSystem(&pFileSystem, saveDataSpaceId, saveDataId));

        char saveDataMetaName[1 + 8 + 5 + 1];
        util::SNPrintf(saveDataMetaName, sizeof(saveDataMetaName), "/%08x.meta", metaType);
        return pFileSystem->OpenFile(pOutFile, saveDataMetaName, static_cast<OpenMode>(OpenMode_Read | OpenMode_Write));
    }

    Result FileSystemProxyCoreImpl::QuerySaveDataTotalSize(int64_t* outValue, int32_t blockSize, int64_t size, int64_t journalSize) NN_NOEXCEPT
    {
        auto params = save::JournalIntegritySaveDataFileSystemDriver::SetUpSaveDataParameters(blockSize, size, journalSize);
        return save::JournalIntegritySaveDataFileSystemDriver::QueryTotalSize(outValue, params.blockSize, params.countExpandMax, params.paramDuplex, params.paramIntegrity, params.countDataBlock, params.countJournalBlock);
    }

    Result FileSystemProxyCoreImpl::CreateSaveDataFileSystem(nn::fs::SaveDataId saveDataId, const nn::fs::SaveDataAttribute& attribute, const nn::fs::SaveDataCreationInfo& creationInfo, const char* saveDataRootPathBuffer, const nn::fs::SaveDataHashSalt& hashSalt, bool skipFormat) NN_NOEXCEPT
    {
        auto size = creationInfo.size;
        auto journalSize = creationInfo.journalSize;
        auto blockSize = creationInfo.blockSize;
        auto ownerId = creationInfo.ownerId;
        auto saveDataSpaceId = creationInfo.spaceId;
        auto isPseudoSaveFs = creationInfo.isPseudoSaveFs;
        auto flags = creationInfo.flags;

        std::shared_ptr<nn::fs::fsa::IFileSystem> pFileSystem;
        NN_RESULT_DO(OpenSaveDataDirectoryFileSystem(&pFileSystem, saveDataSpaceId, saveDataRootPathBuffer, false));

        char saveImageName[1 + 16 + 1];
        util::SNPrintf(saveImageName, sizeof(saveImageName), "/%016llx", saveDataId); // TODO: ディレクトリ分割

#if defined(NN_BUILD_CONFIG_OS_WIN32)
        // Win 版デフォルトではディレクトリセーブデータ FS を利用
        isPseudoSaveFs = true;
        // ターゲット環境変数で指定がある場合、Win 版でも実機向けセーブデータ FS を作成する
        {
            char value[16];
            const char variable[] = "SDK_FS_WIN_USE_TRUE_SAVE_FS";
            size_t outLength = 0;
            if (nn::htc::GetTargetEnvironmentVariableLength(&outLength, variable).IsSuccess() && outLength <= 16)
            {
                if (nn::htc::GetTargetEnvironmentVariable(&outLength, value, outLength, variable).IsSuccess())
                {
                    if (strncmp(value, "TRUE", sizeof(value)) == 0)
                    {
                        isPseudoSaveFs = false;
                    }
                }
            }
        }
#endif

        if (isPseudoSaveFs)
        {
#if !defined(NN_BUILD_CONFIG_OS_WIN)
            if (IsStaticSaveDataIdValueRange(saveDataId))
            {
                NN_SDK_LOG("[fs] Warning: System save data (0x%016llx) was created with default size by auto creation.\n", saveDataId);
            }

            // サイズ決め打ちで正式版セーブデータを作成
            const int64_t SaveDataSize = 16 * 1024 * 1024;
            size        = SaveDataSize;
            journalSize = SaveDataSize;
#else
            // ディレクトリ版セーブデータを作成
            NN_RESULT_DO(EnsureDirectory(pFileSystem.get(), saveImageName));
            NN_RESULT_SUCCESS;
#endif
        }

        // サイズ計算・ファイル作成
        // TODO: システムセーブ向けにブロックサイズ指定可に。
        NN_UNUSED(blockSize);
        int32_t BlockSize = 16 * 1024;
        int64_t totalSize = 0;

        util::optional<save::IntegritySaveDataParameters>        integrityParams;
        util::optional<save::JournalIntegritySaveDataParameters> journalIntegrityParams;
        const bool IsTemporaryFs = (attribute.type == SaveDataType::Temporary);
        if (IsTemporaryFs)
        {
            auto params = save::IntegritySaveDataFileSystemDriver::SetUpSaveDataParameters(BlockSize, size);
            NN_RESULT_DO(save::IntegritySaveDataFileSystemDriver::QueryTotalSize(&totalSize, params.blockSize, params.paramIntegrity, params.countDataBlock));
            integrityParams.emplace(params);
        }
        else
        {
            auto params = save::JournalIntegritySaveDataFileSystemDriver::SetUpSaveDataParameters(BlockSize, size, journalSize);
            NN_RESULT_DO(save::JournalIntegritySaveDataFileSystemDriver::QueryTotalSize(&totalSize, params.blockSize, params.countExpandMax, params.paramDuplex, params.paramIntegrity, params.countDataBlock, params.countJournalBlock));
            journalIntegrityParams.emplace(params);
        }

        NN_RESULT_DO(pFileSystem->CreateFile(saveImageName, totalSize));

        if (skipFormat)
        {
            NN_RESULT_SUCCESS;
        }

        bool isSuccessToFormat = false;

        NN_UTIL_SCOPE_EXIT
        {
            if (!isSuccessToFormat)
            {
                pFileSystem->DeleteFile(saveImageName);
            }
        };

        // フォーマット
        std::shared_ptr<fs::FileStorageBasedFileSystem> fileStorage = fssystem::AllocateShared<fs::FileStorageBasedFileSystem>();
        NN_RESULT_THROW_UNLESS(fileStorage, ResultAllocationMemoryFailedInFileSystemProxyCoreImplA());
        NN_RESULT_DO(fileStorage->Initialize(pFileSystem, saveImageName, static_cast<nn::fs::OpenMode>(nn::fs::OpenMode_Read | nn::fs::OpenMode_Write)));

        if (IsTemporaryFs)
        {
            NN_SDK_ASSERT(integrityParams != util::nullopt);
            auto params = integrityParams.value();
            nn::fs::SubStorage storage(fileStorage.get(), 0, totalSize);
            NN_RESULT_DO(m_fsCreatorInterfaces->saveDataFsCreator->FormatAsIntegritySaveData(storage, params.blockSize, params.paramIntegrity, params.countDataBlock, m_pBufferManager, IsDeviceUniqueMac(saveDataSpaceId)));
        }
        else
        {
            NN_SDK_ASSERT(journalIntegrityParams != util::nullopt);
            auto params = journalIntegrityParams.value();
            nn::fs::SubStorage storage(fileStorage.get(), 0, totalSize);
            NN_RESULT_DO(m_fsCreatorInterfaces->saveDataFsCreator->Format(storage, params.blockSize, params.countExpandMax, params.paramDuplex, params.paramIntegrity, params.countDataBlock, params.countJournalBlock, m_pBufferManager, IsDeviceUniqueMac(saveDataSpaceId), hashSalt));
        }

        // 属性埋め込み
        {
            std::shared_ptr<nn::fssystem::ISaveDataExtraDataAccessor> extraDataAccessor;
            NN_RESULT_DO(m_fsCreatorInterfaces->saveDataFsCreator->CreateExtraDataAccessor(&extraDataAccessor, fileStorage, IsDeviceUniqueMac(saveDataSpaceId), IsTemporaryFs));
            nn::fs::SaveDataExtraData extraData;
            {
                std::memset(&extraData, 0x0, sizeof(extraData));
                std::memcpy(&(extraData.attribute), &attribute, sizeof(nn::fs::SaveDataAttribute));
                extraData.ownerId = ownerId;
                extraData.timeStamp = 0;
                GetSaveDataCommitTimeStamp(&(extraData.timeStamp));
                extraData.commitId = 0;
                m_GenerateRandom(&extraData.commitId, sizeof(extraData.commitId));
                extraData.flags = flags;
                extraData.availableSize = size;
                extraData.journalSize = journalSize;
            }
            NN_RESULT_DO(extraDataAccessor->WriteExtraData(extraData));
            NN_RESULT_DO(extraDataAccessor->CommitExtraData(true));
        }

        isSuccessToFormat = true;

        NN_RESULT_SUCCESS;
    } // NOLINT(impl/function_size)

    Result FileSystemProxyCoreImpl::DeleteSaveDataFileSystem(nn::fs::SaveDataSpaceId saveDataSpaceId, nn::fs::SaveDataId saveDataId, const char* saveDataRootPathBuffer) NN_NOEXCEPT
    {
        g_SaveFileSystemCacheManager.Unregister(saveDataSpaceId, saveDataId);

        std::shared_ptr<nn::fs::fsa::IFileSystem> pFileSystem;
        NN_RESULT_DO(OpenSaveDataDirectoryFileSystem(&pFileSystem, saveDataSpaceId, saveDataRootPathBuffer, false));

        char saveImageName[1 + 16 + 1];
        util::SNPrintf(saveImageName, sizeof(saveImageName), "/%016llx", saveDataId); // TODO: ディレクトリ分割

        DirectoryEntryType type;
        NN_RESULT_DO(pFileSystem->GetEntryType(&type, saveImageName));
        if (type == nn::fs::DirectoryEntryType::DirectoryEntryType_Directory)
        {
            NN_RESULT_DO(pFileSystem->DeleteDirectoryRecursively(saveImageName));
        }
        else
        {
            NN_RESULT_DO(pFileSystem->DeleteFile(saveImageName));
        }

        NN_RESULT_SUCCESS;
    }

    Result FileSystemProxyCoreImpl::ReadSaveDataFileSystemExtraData(nn::fs::SaveDataExtraData* outValue, nn::fs::SaveDataSpaceId saveDataSpaceId, nn::fs::SaveDataId saveDataId, const char* saveDataRootPathBuffer, bool isTemporaryFs) NN_NOEXCEPT
    {
        // SaveFileSystemCacheManager → SaveDataExtraDataAccessorCacheManager の順でロックする規則
        auto scopedLockForSaveFileSystemCacheManager = g_SaveFileSystemCacheManager.GetScopedLock();
        auto scopedLockForSaveDataExtraDataAccessorCacheManager = g_SaveDataExtraDataAccessorCacheManager.GetScopedLock();

        std::shared_ptr<nn::fssystem::ISaveDataExtraDataAccessor> extraDataAccessor;
        if (g_SaveDataExtraDataAccessorCacheManager.GetCache(&extraDataAccessor, saveDataSpaceId, saveDataId).IsSuccess())
        {
            return extraDataAccessor->ReadExtraData(outValue);
        }
        else
        {
            std::shared_ptr<nn::fs::fsa::IFileSystem> pFileSystem;
            NN_RESULT_DO(OpenSaveDataDirectoryFileSystem(&pFileSystem, saveDataSpaceId, saveDataRootPathBuffer, false));

            char saveImageName[1 + 16 + 1];
            util::SNPrintf(saveImageName, sizeof(saveImageName), "/%016llx", saveDataId); // TODO: ディレクトリ分割

            DirectoryEntryType type;
            NN_RESULT_TRY(pFileSystem->GetEntryType(&type, saveImageName))
                NN_RESULT_CATCH(ResultPathNotFound)
                {
                    return ResultTargetNotFound();
                }
            NN_RESULT_END_TRY;

            if (type == nn::fs::DirectoryEntryType::DirectoryEntryType_Directory)
            {
                std::memset(outValue, 0, sizeof(nn::fs::SaveDataExtraData));
                NN_RESULT_SUCCESS;
            }

            std::shared_ptr<fs::FileStorageBasedFileSystem> fileStorage = fssystem::AllocateShared<fs::FileStorageBasedFileSystem>();
            NN_RESULT_THROW_UNLESS(fileStorage, ResultAllocationMemoryFailedInFileSystemProxyCoreImplB());
            NN_RESULT_DO(fileStorage->Initialize(pFileSystem, saveImageName, static_cast<nn::fs::OpenMode>(nn::fs::OpenMode_Read)));

            int64_t size = 0;
            NN_RESULT_DO(fileStorage->GetSize(&size));
            nn::fs::SubStorage storage(fileStorage.get(), 0, size);
            return m_fsCreatorInterfaces->saveDataFsCreator->ReadExtraData(outValue, storage, m_pBufferManager, IsDeviceUniqueMac(saveDataSpaceId), isTemporaryFs);
        }
    }

    Result FileSystemProxyCoreImpl::WriteSaveDataFileSystemExtraData(nn::fs::SaveDataSpaceId saveDataSpaceId, nn::fs::SaveDataId saveDataId, const nn::fs::SaveDataExtraData& value, const char* saveDataRootPathBuffer, nn::fs::SaveDataType saveDataType, bool updateTimeStampAndHash) NN_NOEXCEPT
    {
        // SaveFileSystemCacheManager → SaveDataExtraDataAccessorCacheManager の順でロックする規則
        auto scopedLockForSaveFileSystemCacheManager = g_SaveFileSystemCacheManager.GetScopedLock();
        auto scopedLockForSaveDataExtraDataAccessorCacheManager = g_SaveDataExtraDataAccessorCacheManager.GetScopedLock();

        std::shared_ptr<nn::fssystem::ISaveDataExtraDataAccessor> extraDataAccessor;
        if (g_SaveDataExtraDataAccessorCacheManager.GetCache(&extraDataAccessor, saveDataSpaceId, saveDataId).IsFailure())
        {
            std::shared_ptr<nn::fs::fsa::IFileSystem> saveDataFs;
            // ISaveDataExtraDataAccessor をキャッシュするため一時的にマウントする
            NN_RESULT_DO(OpenSaveDataFileSystem(&saveDataFs, saveDataSpaceId, saveDataId, saveDataRootPathBuffer, false, saveDataType));
            if (g_SaveDataExtraDataAccessorCacheManager.GetCache(&extraDataAccessor, saveDataSpaceId, saveDataId).IsFailure())
            {
                // マウントに成功したのに ISaveDataExtraDataAccessor が無い => Directory セーブなのでなにもせず成功を返す
                NN_RESULT_SUCCESS;
            }
        }

        NN_RESULT_DO(extraDataAccessor->WriteExtraData(value));
        NN_RESULT_DO(extraDataAccessor->CommitExtraData(updateTimeStampAndHash));
        NN_RESULT_SUCCESS;
    }

    Result FileSystemProxyCoreImpl::CorruptSaveDataFileSystem(nn::fs::SaveDataSpaceId saveDataSpaceId, nn::fs::SaveDataId saveDataId, const char* saveDataRootPathBuffer) NN_NOEXCEPT
    {
        g_SaveFileSystemCacheManager.Unregister(saveDataSpaceId, saveDataId);

        std::shared_ptr<nn::fs::fsa::IFileSystem> pFileSystem;
        NN_RESULT_DO(OpenSaveDataDirectoryFileSystem(&pFileSystem, saveDataSpaceId, saveDataRootPathBuffer, false));

        char saveImageName[1 + 16 + 1];
        util::SNPrintf(saveImageName, sizeof(saveImageName), "/%016llx", saveDataId); // TODO: ディレクトリ分割

        DirectoryEntryType type;
        NN_RESULT_DO(pFileSystem->GetEntryType(&type, saveImageName));
        NN_RESULT_THROW_UNLESS(type == nn::fs::DirectoryEntryType::DirectoryEntryType_File, ResultPreconditionViolation());

        // 先頭領域を破壊
        std::unique_ptr<nn::fs::fsa::IFile> file;
        NN_RESULT_DO(pFileSystem->OpenFile(&file, saveImageName, OpenMode_Write));

        const size_t BufferSize = 512;
        const int64_t CorruptionSize = 32 * 1024;
        char buffer[BufferSize];
        memset(buffer, 0xAA, sizeof(buffer));
        for(auto offset = 0LL; offset < CorruptionSize; offset += BufferSize)
        {
            NN_RESULT_DO(file->Write(offset, buffer, sizeof(buffer), WriteOption()));
        }
        NN_RESULT_DO(file->Flush());

        NN_RESULT_SUCCESS;
    }

    Result FileSystemProxyCoreImpl::RecoverSaveDataFileSystemMasterHeader(nn::fs::SaveDataSpaceId saveDataSpaceId, nn::fs::SaveDataId saveDataId) NN_NOEXCEPT
    {
        std::shared_ptr<nn::fs::fsa::IFileSystem> pFileSystem;
        NN_RESULT_DO(OpenSaveDataDirectoryFileSystem(&pFileSystem, saveDataSpaceId, "", true));

        g_SaveFileSystemCacheManager.Unregister(saveDataSpaceId, saveDataId);

        NN_RESULT_DO(m_fsCreatorInterfaces->saveDataFsCreator->RecoverMasterHeader(std::move(pFileSystem), saveDataId, m_pBufferManager, IsDeviceUniqueMac(saveDataSpaceId)));
        NN_RESULT_SUCCESS;
    }

    Result FileSystemProxyCoreImpl::UpdateSaveDataFileSystemMac(nn::fs::SaveDataSpaceId saveDataSpaceId, nn::fs::SaveDataId saveDataId) NN_NOEXCEPT
    {
        std::shared_ptr<nn::fs::fsa::IFileSystem> pFileSystem;
        NN_RESULT_DO(OpenSaveDataDirectoryFileSystem(&pFileSystem, saveDataSpaceId, "", true));

        g_SaveFileSystemCacheManager.Unregister(saveDataSpaceId, saveDataId);

        NN_RESULT_DO(m_fsCreatorInterfaces->saveDataFsCreator->UpdateMac(std::move(pFileSystem), saveDataId, IsDeviceUniqueMac(saveDataSpaceId)));
        NN_RESULT_SUCCESS;
    }


    Result FileSystemProxyCoreImpl::CreatePaddingFile(std::int64_t paddingSize) NN_NOEXCEPT
    {
        std::shared_ptr<nn::fs::fsa::IFileSystem> pFileSystem;
        NN_RESULT_DO(OpenBisFileSystem(&pFileSystem, "", nn::fs::BisPartitionId::User));

        // @User:/Padding/00000000
        NN_RESULT_DO(EnsureDirectory(pFileSystem.get(), PaddingDirectoryName));
        for (int counter = 0; counter <= 0x0FFFFFFF; counter++)
        {
            char filePath[128];
            nn::util::SNPrintf(filePath, sizeof(filePath), "%s/%08x", PaddingDirectoryName, counter);
            NN_RESULT_TRY(pFileSystem->CreateFile(filePath, paddingSize, CreateFileOptionFlag_BigFile))
                NN_RESULT_CATCH(ResultPathAlreadyExists)
                {
                    continue;
                }
            NN_RESULT_END_TRY
            NN_RESULT_SUCCESS;
        }

        return ResultPathAlreadyExists();
    }

    Result FileSystemProxyCoreImpl::DeleteAllPaddingFiles() NN_NOEXCEPT
    {
        std::shared_ptr<nn::fs::fsa::IFileSystem> pFileSystem;
        NN_RESULT_DO(OpenBisFileSystem(&pFileSystem, "", nn::fs::BisPartitionId::User));
        NN_RESULT_TRY(pFileSystem->DeleteDirectoryRecursively(PaddingDirectoryName))
            NN_RESULT_CATCH(ResultPathNotFound)
            {
            }
        NN_RESULT_END_TRY
        NN_RESULT_SUCCESS;
    }


    Result FileSystemProxyCoreImpl::OpenImageDirectoryFileSystem(std::shared_ptr<nn::fs::fsa::IFileSystem>* pOutFileSystem, nn::fs::ImageDirectoryId storageId) NN_NOEXCEPT
    {
        std::shared_ptr<nn::fs::fsa::IFileSystem> pFileSystem;

#if defined(NN_BUILD_CONFIG_OS_WIN32)
        NN_RESULT_DO(m_fsCreatorInterfaces->hostFsCreator->Create(&pFileSystem, GetExecutionDirectoryPath()));

        // TORIAEZU: "実行ディレクトリ/Album[_NAND|_SD]/" をスクリーンショット置き場とする
        const char* const imageDirectoryPath = (storageId == nn::fs::ImageDirectoryId::Nand) ? "/Album_NAND"
            : "/Album_SD";
#else
        // @User / @Sdcard を取得
        switch(storageId)
        {
        case nn::fs::ImageDirectoryId::Nand:
            NN_RESULT_DO(OpenBisFileSystem(&pFileSystem, "", nn::fs::BisPartitionId::User));
            break;
        case nn::fs::ImageDirectoryId::SdCard:
            NN_RESULT_DO(OpenSdCardProxyFileSystem(&pFileSystem));
            break;
        default:
            return nn::fs::ResultInvalidArgument();
        }

        // "@User:/Album" | "@SdCard:/Nintendo/Album" をスクリーンショット置き場とする
        char imageDirectoryPath[64];
        int length;
        if (storageId == nn::fs::ImageDirectoryId::SdCard)
        {
            length = util::SNPrintf(imageDirectoryPath, sizeof(imageDirectoryPath), "/%s/%s", SdCardNintendoRootDirectoryName, ImageDirectoryName);
        }
        else
        {
            length = util::SNPrintf(imageDirectoryPath, sizeof(imageDirectoryPath), "/%s", ImageDirectoryName);
        }
        NN_SDK_ASSERT(static_cast<size_t>(length) < sizeof(imageDirectoryPath));
        NN_UNUSED(length);
#endif

        // スクリーンショット置き場が無かったら作成
        NN_RESULT_DO(EnsureDirectory(pFileSystem.get(), imageDirectoryPath));

        std::shared_ptr<nn::fs::fsa::IFileSystem> subDirFs;
        NN_RESULT_DO(m_fsCreatorInterfaces->subDirFsCreator->Create(&subDirFs, std::move(pFileSystem), imageDirectoryPath));

        *pOutFileSystem = std::move(subDirFs);
        NN_RESULT_SUCCESS;
    }

    Result FileSystemProxyCoreImpl::OpenContentStorageFileSystem(std::shared_ptr<nn::fs::fsa::IFileSystem>* pOutFileSystem, nn::fs::ContentStorageId storageId) NN_NOEXCEPT
    {
        std::shared_ptr<nn::fs::fsa::IFileSystem> pFileSystem;

        // @System / @User / @Sdcard を取得
        switch(storageId)
        {
        case nn::fs::ContentStorageId::System:
            NN_RESULT_DO(OpenBisFileSystem(&pFileSystem, "", nn::fs::BisPartitionId::System));
            break;
        case nn::fs::ContentStorageId::User:
            NN_RESULT_DO(OpenBisFileSystem(&pFileSystem, "", nn::fs::BisPartitionId::User));
            break;
        case nn::fs::ContentStorageId::SdCard:
            NN_RESULT_DO(OpenSdCardProxyFileSystem(&pFileSystem));
            break;
        default:
            return nn::fs::ResultInvalidArgument();
        }

        char contentStoragePath[64];
        int length;
        if (storageId == nn::fs::ContentStorageId::SdCard)
        {
            length = util::SNPrintf(contentStoragePath, sizeof(contentStoragePath), "/%s/%s", SdCardNintendoRootDirectoryName, ContentStorageDirectoryName);
        }
        else
        {
            length = util::SNPrintf(contentStoragePath, sizeof(contentStoragePath), "/%s", ContentStorageDirectoryName);
        }
        NN_SDK_ASSERT(static_cast<size_t>(length) < sizeof(contentStoragePath));
        NN_UNUSED(length);

        // コンテンツディレクトリが無かったら作成
        NN_RESULT_DO(EnsureDirectory(pFileSystem.get(), contentStoragePath));

        std::shared_ptr<nn::fs::fsa::IFileSystem> subDirFs;
        NN_RESULT_DO(m_fsCreatorInterfaces->subDirFsCreator->Create(&subDirFs, std::move(pFileSystem), contentStoragePath));

        // SD のみ暗号化を被せる
        if( storageId == nn::fs::ContentStorageId::SdCard )
        {
            NN_RESULT_DO(m_fsCreatorInterfaces->encryptedFileSystemCreator->Create(&subDirFs, std::move(subDirFs), fscreator::IEncryptedFileSystemCreator::KeyId::SdCardContentStorage, m_SdCardEncryptionSeed));
        }

        *pOutFileSystem = std::move(subDirFs);
        NN_RESULT_SUCCESS;
    }

    Result FileSystemProxyCoreImpl::OpenCloudBackupWorkStorageFileSystem(std::shared_ptr<nn::fs::fsa::IFileSystem>* pOutFileSystem, nn::fs::CloudBackupWorkStorageId storageId) NN_NOEXCEPT
    {
        std::shared_ptr<nn::fs::fsa::IFileSystem> pFileSystem;

        // @User / @Sdcard を取得
        switch(storageId)
        {
        case nn::fs::CloudBackupWorkStorageId::Nand:
            NN_RESULT_DO(OpenBisFileSystem(&pFileSystem, "", nn::fs::BisPartitionId::User));
            break;
        case nn::fs::CloudBackupWorkStorageId::SdCard:
            NN_RESULT_DO(OpenSdCardProxyFileSystem(&pFileSystem));
            break;
        default:
            return nn::fs::ResultInvalidArgument();
        }

        char workStoragePath[64];
        int length;
        if (storageId == nn::fs::CloudBackupWorkStorageId::SdCard)
        {
            length = util::SNPrintf(workStoragePath, sizeof(workStoragePath), "/%s/%s", SdCardNintendoRootDirectoryName, CloudBackupWorkStorageDirectoryName);
        }
        else
        {
            length = util::SNPrintf(workStoragePath, sizeof(workStoragePath), "/%s", CloudBackupWorkStorageDirectoryName);
        }
        NN_SDK_ASSERT(static_cast<size_t>(length) < sizeof(workStoragePath));
        NN_UNUSED(length);

        // ディレクトリが無かったら作成
        NN_RESULT_DO(EnsureDirectory(pFileSystem.get(), workStoragePath));

        std::shared_ptr<nn::fs::fsa::IFileSystem> subDirFs;
        NN_RESULT_DO(m_fsCreatorInterfaces->subDirFsCreator->Create(&subDirFs, std::move(pFileSystem), workStoragePath));

        *pOutFileSystem = std::move(subDirFs);
        NN_RESULT_SUCCESS;
    }


    Result FileSystemProxyCoreImpl::OpenBisFileSystem(std::shared_ptr<nn::fs::fsa::IFileSystem>* pOutFileSystem, const char* rootPath, nn::fs::BisPartitionId id) NN_NOEXCEPT
    {
        return m_fsCreatorInterfaces->bisfsCreator->Create(pOutFileSystem, rootPath, id);
    }

    Result FileSystemProxyCoreImpl::OpenBisStorage(std::shared_ptr<fs::IStorage>* pOutStorage, nn::fs::BisPartitionId id) NN_NOEXCEPT
    {
        std::shared_ptr<fs::IStorage> builtInStorage;
        NN_RESULT_DO(m_fsCreatorInterfaces->builtInStorageCreator->Create(&builtInStorage, id));
        *pOutStorage = std::move(builtInStorage);
        NN_RESULT_SUCCESS;
    }

    Result FileSystemProxyCoreImpl::InvalidateBisCache() NN_NOEXCEPT
    {
        std::shared_ptr<fs::IStorage> builtInStorage;
        NN_RESULT_DO(m_fsCreatorInterfaces->builtInStorageCreator->InvalidateCache());
        NN_RESULT_SUCCESS;
    }

    Result FileSystemProxyCoreImpl::OpenGameCardPartition(std::shared_ptr<fs::IStorage>* pOutStorage, nn::fs::GameCardHandle handle, nn::fs::GameCardPartitionRaw partition) NN_NOEXCEPT
    {
        std::shared_ptr<fs::IStorage> storage;
        switch (partition)
        {
        case nn::fs::GameCardPartitionRaw::NormalReadOnly:
            NN_RESULT_DO(m_fsCreatorInterfaces->gameCardStorageCreator->CreateReadOnly(handle, &storage));
            break;
        case nn::fs::GameCardPartitionRaw::SecureReadOnly:
            NN_RESULT_DO(m_fsCreatorInterfaces->gameCardStorageCreator->CreateSecureReadOnly(handle, &storage));
            break;
        case nn::fs::GameCardPartitionRaw::RootWriteOnly:
            NN_RESULT_DO(m_fsCreatorInterfaces->gameCardStorageCreator->CreateWriteOnly(handle, &storage));
            break;
        default:
            return ResultInvalidArgument();
        }

        *pOutStorage = std::move(storage);
        NN_RESULT_SUCCESS;
    }

    Result FileSystemProxyCoreImpl::OpenGameCardFileSystem(std::shared_ptr<nn::fs::fsa::IFileSystem>* pOutFileSystem, nn::fs::GameCardHandle handle, nn::fs::GameCardPartition partition) NN_NOEXCEPT
    {
        NN_DETAIL_FS_DATA_CORRUPTED_RETRY_RESULT_DO(m_fsCreatorInterfaces->gameCardFileSystemCreator->Create(pOutFileSystem, handle, partition));
        NN_RESULT_SUCCESS;
    }

    Result FileSystemProxyCoreImpl::SetBisRootForHost(nn::fs::BisPartitionId id, const char* rootPath) NN_NOEXCEPT
    {
        return m_fsCreatorInterfaces->bisfsCreator->SetBisRoot(id, rootPath);
    }

    Result FileSystemProxyCoreImpl::OpenSdCardProxyFileSystem(std::shared_ptr<nn::fs::fsa::IFileSystem>* pOutFileSystem) NN_NOEXCEPT
    {
        return m_fsCreatorInterfaces->sdProxyFsCreator->Create(pOutFileSystem);
    }

    Result FileSystemProxyCoreImpl::FormatSdCardProxyFileSystem() NN_NOEXCEPT
    {
        return m_fsCreatorInterfaces->sdProxyFsCreator->Format();
    }

    Result FileSystemProxyCoreImpl::OpenHostFileSystem(std::shared_ptr<nn::fs::fsa::IFileSystem>* pOutFileSystem, const char* rootPath) NN_NOEXCEPT
    {
        std::shared_ptr<nn::fs::fsa::IFileSystem> pFileSystem;

        // TODO: Creater で分岐
#if defined(NN_BUILD_CONFIG_OS_WIN32)
        NN_RESULT_DO(m_fsCreatorInterfaces->hostFsCreator->Create(&pFileSystem));
#else
        NN_RESULT_DO(m_fsCreatorInterfaces->targetFsCreator->Create(&pFileSystem));
#endif

        if (rootPath[0] != '\0')
        {
            std::shared_ptr<nn::fs::fsa::IFileSystem> subDirFs;
            NN_RESULT_DO(m_fsCreatorInterfaces->subDirFsCreator->Create(&subDirFs, std::move(pFileSystem), rootPath, true));
            pFileSystem = std::move(subDirFs);
        }
        else
        {
            DirectoryEntryType type;
            NN_RESULT_TRY(pFileSystem->GetEntryType(&type, "C:/"))
                NN_RESULT_CATCH(ResultTargetNotFound) // PC 未接続時だけそのまま return する
                {
                    NN_RESULT_RETHROW;
                }
                NN_RESULT_CATCH_ALL
                {
                    // do nothing
                }
                NN_RESULT_END_TRY;
        }

        *pOutFileSystem = std::move(pFileSystem);
        NN_RESULT_SUCCESS;
    }

    Result FileSystemProxyCoreImpl::GetRightsId(RightsId* pOutRightsId, uint8_t* pOutKeyGeneration, const char* path, nn::ncm::ProgramId programId) NN_NOEXCEPT
    {
        bool isGameCard = false;
        std::shared_ptr<NcaReader> ncaReader;
        NN_RESULT_DO(ParseNca(&ncaReader, &isGameCard, path, programId.value));

        ncaReader->GetRightsId(pOutRightsId->data, sizeof(pOutRightsId->data));
        *pOutKeyGeneration = ncaReader->GetKeyGeneration();
        NN_RESULT_SUCCESS;
    }


    Result FileSystemProxyCoreImpl::RegisterExternalKey(const RightsId& rightsId, const nn::spl::AccessKey& accessKey) NN_NOEXCEPT
    {
        return m_ExternalKeyManager.Register(rightsId, accessKey);
    }

    Result FileSystemProxyCoreImpl::UnregisterAllExternalKey() NN_NOEXCEPT
    {
        return m_ExternalKeyManager.UnregisterAll();
    }


    Result FileSystemProxyCoreImpl::SetSdCardEncryptionSeed(const EncryptionSeed& seed) NN_NOEXCEPT
    {
        m_SdCardEncryptionSeed = seed;
        m_fsCreatorInterfaces->saveDataFsCreator->SetMacGenerationSeed(&seed, sizeof(seed));
        NN_RESULT_SUCCESS;
    }

    void FileSystemProxyCoreImpl::GetAndClearErrorInfo(fs::FileSystemProxyErrorInfo* pOutValue) NN_NOEXCEPT
    {
        std::lock_guard<os::Mutex> scopedLock(m_ErrorInfoMutex);
        memcpy(pOutValue, &m_ErrorInfo, sizeof(FileSystemProxyErrorInfo));
        memset(&m_ErrorInfo, 0, sizeof(FileSystemProxyErrorInfo));
    }

    void FileSystemProxyCoreImpl::IncrementRomFsRemountForDataCorruptionCount() NN_NOEXCEPT
    {
        std::lock_guard<os::Mutex> scopedLock(m_ErrorInfoMutex);
        m_ErrorInfo.romFsRemountForDataCorruptionCount++;
    }

    void FileSystemProxyCoreImpl::IncrementRomFsUnrecoverableDataCorruptionByRemountCount() NN_NOEXCEPT
    {
        std::lock_guard<os::Mutex> scopedLock(m_ErrorInfoMutex);
        m_ErrorInfo.romFsUnrecoverableDataCorruptionByRemountCount++;
    }

    void FileSystemProxyCoreImpl::IncrementRomFsRecoveredByInvalidateCacheCount() NN_NOEXCEPT
    {
        std::lock_guard<os::Mutex> scopedLock(m_ErrorInfoMutex);
        m_ErrorInfo.romFsRecoveredByInvalidateCacheCount++;
    }

    Result FileSystemProxyCoreImpl::RegisterUpdatePartition(nn::Bit64 programId, const char* path) NN_NOEXCEPT
    {
        NN_RESULT_DO(m_UpdatePartitionPath.Set(programId, path));
        NN_RESULT_SUCCESS;
    }

    Result FileSystemProxyCoreImpl::OpenRegisteredUpdatePartition(std::shared_ptr<nn::fs::fsa::IFileSystem>* pOutFileSystem) NN_NOEXCEPT
    {
        nn::Bit64 updaterProgramId = 0;
        auto path = m_UpdatePartitionPath.Get(&updaterProgramId);
        NN_RESULT_DO(OpenFileSystem(pOutFileSystem, path, FileSystemProxyType::FileSystemProxyType_UpdatePartition, updaterProgramId));
        NN_RESULT_SUCCESS;
    }

    // private

    Result FileSystemProxyCoreImpl::ParseMountName(const char** path, std::shared_ptr<nn::fs::fsa::IFileSystem>* pOutFileSystem, bool *isContinue, bool *isGameCard, nn::fs::GameCardHandle* outGcHandle, bool *isHostFs) NN_NOEXCEPT
    {
        *isContinue = true;
        *isGameCard = false;
        *isHostFs   = false;
        if (strncmp(*path, GameCardFileSystemMountName, strnlen(GameCardFileSystemMountName, MountNameLengthMax)) == 0)
        {
            *path += strnlen(GameCardFileSystemMountName, MountNameLengthMax);

            GameCardPartition partition;
            NN_RESULT_DO(GetGameCardPartitionByMountName(&partition, *path));
            *path += GameCardFileSystemMountNameSuffixLength;

            GameCardHandle handle;
            NN_RESULT_DO(GetDeviceHandleByMountName(&handle, *path));
            *path += sizeof(GameCardHandle) * 2; // "%08x"

            NN_RESULT_DO(OpenGameCardFileSystem(pOutFileSystem, handle, partition));

            *outGcHandle = handle;
            *isGameCard = true;
        }
        else if (strncmp(*path, ContentStorageSystemMountName, strnlen(ContentStorageSystemMountName, MountNameLengthMax)) == 0)
        {
            *path += strnlen(ContentStorageSystemMountName, MountNameLengthMax);
            NN_RESULT_DO(OpenContentStorageFileSystem(pOutFileSystem, ContentStorageId::System));
        }
        else if (strncmp(*path, ContentStorageUserMountName, strnlen(ContentStorageUserMountName, MountNameLengthMax)) == 0)
        {
            *path += strnlen(ContentStorageUserMountName, MountNameLengthMax);
            NN_RESULT_DO(OpenContentStorageFileSystem(pOutFileSystem, ContentStorageId::User));
        }
        else if (strncmp(*path, ContentStorageSdCardMountName, strnlen(ContentStorageSdCardMountName, MountNameLengthMax)) == 0)
        {
            *path += strnlen(ContentStorageSdCardMountName, MountNameLengthMax);
            NN_RESULT_DO(OpenContentStorageFileSystem(pOutFileSystem, ContentStorageId::SdCard));
        }
        else if (strncmp(*path, BisCalibrationFilePartitionMountName, strnlen(BisCalibrationFilePartitionMountName, MountNameLengthMax)) == 0)
        {
            *path += strnlen(BisCalibrationFilePartitionMountName, MountNameLengthMax);
            NN_RESULT_DO(OpenBisFileSystem(pOutFileSystem, "", BisPartitionId::CalibrationFile));
        }
        else if (strncmp(*path, BisSafeModePartitionMountName, strnlen(BisSafeModePartitionMountName, MountNameLengthMax)) == 0)
        {
            *path += strnlen(BisSafeModePartitionMountName, MountNameLengthMax);
            NN_RESULT_DO(OpenBisFileSystem(pOutFileSystem, "", BisPartitionId::SafeMode));
        }
        else if (strncmp(*path, BisUserPartitionMountName, strnlen(BisUserPartitionMountName, MountNameLengthMax)) == 0)
        {
            *path += strnlen(BisUserPartitionMountName, MountNameLengthMax);
            NN_RESULT_DO(OpenBisFileSystem(pOutFileSystem, "", BisPartitionId::User));
        }
        else if (strncmp(*path, BisSystemPartitionMountName, strnlen(BisSystemPartitionMountName, MountNameLengthMax)) == 0)
        {
            *path += strnlen(BisSystemPartitionMountName, MountNameLengthMax);
            NN_RESULT_DO(OpenBisFileSystem(pOutFileSystem, "", BisPartitionId::System));
        }
        else if (strncmp(*path, SdCardFileSystemMountName, strnlen(SdCardFileSystemMountName, MountNameLengthMax)) == 0)
        {
            *path += strnlen(SdCardFileSystemMountName, MountNameLengthMax);
            NN_RESULT_DO(OpenSdCardProxyFileSystem(pOutFileSystem));
        }
        else if (strncmp(*path, HostRootFileSystemMountName, strnlen(HostRootFileSystemMountName, MountNameLengthMax)) == 0)
        {
            *path += strnlen(HostRootFileSystemMountName, MountNameLengthMax);
            NN_RESULT_DO(OpenHostFileSystem(pOutFileSystem, ""));
            *isHostFs = true;
        }
        else if (strncmp(*path, RegisteredUpdatePartitionMountName, strnlen(RegisteredUpdatePartitionMountName, MountNameLengthMax)) == 0)
        {
            *path += strnlen(RegisteredUpdatePartitionMountName, MountNameLengthMax);
            NN_RESULT_DO(OpenRegisteredUpdatePartition(pOutFileSystem));
        }
        else
        {
            return ResultPathNotFound();
        }

        if (strnlen(*path, nn::fs::EntryNameLengthMax) == 0)
        {
            *isContinue = false;
        }

        NN_RESULT_SUCCESS;
    }

    Result FileSystemProxyCoreImpl::CheckDirOrNcaOrNsp(const char** path, bool* isDir) NN_NOEXCEPT
    {
        if (strncmp(*path, MountNameDelemeter, strnlen(MountNameDelemeter, fs::EntryNameLengthMax)) == 0)
        {
            *path += 1; // : をスキップ
        }
        else
        {
            return ResultPathNotFound();
        }

        if (*(*path + strnlen(*path, nn::fs::EntryNameLengthMax) - 1) == '/')
        {
            *isDir = true;
            NN_RESULT_SUCCESS;
        }
        else if (strnlen(*path, nn::fs::EntryNameLengthMax) > 4)
        {
            if (strncmp(*path + strnlen(*path, nn::fs::EntryNameLengthMax) - 4, ".nca", 4) == 0)
            {
                *isDir = false;
                NN_RESULT_SUCCESS;
            }
            if (strncmp(*path + strnlen(*path, nn::fs::EntryNameLengthMax) - 4, ".nsp", 4) == 0)
            {
                *isDir = false;
                NN_RESULT_SUCCESS;
            }
        }
        return ResultPathNotFound();
    }

    Result FileSystemProxyCoreImpl::ParseDir(const char** path, std::shared_ptr<nn::fs::fsa::IFileSystem>* pOutFileSystem, std::shared_ptr<nn::fs::fsa::IFileSystem> pBaseFileSystem, nn::fs::detail::FileSystemProxyType type) NN_NOEXCEPT
    {
        std::shared_ptr<nn::fs::fsa::IFileSystem> subDirFs;
        NN_RESULT_DO(m_fsCreatorInterfaces->subDirFsCreator->Create(&subDirFs, std::move(pBaseFileSystem), *path));
        NN_RESULT_DO(ParseContentTypeForDirectory(pOutFileSystem, std::move(subDirFs), type));
        NN_RESULT_SUCCESS;
    }

    Result FileSystemProxyCoreImpl::ParseNsp(const char** path, std::shared_ptr<nn::fs::fsa::IFileSystem>* pOutFileSystem, std::shared_ptr<nn::fs::fsa::IFileSystem> pBaseFileSystem) NN_NOEXCEPT
    {
        const char* tmpPath = *path;
        // ".nsp" を探索
        const char* NspExtension = ".nsp";
        const size_t NspExtensionSize = 4;
        while(NN_STATIC_CONDITION(true))
        {
            // "*.nsp\0" か "*.nsp/" のときのみ探索終了
            if(strncmp(tmpPath, NspExtension, NspExtensionSize) == 0)
            {
                // 拡張子の次の文字で判断する
                if( *(tmpPath + NspExtensionSize) == '\0' || *(tmpPath + NspExtensionSize) == '/')
                {
                    break;
                }
                tmpPath += NspExtensionSize;
            }
            else
            {
                if (*tmpPath == '\0')
                {
                    return ResultPathNotFound();
                }
                tmpPath += 1;
            }
        }
        tmpPath += NspExtensionSize; // xxx.nsp の終端まで移動

        char nspPath[fs::EntryNameLengthMax + 1];
        NN_RESULT_THROW_UNLESS(static_cast<size_t>(tmpPath - *path) <= sizeof(nspPath)
                               , nn::fs::ResultTooLongPath());
        memcpy(nspPath, *path, tmpPath - *path);
        nspPath[tmpPath - *path] = '\0';

        std::shared_ptr<FileStorageBasedFileSystem> fileStorage = fssystem::AllocateShared<FileStorageBasedFileSystem>();
        NN_RESULT_THROW_UNLESS(fileStorage, ResultAllocationMemoryFailedInFileSystemProxyCoreImplD());
        NN_RESULT_DO(fileStorage->Initialize(std::move(pBaseFileSystem), nspPath, nn::fs::OpenMode_Read));

        NN_RESULT_DO(m_fsCreatorInterfaces->partitionFsCreator->Create(pOutFileSystem, std::move(fileStorage)));
        *path = tmpPath;
        NN_RESULT_SUCCESS;
    }

    Result FileSystemProxyCoreImpl::ParseNca(const char** path, std::shared_ptr<nn::fssystem::NcaReader>* pOutNcaReader, std::shared_ptr<nn::fs::fsa::IFileSystem> pBaseFileSystem, nn::Bit64 programId) NN_NOEXCEPT
    {
        std::shared_ptr<FileStorageBasedFileSystem> fileStorage = fssystem::AllocateShared<FileStorageBasedFileSystem>();
        NN_RESULT_THROW_UNLESS(fileStorage, ResultAllocationMemoryFailedInFileSystemProxyCoreImplE());
        NN_RESULT_DO(fileStorage->Initialize(std::move(pBaseFileSystem), *path, nn::fs::OpenMode_Read));

        std::shared_ptr<NcaReader> ncaReader;
        NN_RESULT_DO(m_fsCreatorInterfaces->storageOnNcaCreator->CreateNcaReader(&ncaReader, std::move(fileStorage)));

        if (programId != CheckThroughId)
        {
            auto ncaId = ncaReader->GetProgramId();
            if (ncaId != CheckThroughId)
            {
                if (ncaId != programId)
                {
                    NN_SDK_LOG("[fs] ResultInvalidNcaId was returned. caller: %016llx, target: %016llx\n", programId, ncaId);
                    return ResultInvalidNcaId();
                }
            }
        }

        *pOutNcaReader = std::move(ncaReader);
        NN_RESULT_SUCCESS;
    }

    Result FileSystemProxyCoreImpl::ParseNca(std::shared_ptr<nn::fssystem::NcaReader>* pOutNcaReader, bool* pOutIsGameCard, const char* path, nn::Bit64 programId) NN_NOEXCEPT
    {
        const char* pPath = path;
        bool isContinue = true;
        GameCardHandle gcHandle = 0;
        bool isHostFs = false;

        std::shared_ptr<nn::fs::fsa::IFileSystem> pFileSystem;
        NN_RESULT_DO(ParseMountName(&pPath, &pFileSystem, &isContinue, pOutIsGameCard, &gcHandle, &isHostFs));
        NN_UNUSED(isHostFs);
        if( !isContinue )
        {
            return nn::fs::ResultPreconditionViolation();
        }

        bool isDirectory = false;
        NN_RESULT_DO(CheckDirOrNcaOrNsp(&pPath, &isDirectory));
        if( isDirectory )
        {
            return nn::fs::ResultTargetNotFound();
        }

        std::shared_ptr<nn::fs::fsa::IFileSystem> pNspFileSystem;
        if( ParseNsp(&pPath, &pNspFileSystem, pFileSystem).IsSuccess() )
        {
            if( *pPath == '\0' )
            {
                return nn::fs::ResultTargetNotFound();
            }
            pFileSystem = std::move(pNspFileSystem);
        }

        return ParseNca(&pPath, pOutNcaReader, std::move(pFileSystem), programId);
    }

    Result FileSystemProxyCoreImpl::ParseContentTypeForDirectory(std::shared_ptr<nn::fs::fsa::IFileSystem>* pOutFileSystem, std::shared_ptr<nn::fs::fsa::IFileSystem> pBaseFileSystem, nn::fs::detail::FileSystemProxyType type) NN_NOEXCEPT
    {
        const char DirPathLength = 16;
        char dirPath[DirPathLength];
        std::memset(dirPath, 0, DirPathLength);
        switch (type)
        {
        case FileSystemProxyType::FileSystemProxyType_Code:
            nn::util::SNPrintf(dirPath, DirPathLength, "/code/");
            break;
        case FileSystemProxyType::FileSystemProxyType_Logo:
            nn::util::SNPrintf(dirPath, DirPathLength, "/logo/");
            break;
        case FileSystemProxyType::FileSystemProxyType_Rom:
        case FileSystemProxyType::FileSystemProxyType_Control:
        case FileSystemProxyType::FileSystemProxyType_Manual:
        case FileSystemProxyType::FileSystemProxyType_Meta:
        case FileSystemProxyType::FileSystemProxyType_UpdatePartition:
            nn::util::SNPrintf(dirPath, DirPathLength, "/data/");
            break;
        case FileSystemProxyType::FileSystemProxyType_Package:
            *pOutFileSystem = std::move(pBaseFileSystem);
            NN_RESULT_SUCCESS;
        default:
            return nn::fs::ResultInvalidArgument();
        }
        std::shared_ptr<nn::fs::fsa::IFileSystem> pFileSystem;
        NN_RESULT_DO(m_fsCreatorInterfaces->subDirFsCreator->Create(&pFileSystem, std::move(pBaseFileSystem), dirPath));
        if (type == FileSystemProxyType::FileSystemProxyType_Code)
        {
            NN_RESULT_DO(m_fsCreatorInterfaces->storageOnNcaCreator->VerifyAcid(pFileSystem.get(), nullptr));
        }
        *pOutFileSystem = std::move(pFileSystem);
        NN_RESULT_SUCCESS;
    }

    Result FileSystemProxyCoreImpl::SetExternalKeyForRightsId(nn::fssystem::NcaReader* pNcaReader) NN_NOEXCEPT
    {
        // RightsId ありの場合、紐づく外部鍵をセット
        const fs::RightsId ZeroRightsId = { { 0 } };
        fs::RightsId rightsId;
        pNcaReader->GetRightsId(rightsId.data, sizeof(rightsId.data));
        bool hasRightsId = !nn::crypto::IsSameBytes(&rightsId, &ZeroRightsId, sizeof(fs::RightsId));

        if (hasRightsId)
        {
            nn::spl::AccessKey accessKey;
            NN_RESULT_DO(m_ExternalKeyManager.Find(&accessKey, rightsId));

            pNcaReader->SetExternalDecryptionKey(&accessKey, sizeof(accessKey));
        }

        NN_RESULT_SUCCESS;
    }

    Result FileSystemProxyCoreImpl::OpenStorageByContentType(std::shared_ptr<nn::fs::IStorage>* pOutStorage, std::shared_ptr<nn::fssystem::NcaReader> pNcaReader, nn::fssystem::NcaFsHeader::FsType *outFsType, nn::fs::detail::FileSystemProxyType type, bool isGameCard, bool canReadPrivateDataContent) NN_NOEXCEPT
    {
        auto contentType = pNcaReader->GetContentType();
        switch (type)
        {
        case FileSystemProxyType::FileSystemProxyType_Code:
            NN_RESULT_THROW_UNLESS(contentType == NcaHeader::ContentType::Program, ResultPreconditionViolation()); // 必要に応じて SystemUpdater も
            break;
        case FileSystemProxyType::FileSystemProxyType_Rom:
            NN_RESULT_THROW_UNLESS(contentType == NcaHeader::ContentType::Program, ResultPreconditionViolation());
            break;
        case FileSystemProxyType::FileSystemProxyType_Logo:
            NN_RESULT_THROW_UNLESS(contentType == NcaHeader::ContentType::Program, ResultPreconditionViolation());
            break;
        case FileSystemProxyType::FileSystemProxyType_Control:
            NN_RESULT_THROW_UNLESS(contentType == NcaHeader::ContentType::Control, ResultPreconditionViolation());
            break;
        case FileSystemProxyType::FileSystemProxyType_Manual:
            NN_RESULT_THROW_UNLESS(contentType == NcaHeader::ContentType::Manual, ResultPreconditionViolation());
            break;
        case FileSystemProxyType::FileSystemProxyType_Meta:
            NN_RESULT_THROW_UNLESS(contentType == NcaHeader::ContentType::Meta, ResultPreconditionViolation());
            break;
        case FileSystemProxyType::FileSystemProxyType_Data:
            NN_RESULT_THROW_UNLESS(contentType == NcaHeader::ContentType::Data || contentType == NcaHeader::ContentType::PublicData, ResultPreconditionViolation());
            break;
        case FileSystemProxyType::FileSystemProxyType_UpdatePartition:
            NN_RESULT_THROW_UNLESS(contentType == NcaHeader::ContentType::Program, ResultPreconditionViolation()); // 必要に応じて SystemUpdater に
            break;
        default:
            return ResultInvalidArgument();
        }

        // Public でない Data コンテンツは権限がないと開けない
        if (contentType == NcaHeader::ContentType::Data)
        {
            NN_RESULT_THROW_UNLESS(canReadPrivateDataContent, ResultPermissionDenied());
        }

        // GameCard 向け nca の場合、 GameCard 上の場合のみ開くことを許可する
        NN_RESULT_THROW_UNLESS(pNcaReader->GetDistributionType() != fssystem::NcaHeader::DistributionType::GameCard || isGameCard, ResultPermissionDenied());

        // GameCard 上の場合、SW AES で復号する
        if (isGameCard)
        {
            pNcaReader->PrioritizeSwAes();
        }

        NN_RESULT_DO(SetExternalKeyForRightsId(pNcaReader.get()));

        int partitionIndex;
        NN_RESULT_DO(GetPartitionIndex(&partitionIndex, type));

        fssystem::NcaFsHeaderReader ncaFsHeaderReader;
        bool verifyHeaderSign2 = (type == FileSystemProxyType_Code);
        NN_RESULT_DO(m_fsCreatorInterfaces->storageOnNcaCreator->Create(pOutStorage, &ncaFsHeaderReader, std::move(pNcaReader), partitionIndex, verifyHeaderSign2));
        *outFsType = ncaFsHeaderReader.GetFsType();
        NN_RESULT_SUCCESS;
    }

    Result FileSystemProxyCoreImpl::OpenStorageWithPatchByContentType(std::shared_ptr<nn::fs::IStorage>* pOutStorage, std::shared_ptr<nn::fssystem::NcaReader> pOriginalNcaReader, std::shared_ptr<nn::fssystem::NcaReader> pCurrentNcaReader, nn::fssystem::NcaFsHeader::FsType* outFsType, nn::fs::detail::FileSystemProxyType type) NN_NOEXCEPT
    {
        // 現状では Rom, Manual のみ対応
        // TODO: システムデータ・追加コンテンツに対応する場合、権限チェックも行うこと
        NN_RESULT_THROW_UNLESS(type == FileSystemProxyType_Rom || type == FileSystemProxyType_Manual, ResultPreconditionViolation());

        auto currentContentType = pCurrentNcaReader->GetContentType();
        NN_RESULT_THROW_UNLESS(currentContentType == NcaHeader::ContentType::Program || currentContentType == NcaHeader::ContentType::Manual, ResultPreconditionViolation());
        NN_RESULT_DO(SetExternalKeyForRightsId(pCurrentNcaReader.get()));

        if (pOriginalNcaReader != nullptr)
        {
            auto originalContentType = pOriginalNcaReader->GetContentType();
            NN_RESULT_THROW_UNLESS(originalContentType == currentContentType, ResultPreconditionViolation());
            NN_RESULT_DO(SetExternalKeyForRightsId(pOriginalNcaReader.get()));
        }

        int partitionIndex;
        NN_RESULT_DO(GetPartitionIndex(&partitionIndex, type));

        const bool VerifyHeaderSign2 = false;
        fssystem::NcaFsHeaderReader ncaFsHeaderReader;
        NN_RESULT_DO(m_fsCreatorInterfaces->storageOnNcaCreator->CreateWithPatch(pOutStorage, &ncaFsHeaderReader, std::move(pOriginalNcaReader), std::move(pCurrentNcaReader), partitionIndex, VerifyHeaderSign2));
        *outFsType = ncaFsHeaderReader.GetFsType();
        NN_RESULT_SUCCESS;
    }

    bool FileSystemProxyCoreImpl::IsSaveEmulated(const char* saveDataRootPathBuffer) NN_NOEXCEPT
    {
        if(strncmp(saveDataRootPathBuffer, "\0", 1) != 0)
        {
            return true;
        }
        return false;
    }

    Result FileSystemProxyCoreImpl::OpenSaveDataDirectoryFileSystem(std::shared_ptr<nn::fs::fsa::IFileSystem>* pOutFileSystem, nn::fs::SaveDataSpaceId saveDataSpaceId, const char* saveDataRootPathBuffer, bool isOpen) NN_NOEXCEPT
    {
        const char* SaveDataAreaDirectoryName = nullptr;
        if(isOpen && saveDataSpaceId == SaveDataSpaceId::User && IsSaveEmulated(saveDataRootPathBuffer))
        {
#if defined(NN_BUILD_CONFIG_OS_WIN32)
            NN_RESULT_DO(m_fsCreatorInterfaces->hostFsCreator->Create(pOutFileSystem, saveDataRootPathBuffer));
            NN_RESULT_SUCCESS;
#else
            std::shared_ptr<nn::fs::fsa::IFileSystem> pBaseFileSystem;
            NN_RESULT_DO(m_fsCreatorInterfaces->targetFsCreator->Create(&pBaseFileSystem));
            NN_RESULT_DO(WrapSubDirectory(pOutFileSystem, pBaseFileSystem, saveDataRootPathBuffer));
            NN_RESULT_SUCCESS;
#endif
        }
        else if (saveDataSpaceId == SaveDataSpaceId::Temporary)
        {
            SaveDataAreaDirectoryName = "/temp";
        }
        else
        {
            SaveDataAreaDirectoryName = "/save";
        }
        if(SaveDataAreaDirectoryName == nullptr)
        {
            return ResultUnknownInFileSystemProxyCoreImpl();
        }
        return OpenSaveDataDirectoryFileSystemImpl(pOutFileSystem, saveDataSpaceId, SaveDataAreaDirectoryName);
    }

    Result FileSystemProxyCoreImpl::OpenSaveDataDirectoryFileSystemImpl(std::shared_ptr<nn::fs::fsa::IFileSystem>* pOutFileSystem, nn::fs::SaveDataSpaceId saveDataSpaceId, const char* rootDirectoryPath, bool isDirectoryCreate) NN_NOEXCEPT
    {
        std::shared_ptr<nn::fs::fsa::IFileSystem> pBaseFileSystem;
        // @User / @System を取得
        switch(saveDataSpaceId)
        {
        case SaveDataSpaceId::System:
            NN_RESULT_DO(OpenBisFileSystem(&pBaseFileSystem, "", nn::fs::BisPartitionId::System));
            NN_RESULT_DO(WrapSubDirectory(pOutFileSystem, std::move(pBaseFileSystem), rootDirectoryPath, isDirectoryCreate));
            break;
        case SaveDataSpaceId::User:
        case SaveDataSpaceId::Temporary:
            NN_RESULT_DO(OpenBisFileSystem(&pBaseFileSystem, "", nn::fs::BisPartitionId::User));
            NN_RESULT_DO(WrapSubDirectory(pOutFileSystem, std::move(pBaseFileSystem), rootDirectoryPath, isDirectoryCreate));
            break;
        case SaveDataSpaceId::SdSystem:
        case SaveDataSpaceId::SdUser:
            {
                NN_RESULT_DO(OpenSdCardProxyFileSystem(&pBaseFileSystem));

                char path[EntryNameLengthMax + 1];
                util::SNPrintf(path, sizeof(path), "/%s%s", SdCardNintendoRootDirectoryName, rootDirectoryPath);
                NN_RESULT_DO(WrapSubDirectory(&pBaseFileSystem, std::move(pBaseFileSystem), path, isDirectoryCreate));

                // SD のみ暗号化を被せる
                NN_RESULT_DO(m_fsCreatorInterfaces->encryptedFileSystemCreator->Create(pOutFileSystem, std::move(pBaseFileSystem), fscreator::IEncryptedFileSystemCreator::KeyId::SdCardSystemSave, m_SdCardEncryptionSeed));
                break;
            }
        case SaveDataSpaceId::ProperSystem:
            NN_RESULT_DO(OpenBisFileSystem(&pBaseFileSystem, "", nn::fs::BisPartitionId::SystemProperPartition));
            NN_RESULT_DO(WrapSubDirectory(pOutFileSystem, std::move(pBaseFileSystem), rootDirectoryPath, isDirectoryCreate));
            break;
        case SaveDataSpaceId::SafeMode:
            NN_RESULT_DO(OpenBisFileSystem(&pBaseFileSystem, "", nn::fs::BisPartitionId::SafeMode));
            NN_RESULT_DO(WrapSubDirectory(pOutFileSystem, std::move(pBaseFileSystem), rootDirectoryPath, isDirectoryCreate));
            break;

        default:
            return nn::fs::ResultInvalidArgument();
        }

        NN_RESULT_SUCCESS;
    }

    Result FileSystemProxyCoreImpl::OpenSaveDataDirectoryFileSystemImpl(std::shared_ptr<nn::fs::fsa::IFileSystem>* pOutFileSystem, nn::fs::SaveDataSpaceId saveDataSpaceId, const char* rootDirectoryPath) NN_NOEXCEPT
    {
        return OpenSaveDataDirectoryFileSystemImpl(pOutFileSystem, saveDataSpaceId, rootDirectoryPath, true);
    }

    Result FileSystemProxyCoreImpl::FormatSdCardFileSystem(bool isUncache) NN_NOEXCEPT
    {
        return m_fsCreatorInterfaces->sdProxyFsCreator->Format(isUncache);
    }

    Result FileSystemProxyCoreImpl::FormatSdCardDryRun() NN_NOEXCEPT
    {
#if defined(NN_BUILD_CONFIG_OS_WIN32)
        NN_RESULT_SUCCESS;
#else // defined(NN_BUILD_CONFIG_OS_WIN32)
        std::shared_ptr<nn::fs::fsa::IFileSystem> pFileSystem;

        // sdStorage 維持もしくは再生成の整合を取るための空実行
        Result result = OpenSdCardProxyFileSystem(&pFileSystem);
        if (nn::fs::ResultSdCardAccessFailed::Includes(result))
        {
            return result;
        }

        // 便宜上、g_FatFileSystemCacheManager を使って sdStorage の生成を排他する
        auto scopedLock = g_FatFileSystemCacheManager.GetScopedLock();
        std::shared_ptr<fs::IStorage> sdStorage;
        if (result.IsFailure())
        {
            // FileSystem が sdStorage を存在させていないため、一時的に sdStorage を生成する
            NN_RESULT_DO(m_fsCreatorInterfaces->sdStorageCreator->Create(&sdStorage));
        }

        uint32_t userAreaSectors = 0;
        uint32_t protectedAreaSectors = 0;
        NN_RESULT_DO(nn::fssrv::detail::GetSdCardUserAreaNumSectors(&userAreaSectors));
        NN_RESULT_DO(nn::fssrv::detail::GetSdCardProtectedAreaNumSectors(&protectedAreaSectors));

        NN_RESULT_DO(nn::fat::detail::FormatDryRun(userAreaSectors, protectedAreaSectors));

        NN_RESULT_SUCCESS;
#endif
    }

    bool FileSystemProxyCoreImpl::IsExFatSupported() NN_NOEXCEPT
    {
#if defined(NN_BUILD_CONFIG_OS_WIN32)
        return false;
#else // defined(NN_BUILD_CONFIG_OS_WIN32)
        return nn::fat::FatFileSystem::IsExFatSupported();
#endif
    }

    Result FlushFatCacheImpl() NN_NOEXCEPT
    {
#if defined(NN_BUILD_CONFIG_OS_WIN32)
        NN_RESULT_SUCCESS;
#else // defined(NN_BUILD_CONFIG_OS_WIN32)
        std::shared_ptr<nn::fs::fsa::IFileSystem> pFileSystem;
        auto scopedLock = g_FatFileSystemCacheManager.GetScopedLock();

        FatDriveId fatDriveIds[] = {
            FatDriveId_BisCalibration, FatDriveId_BisSafeMode, FatDriveId_BisUser,
            FatDriveId_BisSystem, FatDriveId_SdCard };
        for (auto fatDriveId : fatDriveIds)
        {
            pFileSystem = g_FatFileSystemCacheManager.GetCache(fatDriveId);
            if (pFileSystem != nullptr)
            {
                auto result = pFileSystem->Flush();
                if (result.IsFailure())
                {
                    NN_SDK_LOG("[fs] Warning: fat filesystem (%d) flush failed.\n", fatDriveId);
                }
            }
        }
        NN_RESULT_SUCCESS;
#endif
    }


    void FileSystemProxyCoreImpl::SetSdCardAccessibility(bool isAccessible) NN_NOEXCEPT
    {
        m_IsSdCardAccessibleFromNs = isAccessible;
    }
    bool FileSystemProxyCoreImpl::IsSdCardAccessible() NN_NOEXCEPT
    {
        return m_IsSdCardAccessibleFromNs;
    }

    void FileSystemProxyCoreImpl::SetSdCardPortReady() NN_NOEXCEPT
    {
        m_fsCreatorInterfaces->bisfsCreator->SetSdCardPortReady();
    }

    bool FileSystemProxyCoreImpl::IsSignedSystemPartitionOnSdCardValid() NN_NOEXCEPT
    {
        // SignedSystemPartition on SD が利用可能か確認する
        std::shared_ptr<nn::fs::fsa::IFileSystem> bisSystem;
        (void)OpenBisFileSystem(&bisSystem, "", nn::fs::BisPartitionId::System);

        return m_fsCreatorInterfaces->bisfsCreator->IsSignedSystemPartitionOnSdCardValid();
    }

    Result GetCurrentPosixTime(int64_t* pOutPosixTime, int32_t* pOutTimeDifference) NN_NOEXCEPT
    {
        std::lock_guard<os::Mutex> scopedLock(g_PosixTimeMutex);
        if (g_CurrentPosixTimeBase == 0)
        {
            return ResultNotInitialized();
        }
        if (pOutPosixTime != nullptr)
        {
            *pOutPosixTime = g_CurrentPosixTimeBase + GetSystemSeconds();
        }
        if (pOutTimeDifference != nullptr)
        {
            *pOutTimeDifference = g_CurrentTimeDifference;
        }
        NN_RESULT_SUCCESS;
    }

    void SetCurrentPosixTime(int64_t posixTime, int32_t timeDifference) NN_NOEXCEPT
    {
        std::lock_guard<os::Mutex> scopedLock(g_PosixTimeMutex);
        g_CurrentPosixTimeBase = posixTime - GetSystemSeconds();
        g_CurrentTimeDifference = timeDifference;
    }

}
    void InitializeSaveDataFileSystemCacheManager(int maxCacheCount) NN_NOEXCEPT
    {
        g_SaveFileSystemCacheManager.Initialize(maxCacheCount);
    }

    // TODO: creator の一種にする
    FatFileSystemCacheManager* GetFatFileSystemCacheManager()
    {
        return &g_FatFileSystemCacheManager;
    }

}}
