﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <memory>
#include <mutex>
#include <nn/fssrv/sf/fssrv_IFileSystem.h>
#include <nn/fssystem/fs_Utility.h>
#include <nn/fs/fs_QueryRange.h>
#include <nn/fs/detail/fs_IFileSystemPointer.h>
#include <nn/os/os_ReaderWriterLock.h>

#include "fssrv_SharedLock.h"
#include <nn/fssrv/detail/fssrv_IFileSystemProxyForAdapter.h>

namespace nn { namespace fs { namespace fsa {
    class IFile;
    class IDirectory;
    class IFileSystem;
}}}


namespace nn { namespace fssrv {
    class FileSystemProxyImpl;

namespace detail {
    class FileSystemInterfaceAdapter;

    class FileInterfaceAdapter
    {
        NN_DISALLOW_COPY(FileInterfaceAdapter);
    public:
        FileInterfaceAdapter(std::unique_ptr<nn::fs::fsa::IFile>&& fileInterface, FileSystemInterfaceAdapter* parentFileSystem, std::unique_lock<fssystem::SemaphoreAdaptor>&& openCountSemaphore) NN_NOEXCEPT;
        ~FileInterfaceAdapter() NN_NOEXCEPT;
        Result Read(nn::sf::Out<std::int64_t> outValue, std::int64_t offset, const nn::sf::OutBuffer& buffer, std::int64_t size, nn::fssrv::sf::ReadOption option) NN_NOEXCEPT;
        Result Write(std::int64_t offset, const nn::sf::InBuffer& buffer, std::int64_t size, nn::fssrv::sf::WriteOption option) NN_NOEXCEPT;
        Result Flush() NN_NOEXCEPT;
        Result SetSize(std::int64_t size) NN_NOEXCEPT;
        Result GetSize(nn::sf::Out<std::int64_t> outValue) NN_NOEXCEPT;
        Result OperateRange(
            nn::sf::Out<nn::fs::QueryRangeInfo> outValue,
            std::int32_t operationId,
            int64_t offset,
            int64_t size) NN_NOEXCEPT;

    private:
        void InvalidateCacheOnStorage() NN_NOEXCEPT;

    private:
        nn::sf::SharedPointer<FileSystemInterfaceAdapter> m_parentFileSystem;
        std::unique_ptr<nn::fs::fsa::IFile> m_fileInterface;
        std::unique_lock<fssystem::SemaphoreAdaptor> m_OpenCountSemaphore;
    };

    class DirectoryInterfaceAdapter
    {
        NN_DISALLOW_COPY(DirectoryInterfaceAdapter);
    public:
        DirectoryInterfaceAdapter(std::unique_ptr<nn::fs::fsa::IDirectory>&& directoryInterface, FileSystemInterfaceAdapter* parentFileSystem, std::unique_lock<fssystem::SemaphoreAdaptor>&& openCountSemaphore) NN_NOEXCEPT;
        ~DirectoryInterfaceAdapter() NN_NOEXCEPT;
        Result Read(nn::sf::Out<std::int64_t> outValue, const nn::sf::OutBuffer& outEntries) NN_NOEXCEPT;
        Result GetEntryCount(nn::sf::Out<std::int64_t> outValue) NN_NOEXCEPT;
    private:
        nn::sf::SharedPointer<FileSystemInterfaceAdapter> m_parentFileSystem;
        std::unique_ptr<nn::fs::fsa::IDirectory> m_directoryInterface;
        std::unique_lock<fssystem::SemaphoreAdaptor> m_OpenCountSemaphore;
    };

    class FileSystemInterfaceAdapter : public nn::sf::ISharedObject
    {
        NN_DISALLOW_COPY(FileSystemInterfaceAdapter);
    public:
        FileSystemInterfaceAdapter(std::shared_ptr<nn::fs::fsa::IFileSystem>&& fileSystemInterface, FileSystemProxyImpl* pFileSystemProxyImpl, bool isOpenCountLimited) NN_NOEXCEPT;
        FileSystemInterfaceAdapter(std::shared_ptr<nn::fs::fsa::IFileSystem>&& fileSystemInterface, FileSystemProxyImpl* pFileSystemProxyImpl, std::unique_lock<fssystem::SemaphoreAdaptor>&& openCountSemaphore, bool isOpenCountLimited) NN_NOEXCEPT;
        FileSystemInterfaceAdapter(std::shared_ptr<nn::fs::fsa::IFileSystem>&& fileSystemInterface, FileSystemProxyImpl* pFileSystemProxyImpl, bool isOpenCountLimited, bool enableDeepRetry) NN_NOEXCEPT;
        FileSystemInterfaceAdapter(std::shared_ptr<nn::fs::fsa::IFileSystem>&& fileSystemInterface, FileSystemProxyImpl* pFileSystemProxyImpl, std::unique_lock<fssystem::SemaphoreAdaptor>&& openCountSemaphore, bool isOpenCountLimited, bool enableDeepRetry) NN_NOEXCEPT;
        Result CreateFile(const nn::fssrv::sf::Path& path, std::int64_t size, std::uint32_t option) NN_NOEXCEPT;
        Result DeleteFile(const nn::fssrv::sf::Path& path) NN_NOEXCEPT;
        Result CreateDirectory(const nn::fssrv::sf::Path& path) NN_NOEXCEPT;
        Result DeleteDirectory(const nn::fssrv::sf::Path& path) NN_NOEXCEPT;
        Result DeleteDirectoryRecursively(const nn::fssrv::sf::Path& path) NN_NOEXCEPT;
        Result CleanDirectoryRecursively(const nn::fssrv::sf::Path& path) NN_NOEXCEPT;
        Result RenameFile(const nn::fssrv::sf::Path& currentPath, const nn::fssrv::sf::Path& newPath) NN_NOEXCEPT;
        Result RenameDirectory(const nn::fssrv::sf::Path& currentPath, const nn::fssrv::sf::Path& newPath) NN_NOEXCEPT;
        Result GetEntryType(nn::sf::Out<std::uint32_t> outValue, const nn::fssrv::sf::Path& path) NN_NOEXCEPT;
        Result GetFreeSpaceSize(nn::sf::Out<std::int64_t> outValue, const nn::fssrv::sf::Path& path) NN_NOEXCEPT;
        Result GetTotalSpaceSize(nn::sf::Out<std::int64_t> outValue, const nn::fssrv::sf::Path& path) NN_NOEXCEPT;
        Result OpenFile(nn::sf::Out<nn::sf::SharedPointer<nn::fssrv::sf::IFile>> outValue, const nn::fssrv::sf::Path& path, std::uint32_t mode) NN_NOEXCEPT;
        Result OpenDirectory(nn::sf::Out<nn::sf::SharedPointer<nn::fssrv::sf::IDirectory>> outValue, const nn::fssrv::sf::Path& path, std::uint32_t mode) NN_NOEXCEPT;
        Result Commit() NN_NOEXCEPT;
        Result GetFileTimeStampRaw(nn::sf::Out<nn::fs::FileTimeStampRaw> outTimeStamp, const nn::fssrv::sf::Path& path) NN_NOEXCEPT;
        Result QueryEntry(const nn::sf::OutBuffer& outBuffer, const nn::sf::InBuffer& inBuffer, int32_t queryId, const nn::fssrv::sf::Path& path) NN_NOEXCEPT;
        Result GetImpl(nn::sf::Out<nn::fs::detail::IFileSystemPointer> outValue) NN_NOEXCEPT;

    public:
        bool IsDeepRetryEnabled() NN_NOEXCEPT;
        util::optional<SharedLock<os::ReaderWriterLock>> AcquireReaderLockForCacheInvalidation() NN_NOEXCEPT;
        os::ReaderWriterLock& GetReaderWriterLockForCacheInvalidation() NN_NOEXCEPT;
        bool IsAccessFailureDetectionObserved() NN_NOEXCEPT;

    protected:
        ~FileSystemInterfaceAdapter() NN_NOEXCEPT;
    private:
        nn::sf::SharedPointer<detail::IFileSystemProxyForAdapter> m_pFileSystemProxyImpl;
        std::shared_ptr<nn::fs::fsa::IFileSystem> m_fileSystemInterface;
        std::unique_lock<fssystem::SemaphoreAdaptor> m_MountCountSemaphore;
        bool m_IsOpenCountLimited;
        nn::os::ReaderWriterLock m_InvalidateCacheLock;
        bool m_DeepRetryEnabled = false;
    };
}}}
