﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <mutex>
#include <nn/os.h>
#include <nn/nn_TimeSpan.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_SdkLog.h>
#include <nn/fs/fs_Result.h>
#include <nn/fs/fs_ResultPrivate.h>
#include <nn/result/result_HandlingUtility.h>

#include "fssrv_DeviceEventSimulator.h"

namespace nn { namespace fssrv { namespace detail {

    IDeviceEventSimulator::IDeviceEventSimulator(uint32_t timeoutMsec) NN_NOEXCEPT : m_Mutex(true), m_TimeoutMsec(timeoutMsec)
    {
        m_IsAccessFailureSimulationEnabled = false;
        m_IsDeviceDetectionSimulationEnabled = false;
        m_DetectionMode = fs::SimulatingDeviceDetectionMode::NoSimulation;
        m_FailureEvent = fs::SimulatingDeviceAccessFailureEventType::None;
        m_TargetOperationFlag = 0;
        m_IsPersistent = false;
        m_RespondingFailureResult = ResultSuccess();
    }

    IDeviceEventSimulator::~IDeviceEventSimulator() NN_NOEXCEPT
    {
    }

    Result IDeviceEventSimulator::GetCoresspondingResult(fs::SimulatingDeviceAccessFailureEventType eventType) NN_NOEXCEPT
    {
        NN_UNUSED(eventType);
        return ResultSuccess();
    }

    void IDeviceEventSimulator::SetDeviceEvent(uint32_t targetOperationFlag, fs::SimulatingDeviceAccessFailureEventType eventType, Result respondingFailureResult, bool isPersistent) NN_NOEXCEPT
    {
        std::lock_guard<nn::os::Mutex> scopedLock(m_Mutex);

        m_IsAccessFailureSimulationEnabled = true;

        if (respondingFailureResult.IsFailure())
        {
            m_RespondingFailureResult = respondingFailureResult;
        }

        m_TargetOperationFlag = targetOperationFlag;
        m_FailureEvent = eventType;
        m_IsPersistent = isPersistent;
    }

    void IDeviceEventSimulator::ClearDeviceEvent() NN_NOEXCEPT
    {
        std::lock_guard<nn::os::Mutex> scopedLock(m_Mutex);

        m_IsAccessFailureSimulationEnabled = false;

        m_FailureEvent = fs::SimulatingDeviceAccessFailureEventType::None;
        m_TargetOperationFlag = 0;
        m_IsPersistent = false;
        m_RespondingFailureResult = ResultSuccess();
    }

    void IDeviceEventSimulator::SetDetectionSimulationMode(fs::SimulatingDeviceDetectionMode simulatingDetectionMode) NN_NOEXCEPT
    {
        std::lock_guard<nn::os::Mutex> scopedLock(m_Mutex);

        m_IsDeviceDetectionSimulationEnabled = (simulatingDetectionMode != fs::SimulatingDeviceDetectionMode::NoSimulation);

        m_DetectionMode = simulatingDetectionMode;
    }

    void IDeviceEventSimulator::ClearDetectionSimulationMode() NN_NOEXCEPT
    {
        SetDetectionSimulationMode(fs::SimulatingDeviceDetectionMode::NoSimulation);
    }

    Result IDeviceEventSimulator::CheckSimulatedAccessFailureEvent(fs::SimulatingDeviceTargetOperation targetOperation) NN_NOEXCEPT
    {
        // CheckSimulatedDeviceEvent は通常フローで頻繁に呼ばれる事を想定する為、シミュレーションを使用しない場合に Mutex 確保のオーバヘッドが生じないよう努力する
        if (! m_IsAccessFailureSimulationEnabled)
        {
            return ResultSuccess();
        }

        std::lock_guard<nn::os::Mutex> scopedLock(m_Mutex);

        if ((m_TargetOperationFlag & targetOperation) == 0)
        {
            return ResultSuccess();
        }

        Result response = GetCoresspondingResult(m_FailureEvent);

        // 返す Result の指定があればそちらを優先して上書き
        if (response.IsFailure() && m_RespondingFailureResult.IsFailure())
        {
            response = m_RespondingFailureResult;
        }

        if (m_FailureEvent == fs::SimulatingDeviceAccessFailureEventType::AccessTimeoutFailure)
        {
            SimulateTimeout();
        }

        if (! m_IsPersistent)
        {
            ClearDeviceEvent();
        }

        return response;
    }

    bool IDeviceEventSimulator::FilterDetectionState(bool originalDetectionState) NN_NOEXCEPT
    {
        // FilterDetectionState は通常フローで頻繁に呼ばれる事を想定する為、シミュレーションを使用しない場合に Mutex 確保のオーバヘッドが生じないよう努力する
        if (! m_IsDeviceDetectionSimulationEnabled)
        {
            return originalDetectionState;
        }

        std::lock_guard<nn::os::Mutex> scopedLock(m_Mutex);

        switch (m_DetectionMode)
        {
        case fs::SimulatingDeviceDetectionMode::NoSimulation:
            return originalDetectionState;
        case fs::SimulatingDeviceDetectionMode::DeviceAttached:
            return true;
        case fs::SimulatingDeviceDetectionMode::DeviceRemoved:
            return false;
        default:
            break;
        }

        return originalDetectionState;
    }

    void IDeviceEventSimulator::SimulateTimeout() NN_NOEXCEPT
    {
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(m_TimeoutMsec));
    }


    // *** GameCardEventSimulator
    GameCardEventSimulator::GameCardEventSimulator() : IDeviceEventSimulator(DefaultTimeoutMsec)
    {
    }

    GameCardEventSimulator::~GameCardEventSimulator() NN_NOEXCEPT
    {
    }

    Result GameCardEventSimulator::GetCoresspondingResult(fs::SimulatingDeviceAccessFailureEventType eventType) NN_NOEXCEPT
    {
        switch (eventType)
        {
        case fs::SimulatingDeviceAccessFailureEventType::None:
            return ResultSuccess();
        case fs::SimulatingDeviceAccessFailureEventType::AccessFailure:
            return fs::ResultGameCardAccessFailed();
        case fs::SimulatingDeviceAccessFailureEventType::AccessTimeoutFailure:
            return fs::ResultGameCardCardAccessTimeout();
        case fs::SimulatingDeviceAccessFailureEventType::DataCorruption:
            return fs::ResultDataCorrupted();
        default:
            break;
        }

        return fs::ResultInvalidArgument();
    }

    // *** インスタンスの取得
    GameCardEventSimulator& GetGameCardEventSimulator() NN_NOEXCEPT
    {
        NN_FUNCTION_LOCAL_STATIC(GameCardEventSimulator, g_GameCardEventSimulator, );
        return g_GameCardEventSimulator;
    }

}}}
