﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/dd.h>

namespace nn { namespace fssrv { namespace detail {

    static const auto DeviceBufferSize = 8 * 1024 * 1024; //!< デバイス用バッファのサイズです

    /*!
    * @brief       デバイスへのアクセスに使用するバッファを初期化します。
    */
    void InitializeDeviceBuffer() NN_NOEXCEPT;

    /*!
    * @brief       デバイスへのアクセスに使用するバッファの制御を終了します。
    *
    * @pre
    *              - AttachDeviceBufferToGameCardDriver() が未実行か DetachDeviceBufferFromGameCardDriver() が実行済み。
    */
    void FinalizeDeviceBuffer() NN_NOEXCEPT;

    /*!
    * @brief       デバイスアクセス用バッファをゲームカードドライバが使用できる状態にします。
    *
    * @pre
    *              - デバイスアクセス用バッファをゲームカードドライバが使用できる状態でない。
    */
    void AttachDeviceBufferToGameCardDriver() NN_NOEXCEPT;

    /*!
    * @brief       デバイスアクセス用バッファをゲームカードドライバが使用できない状態に戻します。
    *
    * @pre
    *              - デバイスアクセス用バッファをゲームカードドライバが使用できる状態である。
    */
    void DetachDeviceBufferFromGameCardDriver() NN_NOEXCEPT;

    /*!
    * @brief       デバイスへのアクセスに使用するバッファを取得します。
    *
    * @return      デバイスへのアクセスに使用するバッファを返します。
    *
    * @pre
    *              - 初期化済み
    */
    uintptr_t GetDeviceBuffer() NN_NOEXCEPT;

    /*!
    * @brief       デバイスへのアクセスに使用するバッファに対応する仮想アドレスを取得します。
    *
    * @return      デバイスへのアクセスに使用するバッファに対応する仮想アドレスを返します。
    *
    * @pre
    *              - 初期化済み
    */
    nn::dd::DeviceVirtualAddress GetDeviceVirtualAddress() NN_NOEXCEPT;

    /*!
    * @brief       デバイスへのアクセスに使用するバッファを設定します。
    *
    * @param[in]  processAddress  設定するバッファの先頭アドレス
    * @param[in]  size 設定するバッファのサイズ
    *
    * @return      デバイスへのアクセスに使用するバッファの DeviceVirtualAddress
    *
    * @pre
    *              - 初期化済み
    *              - バッファのサイズが 4KB の倍数
    */
    nn::dd::DeviceVirtualAddress MapAdditionalDeviceAddress(uint64_t processAddress, size_t size) NN_NOEXCEPT;

    /*!
    * @brief       MapAdditionalDeviceAddress で設定したバッファを外します。
    *
    * @pre
    *              - 初期化済み
    *              - MapAdditionalDeviceAddress 済み
    */
    void UnmapAdditionalDeviceAddress() NN_NOEXCEPT;

}}}
