﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Abort.h>
#include <nn/os.h>
#include <nn/result/result_HandlingUtility.h>
#include "fssrv_DeviceBuffer.h"

#include <mutex>
#include <nn/os/os_Mutex.h>

namespace nn { namespace fssrv { namespace detail {

    namespace {
#if defined(NN_BUILD_CONFIG_HARDWARE_JETSONTK1) || defined(NN_BUILD_CONFIG_HARDWARE_JETSONTK2) || defined(NN_BUILD_CONFIG_HARDWARE_NX) || defined(NN_BUILD_CONFIG_HARDWARE_JETSONTX2)
#if (defined(NN_DETAIL_FS_SMMU_FOR_SDMMC_ENABLE))
        nn::dd::DeviceAddressSpaceType g_DeviceAddressSpace;
#endif
#endif
        uintptr_t g_DeviceBuffer = 0;

#if defined(NN_BUILD_CONFIG_HARDWARE_JETSONTK1) || defined(NN_BUILD_CONFIG_HARDWARE_JETSONTK2) || defined(NN_BUILD_CONFIG_HARDWARE_NX) || defined(NN_BUILD_CONFIG_HARDWARE_JETSONTX2)
#if (defined(NN_DETAIL_FS_SMMU_FOR_SDMMC_ENABLE))
            const auto MmcDeviceName = nn::dd::DeviceName::DeviceName_Sdmmc4a;
            const auto GcDeviceName = nn::dd::DeviceName::DeviceName_Sdmmc2a;
#if (defined(NN_BUILD_CONFIG_HARDWARE_JETSONTK1))
            const auto SdCardDeviceName = nn::dd::DeviceName::DeviceName_Sdmmc3a;
#else // NN_BUILD_CONFIG_HARDWARE_JETSONTK2
              // NN_BUILD_CONFIG_HARDWARE_NX
            const auto SdCardDeviceName = nn::dd::DeviceName::DeviceName_Sdmmc1a;
#endif
#endif
#endif

        auto g_IsInitialized = false;
        nn::os::Mutex g_AdditionalMapMutex(false);
        uint64_t g_AdditionalMapAddress = 0;
        size_t   g_AdditionalMapSize = 0;

    }

    void InitializeDeviceBuffer() NN_NOEXCEPT
    {
        // この関数でエラーが発生した場合は ABORT します。(失敗したらライブラリ自体が機能しなくなるため)

        if( !g_IsInitialized )
        {
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::os::AllocateMemoryBlock(
                &g_DeviceBuffer,
                DeviceBufferSize));

#if defined(NN_BUILD_CONFIG_HARDWARE_JETSONTK1) || defined(NN_BUILD_CONFIG_HARDWARE_JETSONTK2) || defined(NN_BUILD_CONFIG_HARDWARE_NX) || defined(NN_BUILD_CONFIG_HARDWARE_JETSONTX2)
#if (defined(NN_DETAIL_FS_SMMU_FOR_SDMMC_ENABLE))
            static const auto DeviceAddressSpaceSize = 0x100000000;

            NN_ABORT_UNLESS_RESULT_SUCCESS(dd::CreateDeviceAddressSpace(
                &g_DeviceAddressSpace,
                DeviceAddressSpaceSize));

            NN_ABORT_UNLESS_RESULT_SUCCESS(dd::AttachDeviceAddressSpace(
                &g_DeviceAddressSpace,
                MmcDeviceName));
            NN_ABORT_UNLESS_RESULT_SUCCESS(dd::AttachDeviceAddressSpace(
                &g_DeviceAddressSpace,
                SdCardDeviceName));

            const auto currentProcessHandle = nn::dd::GetCurrentProcessHandle();
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::dd::MapDeviceAddressSpaceAligned(
                &g_DeviceAddressSpace,
                currentProcessHandle,
                static_cast<uint64_t>(g_DeviceBuffer),
                DeviceBufferSize,
                GetDeviceVirtualAddress(),
                nn::dd::MemoryPermission_ReadWrite));
#endif
#endif
            g_IsInitialized = true;
        }
    }

    // TODO: 実行する際には DetachDeviceBufferFromGameCardDriver() のケアが必要
    void FinalizeDeviceBuffer() NN_NOEXCEPT
    {
        if( g_IsInitialized )
        {
#if defined(NN_BUILD_CONFIG_HARDWARE_JETSONTK1) || defined(NN_BUILD_CONFIG_HARDWARE_JETSONTK2) || defined(NN_BUILD_CONFIG_HARDWARE_NX) || defined(NN_BUILD_CONFIG_HARDWARE_JETSONTX2)
#if (defined(NN_DETAIL_FS_SMMU_FOR_SDMMC_ENABLE))
            const auto currentProcessHandle = nn::dd::GetCurrentProcessHandle();
            nn::dd::UnmapDeviceAddressSpace(&g_DeviceAddressSpace,
                currentProcessHandle,
                static_cast<uint64_t>(g_DeviceBuffer),
                DeviceBufferSize,
                GetDeviceVirtualAddress());

            nn::dd::DetachDeviceAddressSpace(&g_DeviceAddressSpace, SdCardDeviceName);
            nn::dd::DetachDeviceAddressSpace(&g_DeviceAddressSpace, MmcDeviceName);

            nn::dd::DestroyDeviceAddressSpace(&g_DeviceAddressSpace);
#endif
#endif
            nn::os::FreeMemoryBlock(g_DeviceBuffer, DeviceBufferSize);
            g_DeviceBuffer = 0;

            g_IsInitialized = false;
        }
    }

    void AttachDeviceBufferToGameCardDriver() NN_NOEXCEPT
    {
        // この関数でエラーが発生した場合は ABORT します。(失敗したらライブラリ自体が機能しなくなるため)

#if defined(NN_BUILD_CONFIG_HARDWARE_JETSONTK1) || defined(NN_BUILD_CONFIG_HARDWARE_JETSONTK2) || defined(NN_BUILD_CONFIG_HARDWARE_NX) || defined(NN_BUILD_CONFIG_HARDWARE_JETSONTX2)
#if (defined(NN_DETAIL_FS_SMMU_FOR_SDMMC_ENABLE))
        NN_ABORT_UNLESS_RESULT_SUCCESS(dd::AttachDeviceAddressSpace(
            &g_DeviceAddressSpace,
            GcDeviceName));
#endif
#endif
    }

    void DetachDeviceBufferFromGameCardDriver() NN_NOEXCEPT
    {
#if defined(NN_BUILD_CONFIG_HARDWARE_JETSONTK1) || defined(NN_BUILD_CONFIG_HARDWARE_JETSONTK2) || defined(NN_BUILD_CONFIG_HARDWARE_NX) || defined(NN_BUILD_CONFIG_HARDWARE_JETSONTX2)
#if (defined(NN_DETAIL_FS_SMMU_FOR_SDMMC_ENABLE))
        nn::dd::DetachDeviceAddressSpace(&g_DeviceAddressSpace, GcDeviceName);
#endif
#endif
    }

    uintptr_t GetDeviceBuffer() NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(g_IsInitialized);
        return g_DeviceBuffer;
    }

    nn::dd::DeviceVirtualAddress GetDeviceVirtualAddress() NN_NOEXCEPT
    {
        return (g_DeviceBuffer % (1 << 22)) + (1 << 22);
    }

    nn::dd::DeviceVirtualAddress GetAdditionalDeviceVirtualAddress(uint64_t processAddress) NN_NOEXCEPT
    {
        nn::dd::DeviceVirtualAddress offset = GetDeviceVirtualAddress() + DeviceBufferSize;
        return (processAddress % (1 << 22)) + nn::util::align_up(offset, (1 << 22));
    }


    nn::dd::DeviceVirtualAddress MapAdditionalDeviceAddress(uint64_t processAddress, size_t size) NN_NOEXCEPT
    {
        std::lock_guard<nn::os::Mutex> scopedLock(g_AdditionalMapMutex);
        auto deviceAddressSpace = GetAdditionalDeviceVirtualAddress(processAddress);
        NN_ABORT_UNLESS((g_AdditionalMapAddress == 0) && (g_AdditionalMapSize == 0));

#if defined(NN_BUILD_CONFIG_HARDWARE_JETSONTK1) || defined(NN_BUILD_CONFIG_HARDWARE_JETSONTK2) || defined(NN_BUILD_CONFIG_HARDWARE_NX) || defined(NN_BUILD_CONFIG_HARDWARE_JETSONTX2)
#if (defined(NN_DETAIL_FS_SMMU_FOR_SDMMC_ENABLE))
        const auto currentProcessHandle = nn::dd::GetCurrentProcessHandle();
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::dd::MapDeviceAddressSpaceAligned(
            &g_DeviceAddressSpace,
            currentProcessHandle,
            processAddress,
            size,
            deviceAddressSpace,
            nn::dd::MemoryPermission_ReadWrite));
#endif
#endif
        NN_UNUSED(size);
        g_AdditionalMapAddress = processAddress;
        g_AdditionalMapSize    = size;

        return deviceAddressSpace;
    }

    void UnmapAdditionalDeviceAddress() NN_NOEXCEPT
    {
        std::lock_guard<nn::os::Mutex> scopedLock(g_AdditionalMapMutex);
        if(g_AdditionalMapAddress == 0 || g_AdditionalMapSize == 0)
        {
            // DO NOTHING;
            return;
        }
#if defined(NN_BUILD_CONFIG_HARDWARE_JETSONTK1) || defined(NN_BUILD_CONFIG_HARDWARE_JETSONTK2) || defined(NN_BUILD_CONFIG_HARDWARE_NX) || defined(NN_BUILD_CONFIG_HARDWARE_JETSONTX2)
#if (defined(NN_DETAIL_FS_SMMU_FOR_SDMMC_ENABLE))
        auto deviceAddressSpace = GetAdditionalDeviceVirtualAddress(g_AdditionalMapAddress);
        const auto currentProcessHandle = nn::dd::GetCurrentProcessHandle();
        nn::dd::UnmapDeviceAddressSpace(
            &g_DeviceAddressSpace,
            currentProcessHandle,
            g_AdditionalMapAddress,
            g_AdditionalMapSize,
            deviceAddressSpace);
#endif
#endif
        g_AdditionalMapAddress = 0;
        g_AdditionalMapSize    = 0;
    }

}}}
