﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_SdkAssert.h>
#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/fs/fs_Result.h>
#include <nn/fs/detail/fs_CommonMountName.h>
#include <nn/fs/fsa/fs_IFileSystem.h>
#include <nn/fs/fsa/fs_Registrar.h>
#include <nn/fssrv/sf/fssrv_IFileSystemProxy.h>
#include <nn/fssrv/sf/fssrv_IDeviceOperator.h>

#include <nn/fs/fs_SdCardForDebug.h>
#include <nn/fs/fs_SdCardPrivate.h>
#include <nn/fs/fs_ErrorInfoPrivate.h>
#include <nn/fs/detail/fs_AccessLog.h>

#include "fs_Library.h"
#include "fs_EventNotifierObjectAdapter.h"
#include "fs_FileSystemProxyServiceObject.h"
#include "fs_FileSystemServiceObjectAdapter.h"
#include "fsa/fs_MountUtility.h"
#include <nn/fs/detail/fs_ResultHandlingUtility.h>

namespace nn { namespace fs {

Result MountSdCard(const char* name) NN_NOEXCEPT
{
    // マウント名チェック
    NN_FS_RESULT_DO(NN_DETAIL_FS_ACCESS_LOG_MOUNT_FOR_PRECONDITION(detail::CheckMountName(name),
        name, NN_DETAIL_FS_ACCESS_LOG_FORMAT_MOUNT, name));

    nn::sf::SharedPointer<nn::fssrv::sf::IFileSystemProxy> fileSystemProxy = detail::GetFileSystemProxyServiceObject();
    nn::sf::SharedPointer<nn::fssrv::sf::IFileSystem> fileSystem;
    // SD カード抜けをハンドリングするためにここは NN_RESULT_DO を使います。
    NN_RESULT_DO(NN_DETAIL_FS_ACCESS_LOG_MOUNT(fileSystemProxy->OpenSdCardFileSystem(&fileSystem) \
        , name, NN_DETAIL_FS_ACCESS_LOG_FORMAT_MOUNT, name));
    std::unique_ptr<fsa::IFileSystem> fileSystemAbstract;
    fileSystemAbstract.reset(new detail::FileSystemServiceObjectAdapter(std::move(fileSystem)));
    NN_FS_RESULT_THROW_UNLESS (fileSystemAbstract, ResultAllocationMemoryFailedInSdCardA());

    NN_FS_RESULT_DO(fsa::Register(name, std::move(fileSystemAbstract)));
    NN_DETAIL_FS_ACCESS_LOG_FSACCESSOR_ENABLE(name);
    NN_RESULT_SUCCESS;
}

Result MountSdCardForDebug(const char* name) NN_NOEXCEPT
{
    return MountSdCard(name);
}

Result OpenSdCardDetectionEventNotifier(std::unique_ptr<IEventNotifier>* outValue) NN_NOEXCEPT
{
    nn::sf::SharedPointer<nn::fssrv::sf::IFileSystemProxy> fileSystemProxy = detail::GetFileSystemProxyServiceObject();
    nn::sf::SharedPointer<nn::fssrv::sf::IEventNotifier> eventNotifier;
    NN_FS_RESULT_DO(fileSystemProxy->OpenSdCardDetectionEventNotifier(nn::sf::Out<nn::sf::SharedPointer<nn::fssrv::sf::IEventNotifier>>(&eventNotifier)));
    outValue->reset(new detail::EventNotifierObjectAdapter(std::move(eventNotifier)));
    NN_FS_RESULT_THROW_UNLESS (*outValue, ResultAllocationMemoryFailedInSdCardB());
    NN_RESULT_SUCCESS;
}

bool IsSdCardInserted() NN_NOEXCEPT
{
    nn::sf::SharedPointer<nn::fssrv::sf::IFileSystemProxy> fileSystemProxy = detail::GetFileSystemProxyServiceObject();
    nn::sf::SharedPointer<nn::fssrv::sf::IDeviceOperator> deviceOperator;
    bool isInserted = false;
    NN_FS_ABORT_UNLESS_RESULT_SUCCESS(fileSystemProxy->OpenDeviceOperator(nn::sf::Out<nn::sf::SharedPointer<nn::fssrv::sf::IDeviceOperator>>(&deviceOperator)));
    NN_FS_ABORT_UNLESS_RESULT_SUCCESS(deviceOperator->IsSdCardInserted(nn::sf::Out<bool>(&isInserted)));
    return isInserted;
}

Result GetSdCardSpeedMode(SdCardSpeedMode* pOutValue) NN_NOEXCEPT
{
    NN_FS_RESULT_THROW_UNLESS(pOutValue != nullptr, ResultNullptrArgument());

    nn::sf::SharedPointer<nn::fssrv::sf::IFileSystemProxy> fileSystemProxy = detail::GetFileSystemProxyServiceObject();
    nn::sf::SharedPointer<nn::fssrv::sf::IDeviceOperator> deviceOperator;
    NN_FS_RESULT_DO(fileSystemProxy->OpenDeviceOperator(nn::sf::Out<nn::sf::SharedPointer<nn::fssrv::sf::IDeviceOperator>>(&deviceOperator)));
    int64_t speedMode = 0;
    NN_FS_RESULT_DO(deviceOperator->GetSdCardSpeedMode(nn::sf::Out<int64_t>(&speedMode)));
    *pOutValue = static_cast<SdCardSpeedMode>(speedMode);
    NN_RESULT_SUCCESS;
}

Result GetSdCardCid(void* outBuffer, size_t size) NN_NOEXCEPT
{
    NN_FS_RESULT_THROW_UNLESS(outBuffer != nullptr, ResultNullptrArgument());

    nn::sf::SharedPointer<nn::fssrv::sf::IFileSystemProxy> fileSystemProxy = detail::GetFileSystemProxyServiceObject();
    nn::sf::SharedPointer<nn::fssrv::sf::IDeviceOperator> deviceOperator;
    NN_FS_RESULT_DO(fileSystemProxy->OpenDeviceOperator(nn::sf::Out<nn::sf::SharedPointer<nn::fssrv::sf::IDeviceOperator>>(&deviceOperator)));
    NN_FS_RESULT_DO(deviceOperator->GetSdCardCid(nn::sf::OutBuffer(reinterpret_cast<char*>(outBuffer), size), static_cast<int64_t>(size)));
    NN_RESULT_SUCCESS;
}

Result GetSdCardUserAreaSize(int64_t* pOutValue) NN_NOEXCEPT
{
    NN_FS_RESULT_THROW_UNLESS(pOutValue != nullptr, ResultNullptrArgument());

    nn::sf::SharedPointer<nn::fssrv::sf::IFileSystemProxy> fileSystemProxy = detail::GetFileSystemProxyServiceObject();
    nn::sf::SharedPointer<nn::fssrv::sf::IDeviceOperator> deviceOperator;
    NN_FS_RESULT_DO(fileSystemProxy->OpenDeviceOperator(nn::sf::Out<nn::sf::SharedPointer<nn::fssrv::sf::IDeviceOperator>>(&deviceOperator)));
    NN_FS_RESULT_DO(deviceOperator->GetSdCardUserAreaSize(pOutValue));
    NN_RESULT_SUCCESS;
}

Result GetSdCardProtectedAreaSize(int64_t* pOutValue) NN_NOEXCEPT
{
    NN_FS_RESULT_THROW_UNLESS(pOutValue != nullptr, ResultNullptrArgument());

    nn::sf::SharedPointer<nn::fssrv::sf::IFileSystemProxy> fileSystemProxy = detail::GetFileSystemProxyServiceObject();
    nn::sf::SharedPointer<nn::fssrv::sf::IDeviceOperator> deviceOperator;
    NN_FS_RESULT_DO(fileSystemProxy->OpenDeviceOperator(nn::sf::Out<nn::sf::SharedPointer<nn::fssrv::sf::IDeviceOperator>>(&deviceOperator)));
    NN_FS_RESULT_DO(deviceOperator->GetSdCardProtectedAreaSize(pOutValue));
    NN_RESULT_SUCCESS;
}

Result GetAndClearSdCardErrorInfo(StorageErrorInfo* pOutStorageErrorInfo, size_t* pOutLogSize, char* pOutLogBuffer, size_t logBufferSize) NN_NOEXCEPT
{
    NN_FS_RESULT_THROW_UNLESS(pOutStorageErrorInfo != nullptr, ResultNullptrArgument());
    NN_FS_RESULT_THROW_UNLESS(pOutLogSize != nullptr, ResultNullptrArgument());
    NN_FS_RESULT_THROW_UNLESS(pOutLogBuffer != nullptr, ResultNullptrArgument());

    nn::sf::SharedPointer<nn::fssrv::sf::IFileSystemProxy> fileSystemProxy = detail::GetFileSystemProxyServiceObject();
    nn::sf::SharedPointer<nn::fssrv::sf::IDeviceOperator> deviceOperator;
    NN_FS_RESULT_DO(fileSystemProxy->OpenDeviceOperator(nn::sf::Out<nn::sf::SharedPointer<nn::fssrv::sf::IDeviceOperator>>(&deviceOperator)));
    int64_t logSize;
    NN_FS_RESULT_DO(deviceOperator->GetAndClearSdCardErrorInfo(nn::sf::Out<StorageErrorInfo>(pOutStorageErrorInfo),
        nn::sf::Out<int64_t>(&logSize), nn::sf::OutBuffer(pOutLogBuffer, logBufferSize), static_cast<int64_t>(logBufferSize)));
    *pOutLogSize = static_cast<size_t>(logSize);
    NN_RESULT_SUCCESS;
}

Result FormatSdCard() NN_NOEXCEPT
{
    nn::sf::SharedPointer<nn::fssrv::sf::IFileSystemProxy> fileSystemProxy = detail::GetFileSystemProxyServiceObject();
    NN_FS_RESULT_DO(fileSystemProxy->FormatSdCardFileSystem());
    NN_RESULT_SUCCESS;
}

Result FormatSdCardDryRun() NN_NOEXCEPT
{
    nn::sf::SharedPointer<nn::fssrv::sf::IFileSystemProxy> fileSystemProxy = detail::GetFileSystemProxyServiceObject();
    NN_FS_RESULT_DO(fileSystemProxy->FormatSdCardDryRun());
    NN_RESULT_SUCCESS;
}

bool IsExFatSupported() NN_NOEXCEPT
{
    nn::sf::SharedPointer<nn::fssrv::sf::IFileSystemProxy> fileSystemProxy = detail::GetFileSystemProxyServiceObject();
    bool isSupported = false;
    NN_FS_ABORT_UNLESS_RESULT_SUCCESS(fileSystemProxy->IsExFatSupported(nn::sf::Out<bool>(&isSupported)));
    return isSupported;
}

Result SetSdCardEncryptionSeed(const EncryptionSeed& seed) NN_NOEXCEPT
{
    nn::sf::SharedPointer<nn::fssrv::sf::IFileSystemProxy> fileSystemProxy = detail::GetFileSystemProxyServiceObject();
    NN_FS_RESULT_DO(fileSystemProxy->SetSdCardEncryptionSeed(seed));
    NN_RESULT_SUCCESS;
}

void SetSdCardAccessibility(bool isAccessible) NN_NOEXCEPT
{
    NN_FS_ABORT_UNLESS_RESULT_SUCCESS(detail::SetSdCardAccessibility(isAccessible));
    return;
}

bool IsSdCardAccessible() NN_NOEXCEPT
{
    nn::sf::SharedPointer<nn::fssrv::sf::IFileSystemProxy> fileSystemProxy = detail::GetFileSystemProxyServiceObject();
    bool isAccessible = false;
    NN_FS_ABORT_UNLESS_RESULT_SUCCESS(fileSystemProxy->IsSdCardAccessible(&isAccessible));
    return isAccessible;
}

namespace detail {

Result SetSdCardAccessibility(bool isAccessible) NN_NOEXCEPT
{
    nn::sf::SharedPointer<nn::fssrv::sf::IFileSystemProxy> fileSystemProxy = detail::GetFileSystemProxyServiceObject();
    NN_FS_RESULT_DO(fileSystemProxy->SetSdCardAccessibility(isAccessible));
    NN_RESULT_SUCCESS;
}

}

}}
