﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/nn_Abort.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_Result.h>
#include <nn/fs/detail/fs_AccessLog.h>
#include <nn/fs/detail/fs_CommonMountName.h>
#include <nn/fs/detail/fs_ResultHandlingUtility.h>
#include <nn/fs/fs_RegisteredUpdatePartition.h>
#include <nn/fs/fsa/fs_IFileSystem.h>
#include <nn/fs/fsa/fs_Registrar.h>
#include <nn/fssrv/sf/fssrv_IFileSystemProxy.h>
#include <nn/util/util_FormatString.h>

#include "fs_Library.h"
#include "fs_FileSystemProxyServiceObject.h"
#include "fs_FileSystemServiceObjectAdapter.h"
#include "fsa/fs_MountUtility.h"

namespace nn { namespace fs {

namespace
{
    class RegisteredUpdatePartitionCommonMountNameGenerator : public fsa::ICommonMountNameGenerator, public detail::Newable
    {
    public:
        virtual Result GenerateCommonMountName(char* name, size_t nameSize) NN_NOEXCEPT NN_OVERRIDE
        {
            const size_t RequiredNameBufferSize = strnlen(detail::RegisteredUpdatePartitionMountName, MountNameLengthMax) + 1 + 1;
            NN_SDK_REQUIRES(nameSize >= static_cast<size_t>(RequiredNameBufferSize));
            NN_UNUSED(RequiredNameBufferSize);
            auto size = nn::util::SNPrintf(name, nameSize, "%s:", detail::RegisteredUpdatePartitionMountName);
            NN_SDK_ASSERT(static_cast<size_t>(size) == RequiredNameBufferSize - 1);
            NN_UNUSED(size);

            NN_RESULT_SUCCESS;
        }
    };
}

Result RegisterUpdatePartition() NN_NOEXCEPT
{
    nn::sf::SharedPointer<nn::fssrv::sf::IFileSystemProxy> fileSystemProxy = detail::GetFileSystemProxyServiceObject();
    NN_FS_RESULT_DO(fileSystemProxy->RegisterUpdatePartition());
    NN_RESULT_SUCCESS;
}

Result MountRegisteredUpdatePartition(const char* name) NN_NOEXCEPT
{
    auto mount = [=]() NN_NOEXCEPT -> Result
    {
        // マウント名チェック
        NN_RESULT_DO(detail::CheckMountNameAcceptingReservedMountName(name));

        nn::sf::SharedPointer<nn::fssrv::sf::IFileSystemProxy> fileSystemProxy = detail::GetFileSystemProxyServiceObject();
        nn::sf::SharedPointer<nn::fssrv::sf::IFileSystem> fileSystem;

        NN_RESULT_DO(fileSystemProxy->OpenRegisteredUpdatePartition(nn::sf::Out<nn::sf::SharedPointer<nn::fssrv::sf::IFileSystem>>(&fileSystem)));

        std::unique_ptr<nn::fs::fsa::IFileSystem> fileSystemAbstract;
        fileSystemAbstract.reset(new detail::FileSystemServiceObjectAdapter(std::move(fileSystem)));
        NN_RESULT_THROW_UNLESS(fileSystemAbstract, ResultAllocationMemoryFailedNew());

        std::unique_ptr<RegisteredUpdatePartitionCommonMountNameGenerator> mountNameGenerator(new RegisteredUpdatePartitionCommonMountNameGenerator());
        NN_RESULT_THROW_UNLESS(mountNameGenerator.get() != nullptr, ResultAllocationMemoryFailedNew());

        NN_RESULT_DO(fsa::Register(name, std::move(fileSystemAbstract), std::move(mountNameGenerator)));
        NN_RESULT_SUCCESS;
    };

    NN_FS_RESULT_DO(NN_DETAIL_FS_ACCESS_LOG_SYSTEM_MOUNT(mount(), name, NN_DETAIL_FS_ACCESS_LOG_FORMAT_MOUNT, name));
    NN_DETAIL_FS_ACCESS_LOG_SYSTEM_FSACCESSOR_ENABLE(name);
    NN_RESULT_SUCCESS;
}

}}
