﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <mutex>

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>

#include <nn/os/os_Mutex.h>
#include <nn/os/os_ReaderWriterLock.h>
#include <nn/util/util_IntrusiveList.h>
#include <nn/util/util_ScopeExit.h>

namespace nn { namespace fs {
    struct ReadOption;
    struct WriteOption;
}}

namespace nn { namespace fs { namespace fsa {
    class IFile;
}}}

namespace nn { namespace fs { namespace detail {

    class Deleter;

    const size_t FilePathHashSize = 4;
    struct FilePathHash : public detail::Newable
    {
        uint8_t data[FilePathHashSize];
    };
    NN_STATIC_ASSERT(std::is_pod<FilePathHash>::value);
    inline bool operator ==(const FilePathHash& lhs, const FilePathHash& rhs) NN_NOEXCEPT
    {
        return std::memcmp(lhs.data, rhs.data, FilePathHashSize) == 0;
    }
    inline bool operator !=(const FilePathHash& lhs, const FilePathHash& rhs) NN_NOEXCEPT
    {
        return !(lhs == rhs);
    }

    class UniqueLock
    {
        NN_DISALLOW_COPY(UniqueLock);
    public:
        UniqueLock() NN_NOEXCEPT
            : m_pMutex(nullptr)
        {
        }
        NN_IMPLICIT UniqueLock(os::ReaderWriterLock& mutex) NN_NOEXCEPT
            : m_pMutex(&mutex)
        {
            m_pMutex->lock();
        }
        NN_IMPLICIT UniqueLock(UniqueLock&& x) NN_NOEXCEPT
            : m_pMutex(x.m_pMutex)
        {
            x.m_pMutex = nullptr;
        }
        UniqueLock& operator=(UniqueLock&& rhs) NN_NOEXCEPT
        {
            UniqueLock tmp(rhs);
            tmp.swap(*this);
            return *this;
        }
        ~UniqueLock() NN_NOEXCEPT
        {
            Detach();
        }
        os::ReaderWriterLock* Get() const NN_NOEXCEPT
        {
            return m_pMutex;
        }
        void Detach() NN_NOEXCEPT
        {
            if (m_pMutex != nullptr)
            {
                m_pMutex->unlock();
                m_pMutex = nullptr;
            }
        }
    private:
        void swap(UniqueLock& other) NN_NOEXCEPT
        {
            std::swap(this->m_pMutex, other.m_pMutex);
        }

        os::ReaderWriterLock* m_pMutex;
    };

    typedef Result (*CalculateHashFunc)(FilePathHash* outValue, const char* path);

    class FileSystemAccessor;
    class FileDataCacheAccessResult;

    struct PathBasedFileDataCacheEntry : public util::IntrusiveListBaseNode<PathBasedFileDataCacheEntry>, public detail::Newable
    {
        FilePathHash hash;
        int hashIndex;
        FileSystemAccessor* pFileSystemAccessor;
        std::unique_ptr<char[], Deleter> path;
        int pathLength;
        void* pBuffer;
        size_t fileSize;
        os::ReaderWriterLock modifyLock;
    };

    class PathBasedFileDataCache
    {
    public:
        PathBasedFileDataCache() NN_NOEXCEPT;
        ~PathBasedFileDataCache() NN_NOEXCEPT;

        Result Read(fsa::IFile* pFile, int fileOpenMode, FileSystemAccessor* pFileSystemAccessor, const FilePathHash& hash, int hashIndex, size_t* outValue, int64_t offset, void* pBuffer, size_t size, const ReadOption& option, FileDataCacheAccessResult* pAccessResult) NN_NOEXCEPT;
        Result Write(fsa::IFile* pFile, int fileOpenMode, FileSystemAccessor* pFileSystemAccessor, const FilePathHash& hash, int hashIndex, int64_t offset, const void* pBuffer, size_t size, const WriteOption& option) NN_NOEXCEPT;

    public:
        bool Add(std::unique_ptr<PathBasedFileDataCacheEntry>&& newEntry) NN_NOEXCEPT;
        UniqueLock Lock() NN_NOEXCEPT;
        UniqueLock Lock(FileSystemAccessor* pFileSystemAccessor, const FilePathHash& hash, const char* path) NN_NOEXCEPT;
        bool Remove(FileSystemAccessor* pFileSystemAccessor, const FilePathHash& hash, int hashIndex) NN_NOEXCEPT;
        bool Remove(UniqueLock&& lock, FileSystemAccessor* pFileSystemAccessor, const FilePathHash& hash, const char* path) NN_NOEXCEPT;
        bool Remove(FileSystemAccessor* pFileSystemAccessor, const FilePathHash& hash, const char* path) NN_NOEXCEPT;
        void Remove(UniqueLock&& lock, FileSystemAccessor* pFileSystemAccessor) NN_NOEXCEPT;
        void Remove(FileSystemAccessor* pFileSystemAccessor) NN_NOEXCEPT;
        bool FindHashIndexOrNext(int* outHashIndex, FileSystemAccessor* pFileSystemAccessor, const FilePathHash& hash, const char* path) NN_NOEXCEPT;
        bool FindHashIndex(int* outHashIndex, FileSystemAccessor* pFileSystemAccessor, const FilePathHash& hash, const char* path) NN_NOEXCEPT;
        Result CalculateHash(FilePathHash* outValue, const char* path) const NN_NOEXCEPT
        {
            return m_CalculateHashFunc(outValue, path);
        }
        void SetDefaultCalculateHashFunc() NN_NOEXCEPT; // for test
        void SetCalculateHashFunc(CalculateHashFunc func) NN_NOEXCEPT; // for test

    private:
        typedef util::IntrusiveList<PathBasedFileDataCacheEntry, util::IntrusiveListBaseNodeTraits<PathBasedFileDataCacheEntry>> PathBasedFileDataCacheEntryList;
        bool FindHashIndexOrNextImpl(int* outHashIndex, FileSystemAccessor* pFileSystemAccessor, const FilePathHash& hash, const char* path) NN_NOEXCEPT;

        PathBasedFileDataCacheEntryList m_List;
        os::ReaderWriterLock m_ListLock;

        CalculateHashFunc m_CalculateHashFunc;
    };

    // TODO: 各 FS に実装したい
    Result GetFilePathHash(FilePathHash* outValue, const char* path) NN_NOEXCEPT;
    void SetDefaultCalculateHashFunc() NN_NOEXCEPT; // for test
    void SetCalculateHashFunc(CalculateHashFunc func) NN_NOEXCEPT; // for test

    UniqueLock LockPathBasedFileDataCacheEntries() NN_NOEXCEPT;
    void InvalidatePathBasedFileDataCacheEntries(UniqueLock&& lock, FileSystemAccessor* pFileSystemAccessor) NN_NOEXCEPT;
    void InvalidatePathBasedFileDataCacheEntries(FileSystemAccessor* pFileSystemAccessor) NN_NOEXCEPT;
    void InvalidatePathBasedFileDataCacheEntry(FileSystemAccessor* pFileSystemAccessor, const FilePathHash& hash, int hashIndex) NN_NOEXCEPT;
    UniqueLock LockPathBasedFileDataCacheEntry(FileSystemAccessor* pFileSystemAccessor, const char* path) NN_NOEXCEPT;
    void InvalidatePathBasedFileDataCacheEntry(UniqueLock&& lock, FileSystemAccessor* pFileSystemAccessor, const char* path) NN_NOEXCEPT;
    void InvalidatePathBasedFileDataCacheEntry(FileSystemAccessor* pFileSystemAccessor, const char* path) NN_NOEXCEPT;
    bool FindPathBasedFileDataCacheEntry(FilePathHash* outValue, int* outHashIndex, FileSystemAccessor* pFileSystemAccessor, const char* path) NN_NOEXCEPT;

    Result ReadViaPathBasedFileDataCache(fsa::IFile* pFile, int fileOpenMode, FileSystemAccessor* pFileSystemAccessor, const FilePathHash& hash, int hashIndex, size_t* outValue, int64_t offset, void* pBuffer, size_t size, const ReadOption& option, FileDataCacheAccessResult* pAccessResult) NN_NOEXCEPT;

    Result WriteViaPathBasedFileDataCache(fsa::IFile* pFile, int fileOpenMode, FileSystemAccessor* pFileSystemAccessor, const FilePathHash& hash, int hashIndex, int64_t offset, const void* pBuffer, size_t size, const WriteOption& option) NN_NOEXCEPT;

}}}
