﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstring>
#include <memory>
#include <nn/nn_Common.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_Result.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/fs/detail/fs_CommonMountName.h>
#include <nn/fs/fsa/fs_IFileSystem.h>
#include <nn/fs/fsa/fs_Registrar.h>
#include <nn/fs/fs_Directory.h>
#include <nn/fs/fs_PathUtility.h>
#include <nn/util/util_FormatString.h>
#include <nn/fs/detail/fs_Newable.h>

#include <nn/fs/fs_Host.h>
#include <nn/fs/detail/fs_OpenHostFileSystem.h>
#include <nn/fs/detail/fs_AccessLog.h>

#include "fs_Library.h"
#include "fs_FileSystemProxyServiceObject.h"
#include "fs_FileSystemServiceObjectAdapter.h"
#include <nn/fs/detail/fs_ResultHandlingUtility.h>
#include "fsa/fs_MountUtility.h"

namespace nn { namespace fs {

using namespace nn::fs::detail;

namespace {

Result OpenHostFileSystemImpl(std::unique_ptr<nn::fs::fsa::IFileSystem>* outValue, const nn::fssrv::sf::FspPath& sfPath) NN_NOEXCEPT
{
    // ユーザーから直接呼び出されないため、この関数内では NN_FS_RESULT_DO を使いません。
    // （アクセスログを追加した時にアクセスログ出力前にアボートしてしまうため）

    nn::sf::SharedPointer<nn::fssrv::sf::IFileSystemProxy> fileSystemProxy = detail::GetFileSystemProxyServiceObject();
    nn::sf::SharedPointer<nn::fssrv::sf::IFileSystem> fileSystem;
    NN_RESULT_DO(fileSystemProxy->OpenHostFileSystem(&fileSystem, sfPath));

    std::unique_ptr<fsa::IFileSystem> fileSystemAbstract;
    fileSystemAbstract.reset(new detail::FileSystemServiceObjectAdapter(std::move(fileSystem)));
    NN_FS_RESULT_THROW_UNLESS(fileSystemAbstract, ResultAllocationMemoryFailedInHostA());

    *outValue = std::move(fileSystemAbstract);
    NN_RESULT_SUCCESS;
}

class HostCommonMountNameGenerator : public fsa::ICommonMountNameGenerator, public Newable
{
public:
    explicit HostCommonMountNameGenerator(const char* path)
    {
        nn::util::SNPrintf(m_Path, sizeof(m_Path), "%s", path);
        const auto length = strnlen(m_Path, sizeof(m_Path));
        if( 0 < length && m_Path[length - 1] == '/' )
        {
            m_Path[length - 1] = '\0';
        }
    }

    virtual Result GenerateCommonMountName(char* name, size_t nameSize) NN_NOEXCEPT NN_OVERRIDE
    {
        const size_t RequiredNameBufferSize = 7 + strnlen(m_Path, EntryNameLengthMax) + 1; // "@Host:/" + m_Path
        NN_RESULT_THROW_UNLESS(nameSize >= RequiredNameBufferSize, nn::fs::ResultTooLongPath());
        auto size = nn::util::SNPrintf(name, nameSize, "@Host:/%s", m_Path);
        NN_SDK_ASSERT(static_cast<size_t>(size) == RequiredNameBufferSize - 1);
        NN_UNUSED(size);

        NN_RESULT_SUCCESS;
    }
private:
    char m_Path[EntryNameLengthMax + 1];
};

class HostRootCommonMountNameGenerator : public fsa::ICommonMountNameGenerator, public Newable
{
public:
    explicit HostRootCommonMountNameGenerator()
    {
    }

    virtual Result GenerateCommonMountName(char* name, size_t nameSize) NN_NOEXCEPT NN_OVERRIDE
    {
        const size_t RequiredNameBufferSize = 7 + 1;
        NN_SDK_REQUIRES(nameSize >= static_cast<size_t>(RequiredNameBufferSize));
        NN_UNUSED(RequiredNameBufferSize);
        auto size = nn::util::SNPrintf(name, nameSize, "@Host:/");
        NN_SDK_ASSERT(static_cast<size_t>(size) == RequiredNameBufferSize - 1);
        NN_UNUSED(size);

        NN_RESULT_SUCCESS;
    }
};

}

Result MountHost(const char* name, const char* rootPath) NN_NOEXCEPT
{
    auto precondition = [=]() NN_NOEXCEPT -> Result
    {
        // マウント名チェック
        NN_RESULT_DO(detail::CheckMountName(name));
        NN_RESULT_SUCCESS;
    };
    NN_FS_RESULT_DO(
        NN_DETAIL_FS_ACCESS_LOG_MOUNT_FOR_PRECONDITION(precondition(),
            name,
            NN_DETAIL_FS_ACCESS_LOG_FORMAT_MOUNT,
            name));

    std::unique_ptr<fsa::IFileSystem> fileSystem;
    // ホストにつながっていない失敗をハンドリングするためここでは NN_RESULT_DO を使います。
    NN_RESULT_DO(
        NN_DETAIL_FS_ACCESS_LOG_MOUNT(
            OpenHostFileSystem(&fileSystem, name, rootPath),
            name,
            NN_DETAIL_FS_ACCESS_LOG_FORMAT_MOUNT,
            name));

    std::unique_ptr<HostCommonMountNameGenerator> mountNameGenerator(new HostCommonMountNameGenerator(rootPath));
    NN_FS_RESULT_THROW_UNLESS(mountNameGenerator.get() != nullptr, ResultAllocationMemoryFailedInHostB());

    NN_FS_RESULT_DO(fsa::Register(name, std::move(fileSystem), std::move(mountNameGenerator)));
    NN_DETAIL_FS_ACCESS_LOG_FSACCESSOR_ENABLE(name);
    NN_RESULT_SUCCESS;
}

Result MountHostRoot() NN_NOEXCEPT
{
    std::unique_ptr<fsa::IFileSystem> fileSystem;

    nn::fssrv::sf::FspPath sfPath;
    sfPath.str[0] = '\0';
    // ホストにつながっていない失敗をハンドリングするためここでは NN_RESULT_DO を使います。
    NN_RESULT_DO(
        NN_DETAIL_FS_ACCESS_LOG_MOUNT(
            OpenHostFileSystemImpl(&fileSystem, sfPath),
            detail::HostRootFileSystemMountName,
            NN_DETAIL_FS_ACCESS_LOG_FORMAT_MOUNT,
            detail::HostRootFileSystemMountName));

    std::unique_ptr<HostRootCommonMountNameGenerator> mountNameGenerator(new HostRootCommonMountNameGenerator());
    NN_FS_RESULT_THROW_UNLESS(mountNameGenerator.get() != nullptr, ResultAllocationMemoryFailedInHostC());

    NN_FS_RESULT_DO(fsa::Register(detail::HostRootFileSystemMountName, std::move(fileSystem), std::move(mountNameGenerator)));
    NN_DETAIL_FS_ACCESS_LOG_FSACCESSOR_ENABLE(detail::HostRootFileSystemMountName);
    NN_RESULT_SUCCESS;
}

void UnmountHostRoot() NN_NOEXCEPT
{
    Unmount(detail::HostRootFileSystemMountName);
}

}}

namespace nn { namespace fs { namespace detail {

Result OpenHostFileSystem(
    std::unique_ptr<fsa::IFileSystem>* pOutValue,
    const char* name,
    const char* rootPath
) NN_NOEXCEPT
{
    // ユーザーから直接呼び出されないため、この関数内では NN_FS_RESULT_DO を使いません。
    // （アクセスログを追加した時にアクセスログ出力前にアボートしてしまうため）

    NN_RESULT_THROW_UNLESS(pOutValue != nullptr, ResultNullptrArgument());
    NN_RESULT_THROW_UNLESS(name != nullptr, ResultInvalidMountName());
    NN_RESULT_THROW_UNLESS(rootPath != nullptr, ResultInvalidPath());

    NN_RESULT_THROW_UNLESS(!detail::IsWindowsDrive(name), ResultInvalidMountName());
    NN_RESULT_THROW_UNLESS(!detail::IsUsedReservedMountName(name), ResultInvalidMountName());

    NN_UNUSED(name);

    nn::fssrv::sf::FspPath sfPath;

    bool isAppendDelemeter = false;
    auto rootPathLength = strnlen(rootPath, sizeof(sfPath.str));
    if (0 < rootPathLength && rootPath[rootPathLength - 1] != '/')
    {
        isAppendDelemeter = true;
        rootPathLength++;
    }
    NN_RESULT_THROW_UNLESS(rootPathLength + 1 < sizeof(sfPath.str), ResultTooLongPath());

    NN_RESULT_DO(FspPathPrintf(&sfPath, "/%s%s", rootPath, isAppendDelemeter ? "/" : ""));

    // UNC のダブルバックスラッシュを維持するため、先頭の連続するスラッシュをバックスラッシュに置換します。
    if( std::strncmp(sfPath.str + 1, "//", 2) == 0 )
    {
        for( auto pChar = sfPath.str + 1; *pChar == '/'; ++pChar )
        {
            *pChar = '\\';
        }
    }

    std::unique_ptr<fsa::IFileSystem> fileSystem;
    NN_RESULT_DO(OpenHostFileSystemImpl(&fileSystem, sfPath));

    *pOutValue = std::move(fileSystem);
    NN_RESULT_SUCCESS;
}

}}}
