﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "fs_FileSystemCreatorInterfaces.h"

#if defined(NN_BUILD_CONFIG_OS_WIN32)
#include <nn/fs/detail/fs_MemoryManagementPrivate.h>
#include <nn/fs/fs_SdCardPrivate.h>
#include <nn/fssrv/fscreator/fssrv_BuiltInStorageCreator.h>
#include <nn/fssrv/fscreator/fssrv_BuiltInStorageFileSystemCreatorHostFs.h>
#include <nn/fssrv/fscreator/fssrv_FatFileSystemCreator.h>
#include <nn/fssrv/fscreator/fssrv_HostFileSystemCreator.h>
#include <nn/fssrv/fscreator/fssrv_PartitionFileSystemCreator.h>
#include <nn/fssrv/fscreator/fssrv_RomFileSystemCreator.h>
#include <nn/fssrv/fscreator/fssrv_SdStorageCreator.h>
#include <nn/fssrv/fscreator/fssrv_SdCardProxyFileSystemCreatorHostFs.h>
#include <nn/fssrv/fscreator/fssrv_StorageOnNcaCreator.h>
#include <nn/fssrv/fscreator/fssrv_SubDirectoryFileSystemCreator.h>
#include <nn/fssrv/fscreator/fssrv_TargetManagerFileSystemCreator.h>
#include <nn/fssrv/fscreator/fssrv_SaveDataFileSystemCreator.h>
#include <nn/fssrv/fscreator/fssrv_GameCardStorageCreator.h>
#include <nn/fssrv/fscreator/fssrv_GameCardFileSystemCreator.h>
#include <nn/fssrv/fscreator/fssrv_EncryptedFileSystemCreator.h>
#include <nn/fssrv/fscreator/fssrv_MemoryStorageCreator.h>
#include <nn/fssrv/fssrv_MemoryReport.h>
#include <nn/fssrv/fssrv_MemoryResourceFromStandardAllocator.h>
#include <nn/fssrv/fssrv_IFileSystemCreator.h>
#include <nn/fssrv/fssrv_FileSystemProxyServer.h>
#include <nn/fssrv/fssrv_SaveDataTransferCryptoConfiguration.h>
#include <nn/fssystem/fs_AllocatorUtility.h>
#include <nn/fssystem/fs_AsynchronousAccess.h>
#include <nn/fssystem/buffers/fs_FileSystemBufferManager.h>
#include <nn/fssrv/fssrv_NcaCryptoConfiguration.h>
#include <nn/mem/mem_StandardAllocator.h>
#include <nn/util/util_Optional.h>
#include <nn/util/util_FormatString.h>
#include <nn/crypto/crypto_AesDecryptor.h>
#include <nn/crypto.h>
#include <nn/htc.h>
#endif

namespace nn { namespace fs { namespace detail {

#if defined(NN_BUILD_CONFIG_OS_WIN32)

using namespace nn::fssystem;

namespace
{
    char g_SaveDataMacKey[] = "const_key______";
}

    void DecryptAesCtrWin(void* pDst, size_t dstSize, int keyTypeValue, const void* pEncryptedKey, size_t encryptedKeySize, const void* pIv, size_t ivSize, const void* pSrc, size_t srcSize) NN_NOEXCEPT
    {
        NN_UNUSED(pDst);
        NN_UNUSED(dstSize);
        NN_UNUSED(keyTypeValue);
        NN_UNUSED(pEncryptedKey);
        NN_UNUSED(encryptedKeySize);
        NN_UNUSED(pIv);
        NN_UNUSED(ivSize);
        NN_UNUSED(pSrc);
        NN_UNUSED(srcSize);

        // TODO: tdf で kek を注入して対応
        NN_ABORT("Hardware AES CTR is not supported at Win.");
    }

    const int AesKeySize = fssystem::NcaCryptoConfiguration::Aes128KeySize;

    unsigned char* GetNcaKeyAreaEncryptionKek(int keyTypeValue)
    {
        static unsigned char s_NcaKeyAreaEncryptionKek[KeyAreaEncryptionKeyCount][AesKeySize] = {
            { 0x3a, 0x7c, 0x3e, 0x38, 0x4a, 0x8f, 0x22, 0xff, 0x4b, 0x21, 0x57, 0x19, 0xb7, 0x81, 0xad, 0x0c },
            { 0 },
            { 0 },
            { 0x3a, 0x7c, 0x3e, 0x38, 0x4a, 0x8f, 0x22, 0xff, 0x4b, 0x21, 0x57, 0x19, 0xb7, 0x81, 0xad, 0x0c },
            { 0 },
            { 0 },
            { 0xce, 0x3b, 0x2a, 0x68, 0x07, 0x9d, 0x99, 0xb1, 0x44, 0x8d, 0x69, 0xdc, 0x5e, 0x94, 0x2e, 0x48 },
            { 0 },
            { 0 },
            { 0xdb, 0x04, 0x79, 0xba, 0x2d, 0x53, 0x95, 0xb2, 0x7c, 0x71, 0xbd, 0x6a, 0xfe, 0x5c, 0x3e, 0xc7 },
            { 0 },
            { 0 },
            { 0xe6, 0xda, 0x78, 0x56, 0xd2, 0x13, 0xf0, 0xaa, 0xcb, 0x2b, 0x34, 0xa7, 0x1a, 0x87, 0xb6, 0xfb },
            { 0 },
            { 0 },
            { 0x02, 0x74, 0x06, 0xfd, 0xf5, 0x96, 0x32, 0xb8, 0x8b, 0xd0, 0x53, 0xaa, 0xf3, 0x06, 0xd3, 0x9b },
            { 0 },
            { 0 },
        };
        static unsigned char s_NcaHeaderKeyEncryptionKek[AesKeySize] = { 0x9e, 0xa4, 0x3e, 0x57, 0xca, 0xb6, 0x29, 0x46, 0xb0, 0x9d, 0x1c, 0x6c, 0xa1, 0x63, 0xaf, 0x8a };
        static unsigned char s_InvalidNcaKeyAreaEncryptionKek[AesKeySize] = { 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 };

        if (keyTypeValue > static_cast<int>(KeyType::NcaHeaderKey) || IsInvalidKeyTypeValue(keyTypeValue))
        {
            return s_InvalidNcaKeyAreaEncryptionKek;
        }
        else if (keyTypeValue == static_cast<int>(KeyType::NcaHeaderKey))
        {
            return s_NcaHeaderKeyEncryptionKek;
        }
        else
        {
            return s_NcaKeyAreaEncryptionKek[keyTypeValue];
        }
    }

    void RegisterNcaKeyAreaEncryptionKek(int keyTypeValue, const void* pKek, size_t kekSize)
    {
        NN_SDK_ASSERT(kekSize == AesKeySize);
        NN_UNUSED(kekSize);
        memcpy(GetNcaKeyAreaEncryptionKek(keyTypeValue), pKek, AesKeySize);
    }

    //! nca の内部鍵を復号
    void GenerateKeyWin(void* pOutKey, size_t outKeySize, const void* pEncryptedKey, size_t encryptedKeySize, int keyTypeValue, const fssystem::NcaCryptoConfiguration& cryptoConfiguration) NN_NOEXCEPT
    {
        NN_UNUSED(cryptoConfiguration);

        auto pKey = GetNcaKeyAreaEncryptionKek(keyTypeValue);

        {
            char ZeroKey[AesKeySize] = { 0 };
            if (memcmp(pKey, ZeroKey, AesKeySize) == 0)
            {
                NN_ABORT("NcaKey(keyTypeValue = %d) is not set.\n", keyTypeValue);
            }
        }

        crypto::AesDecryptor128 aesDecryptor;
        aesDecryptor.Initialize(pKey, AesKeySize);
        aesDecryptor.DecryptBlock(pOutKey, outKeySize, pEncryptedKey, encryptedKeySize);
    }

    void GenerateDeviceUniqueMacForSaveDataWin(void* macBuffer, size_t macBufferSize, const void* dataBuffer, size_t dataBufferSize)
    {
        const auto MacSize = nn::crypto::Aes128CmacGenerator::MacSize;
        NN_SDK_REQUIRES(macBufferSize >= MacSize);
        memset(macBuffer, 0, macBufferSize);
        nn::crypto::GenerateAes128Cmac(macBuffer, MacSize, dataBuffer, dataBufferSize, g_SaveDataMacKey, sizeof(g_SaveDataMacKey));
    }

    void SetSaveDataMacKeyWin(const void* pKey, size_t keySize) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(keySize >= sizeof(g_SaveDataMacKey));
        NN_UNUSED(keySize);
        memcpy(g_SaveDataMacKey, pKey, sizeof(g_SaveDataMacKey));
    }

    void GenerateSeedUniqueMacForSaveDataWin(void* macBuffer, size_t macBufferSize, const void* dataBuffer, size_t dataBufferSize, const void* seedBuffer, size_t seedBufferSize)
    {
        const auto MacSize = nn::crypto::Aes128CmacGenerator::MacSize;
        NN_SDK_REQUIRES(macBufferSize >= MacSize);
        memset(macBuffer, 0, macBufferSize);
        char key[] = "const_key______";

        crypto::AesEncryptor128 aesEncryptor; // seed の混ぜ込み
        aesEncryptor.Initialize(seedBuffer, seedBufferSize);
        aesEncryptor.EncryptBlock(key, sizeof(key), key, sizeof(key));

        nn::crypto::GenerateAes128Cmac(macBuffer, MacSize, dataBuffer, dataBufferSize, key, sizeof(key));
    }

    void GenerateRandomForSaveDataWin(void* pHash, size_t sizeHash) NN_NOEXCEPT
    {
        const auto now = nn::os::GetSystemTick().GetInt64Value();
        static const char Key[] = "HierarchicalIntegrityVerificationStorage::ControlArea";
        char hash[crypto::Sha256Generator::HashSize];
        nn::crypto::GenerateHmacSha256Mac(
            hash, sizeof(hash),
            &now, sizeof(now),
            Key, sizeof(Key)
        );

        NN_SDK_REQUIRES(sizeHash <= sizeof(hash));
        memcpy(pHash, hash, sizeHash);
    }

    void GenerateKeyForSdEncryption(nn::fssrv::fscreator::EncryptedFileSystemCreator::Key* pOutValueKey, int keyCount, nn::fssrv::fscreator::EncryptedFileSystemCreator::KeyId id, const EncryptionSeed& seed) NN_NOEXCEPT
    {
        NN_UNUSED(keyCount);
        NN_SDK_REQUIRES(keyCount == 2);

        // id と seed 固有の鍵を生成
        char mac[crypto::HmacSha256Generator::MacSize];
        nn::crypto::GenerateHmacSha256Mac(mac, sizeof(mac), &id, sizeof(id), seed.value, seed.Size);

        memcpy(&pOutValueKey[0], &mac[0]   , sizeof(pOutValueKey[0]));
        memcpy(&pOutValueKey[1], &mac[0x10], sizeof(pOutValueKey[1]));
    }

    void GenerateRandomForSdEncryption(void* pBuffer, size_t bufferSize) NN_NOEXCEPT
    {
        const auto now = nn::os::GetSystemTick().GetInt64Value();
        char hash[crypto::Sha256Generator::HashSize];
        crypto::GenerateSha256Hash(hash, sizeof(hash), &now, sizeof(now));

        NN_SDK_REQUIRES(bufferSize <= sizeof(hash));
        memcpy(pBuffer, hash, bufferSize);
    }

    void GenerateRandomForSaveDataTransfer(void* pBuffer, size_t bufferSize) NN_NOEXCEPT
    {
        return GenerateRandomForSdEncryption(pBuffer, bufferSize);
    }

    void GenerateAesKeyForSaveDataTransfer(void* pBuffer, size_t bufferSize, fssrv::SaveDataTransferCryptoConfiguration::KeyIndex keyIndex, const void* pKeySource, size_t keySourceSize) NN_NOEXCEPT
    {
        NN_UNUSED(keySourceSize);

        const unsigned char Key[4][16] =
        {
            { 0x50, 0x96, 0xCB, 0x8C, 0x99, 0x16, 0xDB, 0x65, 0x7F, 0xCA, 0x84, 0xD4, 0xEA, 0xAB, 0x17, 0xDE },
            { 0x48, 0x12, 0xe1, 0x49, 0x14, 0x5a, 0xdb, 0x21, 0x13, 0xc3, 0xd9, 0x1a, 0xfc, 0x9b, 0x8f, 0x23 },
            { 0xf0, 0x6d, 0x87, 0x0c, 0x59, 0x06, 0x7f, 0x24, 0xc7, 0x2c, 0xd9, 0x92, 0x18, 0x58, 0xd2, 0xb5 },
            { 0xb9, 0x3f, 0x83, 0x24, 0x38, 0x4e, 0xb1, 0x61, 0x1f, 0x52, 0x09, 0x56, 0xd9, 0x13, 0x57, 0x90 },
        };

        switch (keyIndex)
        {
            case fssrv::SaveDataTransferCryptoConfiguration::KeyIndex::SaveDataTransferToken:
            {
                memcpy(pBuffer, Key[0], bufferSize);
                NN_UNUSED(pKeySource);
                return;
            }
            case fssrv::SaveDataTransferCryptoConfiguration::KeyIndex::SaveDataTransfer:
            {
                memcpy(pBuffer, Key[1], bufferSize);
                NN_SDK_ASSERT_NOT_NULL(pKeySource);
                static_cast<char*>(pBuffer)[0] ^= static_cast<const char*>(pKeySource)[0];
                return;
            }
            case nn::fssrv::SaveDataTransferCryptoConfiguration::KeyIndex::CloudBackUpToken:
            {
                memcpy(pBuffer, Key[2], bufferSize);
                NN_UNUSED(pKeySource);
                return;
            }
            case nn::fssrv::SaveDataTransferCryptoConfiguration::KeyIndex::CloudBackUpData:
            {
                memcpy(pBuffer, Key[3], bufferSize);
                NN_SDK_ASSERT_NOT_NULL(pKeySource);
                static_cast<char*>(pBuffer)[0] ^= static_cast<const char*>(pKeySource)[0];
                return;
            }
            default:
                NN_UNEXPECTED_DEFAULT;
        }
    }

    nn::fssrv::SaveDataTransferCryptoConfiguration g_SaveDataTransferCryptoConfiguration =
    {
        // セーブデータ移行仮鍵
        {
            0xb7, 0x44, 0xfe, 0xa3, 0x88, 0x25, 0xd2, 0xe0, 0x3e, 0xf3, 0x8e, 0x91, 0xcc, 0x94, 0x4c, 0x61,
            0x26, 0xc0, 0xa1, 0x02, 0x0e, 0xe0, 0x1c, 0x7c, 0x67, 0x71, 0x6e, 0x22, 0x15, 0xe8, 0xa1, 0xc9,
            0x0c, 0xf8, 0xce, 0xb9, 0x95, 0x8b, 0x44, 0xf7, 0x0a, 0x17, 0x87, 0x5a, 0x6b, 0xf5, 0x44, 0xf2,
            0xb7, 0xa1, 0x90, 0xe3, 0xb4, 0x9e, 0xab, 0x80, 0x67, 0xd8, 0x0e, 0xbf, 0xa9, 0x81, 0x56, 0x51,
            0xf3, 0x2e, 0x98, 0x7f, 0xee, 0x60, 0x53, 0xad, 0x99, 0xa4, 0x11, 0x0d, 0x07, 0x9d, 0x5d, 0x1a,
            0xe2, 0xad, 0xb4, 0x51, 0x84, 0xa5, 0xa1, 0xf5, 0x39, 0x6b, 0xfc, 0x99, 0x5d, 0xf5, 0x59, 0xdc,
            0xe3, 0x03, 0xee, 0x53, 0x60, 0x6f, 0xb3, 0xda, 0x20, 0x9c, 0xf4, 0x89, 0x8b, 0x65, 0x68, 0x4b,
            0x60, 0x0d, 0x4e, 0xb2, 0x40, 0xfa, 0xfc, 0xd7, 0xc5, 0x1f, 0x81, 0x3d, 0x99, 0xe5, 0x50, 0x58,
            0xd5, 0x42, 0x9c, 0x14, 0x1f, 0x27, 0x90, 0x9e, 0x9b, 0x7e, 0x3b, 0xfd, 0xb0, 0x76, 0xb6, 0xb9,
            0x3d, 0x9e, 0xf6, 0x5c, 0x0d, 0x8c, 0x09, 0xff, 0x4a, 0x55, 0xbb, 0xca, 0x9d, 0xe3, 0x55, 0xad,
            0xf8, 0x9b, 0xfa, 0x02, 0xef, 0x5f, 0xe6, 0x5c, 0xd8, 0x37, 0x8f, 0x2a, 0x5b, 0x19, 0x01, 0xc3,
            0x28, 0xb5, 0x09, 0x3a, 0xfc, 0xc7, 0xa2, 0x40, 0x39, 0x34, 0x9c, 0x90, 0xa2, 0xc2, 0x4c, 0x03,
            0x04, 0xc7, 0x8f, 0x89, 0xaf, 0x57, 0xf0, 0xf3, 0x7f, 0x30, 0xbe, 0xb1, 0xa6, 0xbb, 0x51, 0x5f,
            0x76, 0xf7, 0x21, 0x9a, 0x72, 0xb4, 0x0c, 0x1a, 0x0e, 0x0e, 0x8f, 0x50, 0x91, 0x51, 0x57, 0x50,
            0x56, 0x0a, 0x80, 0xd7, 0x1f, 0x05, 0x1d, 0x69, 0x26, 0x3c, 0xcc, 0xca, 0x4c, 0x6a, 0x03, 0x92,
            0x91, 0x80, 0x71, 0x07, 0xd3, 0x80, 0xf4, 0x1c, 0xd8, 0x95, 0xc2, 0x15, 0x58, 0xfc, 0x57, 0xd5,
        },

        // クラウドバックアップ仮鍵
        {
            0xb7, 0x44, 0xfe, 0xa3, 0x88, 0x25, 0xd2, 0xe0, 0x3e, 0xf3, 0x8e, 0x91, 0xcc, 0x94, 0x4c, 0x61,
            0x26, 0xc0, 0xa1, 0x02, 0x0e, 0xe0, 0x1c, 0x7c, 0x67, 0x71, 0x6e, 0x22, 0x15, 0xe8, 0xa1, 0xc9,
            0x0c, 0xf8, 0xce, 0xb9, 0x95, 0x8b, 0x44, 0xf7, 0x0a, 0x17, 0x87, 0x5a, 0x6b, 0xf5, 0x44, 0xf2,
            0xb7, 0xa1, 0x90, 0xe3, 0xb4, 0x9e, 0xab, 0x80, 0x67, 0xd8, 0x0e, 0xbf, 0xa9, 0x81, 0x56, 0x51,
            0xf3, 0x2e, 0x98, 0x7f, 0xee, 0x60, 0x53, 0xad, 0x99, 0xa4, 0x11, 0x0d, 0x07, 0x9d, 0x5d, 0x1a,
            0xe2, 0xad, 0xb4, 0x51, 0x84, 0xa5, 0xa1, 0xf5, 0x39, 0x6b, 0xfc, 0x99, 0x5d, 0xf5, 0x59, 0xdc,
            0xe3, 0x03, 0xee, 0x53, 0x60, 0x6f, 0xb3, 0xda, 0x20, 0x9c, 0xf4, 0x89, 0x8b, 0x65, 0x68, 0x4b,
            0x60, 0x0d, 0x4e, 0xb2, 0x40, 0xfa, 0xfc, 0xd7, 0xc5, 0x1f, 0x81, 0x3d, 0x99, 0xe5, 0x50, 0x58,
            0xd5, 0x42, 0x9c, 0x14, 0x1f, 0x27, 0x90, 0x9e, 0x9b, 0x7e, 0x3b, 0xfd, 0xb0, 0x76, 0xb6, 0xb9,
            0x3d, 0x9e, 0xf6, 0x5c, 0x0d, 0x8c, 0x09, 0xff, 0x4a, 0x55, 0xbb, 0xca, 0x9d, 0xe3, 0x55, 0xad,
            0xf8, 0x9b, 0xfa, 0x02, 0xef, 0x5f, 0xe6, 0x5c, 0xd8, 0x37, 0x8f, 0x2a, 0x5b, 0x19, 0x01, 0xc3,
            0x28, 0xb5, 0x09, 0x3a, 0xfc, 0xc7, 0xa2, 0x40, 0x39, 0x34, 0x9c, 0x90, 0xa2, 0xc2, 0x4c, 0x03,
            0x04, 0xc7, 0x8f, 0x89, 0xaf, 0x57, 0xf0, 0xf3, 0x7f, 0x30, 0xbe, 0xb1, 0xa6, 0xbb, 0x51, 0x5f,
            0x76, 0xf7, 0x21, 0x9a, 0x72, 0xb4, 0x0c, 0x1a, 0x0e, 0x0e, 0x8f, 0x50, 0x91, 0x51, 0x57, 0x50,
            0x56, 0x0a, 0x80, 0xd7, 0x1f, 0x05, 0x1d, 0x69, 0x26, 0x3c, 0xcc, 0xca, 0x4c, 0x6a, 0x03, 0x92,
            0x91, 0x80, 0x71, 0x07, 0xd3, 0x80, 0xf4, 0x1c, 0xd8, 0x95, 0xc2, 0x15, 0x58, 0xfc, 0x57, 0xd5,
        },
        GenerateAesKeyForSaveDataTransfer,
        GenerateRandomForSaveDataTransfer
    };

    nn::fssrv::SaveDataTransferCryptoConfiguration* GetSaveDataTransferCryptoConfiguration(bool isProd) NN_NOEXCEPT
    {
        NN_SDK_ASSERT(!isProd);
        NN_UNUSED(isProd);
        return &g_SaveDataTransferCryptoConfiguration;
    }


    extern size_t GetExpHeapFreeSizePeak() NN_NOEXCEPT;
    extern void ClearExpHeapFreeSizePeak() NN_NOEXCEPT;

    namespace {
        const int BufferPoolSize = 6 * 1024 * 1024; // fs_Main.cpp と合わせる。TODO: メモリ管理改善
        NN_ALIGNAS(4096) char g_BufferPool[BufferPoolSize];
        nn::mem::StandardAllocator g_BufferAllocator(g_BufferPool, BufferPoolSize);

        const auto DeviceBufferSize = 8 * 1024 * 1024;
        NN_ALIGNAS(4096) char g_DeviceBuffer[DeviceBufferSize];
        NN_ALIGNAS(32) char g_DeviceWorkBuffer[nn::fssystem::BufferPoolWorkSize];

        const auto MaxCacheCount = 1024;
        const auto SizeBlock = 16 * 1024;
        const auto BufferManagerHeapSize = 14 * 1024 * 1024;
        NN_STATIC_ASSERT(BufferManagerHeapSize % nn::os::MemoryBlockUnitSize == 0);
        NN_ALIGNAS(4096) char g_BufferManagerHeap[BufferManagerHeapSize];
        NN_ALIGNAS(4096) char g_BufferManagerWorkBuffer[64 * 1024];
        nn::fssystem::FileSystemBufferManager g_BufferManager;

        const auto PooledThreadCount = 12;
        const auto PooledThreadStackSize = 32 * 1024;
        fssystem::PooledThread g_PooledThreads[PooledThreadCount];
#if 0 // TODO: スタック用のバッファを BufferManager から確保するため静的な確保は見送り。今後調整が必要
        NN_OS_ALIGNAS_THREAD_STACK
            char g_PooledThreadStack[PooledThreadCount * PooledThreadStackSize] = {};
#else
        std::pair<uintptr_t, size_t> g_PooledThreadStack;
#endif
        fssystem::ThreadPool g_ThreadPool(g_PooledThreads, PooledThreadCount);

        nn::util::optional<nn::fssrv::MemoryResourceFromStandardAllocator> g_Allocator;
        nn::util::optional<nn::fssrv::fscreator::RomFileSystemCreator> g_RomFsCreator;
        nn::util::optional<nn::fssrv::fscreator::PartitionFileSystemCreator> g_PartitionFsCreator;
        nn::util::optional<nn::fssrv::fscreator::StorageOnNcaCreator> g_StorageOnNcaCreator;
        nn::util::optional<nn::fssrv::fscreator::FatFileSystemCreator> g_FatFsCreator;
        nn::util::optional<nn::fssrv::fscreator::HostFileSystemCreator> g_HostFsCreator;
        nn::util::optional<nn::fssrv::fscreator::TargetManagerFileSystemCreator> g_TargetFsCreator;
        nn::util::optional<nn::fssrv::fscreator::SubDirectoryFileSystemCreator> g_SubDirFsCreator;
        nn::util::optional<nn::fssrv::fscreator::BuiltInStorageCreator> g_BuiltInStorageCreator;
        nn::util::optional<nn::fssrv::fscreator::SdStorageCreator> g_SdStorageCreator;
        nn::util::optional<nn::fssrv::fscreator::SaveDataFileSystemCreator> g_SaveDataFsCreator;
        nn::util::optional<nn::fssrv::fscreator::GameCardStorageCreator> g_GcStorageCreator;
        nn::util::optional<nn::fssrv::fscreator::GameCardFileSystemCreator> g_GcFsCreator;
        nn::util::optional<nn::fssrv::fscreator::EncryptedFileSystemCreator> g_EncryptedFsCreator;
        nn::util::optional<nn::fssrv::fscreator::MemoryStorageCreator> g_MemoryStorageCreator;
        nn::util::optional<nn::fssrv::fscreator::BuiltInStorageFileSystemCreatorHostFs> g_BisfsCreator;
        nn::util::optional<nn::fssrv::fscreator::SdCardProxyFileSystemCreatorHostFs> g_SdProxyFsCreator;


        nn::fssrv::fscreator::FileSystemCreatorInterfaces g_FsCreatorInterfaces = {};

        nn::fssystem::NcaCryptoConfiguration g_NcaCryptoConfiguration;

        auto g_BufferManagerMemoryReport = nn::fssrv::MemoryReportCreator::CreateFunctionalMemoryReport(
            [&]() NN_NOEXCEPT -> size_t
            {
                return g_BufferManager.GetFreeSizePeak();
            },
            [&]() NN_NOEXCEPT -> size_t
            {
                return g_BufferManager.GetTotalAllocatableSizePeak();
            },
            [&]() NN_NOEXCEPT -> size_t
            {
                return g_BufferManager.GetRetriedCount();
            },
            [&]() NN_NOEXCEPT
            {
                g_BufferManager.ClearPeak();
            }
        );
        auto g_ExpHeapMemoryReport = nn::fssrv::MemoryReportCreator::CreateFunctionalMemoryReport(
            [&]() NN_NOEXCEPT -> size_t
            {
                return GetExpHeapFreeSizePeak();
            },
            [&]() NN_NOEXCEPT -> size_t
            {
                return 0;
            },
            [&]() NN_NOEXCEPT -> size_t
            {
                return 0;
            },
            [&]() NN_NOEXCEPT
            {
                ClearExpHeapFreeSizePeak();
            }
        );
        auto g_BufferPoolMemoryReport = nn::fssrv::MemoryReportCreator::CreateFunctionalMemoryReport(
            [&]() NN_NOEXCEPT -> size_t
            {
                return g_Allocator->GetFreeSizePeak();
            },
            [&]() NN_NOEXCEPT -> size_t
            {
                return 0;
            },
            [&]() NN_NOEXCEPT -> size_t
            {
                return 0;
            },
            [&]() NN_NOEXCEPT
            {
                g_Allocator->ClearPeak();
            }
        );

        nn::fssrv::MemoryReportCreator g_MemoryReport(&g_BufferManagerMemoryReport, &g_ExpHeapMemoryReport, &g_BufferPoolMemoryReport);
    }

    template <int KeySize>
    bool LoadKey(unsigned char* pOutBuffer, size_t outBufferSize, const char* pVariableName) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(outBufferSize == KeySize);
        NN_UNUSED(outBufferSize);

        const size_t KeyStringSize = KeySize * 2;
        char keyString[KeyStringSize + 1];
        size_t outLength = 0;
        if (nn::htc::GetTargetEnvironmentVariableLength(&outLength, pVariableName).IsSuccess() && outLength == sizeof(keyString))
        {
            if (nn::htc::GetTargetEnvironmentVariable(&outLength, keyString, sizeof(keyString), pVariableName).IsSuccess())
            {
                // TODO: valueType = binary を利用する等
                char keyValue[KeySize];
                for(int i = 0; i < KeySize; i++)
                {
                    char hex[2 + 1] = {0};
                    memcpy(hex, &keyString[i * 2], 2);
                    // strtol() は文字列の最大長を指定できないが、hex は必ず NUL 終端されているはずなので問題ない
                    keyValue[i] = static_cast<char>(strtol(hex, nullptr, 16));
                }

                memcpy(pOutBuffer, keyValue, KeySize);
                return true;
            }
        }

        return false;
    }

    void InitializeFileSystemCreatorInterfaces() NN_NOEXCEPT
    {
        nn::fssystem::InitializeAllocator(nn::fs::detail::Allocate, nn::fs::detail::Deallocate);

        const bool IsProd = false;

        // ターゲット環境変数から index = 1, 2 の鍵値を取得
        g_NcaCryptoConfiguration = *nn::fssrv::GetNcaDefaultCryptoConfiguration(IsProd);
        {
            const size_t RsaKeySize = fssystem::NcaCryptoConfiguration::Rsa2048KeyModulusSize;
            g_NcaCryptoConfiguration.pGenerateKey = GenerateKeyWin;
            g_NcaCryptoConfiguration.pDecryptAesCtr = DecryptAesCtrWin;
            g_NcaCryptoConfiguration.isAvailableRawHeader = !IsProd;

            struct {
                const char* VariableName;
                int KeyTypeValue;
            } Keys[] = {
                {"SDK_FS_WIN_NCA_KEY_AREA_ENCRYPTION_KEY_1",       fssystem::GetKeyTypeValue(1, 0)},
                {"SDK_FS_WIN_NCA_KEY_AREA_ENCRYPTION_KEY_2",       fssystem::GetKeyTypeValue(2, 0)},
                {"SDK_FS_WIN_NCA_KEY_AREA_ENCRYPTION_KEY_1_GEN_2", fssystem::GetKeyTypeValue(1, 2)},
                {"SDK_FS_WIN_NCA_KEY_AREA_ENCRYPTION_KEY_2_GEN_2", fssystem::GetKeyTypeValue(2, 2)},
                {"SDK_FS_WIN_NCA_KEY_AREA_ENCRYPTION_KEY_1_GEN_3", fssystem::GetKeyTypeValue(1, 3)},
                {"SDK_FS_WIN_NCA_KEY_AREA_ENCRYPTION_KEY_2_GEN_3", fssystem::GetKeyTypeValue(2, 3)},
                {"SDK_FS_WIN_NCA_KEY_AREA_ENCRYPTION_KEY_1_GEN_4", fssystem::GetKeyTypeValue(1, 4)},
                {"SDK_FS_WIN_NCA_KEY_AREA_ENCRYPTION_KEY_2_GEN_4", fssystem::GetKeyTypeValue(2, 4)},
                {"SDK_FS_WIN_NCA_KEY_AREA_ENCRYPTION_KEY_1_GEN_5", fssystem::GetKeyTypeValue(1, 5)},
                {"SDK_FS_WIN_NCA_KEY_AREA_ENCRYPTION_KEY_2_GEN_5", fssystem::GetKeyTypeValue(2, 5)},
            };

            for(auto key : Keys)
            {
                auto pKey = GetNcaKeyAreaEncryptionKek(key.KeyTypeValue);
                LoadKey<AesKeySize>(pKey, AesKeySize, key.VariableName);
            }

            LoadKey<AesKeySize>(g_NcaCryptoConfiguration.headerEncryptionKeySource, AesKeySize, "SDK_FS_WIN_NCA_HEADER_ENCRYPTION_KEY_SOURCE");
            LoadKey<RsaKeySize>(g_NcaCryptoConfiguration.headerSign1KeyModulus, RsaKeySize,     "SDK_FS_WIN_NCA_HEADER_SIGN_1_VERIFICATION_KEY");
        }

        fssystem::InitializeBufferPool(
            g_DeviceBuffer,
            sizeof(g_DeviceBuffer),
            g_DeviceWorkBuffer,
            sizeof(g_DeviceWorkBuffer));

        g_BufferManager.Initialize(
            MaxCacheCount,
            reinterpret_cast<uintptr_t>(g_BufferManagerHeap),
            BufferManagerHeapSize,
            SizeBlock,
            g_BufferManagerWorkBuffer,
            sizeof(g_BufferManagerWorkBuffer));

        nn::fssrv::SetMemoryReportCreator(&g_MemoryReport);

#if 0 // TODO: スタック用のバッファを BufferManager から確保するため静的な確保は見送り。今後調整が必要
        g_ThreadPool.Initialize(g_PooledThreadStack, PooledThreadCount * PooledThreadStackSize);
#else
        g_PooledThreadStack = g_BufferManager.AllocateBuffer(PooledThreadCount * PooledThreadStackSize);
        g_ThreadPool.Initialize(
            reinterpret_cast<char*>(g_PooledThreadStack.first),
            PooledThreadCount * PooledThreadStackSize,
            [](char* buffer, size_t bufferSize, void* pArgument) NN_NOEXCEPT
            {
                NN_SDK_REQUIRES_EQUAL(reinterpret_cast<uintptr_t>(buffer), g_PooledThreadStack.first);
                NN_UNUSED(buffer);
                NN_UNUSED(bufferSize);
                NN_UNUSED(pArgument);
                g_BufferManager.DeallocateBuffer(g_PooledThreadStack.first, g_PooledThreadStack.second);
            },
            nullptr
        );
#endif
        fssystem::RegisterThreadPool(&g_ThreadPool);

        g_Allocator.emplace(&g_BufferAllocator);

        g_RomFsCreator.emplace(&g_Allocator.value());
        g_PartitionFsCreator.emplace();
        g_StorageOnNcaCreator.emplace(&g_Allocator.value(), g_NcaCryptoConfiguration, IsProd, &g_BufferManager);
        g_FatFsCreator.emplace(&g_Allocator.value());
        g_HostFsCreator.emplace(!IsProd);
        g_TargetFsCreator.emplace();
        g_SubDirFsCreator.emplace();

        nn::fssrv::fscreator::BuiltInStorageCreator::Configuration bisCreatorConfig;
        bisCreatorConfig.enablePackage2HashVerification   = false;
        bisCreatorConfig.getKeyFunc = nullptr; // TODO: Win 版対応
        bisCreatorConfig.pSignedSystemPartitionSignKeyPublicModulus = nullptr;
        g_BuiltInStorageCreator.emplace(bisCreatorConfig);
        g_BisfsCreator.emplace(&g_HostFsCreator.value());

        g_SdStorageCreator.emplace();
        g_SdProxyFsCreator.emplace(&g_HostFsCreator.value());

        // 使用可能なセーブデータ最低バージョン
        // 4: 5NUP よりも古い本体で作成したセーブデータ
        // 5: 5NUP 以降で作成したセーブデータ
        // TODO: セーブデータバージョン 5 正式対応後には 5 に書き換える必要があります。
        const int SaveDataMinimumVersion = 4;
        g_SaveDataFsCreator.emplace(&g_Allocator.value(), &g_BufferManager, GenerateDeviceUniqueMacForSaveDataWin, GenerateSeedUniqueMacForSaveDataWin, GenerateRandomForSaveDataWin, SaveDataMinimumVersion);

        g_GcStorageCreator.emplace(&g_Allocator.value());
        g_GcFsCreator.emplace(&g_Allocator.value(), &g_GcStorageCreator.value());

        nn::fssrv::fscreator::EncryptedFileSystemCreator::Configuration encryptedFsConfig;
        encryptedFsConfig.pGenerateRandomFunction = GenerateRandomForSdEncryption;
        encryptedFsConfig.pGetKeyFunction         = GenerateKeyForSdEncryption;
        g_EncryptedFsCreator.emplace(encryptedFsConfig);
        g_MemoryStorageCreator.emplace();

        g_FsCreatorInterfaces.romFsCreator = &g_RomFsCreator.value();
        g_FsCreatorInterfaces.partitionFsCreator = &g_PartitionFsCreator.value();
        g_FsCreatorInterfaces.storageOnNcaCreator = &g_StorageOnNcaCreator.value();
        g_FsCreatorInterfaces.fatFsCreator = &g_FatFsCreator.value();
        g_FsCreatorInterfaces.hostFsCreator = &g_HostFsCreator.value();
        g_FsCreatorInterfaces.targetFsCreator = &g_TargetFsCreator.value();
        g_FsCreatorInterfaces.subDirFsCreator = &g_SubDirFsCreator.value();
        g_FsCreatorInterfaces.builtInStorageCreator = &g_BuiltInStorageCreator.value();
        g_FsCreatorInterfaces.sdStorageCreator = &g_SdStorageCreator.value();
        g_FsCreatorInterfaces.saveDataFsCreator = &g_SaveDataFsCreator.value();
        g_FsCreatorInterfaces.gameCardStorageCreator = &g_GcStorageCreator.value();
        g_FsCreatorInterfaces.gameCardFileSystemCreator = &g_GcFsCreator.value();
        g_FsCreatorInterfaces.encryptedFileSystemCreator = &g_EncryptedFsCreator.value();
        g_FsCreatorInterfaces.memoryStorageCreator = &g_MemoryStorageCreator.value();
        g_FsCreatorInterfaces.bisfsCreator = &g_BisfsCreator.value();
        g_FsCreatorInterfaces.sdProxyFsCreator = &g_SdProxyFsCreator.value();

        // http://spdlybra.nintendo.co.jp/confluence/pages/viewpage.action?pageId=109326482
        // システムプロセス、システムデータ、本体機能は、本体保存メモリーのみに配置される想定で速度エミュレーションを行う
        nn::fssrv::InternalProgramIdRangeForSpeedEmulation internalProgramIdRangeForSpeedEmulation = { 0x0100000000000000ULL, 0x0100000000001FFFULL };

        nn::fssrv::InitializeFileSystemProxy(&g_FsCreatorInterfaces, &g_BufferManager, true, &g_SaveDataTransferCryptoConfiguration, &internalProgramIdRangeForSpeedEmulation, GenerateRandomForSaveDataWin);

    } // NOLINT(impl/function_size)
#else
    void InitializeFileSystemCreatorInterfaces() NN_NOEXCEPT
    {
        return;
    }
#endif

}}}
