﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_SdkAssert.h>
#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/fs/fs_PathUtility.h>
#include <nn/fs/detail/fs_AccessLog.h>
#include <nn/fs/detail/fs_CommonMountName.h>
#include <nn/fs/detail/fs_FileSystemProxyTypes.h>
#include <nn/fs/fsa/fs_IFileSystem.h>
#include <nn/fs/fsa/fs_Registrar.h>
#include <nn/util/util_FormatString.h>

#include <nn/fs/fs_Content.h>

#include "fs_FileSystemProxyServiceObject.h"
#include "fs_FileSystemServiceObjectAdapter.h"
#include "fsa/fs_MountUtility.h"
#include <nn/fs/detail/fs_ResultHandlingUtility.h>

namespace nn { namespace fs {

namespace {

nn::fs::detail::FileSystemProxyType ConvertToFileSystemProxyType(ContentType contentType) NN_NOEXCEPT
{
    switch (contentType)
    {
    case ContentType_Control:
        return detail::FileSystemProxyType::FileSystemProxyType_Control;
    case ContentType_Manual:
        return detail::FileSystemProxyType::FileSystemProxyType_Manual;
    case ContentType_Meta:
        return detail::FileSystemProxyType::FileSystemProxyType_Meta;
    case ContentType_Data:
        return detail::FileSystemProxyType::FileSystemProxyType_Data;
    default:
        NN_UNEXPECTED_DEFAULT;
    }
}

Result MountContentImpl(const char* name, const char* path, Bit64 id, nn::fs::detail::FileSystemProxyType type) NN_NOEXCEPT
{
    // ユーザーから直接呼び出されないため、この関数内では NN_FS_RESULT_DO を使いません。
    // （アクセスログを追加した時にアクセスログ出力前にアボートしてしまうため）

    nn::fssrv::sf::FspPath sfPath;
    NN_RESULT_DO(FspPathPrintf(&sfPath, "%s", path));

    nn::sf::SharedPointer<nn::fssrv::sf::IFileSystemProxy> fileSystemProxy = detail::GetFileSystemProxyServiceObject();
    nn::sf::SharedPointer<nn::fssrv::sf::IFileSystem> fileSystem;
    NN_RESULT_DO(fileSystemProxy->OpenFileSystemWithId(&fileSystem, sfPath, id, type));

    std::unique_ptr<fsa::IFileSystem> fileSystemAbstract;
    fileSystemAbstract.reset(new detail::FileSystemServiceObjectAdapter(std::move(fileSystem)));
    NN_RESULT_THROW_UNLESS(fileSystemAbstract, ResultAllocationMemoryFailedInContentA());

    NN_RESULT_DO(fsa::Register(name, std::move(fileSystemAbstract)));
    NN_RESULT_SUCCESS;
}

}

Result MountContent(const char* name, const char* path, ContentType contentType) NN_NOEXCEPT
{
    NN_FS_ABORT_UNLESS_WITH_RESULT(NN_DETAIL_FS_ACCESS_LOG_SYSTEM_MOUNT_FOR_PRECONDITION(contentType == ContentType_Meta,
        name, NN_DETAIL_FS_ACCESS_LOG_FORMAT_MOUNTCONTENT_PATH(name, path, contentType)), ResultInvalidArgument());
    ncm::ProgramId id = {};
    return MountContent(name, path, id, contentType);
}

Result MountContent(const char* name, const char* path, ncm::ProgramId id, ContentType contentType) NN_NOEXCEPT
{
    auto mount = [=]() NN_NOEXCEPT -> Result
    {
        // マウント名チェック
        NN_RESULT_DO(detail::CheckMountNameAcceptingReservedMountName(name));

        auto type = ConvertToFileSystemProxyType(contentType);
        NN_RESULT_THROW_UNLESS(path != nullptr, ResultInvalidPath());
        NN_RESULT_DO(MountContentImpl(name, path, id.value, type));
        NN_RESULT_SUCCESS;
    };

    NN_FS_RESULT_DO(NN_DETAIL_FS_ACCESS_LOG_SYSTEM_MOUNT(mount(),
        name, NN_DETAIL_FS_ACCESS_LOG_FORMAT_MOUNTCONTENT_PATH_AND_PROGRAMID(name, path, id, contentType)));
    NN_DETAIL_FS_ACCESS_LOG_SYSTEM_FSACCESSOR_ENABLE(name);
    NN_RESULT_SUCCESS;
}

Result MountContent(const char* name, const char* path, ncm::DataId id, ContentType contentType) NN_NOEXCEPT
{
    auto mount = [=]() NN_NOEXCEPT -> Result
    {
        // マウント名チェック
        NN_RESULT_DO(detail::CheckMountNameAcceptingReservedMountName(name));

        auto type = ConvertToFileSystemProxyType(contentType);
        NN_RESULT_THROW_UNLESS(path != nullptr, ResultInvalidPath());
        NN_RESULT_DO(MountContentImpl(name, path, id.value, type));
        NN_RESULT_SUCCESS;
    };
    NN_FS_RESULT_DO(NN_DETAIL_FS_ACCESS_LOG_SYSTEM_MOUNT(mount(),
        name, NN_DETAIL_FS_ACCESS_LOG_FORMAT_MOUNTCONTENT_PATH_AND_DATAID(name, path, id, contentType)));
    NN_DETAIL_FS_ACCESS_LOG_SYSTEM_FSACCESSOR_ENABLE(name);
    NN_RESULT_SUCCESS;
}

Result MountContent(const char* name, ncm::ProgramId id, ContentType contentType) NN_NOEXCEPT
{
    auto mount = [=]() NN_NOEXCEPT -> Result
    {
        // マウント名チェック
        NN_RESULT_DO(detail::CheckMountNameAcceptingReservedMountName(name));

        auto type = ConvertToFileSystemProxyType(contentType);

        nn::sf::SharedPointer<nn::fssrv::sf::IFileSystemProxy> fileSystemProxy = detail::GetFileSystemProxyServiceObject();
        nn::sf::SharedPointer<nn::fssrv::sf::IFileSystem> fileSystem;
        NN_RESULT_DO(fileSystemProxy->OpenFileSystemWithPatch(&fileSystem, id, type));

        std::unique_ptr<fsa::IFileSystem> fileSystemAbstract;
        fileSystemAbstract.reset(new detail::FileSystemServiceObjectAdapter(std::move(fileSystem)));
        NN_RESULT_THROW_UNLESS(fileSystemAbstract, ResultAllocationMemoryFailedInContentA());

        NN_RESULT_DO(fsa::Register(name, std::move(fileSystemAbstract)));
        NN_RESULT_SUCCESS;
    };

    NN_FS_RESULT_DO(NN_DETAIL_FS_ACCESS_LOG_SYSTEM_MOUNT(mount(),
        name, NN_DETAIL_FS_ACCESS_LOG_FORMAT_MOUNTCONTENT_PROGRAMID(name, id, contentType)));
    NN_DETAIL_FS_ACCESS_LOG_SYSTEM_FSACCESSOR_ENABLE(name);
    NN_RESULT_SUCCESS;
}


}}
