﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstring>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_Abort.h>
#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/fs/fs_Directory.h>
#include <nn/fs/fs_Mount.h>
#include <nn/fs/detail/fs_AccessLog.h>
#include <nn/fs/detail/fs_CommonMountName.h>
#include <nn/fs/fsa/fs_IFileSystem.h>
#include <nn/fs/fsa/fs_Registrar.h>
#include <nn/fs/fs_IStorage.h>
#include <nn/fs/fs_PathUtility.h>
#include <nn/fssrv/sf/fssrv_IFileSystemProxy.h>
#include <nn/util/util_FormatString.h>

#include <nn/fs/fs_Bis.h>

#include "fs_Library.h"
#include "fs_FileSystemProxyServiceObject.h"
#include "fs_FileSystemServiceObjectAdapter.h"
#include <nn/fs/detail/fs_ResultHandlingUtility.h>
#include "fs_StorageServiceObjectAdapter.h"
#include "fsa/fs_MountUtility.h"

namespace nn { namespace fs {

using namespace nn::fs::detail;

namespace {

class BisCommonMountNameGenerator : public fsa::ICommonMountNameGenerator, public Newable
{
public:
    explicit BisCommonMountNameGenerator(BisPartitionId id)
        : m_Id(id)
    {
    }

    virtual Result GenerateCommonMountName(char* name, size_t nameSize) NN_NOEXCEPT NN_OVERRIDE
    {
        const size_t RequiredNameBufferSize = strnlen(GetBisMountName(m_Id), MountNameLengthMax) + 1 + 1;
        NN_SDK_REQUIRES(nameSize >= static_cast<size_t>(RequiredNameBufferSize));
        NN_UNUSED(RequiredNameBufferSize);
        auto size = nn::util::SNPrintf(name, nameSize, "%s:", GetBisMountName(m_Id));
        NN_SDK_ASSERT(static_cast<size_t>(size) == RequiredNameBufferSize - 1);
        NN_UNUSED(size);

        NN_RESULT_SUCCESS;
    }

private:
    const BisPartitionId m_Id;
};

} // namespace

namespace detail {

Result MountBis(const char* name, BisPartitionId id, const char* rootPath) NN_NOEXCEPT
{
    auto mount = [=]() NN_NOEXCEPT -> Result
    {
        NN_UNUSED(rootPath); // TODO: 削除

        // マウント名チェック
        NN_RESULT_DO(CheckMountNameAcceptingReservedMountName(name));

        nn::sf::SharedPointer<nn::fssrv::sf::IFileSystemProxy> fileSystemProxy = GetFileSystemProxyServiceObject();
        nn::fssrv::sf::FspPath sfPath;
        sfPath.str[0] = '\0';

        nn::sf::SharedPointer<nn::fssrv::sf::IFileSystem> fileSystem;
        NN_RESULT_DO(fileSystemProxy->OpenBisFileSystem(&fileSystem, sfPath, static_cast<uint32_t>(id)));

        std::unique_ptr<BisCommonMountNameGenerator> mountNameGenerator(new BisCommonMountNameGenerator(id));
        NN_RESULT_THROW_UNLESS(mountNameGenerator.get() != nullptr, ResultAllocationMemoryFailedInBisA());

        std::unique_ptr<fsa::IFileSystem> fileSystemAbstract;
        fileSystemAbstract.reset(new detail::FileSystemServiceObjectAdapter(std::move(fileSystem)));
        NN_RESULT_THROW_UNLESS(fileSystemAbstract, ResultAllocationMemoryFailedInBisB());

        NN_RESULT_DO(fsa::Register(name, std::move(fileSystemAbstract), std::move(mountNameGenerator)));
        NN_RESULT_SUCCESS;
    };

    NN_FS_RESULT_DO(NN_DETAIL_FS_ACCESS_LOG_SYSTEM_MOUNT(mount(),
        name, NN_DETAIL_FS_ACCESS_LOG_FORMAT_MOUNTBIS(name, id, rootPath)));
    NN_DETAIL_FS_ACCESS_LOG_SYSTEM_FSACCESSOR_ENABLE(name);
    NN_RESULT_SUCCESS;
}

} // namespace detail

const char* GetBisMountName(BisPartitionId id)
{
    switch (id)
    {
    case BisPartitionId::BootPartition1Root:
    case BisPartitionId::BootPartition2Root:
    case BisPartitionId::UserDataRoot:
    case BisPartitionId::BootConfigAndPackage2Part1:
    case BisPartitionId::BootConfigAndPackage2Part2:
    case BisPartitionId::BootConfigAndPackage2Part3:
    case BisPartitionId::BootConfigAndPackage2Part4:
    case BisPartitionId::BootConfigAndPackage2Part5:
    case BisPartitionId::BootConfigAndPackage2Part6:
    case BisPartitionId::CalibrationBinary:
        NN_ABORT("The partition specified is not mountable.");
    case BisPartitionId::CalibrationFile:
        return detail::BisCalibrationFilePartitionMountName;
    case BisPartitionId::SafeMode:
        return detail::BisSafeModePartitionMountName;
    case BisPartitionId::User:
        return detail::BisUserPartitionMountName;
    case BisPartitionId::System:
        return detail::BisSystemPartitionMountName;
    default:
        NN_UNEXPECTED_DEFAULT;
    }
}

void SetBisRootForHost(BisPartitionId id, const char* rootPath) NN_NOEXCEPT
{
    size_t length = strnlen(rootPath, EntryNameLengthMax + 1);
    NN_FS_ABORT_UNLESS_WITH_RESULT(length <= EntryNameLengthMax, ResultTooLongPath(), "Bis root path is too long.\n");

    nn::fssrv::sf::FspPath sfPath;
    if (length > 0)
    {
        NN_FS_ABORT_UNLESS_RESULT_SUCCESS(FspPathPrintf(
            &sfPath,
            "%s%s",
            rootPath,
            (rootPath[length - 1] == '/') ? "" : "/"));
    }
    else
    {
        sfPath.str[0] = '\0';
    }

    nn::sf::SharedPointer<nn::fssrv::sf::IFileSystemProxy> fileSystemProxy = detail::GetFileSystemProxyServiceObject();
    NN_FS_ABORT_UNLESS_RESULT_SUCCESS(fileSystemProxy->SetBisRootForHost(static_cast<uint32_t>(id), sfPath));
}

Result MountBis(const char* name, BisPartitionId id) NN_NOEXCEPT
{
    return detail::MountBis(name, id, nullptr);
}

Result MountBis(BisPartitionId id, const char* rootPath) NN_NOEXCEPT
{
    return detail::MountBis(GetBisMountName(id), id, rootPath);
}

Result OpenBisPartition(std::unique_ptr<nn::fs::IStorage>* outValue, BisPartitionId id) NN_NOEXCEPT
{
    nn::sf::SharedPointer<nn::fssrv::sf::IFileSystemProxy> fileSystemProxy = detail::GetFileSystemProxyServiceObject();
    nn::sf::SharedPointer<nn::fssrv::sf::IStorage> storage;
    NN_FS_RESULT_DO(fileSystemProxy->OpenBisStorage(&storage, static_cast<uint32_t>(id)));

    std::unique_ptr<nn::fs::IStorage> storageAbstract;
    storageAbstract.reset(new detail::StorageServiceObjectAdapter(std::move(storage)));
    NN_FS_RESULT_THROW_UNLESS(storageAbstract, ResultAllocationMemoryFailedInBisC());

    *outValue = std::move(storageAbstract);
    NN_RESULT_SUCCESS;
}

Result InvalidateBisCache() NN_NOEXCEPT
{
    nn::sf::SharedPointer<nn::fssrv::sf::IFileSystemProxy> fileSystemProxy = detail::GetFileSystemProxyServiceObject();
    NN_FS_ABORT_UNLESS_RESULT_SUCCESS(fileSystemProxy->InvalidateBisCache());
    NN_RESULT_SUCCESS;
}

}}
