﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_SdkAssert.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/fs/fs_FileSystem.h>
#include <nn/fs/fs_FileSystemPrivate.h>
#include <nn/fs/fs_Result.h>
#include <nn/fs/fs_ResultPrivate.h>
#include <nn/fs/fs_SaveDataPrivate.h>
#include <nn/fs/fs_SaveDataTransaction.h>
#include <nn/fs/fs_Transaction.h>
#include <nn/fs/detail/fs_AccessLog.h>
#include "shim/fs_Library.h"
#include "fs_FileSystemAccessor.h"
#include "fs_MountUtility.h"
#include "fs_UserMountTable.h"
#include <nn/fs/detail/fs_ResultHandlingUtility.h>
#include "../fs/shim/fs_FileSystemProxyServiceObject.h"

namespace nn { namespace fs {

Result CreateFile(const char* path, int64_t size) NN_NOEXCEPT
{
    return CreateFile(path, size, 0);
}

Result DeleteFile(const char* path) NN_NOEXCEPT
{
    detail::FileSystemAccessor* system;
    NN_FS_RESULT_DO(NN_DETAIL_FS_ACCESS_LOG_FILESYSTEM_FOR_FINDFILESYSTEM(
        detail::FindFileSystem(&system, path), NN_DETAIL_FS_ACCESS_LOG_FORMAT_PATH, path));

    NN_FS_RESULT_DO(NN_DETAIL_FS_ACCESS_LOG_FILESYSTEM(system->DeleteFile(detail::GetSubPath(path)), nullptr, system, NN_DETAIL_FS_ACCESS_LOG_FORMAT_PATH, path));
    NN_RESULT_SUCCESS;
}

Result CreateDirectory(const char* path) NN_NOEXCEPT
{
    detail::FileSystemAccessor* system;
    NN_FS_RESULT_DO(NN_DETAIL_FS_ACCESS_LOG_FILESYSTEM_FOR_FINDFILESYSTEM(
        detail::FindFileSystem(&system, path), NN_DETAIL_FS_ACCESS_LOG_FORMAT_PATH, path));

    NN_FS_RESULT_DO(NN_DETAIL_FS_ACCESS_LOG_FILESYSTEM(system->CreateDirectory(detail::GetSubPath(path)), nullptr, system, NN_DETAIL_FS_ACCESS_LOG_FORMAT_PATH, path));
    NN_RESULT_SUCCESS;
}

Result DeleteDirectory(const char* path) NN_NOEXCEPT
{
    detail::FileSystemAccessor* system;
    NN_FS_RESULT_DO(NN_DETAIL_FS_ACCESS_LOG_FILESYSTEM_FOR_FINDFILESYSTEM(
        detail::FindFileSystem(&system, path), NN_DETAIL_FS_ACCESS_LOG_FORMAT_PATH, path));

    NN_FS_RESULT_DO(NN_DETAIL_FS_ACCESS_LOG_FILESYSTEM(system->DeleteDirectory(detail::GetSubPath(path)), nullptr, system, NN_DETAIL_FS_ACCESS_LOG_FORMAT_PATH, path));
    NN_RESULT_SUCCESS;
}

Result DeleteDirectoryRecursively(const char* path) NN_NOEXCEPT
{
    detail::FileSystemAccessor* system;
    NN_FS_RESULT_DO(NN_DETAIL_FS_ACCESS_LOG_FILESYSTEM_FOR_FINDFILESYSTEM(
        detail::FindFileSystem(&system, path), NN_DETAIL_FS_ACCESS_LOG_FORMAT_PATH, path));

    NN_FS_RESULT_DO(NN_DETAIL_FS_ACCESS_LOG_FILESYSTEM(system->DeleteDirectoryRecursively(detail::GetSubPath(path)), nullptr, system, NN_DETAIL_FS_ACCESS_LOG_FORMAT_PATH, path));
    NN_RESULT_SUCCESS;
}

Result CleanDirectoryRecursively(const char* path) NN_NOEXCEPT
{
    detail::FileSystemAccessor* system;
    NN_FS_RESULT_DO(NN_DETAIL_FS_ACCESS_LOG_FILESYSTEM_FOR_FINDFILESYSTEM(
        detail::FindFileSystem(&system, path), NN_DETAIL_FS_ACCESS_LOG_FORMAT_PATH, path));

    NN_FS_RESULT_DO(NN_DETAIL_FS_ACCESS_LOG_FILESYSTEM(system->CleanDirectoryRecursively(detail::GetSubPath(path)), nullptr, system, NN_DETAIL_FS_ACCESS_LOG_FORMAT_PATH, path));
    NN_RESULT_SUCCESS;
}

Result RenameFile(const char* currentPath, const char* newPath) NN_NOEXCEPT
{
    detail::FileSystemAccessor* pCurrentSystem;
    detail::FileSystemAccessor* pNewSystem;
    NN_FS_RESULT_DO(NN_DETAIL_FS_ACCESS_LOG_FILESYSTEM_FOR_FINDFILESYSTEM(
        detail::FindFileSystem(&pCurrentSystem, currentPath), NN_DETAIL_FS_ACCESS_LOG_FORMAT_RENAME, currentPath, newPath));
    NN_FS_RESULT_DO(NN_DETAIL_FS_ACCESS_LOG_FILESYSTEM_FOR_FINDFILESYSTEM(
        detail::FindFileSystem(&pNewSystem, newPath), NN_DETAIL_FS_ACCESS_LOG_FORMAT_RENAME, currentPath, newPath));

    auto rename = [=]() NN_NOEXCEPT -> Result
    {
        if( pCurrentSystem != pNewSystem )
        {
            NN_RESULT_DO(nn::fs::ResultRenameToOtherFileSystem());
        }
        NN_RESULT_DO(pCurrentSystem->RenameFile(detail::GetSubPath(currentPath), detail::GetSubPath(newPath)));
        NN_RESULT_SUCCESS;
    };

    NN_FS_RESULT_DO(NN_DETAIL_FS_ACCESS_LOG_FILESYSTEM(rename(), nullptr, pCurrentSystem, NN_DETAIL_FS_ACCESS_LOG_FORMAT_RENAME, currentPath, newPath));
    NN_RESULT_SUCCESS;
}

Result RenameDirectory(const char* currentPath, const char* newPath) NN_NOEXCEPT
{
    detail::FileSystemAccessor* pCurrentSystem;
    detail::FileSystemAccessor* pNewSystem;
    NN_FS_RESULT_DO(NN_DETAIL_FS_ACCESS_LOG_FILESYSTEM_FOR_FINDFILESYSTEM(
        detail::FindFileSystem(&pCurrentSystem, currentPath), NN_DETAIL_FS_ACCESS_LOG_FORMAT_RENAME, currentPath, newPath));
    NN_FS_RESULT_DO(NN_DETAIL_FS_ACCESS_LOG_FILESYSTEM_FOR_FINDFILESYSTEM(
        detail::FindFileSystem(&pNewSystem, newPath), NN_DETAIL_FS_ACCESS_LOG_FORMAT_RENAME, currentPath, newPath));

    auto rename = [=]() NN_NOEXCEPT -> Result
    {
        if( pCurrentSystem != pNewSystem )
        {
            NN_RESULT_DO(nn::fs::ResultRenameToOtherFileSystem());
        }

        NN_RESULT_DO(pCurrentSystem->RenameDirectory(detail::GetSubPath(currentPath), detail::GetSubPath(newPath)));
        NN_RESULT_SUCCESS;
    };

    NN_FS_RESULT_DO(NN_DETAIL_FS_ACCESS_LOG_FILESYSTEM(rename(), nullptr, pCurrentSystem, NN_DETAIL_FS_ACCESS_LOG_FORMAT_RENAME, currentPath, newPath));
    NN_RESULT_SUCCESS;
}

Result GetEntryType(DirectoryEntryType* outValue, const char* path) NN_NOEXCEPT
{
    detail::FileSystemAccessor* system;
    NN_FS_RESULT_DO(NN_DETAIL_FS_ACCESS_LOG_FILESYSTEM_FOR_FINDFILESYSTEM(
        detail::FindFileSystem(&system, path), NN_DETAIL_FS_ACCESS_LOG_FORMAT_PATH, path));

    NN_FS_RESULT_DO(NN_DETAIL_FS_ACCESS_LOG_FILESYSTEM(system->GetEntryType(outValue, detail::GetSubPath(path)), nullptr, system, NN_DETAIL_FS_ACCESS_LOG_FORMAT_PATH, path));
    NN_RESULT_SUCCESS;
}

Result OpenFile(FileHandle* outValue, const char* path, int mode) NN_NOEXCEPT
{
    detail::FileSystemAccessor* system;
    NN_FS_RESULT_DO(NN_DETAIL_FS_ACCESS_LOG_FILESYSTEM_FOR_FINDFILESYSTEM(
        detail::FindFileSystem(&system, path), NN_DETAIL_FS_ACCESS_LOG_FORMAT_PATH NN_DETAIL_FS_ACCESS_LOG_FORMAT_OPENMODE, path, mode));

    std::unique_ptr<detail::FileAccessor> file;

    auto openImpl = [&]() NN_NOEXCEPT -> Result
    {
        NN_RESULT_THROW_UNLESS(outValue != nullptr, ResultNullptrArgument());
        NN_RESULT_DO(system->OpenFile(&file, detail::GetSubPath(path), static_cast<OpenMode>(mode)));
        NN_RESULT_SUCCESS;
    };

    NN_FS_RESULT_DO(NN_DETAIL_FS_ACCESS_LOG_FILESYSTEM(openImpl(), file.get(), system, NN_DETAIL_FS_ACCESS_LOG_FORMAT_PATH NN_DETAIL_FS_ACCESS_LOG_FORMAT_OPENMODE, path, mode));
    outValue->handle = file.release();
    NN_RESULT_SUCCESS;
}

Result OpenFile(FileHandle* outValue, std::unique_ptr<fsa::IFile>&& file, int mode) NN_NOEXCEPT
{
    NN_FS_RESULT_THROW_UNLESS(outValue != nullptr, ResultNullptrArgument());

    std::unique_ptr<detail::FileAccessor> fileAccessor(new detail::FileAccessor(std::move(file), nullptr, mode));
    NN_FS_RESULT_THROW_UNLESS(fileAccessor != nullptr, nn::fs::ResultAllocationMemoryFailedNew());
    outValue->handle = fileAccessor.release();

    NN_RESULT_SUCCESS;
}

Result OpenDirectory(DirectoryHandle* outValue, const char* path, int mode) NN_NOEXCEPT
{
    detail::FileSystemAccessor* system;
    NN_FS_RESULT_DO(NN_DETAIL_FS_ACCESS_LOG_FILESYSTEM_FOR_FINDFILESYSTEM(
        detail::FindFileSystem(&system, path), NN_DETAIL_FS_ACCESS_LOG_FORMAT_PATH, path));

    std::unique_ptr<detail::DirectoryAccessor> directory;

    auto openImpl = [&]() NN_NOEXCEPT->Result
    {
        NN_RESULT_THROW_UNLESS(outValue != nullptr, ResultNullptrArgument());
        NN_RESULT_DO(system->OpenDirectory(&directory, detail::GetSubPath(path), static_cast<OpenDirectoryMode>(mode)));
        NN_RESULT_SUCCESS;
    };

    NN_FS_RESULT_DO(NN_DETAIL_FS_ACCESS_LOG_FILESYSTEM(openImpl(), directory.get(), system, NN_DETAIL_FS_ACCESS_LOG_FORMAT_PATH, path));
    outValue->handle = directory.release();
    NN_RESULT_SUCCESS;
}

namespace {
    Result CommitImpl(const char* name, const char* commitApiName) NN_NOEXCEPT
    {
        NN_UNUSED(commitApiName);

        detail::FileSystemAccessor* system;
        NN_FS_RESULT_DO(NN_DETAIL_FS_ACCESS_LOG_FILESYSTEM_FOR_FINDFILESYSTEM_EXPLICIT_FUNCTION_NAME(
            detail::Find(&system, name), commitApiName, NN_DETAIL_FS_ACCESS_LOG_FORMAT_MOUNT, name));

        NN_FS_RESULT_DO(NN_DETAIL_FS_ACCESS_LOG_FILESYSTEM_EXPLICIT_FUNCTION_NAME(system->Commit(), nullptr, system, commitApiName, NN_DETAIL_FS_ACCESS_LOG_FORMAT_MOUNT, name));
        NN_RESULT_SUCCESS;
    }
}

Result Commit(const char* name) NN_NOEXCEPT
{
    return CommitImpl(name, NN_CURRENT_FUNCTION_NAME);
}

Result CommitSaveData(const char* name) NN_NOEXCEPT
{
    return CommitImpl(name, NN_CURRENT_FUNCTION_NAME);
}

Result Commit(const char* const* nameArray, int nameCount) NN_NOEXCEPT
{
    NN_FS_RESULT_THROW_UNLESS(nameArray != nullptr, ResultNullptrArgument());
    NN_FS_RESULT_THROW_UNLESS(nameCount >= 0, ResultOutOfRange());
    NN_FS_RESULT_THROW_UNLESS(nameCount <= detail::CommittableCountMax, ResultOutOfRange());

    if( nameCount >= 1 )
    {
        // マウント名の重複チェック
        for( int i = 0; i < nameCount - 1; ++i )
        {
            for( int j = i + 1; j < nameCount; ++j )
            {
                if( std::strcmp(nameArray[i], nameArray[j]) == 0 )
                {
                    NN_RESULT_THROW(ResultMountNameAlreadyExists());
                }
            }
        }

        auto fileSystemProxy = detail::GetFileSystemProxyServiceObject();

        nn::sf::SharedPointer<fssrv::sf::IMultiCommitManager> pCommitManager;
        NN_RESULT_DO(fileSystemProxy->OpenMultiCommitManager(&pCommitManager));

        for( int i = 0; i < nameCount; ++i )
        {
            const auto name = nameArray[i];

            // マウント名から FileSystemAccessor -> FileSystemServiceObjectAdapter -> nn::sf::SharedPointer<nn::fssrv::sf::IFileSystem> を掘り起こす

            detail::FileSystemAccessor* pAccessor;
            NN_FS_RESULT_DO(NN_DETAIL_FS_ACCESS_LOG_FILESYSTEM_FOR_FINDFILESYSTEM_EXPLICIT_FUNCTION_NAME(
                detail::Find(&pAccessor, name), NN_CURRENT_FUNCTION_NAME, NN_DETAIL_FS_ACCESS_LOG_FORMAT_MOUNT, name));

            NN_SDK_ASSERT(pAccessor != nullptr);

            nn::sf::SharedPointer<nn::fssrv::sf::IFileSystem> pFileSystem(pAccessor->GetMultiCommitTarget());
            if( pFileSystem != nullptr )
            {
                // ToDo: AccessLog 対応
                NN_RESULT_DO(pCommitManager->Add(pFileSystem));
            }
            else
            {
                NN_RESULT_THROW(ResultUnsupportedOperation());
            }
        }

        if( nameCount == 1 )
        {
            NN_RESULT_DO(CommitImpl(nameArray[0], NN_CURRENT_FUNCTION_NAME));
        }
        else
        {
            // ToDo: AccessLog 対応
            NN_RESULT_DO(pCommitManager->Commit());
        }
    }

    NN_RESULT_SUCCESS;
}

}}
