﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdio>
#include <cstring>
#include <nn/fs/fs_File.h>
#include <nn/fs/fs_Result.h>
#include <nn/fs/fs_QueryRange.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/fs/detail/fs_AccessLog.h>
#include "fs_FileAccessor.h"
#include "fs_FileSystemAccessor.h"
#include <nn/fs/detail/fs_IFileDataCache.h>
#include <nn/fs/detail/fs_ResultHandlingUtility.h>
#include "shim/fs_FileDataCacheDetail.h"

namespace nn { namespace fs {

namespace {
    detail::FileAccessor& Get(FileHandle handle) NN_NOEXCEPT
    {
        return *reinterpret_cast<detail::FileAccessor*>(handle.handle);
    }
}


void CloseFile(FileHandle handle) NN_NOEXCEPT
{
    NN_DETAIL_FS_ACCESS_LOG((delete &Get(handle), nn::ResultSuccess()), handle, "");
}

Result ReadFile(FileHandle handle, int64_t offset, void *buffer, size_t size, const ReadOption& option) NN_NOEXCEPT
{
    size_t readSize;
    NN_FS_RESULT_DO(ReadFile(&readSize, handle, offset, buffer, size, option));
    NN_FS_RESULT_THROW_UNLESS(readSize == size, ResultOutOfRange());
    NN_RESULT_SUCCESS;
}

Result ReadFile(FileHandle handle, int64_t offset, void *buffer, size_t size) NN_NOEXCEPT
{
    return ReadFile(handle, offset, buffer, size, ReadOption());
}

Result ReadFile(size_t* outValue, FileHandle handle, int64_t offset, void *buffer, size_t size, const ReadOption& option) NN_NOEXCEPT
{
    // ReadFile のログは FileAccessor 内で生成する
    NN_FS_RESULT_DO(Get(handle).Read(outValue, offset, buffer, size, option));
    NN_RESULT_SUCCESS;
}

Result ReadFile(size_t* outValue, FileHandle handle, int64_t offset, void *buffer, size_t size) NN_NOEXCEPT
{
    return ReadFile(outValue, handle, offset, buffer, size, ReadOption());
}

Result WriteFile(FileHandle handle, int64_t offset, const void* buffer, size_t size, const WriteOption& option) NN_NOEXCEPT
{
    NN_FS_RESULT_DO(NN_DETAIL_FS_ACCESS_LOG(Get(handle).Write(offset, buffer, size, option), handle,
        (((option.flags & WriteOptionFlag_Flush) != 0) ? NN_DETAIL_FS_ACCESS_LOG_FORMAT_WRITEFILE_WITH_FLUSH : NN_DETAIL_FS_ACCESS_LOG_FORMAT_OFFSET_AND_SIZE), offset, size));
    NN_RESULT_SUCCESS;
}

Result FlushFile(FileHandle handle) NN_NOEXCEPT
{
    NN_FS_RESULT_DO(NN_DETAIL_FS_ACCESS_LOG(Get(handle).Flush(), handle, ""));
    NN_RESULT_SUCCESS;
}

Result SetFileSize(FileHandle handle, int64_t size) NN_NOEXCEPT
{
    NN_FS_RESULT_DO(NN_DETAIL_FS_ACCESS_LOG(Get(handle).SetSize(size), handle, NN_DETAIL_FS_ACCESS_LOG_FORMAT_SIZE, size));
    NN_RESULT_SUCCESS;
}

Result GetFileSize(int64_t* outValue, FileHandle handle) NN_NOEXCEPT
{
    NN_FS_RESULT_DO(Get(handle).GetSize(outValue));
    NN_RESULT_SUCCESS;
}

int GetFileOpenMode(FileHandle handle) NN_NOEXCEPT
{
    return Get(handle).GetOpenMode();
}

Result QueryRange(QueryRangeInfo* outValue, FileHandle handle, int64_t offset, int64_t size) NN_NOEXCEPT
{
    NN_FS_RESULT_DO(Get(handle).OperateRange(
        outValue,
        sizeof(QueryRangeInfo),
        OperationId::QueryRange,
        offset,
        size,
        nullptr,
        0));
    NN_RESULT_SUCCESS;
}

}}
