﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <mutex>
#include <cstring>
#include <nn/os/os_Mutex.h>
#include <nn/nn_Abort.h>
#include <nn/nn_SdkAssert.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/fs/fs_Result.h>
#include <nn/fs/fs_ResultPrivate.h>
#include <nn/fs/detail/fs_ResultHandlingUtility.h>
#include "fs_MountTable.h"

namespace nn { namespace fs { namespace detail {

    namespace
    {
        bool Matches(const FileSystemAccessor& accessor, const char* name)
        {
            return std::strncmp(accessor.GetName(), name, MountName::LENGTH) == 0;
        }
    }

    Result MountTable::Mount(std::unique_ptr<FileSystemAccessor>&& fileSystem) NN_NOEXCEPT
    {
        std::lock_guard<os::Mutex> scopedLock(m_ListMutex);

        NN_RESULT_THROW_UNLESS(CanAcceptMountName(fileSystem->GetName()), ResultMountNameAlreadyExists());

        m_List.push_back(*fileSystem.release());

        NN_RESULT_SUCCESS;
    }

    void MountTable::Unmount(const char* name) NN_NOEXCEPT
    {
        std::lock_guard<os::Mutex> scopedLock(m_ListMutex);

        for (auto iter = m_List.cbegin(); iter != m_List.cend(); iter++)
        {
            if (Matches(*iter, name))
            {
                auto p = iter.operator->();
                m_List.erase(iter);
                delete p;
                return;
            }
        }

        NN_FS_ABORT_UNLESS_WITH_RESULT(false, ResultNotMounted(), "%s is not mounted.", name);
    }

    Result MountTable::Find(FileSystemAccessor** outValue, const char* name) NN_NOEXCEPT
    {
        std::lock_guard<os::Mutex> scopedLock(m_ListMutex);

        for (auto iter = m_List.begin(); iter != m_List.end(); iter++)
        {
            if (Matches(*iter, name))
            {
                *outValue = iter.operator->();
                NN_RESULT_SUCCESS;
            }
        }

        return ResultNotMounted();
    }

    bool MountTable::CanAcceptMountName(const char* name)
    {
        NN_SDK_ASSERT(m_ListMutex.IsLockedByCurrentThread());

        for (const auto& node : m_List)
        {
            if (Matches(node, name))
            {
                return false;
            }
        }

        return true;
    }
}}}
