﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_SdkAssert.h>
#include <nn/nn_Abort.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/fs/fs_Result.h>
#include <nn/fs/fsa/fs_IFile.h>
#include <nn/fs/detail/fs_AccessLog.h>
#include <nn/fs/detail/fs_IFileDataCache.h>
#include <nn/fs/detail/fs_ResultHandlingUtility.h>
#include "fs_FileAccessor.h"
#include "fs_FileSystemAccessor.h"
#include "fs_ScopedSetter.h"
#include "shim/fs_FileDataCacheAccessor.h"
#include "shim/fs_FileDataCacheDetail.h"
#include "shim/fs_PathBasedFileDataCache.h"

namespace nn { namespace fs { namespace detail {

    FileAccessor::FileAccessor(std::unique_ptr<fsa::IFile>&& file, FileSystemAccessor* pParent, int mode) NN_NOEXCEPT
        : m_Impl(std::move(file)), m_pParent(pParent), m_WriteState(WriteStateNon), m_LastWriteResult(ResultSuccess()), m_OpenMode(mode)
    {
    }

    FileAccessor::~FileAccessor() NN_NOEXCEPT
    {
        if (m_LastWriteResult.IsSuccess())
        {
            NN_FS_ABORT_UNLESS_WITH_RESULT(m_WriteState != WriteStateNeedsFlush, ResultNeedFlush());
        }
        m_Impl.reset();

        if (m_pParent != nullptr)
        {
            m_pParent->NotifyCloseFile(this);
        }
    }

    Result FileAccessor::ReadWithCacheAccessLog(size_t* outValue, int64_t offset, void* buffer, size_t size, const ReadOption& option, bool usePathBasedFileDataCache, bool useFileDataCache) NN_NOEXCEPT
    {
        FileHandle handle = { this }; // TORIAEZU

        detail::FileDataCacheAccessResult cacheAccessResult;
        Result result = fs::ResultUnexpected();
        bool cacheUsed = false;

        NN_DETAIL_FS_ACCESS_LOG_RECORD_TIME(start);

        if (usePathBasedFileDataCache)
        {
            result = ReadViaPathBasedFileDataCache(m_Impl.get(), GetOpenMode(), m_pParent, *m_FilePathHash, m_FilePathHashIndex, outValue, offset, buffer, size, option, &cacheAccessResult);
            cacheUsed = true;
        }
        else if (useFileDataCache)
        {
            GlobalFileDataCacheAccessorReadableScopedPointer fileDataCacheAccessor;
            if (TryGetGlobalFileDataCacheAccessor(&fileDataCacheAccessor))
            {
                result = fileDataCacheAccessor->Read(m_Impl.get(), outValue, offset, buffer, size, option, &cacheAccessResult);
                cacheUsed = true;
            }
        }

        NN_DETAIL_FS_ACCESS_LOG_RECORD_TIME(end);

        if (cacheUsed)
        {
            if (cacheAccessResult.IsFileDataCacheUsed())
            {
                if (cacheAccessResult.ExceededMaxCacheFetchedRegionCount())
                {
                    return NN_DETAIL_FS_ACCESS_LOG_EXPLICIT_RESULT_AND_TIME(
                        result, start, end, handle, "ReadFile", NN_DETAIL_FS_ACCESS_LOG_FORMAT_OFFSET_SIZE_CACHE_ACCESS_RESULT_GE_9(offset, size, cacheAccessResult));
                }
                else
                {
                    switch (cacheAccessResult.GetCacheFetchedRegionCount())
                    {
                    case 0:
                        return NN_DETAIL_FS_ACCESS_LOG_EXPLICIT_RESULT_AND_TIME(
                            result, start, end, handle, "ReadFile", NN_DETAIL_FS_ACCESS_LOG_FORMAT_OFFSET_SIZE_CACHE_ACCESS_RESULT_0(offset, size, cacheAccessResult));
                    case 1:
                        return NN_DETAIL_FS_ACCESS_LOG_EXPLICIT_RESULT_AND_TIME(
                            result, start, end, handle, "ReadFile", NN_DETAIL_FS_ACCESS_LOG_FORMAT_OFFSET_SIZE_CACHE_ACCESS_RESULT_1(offset, size, cacheAccessResult));
                    case 2:
                        return NN_DETAIL_FS_ACCESS_LOG_EXPLICIT_RESULT_AND_TIME(
                            result, start, end, handle, "ReadFile", NN_DETAIL_FS_ACCESS_LOG_FORMAT_OFFSET_SIZE_CACHE_ACCESS_RESULT_2(offset, size, cacheAccessResult));
                    case 3:
                        return NN_DETAIL_FS_ACCESS_LOG_EXPLICIT_RESULT_AND_TIME(
                            result, start, end, handle, "ReadFile", NN_DETAIL_FS_ACCESS_LOG_FORMAT_OFFSET_SIZE_CACHE_ACCESS_RESULT_3(offset, size, cacheAccessResult));
                    case 4:
                        return NN_DETAIL_FS_ACCESS_LOG_EXPLICIT_RESULT_AND_TIME(
                            result, start, end, handle, "ReadFile", NN_DETAIL_FS_ACCESS_LOG_FORMAT_OFFSET_SIZE_CACHE_ACCESS_RESULT_4(offset, size, cacheAccessResult));
                    case 5:
                        return NN_DETAIL_FS_ACCESS_LOG_EXPLICIT_RESULT_AND_TIME(
                            result, start, end, handle, "ReadFile", NN_DETAIL_FS_ACCESS_LOG_FORMAT_OFFSET_SIZE_CACHE_ACCESS_RESULT_5(offset, size, cacheAccessResult));
                    case 6:
                        return NN_DETAIL_FS_ACCESS_LOG_EXPLICIT_RESULT_AND_TIME(
                            result, start, end, handle, "ReadFile", NN_DETAIL_FS_ACCESS_LOG_FORMAT_OFFSET_SIZE_CACHE_ACCESS_RESULT_6(offset, size, cacheAccessResult));
                    case 7:
                        return NN_DETAIL_FS_ACCESS_LOG_EXPLICIT_RESULT_AND_TIME(
                            result, start, end, handle, "ReadFile", NN_DETAIL_FS_ACCESS_LOG_FORMAT_OFFSET_SIZE_CACHE_ACCESS_RESULT_7(offset, size, cacheAccessResult));
                    case 8:
                        return NN_DETAIL_FS_ACCESS_LOG_EXPLICIT_RESULT_AND_TIME(
                            result, start, end, handle, "ReadFile", NN_DETAIL_FS_ACCESS_LOG_FORMAT_OFFSET_SIZE_CACHE_ACCESS_RESULT_8(offset, size, cacheAccessResult));
                    default:
                        NN_UNEXPECTED_DEFAULT;
                    }
                }
            }
            else
            {
                return NN_DETAIL_FS_ACCESS_LOG_EXPLICIT_RESULT_AND_TIME(result, start, end, handle, "ReadFile", NN_DETAIL_FS_ACCESS_LOG_FORMAT_OFFSET_AND_SIZE, offset, size);
            }
        }
        else
        {
            return NN_DETAIL_FS_ACCESS_LOG_EXPLICIT_FUNCTION_NAME(
                m_Impl->Read(outValue, offset, buffer, size, option), handle, "ReadFile", NN_DETAIL_FS_ACCESS_LOG_FORMAT_OFFSET_AND_SIZE, offset, size);
        }
    }

    Result FileAccessor::ReadWithoutCacheAccessLog(size_t* outValue, int64_t offset, void* buffer, size_t size, const ReadOption& option) NN_NOEXCEPT
    {
        return m_Impl->Read(outValue, offset, buffer, size, option);
    }

    Result FileAccessor::Read(size_t* outValue, int64_t offset, void* buffer, size_t size, const ReadOption& option) NN_NOEXCEPT
    {
        FileHandle handle = { this }; // TORIAEZU

        NN_RESULT_THROW_UNLESS(m_LastWriteResult.IsSuccess(), NN_DETAIL_FS_ACCESS_LOG_EXPLICIT_FUNCTION_NAME(m_LastWriteResult, handle, "ReadFile", NN_DETAIL_FS_ACCESS_LOG_FORMAT_OFFSET_AND_SIZE, offset, size));

        const bool usePathBasedFileDataCache = m_pParent != nullptr && m_FilePathHash != nullptr;
        const bool useFileDataCache = detail::IsGlobalFileDataCacheEnabled() && m_pParent != nullptr && m_pParent->IsFileDataCacheAttachable();

        if (usePathBasedFileDataCache || useFileDataCache)
        {
            return ReadWithCacheAccessLog(outValue, offset, buffer, size, option, usePathBasedFileDataCache, useFileDataCache);
        }
        else
        {
            return NN_DETAIL_FS_ACCESS_LOG_EXPLICIT_FUNCTION_NAME(
                ReadWithoutCacheAccessLog(outValue, offset, buffer, size, option),
                handle, "ReadFile", NN_DETAIL_FS_ACCESS_LOG_FORMAT_OFFSET_AND_SIZE, offset, size);
        }
    }

    Result FileAccessor::Write(int64_t offset, const void* buffer, size_t size, const WriteOption& option) NN_NOEXCEPT
    {
        NN_RESULT_DO(m_LastWriteResult);

        auto setter = MakeScopedSetter(m_WriteState, WriteStateFailed);

        if (m_FilePathHash != nullptr)
        {
            NN_RESULT_DO(UpdateLastResult(WriteViaPathBasedFileDataCache(m_Impl.get(), GetOpenMode(), m_pParent, *m_FilePathHash, m_FilePathHashIndex, offset, buffer, size, option)));
        }
        else
        {
            NN_RESULT_DO(UpdateLastResult(m_Impl->Write(offset, buffer, size, option)));
        }

        setter.Set( (option.flags & WriteOptionFlag_Flush) != 0 ? WriteStateNon : WriteStateNeedsFlush );

        NN_RESULT_SUCCESS;
    }

    Result FileAccessor::Flush() NN_NOEXCEPT
    {
        NN_RESULT_DO(m_LastWriteResult);

        auto setter = MakeScopedSetter(m_WriteState, WriteStateFailed);

        NN_RESULT_DO(UpdateLastResult(m_Impl->Flush()));

        setter.Set(WriteStateNon);

        NN_RESULT_SUCCESS;
    }

    Result FileAccessor::SetSize(int64_t size) NN_NOEXCEPT
    {
        NN_RESULT_DO(m_LastWriteResult);

        WriteState lastWriteState = m_WriteState;
        auto setter = MakeScopedSetter(m_WriteState, WriteStateFailed);

        NN_RESULT_DO(UpdateLastResult(m_Impl->SetSize(size)));

        if (m_FilePathHash != nullptr)
        {
            InvalidatePathBasedFileDataCacheEntry(m_pParent, *m_FilePathHash, m_FilePathHashIndex);
        }

        setter.Set(lastWriteState);

        NN_RESULT_SUCCESS;
    }

    Result FileAccessor::GetSize(int64_t *outValue) NN_NOEXCEPT
    {
        NN_RESULT_DO(m_LastWriteResult);

        return m_Impl->GetSize(outValue);
    }

    int FileAccessor::GetOpenMode() const NN_NOEXCEPT
    {
        return m_OpenMode;
    }

    WriteState FileAccessor::GetWriteState() NN_NOEXCEPT
    {
        return m_WriteState;
    }

    Result FileAccessor::OperateRange(
        void* outBuffer,
        size_t outBufferSize,
        OperationId operationId,
        int64_t offset,
        int64_t size,
        const void* inBuffer,
        size_t inBufferSize) NN_NOEXCEPT
    {
        return m_Impl->OperateRange(
            outBuffer, outBufferSize, operationId, offset, size, inBuffer, inBufferSize
        );
    }

    Result FileAccessor::UpdateLastResult(Result result) NN_NOEXCEPT
    {
        if( nn::fs::ResultUsableSpaceNotEnough::Includes(result) )
        {
            return result;
        }
        m_LastWriteResult = result;
        return m_LastWriteResult;
    }

    void FileAccessor::SetFilePathHash(std::unique_ptr<FilePathHash>&& filePathHash, int index) NN_NOEXCEPT
    {
        m_FilePathHash = std::move(filePathHash);
        m_FilePathHashIndex = index;
    }

}}}
