﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <memory>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_FormatString.h>

#include <nn/fs/fs_DbmRomTypes.h>
#include <nn/fs/fs_DbmHierarchicalRomFileTable.h>
#include <nn/fs/detail/fs_Log.h>

namespace nn { namespace fs {

NN_DEFINE_STATIC_CONSTANT(
    const HierarchicalRomFileTable::StoragePosition HierarchicalRomFileTable::PositionNone
);

NN_DEFINE_STATIC_CONSTANT(
    const HierarchicalRomFileTable::StoragePosition HierarchicalRomFileTable::PositionRootDirectory
);

NN_DEFINE_STATIC_CONSTANT(const uint32_t HierarchicalRomFileTable::DirectoryCountSystemReserved);

NN_DEFINE_STATIC_CONSTANT(const uint32_t HierarchicalRomFileTable::MaxKeyLength);

/*!
    @brief ディレクトリエントリー用ストレージに必要なサイズを取得します。

           1 エントリーに対して大きめなサイズを要求します。

           初回ファイルツリー作成(メタデータ構築)時に使用します。

    @param[in] countDirectoryEntry ディレクトリエントリー数

    @return ディレクトリエントリー用ストレージに必要なサイズ
*/
int64_t HierarchicalRomFileTable::QueryDirectoryEntryStorageSize(
             uint32_t countDirectoryEntry
         ) NN_NOEXCEPT
{
    return DirectoryEntryMapTable::QueryKeyValueStorageSize(
               countDirectoryEntry + DirectoryCountSystemReserved
           ) + (countDirectoryEntry + DirectoryCountSystemReserved)
                * (RomPathTool::MaxPathLength + 1) * sizeof(RomPathChar);
}

/*!
    @brief ファイルエントリー用ストレージに必要なサイズを取得します。

           1 エントリーに対して大きめなサイズを要求します。

           初回ファイルツリー作成(メタデータ構築)時に使用します。

    @param[in] countFileEntry ファイルエントリー数

    @return ファイルエントリー用ストレージに必要なサイズ
*/
int64_t HierarchicalRomFileTable::QueryFileEntryStorageSize(
             uint32_t countFileEntry
         ) NN_NOEXCEPT
{
    return FileEntryMapTable::QueryKeyValueStorageSize(countFileEntry)
        + countFileEntry * (RomPathTool::MaxPathLength + 1) * sizeof(RomPathChar);
}

/*!
    @brief ディレクトリエントリーバケット用ストレージに必要なサイズを取得します。

    @param[in] countDirectoryBucket バケット数

    @return ディレクトリエントリーバケット用ストレージに必要なサイズ
*/
int64_t HierarchicalRomFileTable::QueryDirectoryEntryBucketStorageSize(
             int64_t countDirectoryBucket
         ) NN_NOEXCEPT
{
    return DirectoryEntryMapTable::QueryBucketStorageSize(countDirectoryBucket);
}

/*!
    @brief ファイルエントリーバケット用ストレージに必要なサイズを取得します。

    @param[in] countFileBucket バケット数

    @return ファイルエントリーバケット用ストレージに必要なサイズを取得します。
*/
int64_t HierarchicalRomFileTable::QueryFileEntryBucketStorageSize(
             int64_t countFileBucket
         ) NN_NOEXCEPT
{
    return FileEntryMapTable::QueryBucketStorageSize(countFileBucket);
}

/*!
    @brief ファイル管理テーブルをフォーマットします。

           バケットストレージを初期化します。使用する前に一度呼び出します。

    @param[in] directoryBucket      ディレクトリエントリーバケット用ストレージ
    @param[in] fileBucket           ファイルエントリーバケット用ストレージ

    @retval ResultSuccess       成功しました。
    @retval ResultOutOfRange    ストレージの範囲外を書き込もうとしました。
    @retval それ以外            ストレージの書き込みで失敗しました。

    @return 関数の処理結果を返します。
*/
Result HierarchicalRomFileTable::Format(
           SubStorage directoryBucket,
           SubStorage fileBucket
       ) NN_NOEXCEPT
{
    int64_t sizeDirectoryBucket = 0;
    NN_RESULT_DO(directoryBucket.GetSize(&sizeDirectoryBucket));

    // ディレクトリエントリー用テーブルをフォーマットします。
    NN_RESULT_DO(
        DirectoryEntryMapTable::Format(
            directoryBucket,
            DirectoryEntryMapTable::QueryBucketCount(sizeDirectoryBucket)
        )
    );

    int64_t sizeFileBucket = 0;
    NN_RESULT_DO(fileBucket.GetSize(&sizeFileBucket));

    // ファイルエントリー用テーブルをフォーマットします。
    NN_RESULT_DO(
        FileEntryMapTable::Format(
            fileBucket,
            FileEntryMapTable::QueryBucketCount(sizeFileBucket)
        )
    );

    NN_RESULT_SUCCESS;
}

/*!
    @brief コンストラクタです。
*/
HierarchicalRomFileTable::HierarchicalRomFileTable() NN_NOEXCEPT
{
}

/*!
    @brief ストレージをマウントします。

    @param[in] directoryBucket      ディレクトリバケット用ストレージ
    @param[in] directoryEntry       ディレクトリエントリー用ストレージ
    @param[in] fileBucket           ファイルエントリーバケット用ストレージ
    @param[in] fileEntry            ファイルエントリー用ストレージ

    @return 関数の処理結果を返します。

    @retval ResultSuccess   成功しました。
*/
Result HierarchicalRomFileTable::Initialize(
           SubStorage directoryBucket,
           SubStorage directoryEntry,
           SubStorage fileBucket,
           SubStorage fileEntry
       ) NN_NOEXCEPT
{
    int64_t sizeDirectoryBucket = 0;
    NN_RESULT_DO(directoryBucket.GetSize(&sizeDirectoryBucket));

    // ディレクトリ管理用ストレージをマウントします。
    NN_RESULT_DO(
        m_TableDirectory.Initialize(
            directoryBucket,
            DirectoryEntryMapTable::QueryBucketCount(sizeDirectoryBucket),
            directoryEntry
        )
    );

    int64_t sizeFileBucket = 0;
    NN_RESULT_DO(fileBucket.GetSize(&sizeFileBucket));

    // ファイル管理用ストレージをマウントします。
    NN_RESULT_DO(
        m_TableFile.Initialize(
            fileBucket,
            FileEntryMapTable::QueryBucketCount(sizeFileBucket),
            fileEntry
        )
    );

    NN_RESULT_SUCCESS;
}

/*!
    @brief ストレージをアンマウントします。
*/
void HierarchicalRomFileTable::Finalize() NN_NOEXCEPT
{
    // ディレクトリ管理用ストレージをアンマウントします。
    m_TableDirectory.Finalize();

    // ファイル管理用ストレージをアンマウントします。
    m_TableFile.Finalize();
}

/*!
    @brief ルートディレクトリエントリーを作成します。

    @return 関数の処理結果を返します。

    @retval ResultSuccess           成功しました。
    @retval ResultDbmAlreadyExists  すでにルートディレクトリが作成されています。
    @retval ResultDbmKeyFull        割り当て可能領域はありません。
    @retval ResultOutOfRange        ストレージの範囲外をアクセスしようとしました。
    @retval それ以外                ストレージの読み書きで失敗しました。
*/
Result HierarchicalRomFileTable::CreateRootDirectory() NN_NOEXCEPT
{
    StoragePosition rootPosition = 0;
    EntryKey rootKey = {};
    rootKey.key.parentDirectory = PositionRootDirectory;
    RomPathTool::InitEntryName(&rootKey.name);
    DirectoryRomEntry rootEntry = {};
    rootEntry.posNext = PositionNone;
    rootEntry.posDirectory = PositionNone;
    rootEntry.posFile = PositionNone;
#ifdef ENABLE_DIRECTORY_METADATA
    std::memset(&rootEntry.info, 0, sizeof(DirectoryInfo));
#endif
    return m_TableDirectory.Add(&rootPosition, rootKey, rootEntry);
}

/*!
    @brief ディレクトリを作成します。

    @param[out] pOutValue ディレクトリ ID
    @param[in] pFullPath ディレクトリパス
    @param[in] dirInfo ディレクトリ情報

    @return 関数の処理結果を返します。

    @retval ResultSuccess               成功しました。
    @retval ResultDbmAlreadyExists      すでにディレクトリが存在します。
    @retval ResultDbmDirectoryEntryFull これ以上ディレクトリを作成できません。

    @pre    pOutValue != nullptr
    @pre    pFullPath != nullptr
*/
Result HierarchicalRomFileTable::CreateDirectory(
           RomDirectoryId* pOutValue, const RomPathChar* pFullPath, const DirectoryInfo& dirInfo
       ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);
    NN_SDK_REQUIRES_NOT_NULL(pFullPath);

    // 親ディレクトリのエントリーを探索し、新規ディレクトリ用のキーを生成します。
    DirectoryRomEntry parentEntry = {};
    EntryKey newKey = {};
    NN_RESULT_DO(
        FindDirectoryRecursive(
            &newKey,
            &parentEntry,
            pFullPath
        )
    );

    // 同名のファイル、ディレクトリがすでに存在しているかどうかチェックします。
    // 存在した場合は、すでにエントリーが存在していることを結果として返します。
    NN_RESULT_DO(
        CheckSameEntryExists(
            newKey,
            ResultDbmAlreadyExists()
        )
    );

    // 新規ディレクトリエントリーを初期化します。
    DirectoryRomEntry newEntry = {};
    newEntry.posNext = PositionNone;
    newEntry.posDirectory = PositionNone;
    newEntry.posFile = PositionNone;
#ifdef ENABLE_DIRECTORY_METADATA
    newEntry.info = dirInfo;
#else
    NN_UNUSED(dirInfo);
#endif

    // 新規ディレクトリエントリーをハッシュテーブルに登録します。
    StoragePosition newPosition = 0;
    Result result = m_TableDirectory.Add(&newPosition, newKey, newEntry);
    if( result.IsFailure() )
    {
        if (ResultDbmKeyFull::Includes(result))
        {
            return ResultDbmDirectoryEntryFull();
        }
        return result;
    }

    // キーバリューストレージ内でのアドレスをディレクトリIDとして返します。
    *pOutValue = PositionToDirectoryId(newPosition);

    // 新規ディレクトリエントリーを親ディレクトリにおける子ディレクトリリストの末尾として追加します。
    if( parentEntry.posDirectory == PositionNone )
    {
        parentEntry.posDirectory = newPosition;

        // 親ディレクトリのリンク情報を更新します。
        NN_RESULT_DO(
            m_TableDirectory.SetByPosition(
                newKey.key.parentDirectory, parentEntry
            )
        );
    }
    else
    {
        StoragePosition posTail = parentEntry.posDirectory;
        for( ; ; )
        {
            RomEntryKey curKey = {};
            DirectoryRomEntry curEntry = {};
            NN_RESULT_DO(
                m_TableDirectory.GetByPosition(&curKey, &curEntry, posTail)
            );
            if( curEntry.posNext == PositionNone )
            {
                curEntry.posNext = newPosition;

                // リンク情報を更新します。
                NN_RESULT_DO(
                    m_TableDirectory.SetByPosition(posTail, curEntry)
                );

                break;
            }

            posTail = curEntry.posNext;
        }
    }

    NN_RESULT_SUCCESS;
}

/*!
    @brief ファイルを作成します。

    @param[out] pOutValue ファイル ID
    @param[in] pFullPath ファイルパス
    @param[in] fileInfo ファイル情報

    @return 関数の処理結果を返します。

    @retval ResultSuccess                   成功しました。
    @retval ResultDbmAlreadyExists          すでにファイルが存在します。
    @retval ResultDbmFileEntryFull          これ以上ファイルを作成できません。
    @retval ResultDbmDirectoryNotFound      作成先ディレクトリが見つかりませんでした。
    @retval ResultDbmInvalidOperation       不正な操作です。

    @pre    pOutValue != nullptr
    @pre    pFullPath != nullptr
*/
Result HierarchicalRomFileTable::CreateFile(
           RomFileId* pOutValue, const RomPathChar* pFullPath, const FileInfo& fileInfo
       ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);
    NN_SDK_REQUIRES_NOT_NULL(pFullPath);

    // 親ディレクトリのエントリーを探索し、新規ファイル用のキーを生成します。
    DirectoryRomEntry parentEntry = {};
    EntryKey newKey = {};
    NN_RESULT_DO(
        FindFileRecursive(
            &newKey,
            &parentEntry,
            pFullPath
        )
    );

    // 同名のファイル、ディレクトリがすでに存在しているかどうかチェックします。
    // 存在した場合は、すでにエントリーが存在していることを結果として返します。
    NN_RESULT_DO(CheckSameEntryExists(newKey, ResultDbmAlreadyExists()));

    // 新規ファイルエントリーを初期化します。
    FileRomEntry newEntry = {};
    newEntry.posNext = PositionNone;
    newEntry.info = fileInfo;

    // 新規ファイルエントリーをハッシュテーブルに登録します。
    StoragePosition newPosition = PositionNone;
    Result result = m_TableFile.Add(&newPosition, newKey, newEntry);
    if( result.IsFailure() )
    {
        if( ResultDbmKeyFull::Includes(result) )
        {
            return ResultDbmFileEntryFull();
        }
        return result;
    }

    // キーバリューストレージ内でのアドレスをファイルIDとして返します。
    *pOutValue = PositionToFileId(newPosition);

    // 新規ファイルエントリーを親ディレクトリにおける子ファイルリストの末尾として追加します。
    if( parentEntry.posFile == PositionNone )
    {
        parentEntry.posFile = newPosition;

        // 親ディレクトリのリンク情報を更新します。
        NN_RESULT_DO(
            m_TableDirectory.SetByPosition(
                newKey.key.parentDirectory, parentEntry
            )
        );
    }
    else
    {
        StoragePosition posTail = parentEntry.posFile;
        for( ; ; )
        {
            RomEntryKey curKey = {};
            FileRomEntry curEntry = {};
            NN_RESULT_DO(
                m_TableFile.GetByPosition(&curKey, &curEntry, posTail)
            );
            if( curEntry.posNext == PositionNone )
            {
                curEntry.posNext = newPosition;

                // リンク情報を更新します。
                NN_RESULT_DO(m_TableFile.SetByPosition(posTail, curEntry));

                break;
            }

            posTail = curEntry.posNext;
        }
    }

    NN_RESULT_SUCCESS;
}

/*!
    @brief 指定したフルパスをディレクトリ ID に変換します。

    @param[out] pOutValue ディレクトリ ID
    @param[in] pFullPath ディレクトリパス

    @return 関数の処理結果を返します。

    @retval ResultSuccess                   成功しました。
    @retval ResultDbmDirectoryNameTooLong   ディレクトリ名が長過ぎます。
    @retval ResultDbmDirectoryNotFound      親ディレクトリが見つかりませんでした。
    @retval ResultDbmInvalidOperation       不正な操作です。

    @pre    pOutValue != nullptr
    @pre    pFullPath != nullptr
*/
Result HierarchicalRomFileTable::ConvertPathToDirectoryId(
           RomDirectoryId* pOutValue,
           const RomPathChar* pFullPath
       ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);
    NN_SDK_REQUIRES_NOT_NULL(pFullPath);

    // 親ディレクトリのエントリーを探索し、対象ディレクトリ用キ－を生成します。
    DirectoryRomEntry parentEntry = {};
    EntryKey key = {};
    NN_RESULT_DO(
        FindDirectoryRecursive(
            &key,
            &parentEntry,
            pFullPath
        )
    );

    // ディレクトリエントリーを取得します。
    StoragePosition pos = 0;
    DirectoryRomEntry entry = {};
    NN_RESULT_DO(GetDirectoryEntry(&pos, &entry, key));

    *pOutValue = PositionToDirectoryId(pos);

    NN_RESULT_SUCCESS;
}

/*!
    @brief 指定したフルパスをファイル ID に変換します。

    @param[out] pOutValue ファイル ID
    @param[in] pFullPath ファイルパス

    @return 関数の処理結果を返します。

    @retval ResultSuccess                   成功しました。
    @retval ResultDbmDirectoryNameTooLong   ディレクトリ名が長過ぎます。
    @retval ResultDbmDirectoryNotFound      親ディレクトリが見つかりませんでした。
    @retval ResultDbmFileNotFound           ファイルが見つかりませんでした。
    @retval ResultDbmInvalidOperation       不正な操作です。

    @pre    pOutValue != nullptr
    @pre    pFullPath != nullptr
*/
Result HierarchicalRomFileTable::ConvertPathToFileId(
           RomFileId* pOutValue,
           const RomPathChar* pFullPath
       ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);
    NN_SDK_REQUIRES_NOT_NULL(pFullPath);

    // 親ディレクトリのエントリーを探索し、対象ディレクトリ用キ－を生成します。
    DirectoryRomEntry parentEntry = {};
    EntryKey key = {};
    NN_RESULT_DO(
        FindDirectoryRecursive(
            &key,
            &parentEntry,
            pFullPath
        )
    );

    // ファイルエントリーを取得します。
    StoragePosition pos = 0;
    FileRomEntry entry = {};
    NN_RESULT_DO(GetFileEntry(&pos, &entry, key));

    *pOutValue = PositionToFileId(pos);

    NN_RESULT_SUCCESS;
}

/*!
    @brief 指定したディレクトリ情報を取得します。

    @param[out] pOutValue ディレクトリ情報
    @param[in] pFullPath ディレクトリパス

    @return 関数の処理結果を返します。

    @retval ResultSuccess                   成功しました。
    @retval ResultDbmDirectoryNameTooLong   ディレクトリ名が長過ぎます。
    @retval ResultDbmDirectoryNotFound      ディレクトリが見つかりませんでした。
    @retval ResultDbmInvalidOperation       不正な操作です。

    @pre    pOutValue != nullptr
    @pre    pFullPath != nullptr
*/
Result HierarchicalRomFileTable::GetDirectoryInformation(
           DirectoryInfo* pOutValue,
           const RomPathChar* pFullPath
       ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);
    NN_SDK_REQUIRES_NOT_NULL(pFullPath);

    // 親ディレクトリのエントリーを探索し、対象ディレクトリ用キ－を生成します。
    DirectoryRomEntry parentEntry = {};
    EntryKey key = {};
    NN_RESULT_DO(
        FindDirectoryRecursive(
            &key,
            &parentEntry,
            pFullPath
        )
    );

    return GetDirectoryInformation(pOutValue, key);
}

/*!
    @brief 指定したディレクトリ情報を取得します。

    @param[out] pOutValue ディレクトリ情報
    @param[in] directoryId ディレクトリ ID

    @return 関数の処理結果を返します。

    @retval ResultSuccess               成功しました。
    @retval ResultDbmDirectoryNotFound  ディレクトリが見つかりませんでした。
    @retval ResultDbmInvalidOperation   不正な操作です。

    @pre    pOutValue != nullptr
*/
Result HierarchicalRomFileTable::GetDirectoryInformation(
           DirectoryInfo* pOutValue,
           RomDirectoryId directoryId
       ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);

    // ディレクトリエントリーを取得します。
    DirectoryRomEntry entry = {};
    NN_RESULT_DO(GetDirectoryEntry(&entry, directoryId));

#ifdef ENABLE_DIRECTORY_METADATA
    // ディレクトリの付加情報をコピーします。
    *pOutValue = entry.info;
#else
    NN_UNUSED(pOutValue);
#endif

    NN_RESULT_SUCCESS;
}

/*!
    @brief ファイルを開きます。

    @param[out] pOutValue ファイル情報
    @param[in] pFullPath ファイルパス

    @return 関数の処理結果を返します。

    @retval ResultSuccess                   成功しました。
    @retval ResultDbmDirectoryNameTooLong   ディレクトリ名が長過ぎます。
    @retval ResultDbmDirectoryNotFound      親ディレクトリが見つかりませんでした。
    @retval ResultDbmFileNotFound           ファイルが見つかりませんでした。
    @retval ResultDbmInvalidOperation       不正な操作です。

    @pre    pOutValue != nullptr
    @pre    pFullPath != nullptr
*/
Result HierarchicalRomFileTable::OpenFile(
           FileInfo* pOutValue, const RomPathChar* pFullPath
       ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);
    NN_SDK_REQUIRES_NOT_NULL(pFullPath);

    // 親ディレクトリのエントリーを探索し、対象ファイル用キ－を生成します。
    DirectoryRomEntry parentEntry = {};
    EntryKey fileKey = {};
    NN_RESULT_DO(
        FindFileRecursive(
            &fileKey,
            &parentEntry,
            pFullPath
        )
    );

    return OpenFile(pOutValue, fileKey);
}

/*!
    @brief ファイルを開きます。

    @param[out] pOutValue ファイル情報
    @param[in] fileId ファイル ID

    @return 関数の処理結果を返します。

    @retval ResultSuccess               成功しました。
    @retval ResultDbmFileNotFound       ファイルが見つかりませんでした。
    @retval ResultDbmInvalidOperation   不正な操作です。

    @pre    pOutValue != nullptr
*/
Result HierarchicalRomFileTable::OpenFile(
           FileInfo* pOutValue, RomFileId fileId
       ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);

    // ファイルエントリーを取得します。
    FileRomEntry fileEntry = {};
    NN_RESULT_DO(GetFileEntry(&fileEntry, fileId));

    // ファイルの付加情報をコピーします。
    *pOutValue = fileEntry.info;

    NN_RESULT_SUCCESS;
}

/*!
    @brief 指定したディレクトリ以下のディレクトリとファイル列挙を開始します。

           イテレータを @ref FindNextDirectory や @ref FindNextFile に
           渡すことでディレクトリ名、ファイル名をイテレーションできます。
           また、イテレーションはファイル、ディレクトリ各々に対して行うことができます。

    @param[in,out] pOutValue イテレータ
    @param[in] pFullPath ディレクトリパス

    @return 関数の処理結果を返します。

    @retval ResultSuccess                   成功しました。
    @retval ResultDbmDirectoryNameTooLong   ディレクトリ名が長過ぎます。
    @retval ResultDbmDirectoryNotFound      ディレクトリが見つかりませんでした。
    @retval ResultDbmInvalidOperation       不正な操作です。

    @pre    pOutValue != nullptr
    @pre    pFullPath != nullptr
*/
Result HierarchicalRomFileTable::FindOpen(
           FindPosition* pOutValue, const RomPathChar* pFullPath
       ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);
    NN_SDK_REQUIRES_NOT_NULL(pFullPath);

    // 親ディレクトリのエントリーを探索し、
    // イテレーション対象ディレクトリ用キ－を生成します。
    DirectoryRomEntry parentEntry = {};
    EntryKey directoryKey = {};
    NN_RESULT_DO(
        FindDirectoryRecursive(
            &directoryKey,
            &parentEntry,
            pFullPath
        )
    );

    return FindOpen(pOutValue, directoryKey);
}

/*!
    @brief 指定したディレクトリ以下のディレクトリとファイル列挙を開始します。

           イテレータを @ref FindNextDirectory や @ref FindNextFile に
           渡すことでディレクトリ名、ファイル名をイテレーションできます。
           また、イテレーションはファイル、ディレクトリ各々に対して行うことができます。

    @param[out] pOutValue イテレータ
    @param[in] directoryId ディレクトリ ID

    @return 関数の処理結果を返します。

    @retval ResultSuccess               成功しました。
    @retval ResultDbmDirectoryNotFound  ディレクトリが見つかりませんでした。
    @retval ResultDbmInvalidOperation   不正な操作です。

    @pre    pOutValue != nullptr
*/
Result HierarchicalRomFileTable::FindOpen(
           FindPosition* pOutValue, RomDirectoryId directoryId
       ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);

    pOutValue->nextPositionDirectory = PositionNone;
    pOutValue->nextPositionFile = PositionNone;

    // イテレーション対象のディレクトリエントリーを取得します。
    DirectoryRomEntry directoryEntry = {};
    NN_RESULT_DO(GetDirectoryEntry(&directoryEntry, directoryId));

    // 子ディレクトリ、ファイルリストへのインデックスを取得します。
    pOutValue->nextPositionDirectory = directoryEntry.posDirectory;
    pOutValue->nextPositionFile = directoryEntry.posFile;

    NN_RESULT_SUCCESS;
}

/*!
    @brief ディレクトリ以下の次のディレクトリを取得します。

           この関数の前に @ref FindOpen でイテレータを初期化する必要があります。

    @param[out]     pOutValue       ディレクトリ名を格納するバッファ
    @param[in,out]  pFindPosition   イテレータ
    @param[in]      length          pOutValue のバッファ文字数

    @return 関数の処理結果を返します。

    @retval ResultSuccess               成功しました。
    @retval ResultDbmFindFinished       イテレーションが完了しました。

    @pre    pFindPosition != nullptr
    @pre    DirectoryName != nullptr
    @pre    length > RomPathTool::MaxPathLength
*/
Result HierarchicalRomFileTable::FindNextDirectory(
           RomPathChar* pOutValue,
           FindPosition* pFindPosition,
           size_t length
       ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pFindPosition);
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);
    NN_SDK_REQUIRES_LESS(RomPathTool::MaxPathLength, length);
    NN_UNUSED(length);

    // 子ディレクトリリストの終端チェックを行います。
    if( pFindPosition->nextPositionDirectory == PositionNone )
    {
        // 探索は終了しました
        return ResultDbmFindFinished();
    }

    // 現在のディレクトリエントリーを取得します。
    RomEntryKey key = {};
    DirectoryRomEntry entry = {};
    size_t extraSize = 0;
    NN_RESULT_DO(
        m_TableDirectory.GetByPosition(
            &key, &entry, pOutValue, &extraSize, pFindPosition->nextPositionDirectory
        )
    );
    NN_SDK_ASSERT_LESS_EQUAL(extraSize / sizeof(RomPathChar), RomPathTool::MaxPathLength);
    pOutValue[extraSize / sizeof(RomPathChar)] = StringTraitsRom::Nul;

    // 次のディレクトリへのインデックスに更新します。
    pFindPosition->nextPositionDirectory = entry.posNext;

    NN_RESULT_SUCCESS;
}

/*!
    @brief ディレクトリ以下の次のファイルを取得します。

           この関数の前に @ref FindOpen でイテレータを初期化する必要があります。

    @param[out]     pOutValue       ベースファイル名
    @param[in,out]  pFindPosition   イテレータ
    @param[in]      length          pOutValue のバッファ文字数

    @return 関数の処理結果を返します。

    @retval ResultSuccess               成功しました。
    @retval ResultDbmFindFinished       イテレーションが完了しました。

    @pre    pFindPosition != nullptr
    @pre    pOutValue != nullptr
    @pre    length > RomPathTool::MaxPathLength
*/
Result HierarchicalRomFileTable::FindNextFile(
           RomPathChar* pOutValue,
           FindPosition* pFindPosition,
           size_t length
       ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pFindPosition);
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);
    NN_SDK_REQUIRES_LESS(RomPathTool::MaxPathLength, length);
    NN_UNUSED(length);

    // 子ファイルリストの終端チェックを行います。
    if( pFindPosition->nextPositionFile == PositionNone )
    {
        // 探索は終了しました
        return ResultDbmFindFinished();
    }

    // 現在のファイルエントリーを取得します。
    RomEntryKey key = {};
    FileRomEntry entry = {};
    size_t extraSize = 0;
    NN_RESULT_DO(
        m_TableFile.GetByPosition(
            &key, &entry, pOutValue, &extraSize, pFindPosition->nextPositionFile
        )
    );
    NN_SDK_ASSERT_LESS_EQUAL(extraSize / sizeof(RomPathChar), RomPathTool::MaxPathLength);
    pOutValue[extraSize / sizeof(RomPathChar)] = StringTraitsRom::Nul;

    // 次のファイルへのインデックスに更新します。
    pFindPosition->nextPositionFile = entry.posNext;

    NN_RESULT_SUCCESS;
}

/*!
    @brief ROM ファイルシステムに必要なバッファサイズを取得します。

    @param[out] pOutSizeDirectoryEntry ディレクトリエントリに必要なサイズ
    @param[out] pOutSizeFileEntry ファイルエントリに必要なサイズ

    @return 関数の処理結果を返します。

    @retval ResultSuccess   成功しました。

    @pre    pOutSizeDirectoryEntry != nullptr
    @pre    pOutSizeFileEntry != nullptr
*/
Result HierarchicalRomFileTable::QueryRomFileSystemSize(
           int64_t* pOutSizeDirectoryEntry,
           int64_t* pOutSizeFileEntry
       ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutSizeDirectoryEntry);
    NN_SDK_REQUIRES_NOT_NULL(pOutSizeFileEntry);

    *pOutSizeDirectoryEntry = m_TableDirectory.GetTotalEntrySize();
    *pOutSizeFileEntry = m_TableFile.GetTotalEntrySize();

    NN_RESULT_SUCCESS;
}

/*!
    @brief ファイル構成をダンプします。

           デバッグ用途の関数です。内部でスタックを多く消費します。

    @return 関数の処理結果を返します。

    @retval ResultSuccess       成功しました。
    @retval ResultOutOfRange    ストレージの範囲外を読み込もうとしました。
    @retval それ以外            ストレージの読み込みで失敗しました。
*/
Result HierarchicalRomFileTable::DumpTree() NN_NOEXCEPT
{
    NN_DETAIL_FS_TRACE("==== DUMP START ====\n");

    // ルートディレクトリ以下をダンプします。
    NN_RESULT_DO(DumpDirectory(PositionRootDirectory, 0));

    NN_DETAIL_FS_TRACE("==== DUMP FINISHED ====\n");

    NN_RESULT_SUCCESS;
}

/*!
    @brief 親の親ディレクトリを取得します。

    @param[out] pOutPosition 親の親ディレクトリエントリーのストレージ位置
    @param[out] pOutDirectoryKey 親の親ディレクトリキー
    @param[out] pOutDirectoryEntry 親の親ディレクトリエントリー
    @param[in] pos 親ディレクトリエントリーのストレージ位置
    @param[in] name 親ディレクトリ名
    @param[in] pFileName フルパス

    @return 関数の処理結果を返します。

    @retval ResultSuccess               成功しました。
    @retval ResultDbmDirectoryNotFound  ディレクトリが見つかりませんでした。
    @retval ResultDbmInvalidOperation   不正な操作です。
    @retval ResultDbmInvalidPathFormat  ルートディレクトリの親を取得しようとしました。

    @pre    pOutPosition != nullptr
    @pre    pOutDirectoryKey != nullptr
    @pre    pOutDirectoryEntry != nullptr
    @pre    pFullPath != nullptr
*/
Result HierarchicalRomFileTable::GetGrandparent(
           StoragePosition* pOutPosition,
           EntryKey* pOutDirectoryKey,
           DirectoryRomEntry* pOutDirectoryEntry,
           StoragePosition pos,
           RomPathTool::RomEntryName name,
           const RomPathChar* pFullPath
       ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutPosition);
    NN_SDK_REQUIRES_NOT_NULL(pOutDirectoryKey);
    NN_SDK_REQUIRES_NOT_NULL(pOutDirectoryEntry);
    NN_SDK_REQUIRES_NOT_NULL(pFullPath);

    // 親の親ディレクトリ ID を取得します。
    RomEntryKey grandparentKey = {};
    DirectoryRomEntry grandparentEntry = {};
    NN_RESULT_DO(m_TableDirectory.GetByPosition(&grandparentKey, &grandparentEntry, pos));
    pOutDirectoryKey->key.parentDirectory = grandparentKey.parentDirectory;

    // 親ディレクトリ名を取得します。
    NN_RESULT_DO(RomPathTool::GetParentDirectoryName(&pOutDirectoryKey->name, name, pFullPath));

    // 親ディレクトリを取得します。
    NN_RESULT_DO(GetDirectoryEntry(pOutPosition, pOutDirectoryEntry, *pOutDirectoryKey));

    NN_RESULT_SUCCESS;
}

/*!
    @brief 指定した親ディレクトを取得します。

    @param[out] pOutParentPosition 親ディレクトリインデックス
    @param[out] pOutParentDirectoryKey 親ディレクトリキー
    @param[out] pOutParentDirectoryEntry 親ディレクトリエントリー
    @param[in] pParser パスパーサー
    @param[in] pFullPath フルパス

    @return 関数の処理結果を返します。

    @retval ResultSuccess                   成功しました。
    @retval ResultDbmDirectoryNameTooLong   ディレクトリ名が長過ぎます。
    @retval ResultDbmDirectoryNotFound      ディレクトリが見つかりませんでした。
    @retval ResultDirectoryUnobtainable     ルートディレクトリの親ディレクトリは取得できません。

    @pre    pOutParentPosition != nullptr
    @pre    pOutParentDirectoryKey != nullptr
    @pre    pOutParentDirectoryEntry != nullptr
    @pre    pParser != nullptr
    @pre    pFullPath != nullptr
*/
Result HierarchicalRomFileTable::FindParentDirectoryRecursive(
           StoragePosition* pOutParentPosition,
           EntryKey* pOutParentDirectoryKey,
           DirectoryRomEntry* pOutParentDirectoryEntry,
           RomPathTool::PathParser* pParser,
           const RomPathChar* pFullPath
       ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutParentPosition);
    NN_SDK_REQUIRES_NOT_NULL(pOutParentDirectoryKey);
    NN_SDK_REQUIRES_NOT_NULL(pOutParentDirectoryEntry);
    NN_SDK_REQUIRES_NOT_NULL(pParser);
    NN_SDK_REQUIRES_NOT_NULL(pFullPath);

    StoragePosition dirPosition = PositionRootDirectory;
    EntryKey directoryKey = {};
    DirectoryRomEntry directoryEntry = {};

    // ルートディレクトリを取得します。
    directoryKey.key.parentDirectory = PositionRootDirectory;
    NN_RESULT_DO(pParser->GetNextDirectoryName(&directoryKey.name));
    NN_RESULT_DO(GetDirectoryEntry(&dirPosition, &directoryEntry, directoryKey));

    StoragePosition parentPosition = dirPosition;
    while( !pParser->IsParseFinished() )
    {
        EntryKey oldKey = directoryKey;

        // 次のディレクトリ名を取得します。
        NN_RESULT_DO(pParser->GetNextDirectoryName(&directoryKey.name));

        if( RomPathTool::IsCurrentDirectory(directoryKey.name) )
        {
            // "."の場合何もしない。
            directoryKey = oldKey;
            continue;
        }
        else if( RomPathTool::IsParentDirectory(directoryKey.name) )
        {
            if( parentPosition == PositionRootDirectory )
            {
                // ルートディレクトリの親ディレクトリは取得できません。
                return ResultDirectoryUnobtainable();
            }

            // ".."の場合は親ディレクトリを取得します。
            NN_RESULT_DO(
                GetGrandparent(
                    &parentPosition,
                    &directoryKey,
                    &directoryEntry,
                    directoryKey.key.parentDirectory,
                    directoryKey.name,
                    pFullPath
                )
            );
        }
        else
        {
            // 親ディレクトリ ID + ディレクトリ名から、ディレクトリエントリーを取得します。
            directoryKey.key.parentDirectory = parentPosition;
            const Result result = GetDirectoryEntry(&dirPosition, &directoryEntry, directoryKey);
            if( nn::fs::ResultDbmInvalidOperation::Includes(result) )
            {
                // 対象がファイルだった場合
                return nn::fs::ResultDbmNotFound();
            }
            NN_RESULT_DO(result);

            parentPosition = dirPosition;
        }
    }

    // 親ディレクトリ情報を設定します。
    *pOutParentPosition = parentPosition;
    *pOutParentDirectoryKey = directoryKey;
    *pOutParentDirectoryEntry = directoryEntry;

    NN_RESULT_SUCCESS;
}

/*!
    @brief 指定したパスからディレクトまたはファイルを取得します。

    @param[out] pOutKey 取得したディレクトリまたはファイルエントリーのキー
    @param[out] pOutParentDirectoryEntry 親ディレクトリエントリー
    @param[in] isFindDirectory ディレクトリを取得するなら true,
                               ファイルを取得するなら false
    @param[in] pFullPath 分離するフルパス

    @return 関数の処理結果を返します。

    @retval ResultSuccess                   成功しました。
    @retval ResultDbmDirectoryNameTooLong   ディレクトリ名が長過ぎます。
    @retval ResultDbmDirectoryNotFound      ディレクトリが見つかりませんでした。
    @retval ResultDirectoryUnobtainable     ルートディレクトリの親ディレクトリは取得できません。
    @retval ResultDbmInvalidOperation       不正な操作です。

    @pre    pOutKey != nullptr
    @pre    pOutParentDirectoryEntry != nullptr
    @pre    pFullPath != nullptr
*/
Result HierarchicalRomFileTable::FindPathRecursive(
           EntryKey* pOutKey,
           DirectoryRomEntry* pOutParentDirectoryEntry,
           bool isFindDirectory,
           const RomPathChar* pFullPath
       ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutParentDirectoryEntry);
    NN_SDK_REQUIRES_NOT_NULL(pOutKey);
    NN_SDK_REQUIRES_NOT_NULL(pFullPath);

    // フルパスを解析するためのパーサを初期化します。
    RomPathTool::PathParser parser;
    NN_RESULT_DO(parser.Initialize(pFullPath));

    // 親ディレクトリを探索します。
    EntryKey parentDirectoryKey = {};
    StoragePosition parentPosition = 0;
    NN_RESULT_DO(
        FindParentDirectoryRecursive(
            &parentPosition,
            &parentDirectoryKey,
            pOutParentDirectoryEntry,
            &parser,
            pFullPath
        )
    );

    // 最後のパス部分を取得します。
    if( isFindDirectory )
    {
        RomPathTool::RomEntryName name = {};

        // パスの最後の部分をディレクトリ名として取得します。
        NN_RESULT_DO(parser.GetAsDirectoryName(&name));

        if( RomPathTool::IsCurrentDirectory(name) )
        {
            // 最後のパスが"."であった場合

            // 親ディレクトリとして取得したキーをカレントディレクトリとします。
            *pOutKey = parentDirectoryKey;

            // 親ディレクトリを更新します。
            if( pOutKey->key.parentDirectory != PositionRootDirectory )
            {
                StoragePosition pos = 0;
                NN_RESULT_DO(
                    GetGrandparent(
                        &pos,
                        &parentDirectoryKey,
                        pOutParentDirectoryEntry,
                        pOutKey->key.parentDirectory,
                        pOutKey->name,
                        pFullPath
                    )
                );
            }
        }
        else if( RomPathTool::IsParentDirectory(name) )
        {
            // 最後のパスが".."であった場合は、カレントディレクトリ情報を
            // 取得しつつ親ディレクトリ方法を更新します。

            if( parentPosition == PositionRootDirectory )
            {
                // ルートディレクトリの親ディレクトリは取得できません。
                return ResultDirectoryUnobtainable();
            }

            StoragePosition pos = 0;

            // 親の親ディレクトリ ID を取得します。
            DirectoryRomEntry curEntry = {};
            NN_RESULT_DO(
                GetGrandparent(
                    &pos,
                    pOutKey,
                    &curEntry,
                    parentDirectoryKey.key.parentDirectory,
                    parentDirectoryKey.name,
                    pFullPath
                )
            );

            // 親ディレクトリを更新します。
            if( pOutKey->key.parentDirectory != PositionRootDirectory )
            {
                NN_RESULT_DO(
                    GetGrandparent(
                        &pos,
                        &parentDirectoryKey,
                        pOutParentDirectoryEntry,
                        pOutKey->key.parentDirectory,
                        pOutKey->name,
                        pFullPath
                    )
                );
            }
        }
        else
        {
            pOutKey->name = name;

            // 親ディレクトリ ID を設定します。
            // (pOutKey->name.length == 0 であればルートディレクトリを示しています)
            pOutKey->key.parentDirectory
                = (pOutKey->name.length > 0) ? parentPosition : PositionRootDirectory;
        }
    }
    else
    {
        // パスがルートの親を示していないかチェックします。
        {
            RomPathTool::RomEntryName name = {};
            NN_RESULT_DO(parser.GetAsDirectoryName(&name));
            if( RomPathTool::IsParentDirectory(name) && parentPosition == PositionRootDirectory )
            {
                return ResultDirectoryUnobtainable();
            }
        }

        // パスがディレクトリを示していたかどうかチェックします。
        if( parser.IsDirectoryPath() )
        {
            // 不正な操作です。
            return ResultDbmInvalidOperation();
        }

        // パスの最後の部分を取得します。
        pOutKey->key.parentDirectory = parentPosition;
        NN_RESULT_DO(parser.GetAsFileName(&pOutKey->name));
    }

    NN_RESULT_SUCCESS;
}

/*!
    @brief 指定したパスから親ディレクトリとディレクトリキーを取得します。

    @param[out] pOutDirectoryKey ディレクトリキー
    @param[out] pOutParentDirectoryEntry 親ディレクトリエントリー
    @param[in] pFullPath 分離するフルパス

    @return 関数の処理結果を返します。

    @retval ResultSuccess                   成功しました。
    @retval ResultDbmDirectoryNameTooLong   ディレクトリ名が長過ぎます。
    @retval ResultDbmDirectoryNotFound      ディレクトリが見つかりませんでした。
    @retval ResultDirectoryUnobtainable     ルートディレクトリの親ディレクトリは取得できません。
    @retval ResultDbmInvalidOperation       不正な操作です。

    @pre    pOutParentDirectoryEntry != nullptr
    @pre    pOutDirectoryKey != nullptr
    @pre    pFullPath != nullptr
*/
Result HierarchicalRomFileTable::FindDirectoryRecursive(
           EntryKey* pOutDirectoryKey,
           DirectoryRomEntry* pOutParentDirectoryEntry,
           const RomPathChar* pFullPath
       ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutParentDirectoryEntry);
    NN_SDK_REQUIRES_NOT_NULL(pOutDirectoryKey);
    NN_SDK_REQUIRES_NOT_NULL(pFullPath);

    return FindPathRecursive(
               pOutDirectoryKey,
               pOutParentDirectoryEntry,
               true,
               pFullPath
           );
}

/*!
    @brief 指定したパスから親ディレクトリとファイルキーを取得します。

    @param[out] pOutFileKey ファイルキー
    @param[out] pOutParentDirectoryEntry 親ディレクトリエントリー
    @param[in] pFullPath 分離するフルパス

    @return 関数の処理結果を返します。

    @retval ResultSuccess                   成功しました。
    @retval ResultDbmDirectoryNameTooLong   ディレクトリ名が長過ぎます。
    @retval ResultDbmDirectoryNotFound      ディレクトリが見つかりませんでした。
    @retval ResultDirectoryUnobtainable     ルートディレクトリの親ディレクトリは取得できません。
    @retval ResultDbmInvalidOperation       不正な操作です。

    @pre    pOutParentDirectoryEntry != nullptr
    @pre    pOutFileKey != nullptr
    @pre    pFullPath != nullptr
*/
Result HierarchicalRomFileTable::FindFileRecursive(
           EntryKey* pOutFileKey,
           DirectoryRomEntry* pOutParentDirectoryEntry,
           const RomPathChar* pFullPath
       ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutParentDirectoryEntry);
    NN_SDK_REQUIRES_NOT_NULL(pOutFileKey);
    NN_SDK_REQUIRES_NOT_NULL(pFullPath);

    return FindPathRecursive(
               pOutFileKey,
               pOutParentDirectoryEntry,
               false,
               pFullPath
           );
}

/*!
    @brief 指定したエントリーと同名のファイル、ディレクトリが存在するかどうかを取得します。

    @param[in] key キー
    @param[in] resultIfExist 同名のエントリーが存在したときの結果

    @retval ResultSuccess   同名のエントリーは存在しません。
    @retval resultIfExist   同名のエントリーが存在します。
    @retval それ以外        ストレージによるエラーです。

    @return 関数の処理結果を返します。
*/
Result HierarchicalRomFileTable::CheckSameEntryExists(
         const EntryKey& key,
         Result resultIfExist
     ) NN_NOEXCEPT
{
    // 同名ディレクトリの存在をチェックします。
    {
        StoragePosition pos = 0;
        DirectoryRomEntry directoryEntry = {};
        const Result result = m_TableDirectory.Get(&pos, &directoryEntry, key);
        if (! ResultDbmKeyNotFound::Includes(result))
        {
            if (result.IsFailure())
            {
                // ストレージによるエラーです。
                return result;
            }
            else
            {
                // 同名のエントリーが存在します。
                return resultIfExist;
            }
        }
    }

    // 同名ファイルの存在をチェックします。
    {
        StoragePosition pos = 0;
        FileRomEntry fileEntry = {};
        const Result result = m_TableFile.Get(&pos, &fileEntry, key);
        if (! ResultDbmKeyNotFound::Includes(result))
        {
            if (result.IsFailure())
            {
                // ストレージによるエラーです。
                return result;
            }
            else
            {
                // 同名のエントリーが存在します。
                return resultIfExist;
            }
        }
    }

    // 同名のエントリーは存在しません。
    NN_RESULT_SUCCESS;
}

/*!
    @brief ディレクトリエントリーを取得します。

    @param[out] pOutPosition ディレクトリエントリーのストレージ位置
    @param[out] pOutEntry ディレクトリエントリー
    @param[in] key ディレクトリキー

    @return 関数の処理結果を返します。

    @retval ResultSuccess                   成功しました。
    @retval ResultDbmDirectoryNotFound      ディレクトリが見つかりませんでした。
    @retval ResultDbmInvalidOperation       不正な操作です。

    @pre    pOutPosition != nullptr
    @pre    pOutEntry != nullptr
*/
Result HierarchicalRomFileTable::GetDirectoryEntry(
           StoragePosition* pOutPosition,
           DirectoryRomEntry* pOutEntry,
           const EntryKey& key
       ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutPosition);
    NN_SDK_REQUIRES_NOT_NULL(pOutEntry);

    Result result = m_TableDirectory.Get(pOutPosition, pOutEntry, key);
    if( result.IsFailure() )
    {
        if( ResultDbmKeyNotFound::Includes(result) )
        {
            // ファイルをディレクトリとして使用していないかどうかチェックします。
            StoragePosition filePosition = 0;
            FileRomEntry fileEntry = {};
            result = m_TableFile.Get(&filePosition, &fileEntry, key);
            if( result.IsFailure() )
            {
                if( !ResultDbmKeyNotFound::Includes(result) )
                {
                    return result;
                }
                // ディレクトリが見つかりませんでした。
                return ResultDbmDirectoryNotFound();
            }
            else
            {
                // 不正な操作です。
                return ResultDbmInvalidOperation();
            }
        }
    }
    return result;
}

/*!
    @brief ディレクトリエントリーを取得します。

    @param[out] pOutValue ディレクトリエントリー
    @param[in] directoryId ディレクトリ ID

    @return 関数の処理結果を返します。

    @retval ResultSuccess               成功しました。
    @retval ResultDbmDirectoryNotFound  ディレクトリが見つかりませんでした。
    @retval ResultDbmInvalidOperation   不正な操作です。

    @pre    pOutValue != nullptr
*/
Result HierarchicalRomFileTable::GetDirectoryEntry(
           DirectoryRomEntry* pOutValue,
           RomDirectoryId directoryId
       ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);

    StoragePosition pos = DirectoryIdToPosition(directoryId);

    RomEntryKey key = {};
    Result result = m_TableDirectory.GetByPosition(&key, pOutValue, pos);
    if( result.IsFailure() )
    {
        if( ResultDbmKeyNotFound::Includes(result) )
        {
            // ファイルをディレクトリとして使用していないかどうかチェックします。
            FileRomEntry fileEntry = {};
            result = m_TableFile.GetByPosition(&key, &fileEntry, pos);
            if( result.IsFailure() )
            {
                if( !ResultDbmKeyNotFound::Includes(result) )
                {
                    return result;
                }
                // ディレクトリが見つかりませんでした。
                return ResultDbmDirectoryNotFound();
            }
            else
            {
                // 不正な操作です。
                return ResultDbmInvalidOperation();
            }
        }
    }
    return result;
}

/*!
    @brief ファイルエントリーを取得します。

    @param[out] pOutPosition ファイルエントリーのストレージ位置
    @param[out] pOutEntry ファイルエントリー
    @param[in] key ファイルキー

    @return 関数の処理結果を返します。

    @retval ResultSuccess               成功しました。
    @retval ResultDbmFileNotFound       ファイルが見つかりませんでした。
    @retval ResultDbmInvalidOperation   不正な操作です。

    @pre    pOutPosition != nullptr
    @pre    pOutEntry != nullptr
*/
Result HierarchicalRomFileTable::GetFileEntry(
           StoragePosition* pOutPosition,
           FileRomEntry* pOutEntry,
           const EntryKey& key
       ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutPosition);
    NN_SDK_REQUIRES_NOT_NULL(pOutEntry);

    Result result = m_TableFile.Get(pOutPosition, pOutEntry, key);
    if( result.IsFailure() )
    {
        if( ResultDbmKeyNotFound::Includes(result) )
        {
            // ディレクトリをファイルとして使用していないかどうかチェックします。
            StoragePosition dirPosition = 0;
            DirectoryRomEntry directoryEntry = {};
            result = m_TableDirectory.Get(&dirPosition, &directoryEntry, key);
            if( result.IsFailure() )
            {
                if( !ResultDbmKeyNotFound::Includes(result) )
                {
                    return result;
                }
                // ファイルが見つかりませんでした。
                return ResultDbmFileNotFound();
            }
            else
            {
                // 不正な操作です。
                return ResultDbmInvalidOperation();
            }
        }
    }
    return result;
}

/*!
    @brief ファイルエントリーを取得します。

    @param[out] pOutValue ファイルエントリー
    @param[in] fileId ファイル ID

    @return 関数の処理結果を返します。

    @retval ResultSuccess               成功しました。
    @retval ResultDbmFileNotFound       ファイルが見つかりませんでした。
    @retval ResultDbmInvalidOperation   不正な操作です。

    @pre    pOutValue != nullptr
*/
Result HierarchicalRomFileTable::GetFileEntry(
           FileRomEntry* pOutValue,
           RomFileId fileId
       ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);

    StoragePosition pos = FileIdToPosition(fileId);

    RomEntryKey key = {};
    Result result = m_TableFile.GetByPosition(&key, pOutValue, pos);
    if( result.IsFailure() )
    {
        if( ResultDbmKeyNotFound::Includes(result) )
        {
            // ディレクトリをファイルとして使用していないかどうかチェックします。
            DirectoryRomEntry directoryEntry = {};
            result = m_TableDirectory.GetByPosition(&key, &directoryEntry, pos);
            if( result.IsFailure() )
            {
                if( !ResultDbmKeyNotFound::Includes(result) )
                {
                    return result;
                }
                // ファイルが見つかりませんでした。
                return ResultDbmFileNotFound();
            }
            else
            {
                // 不正な操作です。
                return ResultDbmInvalidOperation();
            }
        }
    }
    return result;
}

/*!
    @brief 指定したディレクトリ情報を取得します。

    @param[out] pOutValue ディレクトリ情報
    @param[in] directoryKey ディレクトリキー

    @retval ResultSuccess               成功しました。
    @retval ResultDbmDirectoryNotFound  ディレクトリが見つかりませんでした。
    @retval ResultDbmInvalidOperation   不正な操作です。

    @return 関数の処理結果を返します。

    @pre    pOutValue != nullptr
*/
Result HierarchicalRomFileTable::GetDirectoryInformation(
           DirectoryInfo* pOutValue,
           const EntryKey& directoryKey
       ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);

    // ディレクトリエントリーを取得します。
    StoragePosition pos = 0;
    DirectoryRomEntry entry = {};
    NN_RESULT_DO(GetDirectoryEntry(&pos, &entry, directoryKey));

#ifdef ENABLE_DIRECTORY_METADATA
    // ディレクトリの付加情報をコピーします。
    *pOutValue = entry.info;
#else
    NN_UNUSED(pOutValue);
#endif

    NN_RESULT_SUCCESS;
}

/*!
    @brief ファイルを開きます。

    @param[out] pOutValue ファイル情報
    @param[in] fileKey ファイルキー

    @return 関数の処理結果を返します。

    @retval ResultSuccess               成功しました。
    @retval ResultDbmFileNotFound       ファイルが見つかりませんでした。
    @retval ResultDbmInvalidOperation   不正な操作です。

    @pre    pOutValue != nullptr
*/
Result HierarchicalRomFileTable::OpenFile(
           FileInfo* pOutValue, const EntryKey& fileKey
       ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);

    // ファイルエントリーを取得します。
    FileRomEntry fileEntry = {};
    StoragePosition pos = 0;
    NN_RESULT_DO(GetFileEntry(&pos, &fileEntry, fileKey));

    // ファイルの付加情報をコピーします。
    *pOutValue = fileEntry.info;

    NN_RESULT_SUCCESS;
}

/*!
    @brief 指定したディレクトリ以下のディレクトリとファイル列挙を開始します。

           イテレータを @ref FindNextDirectory、@ref FindNextFile に
           渡すことでディレクトリ名、ファイル名をイテレーションできます。
           また、イテレーションはファイル、ディレクトリ各々に対して行うことができます。

    @param[in,out] pOutValue イテレータ
    @param[in] directoryKey ディレクトリキー

    @return 関数の処理結果を返します。

    @retval ResultSuccess               成功しました。
    @retval ResultDbmDirectoryNotFound  ディレクトリが見つかりませんでした。
    @retval ResultDbmInvalidOperation   不正な操作です。

    @pre    pOutValue != nullptr
*/
Result HierarchicalRomFileTable::FindOpen(
           FindPosition* pOutValue, const EntryKey& directoryKey
       ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);

    pOutValue->nextPositionDirectory = PositionNone;
    pOutValue->nextPositionFile = PositionNone;

    // イテレーション対象のディレクトリエントリーを取得します。
    StoragePosition pos = 0;
    DirectoryRomEntry directoryEntry = {};
    NN_RESULT_DO(GetDirectoryEntry(&pos, &directoryEntry, directoryKey));

    // 子ディレクトリ、ファイルリストへのインデックスを取得します。
    pOutValue->nextPositionDirectory = directoryEntry.posDirectory;
    pOutValue->nextPositionFile = directoryEntry.posFile;

    NN_RESULT_SUCCESS;
}

/*!
    @brief 指定したディレクトリ以下のファイルツリーをデバッグ表示します。

    @param[in] pRootKey ディレクトリキー
    @param[in] pRootEntry ディレクトリエントリー
    @param[in] depth ファイル階層の深さ

    @return 関数の処理結果を返します。

    @retval ResultSuccess       成功しました。
    @retval ResultOutOfRange    ストレージの範囲外を読み込もうとしました。
    @retval それ以外            ストレージの読み込みで失敗しました。
*/
Result HierarchicalRomFileTable::DumpDirectory(
           StoragePosition posParent, int depth
       ) NN_NOEXCEPT
{
    static const int ReadBufLength = RomPathTool::MaxPathLength + 1;
    const auto readBufDeleter = [](RomPathChar* ptr) NN_NOEXCEPT
    {
        nn::fs::detail::Deallocate(ptr, sizeof(RomPathChar) * ReadBufLength);
    };
    std::unique_ptr<RomPathChar, decltype(readBufDeleter)> readBuf(
        reinterpret_cast<RomPathChar*>(detail::Allocate(sizeof(RomPathChar) * ReadBufLength)),
        readBufDeleter
    );
    NN_RESULT_THROW_UNLESS(readBuf != nullptr, nn::fs::ResultAllocationMemoryFailedInDbmHierarchicalRomFileTable());

    DirectoryRomEntry directoryEntry = {};

    static const int StrBufLength = 1024;
    const auto strBufDeleter = [](RomPathChar* ptr) NN_NOEXCEPT
    {
        nn::fs::detail::Deallocate(ptr, sizeof(RomPathChar) * StrBufLength);
    };
    std::unique_ptr<RomPathChar, decltype(strBufDeleter)> strBuf(
        reinterpret_cast<RomPathChar*>(nn::fs::detail::Allocate(sizeof(RomPathChar) * StrBufLength)),
        strBufDeleter
    );
    NN_RESULT_THROW_UNLESS(strBuf != nullptr, nn::fs::ResultAllocationMemoryFailedInDbmHierarchicalRomFileTable());

    {
        size_t length = 0;

        EntryKey directoryKey = {};
        NN_RESULT_DO(
            m_TableDirectory.GetByPosition(
                &directoryKey.key, &directoryEntry, readBuf.get(), &length, posParent
            )
        );
        NN_SDK_ASSERT_LESS_EQUAL(length / sizeof(RomPathChar), RomPathTool::MaxPathLength);
        readBuf.get()[length / sizeof(RomPathChar)] = StringTraitsRom::Nul;

        for( int i = 0; i < depth * 2; i++ )
        {
            NN_DETAIL_FS_TRACE(" ");
        }

        if( length > 0 )
        {
            nn::util::SNPrintf(
                strBuf.get(), StrBufLength, "+ / %s [%08X]\n", readBuf.get(), posParent
            );
        }
        else
        {
            nn::util::SNPrintf(
                strBuf.get(), StrBufLength, "+ / (null) [%08X]\n", posParent
            );
        }
        NN_DETAIL_FS_TRACE("%s", strBuf.get());
    }

    StoragePosition posDirectory = directoryEntry.posDirectory;
    while( posDirectory != PositionNone )
    {
        // 再帰的にダンプを行います。
        // ディレクトリ数が多いとスタックオーバーフローの可能性があります。
        NN_RESULT_DO(DumpDirectory(posDirectory, depth + 1));

        // 次のディレクトリエントリーを取得します。
        RomEntryKey nextDirKey = {};
        DirectoryRomEntry nextDirEntry = {};
        NN_RESULT_DO(
            m_TableDirectory.GetByPosition(
                &nextDirKey, &nextDirEntry, posDirectory
            )
        );

        // 兄弟リンクを辿ります。
        posDirectory = nextDirEntry.posNext;
    }

    StoragePosition posFile = directoryEntry.posFile;
    while( posFile != PositionNone )
    {
        // 現在のファイルエントリーを取得します。
        RomEntryKey fileKey = {};
        FileRomEntry fileEntry = {};
        size_t length = 0;
        NN_RESULT_DO(m_TableFile.GetByPosition(&fileKey, &fileEntry, readBuf.get(), &length, posFile));
        NN_SDK_ASSERT_LESS_EQUAL(length / sizeof(RomPathChar), RomPathTool::MaxPathLength);
        readBuf.get()[length / sizeof(RomPathChar)] = StringTraitsRom::Nul;

        for( int i = 0; i < (depth + 1) * 2; i++ )
        {
            NN_DETAIL_FS_TRACE(" ");
        }
        nn::util::SNPrintf(
            strBuf.get(), StrBufLength, "+ / %s [%08X]\n", readBuf.get(), posFile
        );
        NN_DETAIL_FS_TRACE("%s", strBuf.get());

        // 兄弟リンクを辿ります。
        posFile = fileEntry.posNext;
    }

    NN_RESULT_SUCCESS;
}


}}
