﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdio>
#include <cstring>
#include <climits>
#include <algorithm>
#include <mutex>
#include <nn/os.h>
#include <nn/nn_Abort.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_SdkLog.h>
#include <nn/os/os_Mutex.h>
#include <nn/time.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/util/util_FormatString.h>
#include <nn/util/util_Optional.h>
#include <nn/fs/fs_Result.h>
#include <nn/fs/fs_ResultPrivate.h>
#include <nn/fs/fs_QueryRange.h>
#include <nn/fs/fsa/fs_IFile.h>
#include <nn/fs/fsa/fs_IDirectory.h>
#include <nn/fs/detail/fs_Log.h>

#include <nn/fat/fat_FatFileSystem.h>

#include <nn/fs/fs_IStorage.h>
#include <nn/fs/fs_PathUtility.h>
#include <nn/fssystem/fs_Assert.h>
#include "detail/fat_FatFileSystemStorageAdapter.h"

#include "prfile2/pf_api.h"
#include "prfile2/pdm_api.h"
#include "prfile2/pf_sm_safe_api.h"
#include "prfile2/pf_config.h"

namespace {

    // TORIAEZU
    static const int FatCachePageSize   = 32; // 32 セクタ = 16kB
    static const int FatCachePageCount  = 8;
    static const int DataCachePageSize  = 32; // 32 セクタ = 16kB
    static const int DataCachePageCount = 8;

    static const uint64_t PfFilePosBitLength = sizeof(nne::prfile2::PF_FPOS_T) * 8;
    static const uint64_t PfFilePosMaxValue = ( ( ( static_cast<uint64_t>(1) << (PfFilePosBitLength - 1)) - 1) << 1) + 1;
    static const int64_t PfFilePosMaxValueForInt64 = (PfFilePosMaxValue > static_cast<uint64_t>(INT64_MAX)) ? INT64_MAX : static_cast<int64_t>(PfFilePosMaxValue);
}

using namespace nn::fs;
using namespace nn::fs::detail;
using namespace nne::prfile2;

namespace {
    nn::fat::FatCalendarTimeGetter g_CurrentCalendarTimeGetter = nullptr;

    void GetPfTimeStamp(PF_SYS_DATE* date, PF_SYS_TIME* time)
    {
        nn::time::CalendarTime calendar;
        if ((g_CurrentCalendarTimeGetter != nullptr) &&
            g_CurrentCalendarTimeGetter(&calendar).IsSuccess())
        {
            date->sys_year  = calendar.year;
            date->sys_month = calendar.month;
            date->sys_day   = calendar.day;
            time->sys_hour  = calendar.hour;
            time->sys_min   = calendar.minute;
            time->sys_sec   = calendar.second;
        }
        else
        {
            /* Set dummy time */
            date->sys_year  = 1980;
            date->sys_month = 1;
            date->sys_day   = 1;
            time->sys_hour  = 0;
            time->sys_min   = 0;
            time->sys_sec   = 0;
        }
    }
}

namespace nn { namespace fat {

    void SetCurrentCalendarTimeGetter(FatCalendarTimeGetter currentCalendarTimeGetter)
    {
        g_CurrentCalendarTimeGetter = currentCalendarTimeGetter;
    }

    void FatErrorInfoSetter::SetError(int error, int extraError, const char* name) NN_NOEXCEPT
    {
        if (m_pFatError != nullptr)
        {
            NN_SDK_REQUIRES(m_pMutex != nullptr);
            std::lock_guard<os::Mutex> scopedLock(*m_pMutex);

            m_pFatError->error = error;
            m_pFatError->extraError = extraError;
            m_pFatError->driveId = m_driveId;
            std::strncpy(m_pFatError->name, name, FatErrorNameMaxLength - 1);
        }
    }

    namespace detail {

        class NnResultMaker : public Newable
        {
        public:
            NnResultMaker() NN_NOEXCEPT : m_HasDriveLetter(false), m_DriveLetterStore(0), m_pFatStorageAdapter(nullptr), m_ResultInvalidFatFormat(ResultInvalidFatFormat()) {}

            void Initialize(detail::FatStorageAdapter* pFatStorageAdapter, std::unique_ptr<FatErrorInfoSetter>&& pFatErrorInfoSetter, Result resultInvalidFatFormat, Result resultUsableSpaceNotEnough) NN_NOEXCEPT
            {
                m_pFatStorageAdapter = pFatStorageAdapter;
                m_pFatErrorInfoSetter = std::move(pFatErrorInfoSetter);
                m_ResultInvalidFatFormat = resultInvalidFatFormat;
                m_ResultUsableSpaceNotEnough = resultUsableSpaceNotEnough;
            }

            void StoreDriveLetter(const char driveLetter) NN_NOEXCEPT
            {
                m_DriveLetterStore = driveLetter;
                m_HasDriveLetter = true;
            }

            void Finalize() NN_NOEXCEPT
            {
                m_HasDriveLetter = false;
                m_DriveLetterStore = 0;
                m_pFatStorageAdapter = nullptr;
            }

            Result GetNnResultFromPfFileError(PF_FILE* pfFile, const char* name) NN_NOEXCEPT;
            Result GetNnResultFromPfError(const char* name) NN_NOEXCEPT;
            Result GetNnResultFromPfFormatError() NN_NOEXCEPT;

            Result GetNnResultFromPdmError(int pdmError, const char* name) NN_NOEXCEPT;
            Result GetResultInvalidfatFormat(int error, const char* name) NN_NOEXCEPT;

        private:
            Result ConvertPfErrorToNnResult(int pfError, int pfInError, const char* name) NN_NOEXCEPT;

        private:
            bool m_HasDriveLetter;
            char m_DriveLetterStore;
            detail::FatStorageAdapter* m_pFatStorageAdapter;
            std::unique_ptr<FatErrorInfoSetter> m_pFatErrorInfoSetter;
            Result m_ResultInvalidFatFormat;
            Result m_ResultUsableSpaceNotEnough;
        };

        Result NnResultMaker::ConvertPfErrorToNnResult(int pfError, int pfInError, const char* name) NN_NOEXCEPT
        {
            int pfSafeError = 0;

            switch (pfError)
            {
            case PF_ERR_EINVAL:
            case PF_ERR_ENOENT:
                return fs::ResultPathNotFound();

            case PF_ERR_EEXIST:
                return fs::ResultPathAlreadyExists();

            case PF_ERR_ENOTEMPTY:
                return fs::ResultDirectoryNotEmpty();

            case PF_ERR_ENOSPC:
                return m_ResultUsableSpaceNotEnough;

            case PF_ERR_EFBIG:
                return fs::ResultOutOfRange();

            case PF_ERR_ENOMEM:
                return fs::ResultAllocationMemoryFailedInFatFileSystemA();

            case PF_ERR_ENFILE:
                return fs::ResultFatFsTooManyFilesOpenedS();
            case PF_ERR_EMFILE:
                return fs::ResultFatFsTooManyFilesOpenedU();

            case PF_ERR_EACCES:
            case PF_ERR_ENOLCK:
                return fs::ResultTargetLocked();

            case PF_ERR_EBADF:
                return fs::ResultInvalidFatFileNumber();

            // オペレーションによっては返り得る
            case PF_ERR_EPERM:
                return fs::ResultFatFsLockError();
            case PF_ERR_EISDIR:
                return fs::ResultFatFsNotAFile();

            case PF_ERR_EEXFAT_NOTSPRT:
                return fs::ResultExFatUnavailable();

            // ファイルシステムが壊れているかも（非FAT形式をMountしようとした場合にも返る）
            case PF_ERR_ENOEXEC:
                m_pFatErrorInfoSetter->SetError(pfInError, 0, name);
                return m_ResultInvalidFatFormat;

            case PF_ERR_EIO:
            case PF_ERR_DFNC:
                {
                    // PrFile2 の先で動くドライバのエラー Result を PrFile2 内で直接扱えない為、一旦格納しておいたものを返す
                    m_pFatErrorInfoSetter->SetError(pfInError, 0, name);
                    Result result = m_pFatStorageAdapter->GetLastErrorResult();
                    if (result.IsFailure())
                    {
                        return result;
                    }
                    else
                    {
                        // pFatStorageAdapter と PrFILE2 が一致しない想定しない状況
                        return fs::ResultFatFsStorageStateMissmatch();
                    }
                }

            case PF_ERR_EMOD_SAFE:
                if (m_HasDriveLetter)
                {
                    pfSafeError = pf_module_errnum(m_DriveLetterStore, PF_MODULE_ID_SAFE);
                    NN_DETAIL_FS_ERROR("prfile2 fatsafe error (%s:%d)\n", name, pfSafeError);
                }
                m_pFatErrorInfoSetter->SetError(pfInError, pfSafeError, name);
                return fs::ResultFatFsModuleSafeError();

            // 設計上は返らないはずのエラー
            case PF_ERR_EBUSY:
            case PF_ERR_ESRCH:
            case PF_ERR_EMOD_NOTREG:
            case PF_ERR_EMOD_NOTSPRT:
            case PF_ERR_EMOD_FCS:
            case PF_ERR_ENOMEDIUM:
            case PF_ERR_ENOSYS:
            case PF_ERR_ENODEV:
            case PF_ERR_SYSTEM:
                m_pFatErrorInfoSetter->SetError(pfInError, 0, name);
                NN_DETAIL_FS_ERROR("prfile2 system error (%s:%d,%d)\n", name, pfError, pfInError);
                return fs::ResultFatFsUnexpectedSystemError();

            default:
                m_pFatErrorInfoSetter->SetError(pfInError, 0, name);
                NN_DETAIL_FS_ERROR("prfile2 unclassified error (%s:%d)\n", name, pfError);
                return fs::ResultFatFsUnclassified();
            }
        }

        Result NnResultMaker::GetNnResultFromPdmError(int pdmError, const char* name) NN_NOEXCEPT
        {
            m_pFatErrorInfoSetter->SetError(0, pdmError, name);

            switch (pdmError)
            {
            // ファイルシステムが壊れているかも
            case PDM_ERR_INVALID_MASTER_BOOT:
            case PDM_ERR_INVALID_BOOT_SECTOR:
            case PDM_ERR_INVALID_BPB:
            case PDM_ERR_NOT_EXIST_MBR:
            case PDM_ERR_NOT_EXIST_EPBR:
            case PDM_ERR_NOT_EXIST_PARTITION:
                return m_ResultInvalidFatFormat;

            case PDM_ERR_DRIVER_ERROR:
                {
                    // PrFile2 の先で動くドライバのエラー Result を PrFile2 内で直接扱えない為、一旦格納しておいたものを返す
                    Result result = m_pFatStorageAdapter->GetLastErrorResult();
                    if (result.IsFailure())
                    {
                        return result;
                    }
                    else
                    {
                        // pFatStorageAdapter と PrFILE2 が一致しない想定しない状況
                        return fs::ResultFatFsStorageStateMissmatch();
                    }
                }

            // 設計上は返らないはずのエラー
            case PDM_ERR_INVALID_PARAMETER:
            case PDM_ERR_NOT_EXIST_FREE_DISK_STRUCT:
            case PDM_ERR_NOT_EXIST_PARTITION_STRUCT:
            case PDM_ERR_NOT_EXIST_FREE_PARTITION_STRUCT:
            case PDM_ERR_STATE_OPENED:
            case PDM_ERR_STATE_CLOSED:
            case PDM_ERR_STATE_LOCKED:
            case PDM_ERR_STATE_UNLOCKED:
            case PDM_ERR_ACCESS_PERMISSION:
            case PDM_ERR_WRITE_PROTECTED:
            case PDM_ERR_MEDIA_EJECTED:
            case PDM_ERR_OUT_OF_RANGE:
            case PDM_ERR_SYSTEM_CALL_ERROR:
            case PDM_ERR_LOCK_ERROR:
            case PDM_ERR_UNSUPPORT_DISK_FORMAT:
            case PDM_ERR_EXFAT_FORMAT_UNSUPPORTED:
                NN_DETAIL_FS_ERROR("pdm system error (%s:%d)\n", name, pdmError);
                return fs::ResultFatFsUnexpectedSystemError();

            default:
                NN_DETAIL_FS_ERROR("pdm unclassified error (%s:%d)\n", name, pdmError);
                return fs::ResultFatFsUnclassified();
            }
        }

        Result NnResultMaker::GetResultInvalidfatFormat(int error, const char* name) NN_NOEXCEPT
        {
            m_pFatErrorInfoSetter->SetError(0, error, name);
            return m_ResultInvalidFatFormat;
        }

        Result NnResultMaker::GetNnResultFromPfFileError(PF_FILE* pfFile, const char* name) NN_NOEXCEPT
        {
            int pfError = pf_ferror(pfFile);
            if (pfError == PF_RET_ERR)
            {
                return fs::ResultInvalidFatFileNumber();
            }
            int pfInError = pf_inferror(pfFile);
            return ConvertPfErrorToNnResult(pfError, pfInError, name);
        }

        Result NnResultMaker::GetNnResultFromPfError(const char* name) NN_NOEXCEPT
        {
            int pfError = pf_errnum();
            int pfInError = pf_inerrnum();
            return ConvertPfErrorToNnResult(pfError, pfInError, name);
        }

        Result NnResultMaker::GetNnResultFromPfFormatError() NN_NOEXCEPT
        {
            Result result = m_pFatStorageAdapter->GetFormatErrorResult();
            if (result.IsFailure())
            {
                return result;
            }

            return GetNnResultFromPfError("format");
        }
    }

    namespace
    {
        bool g_IsPrfile2Initialized = false;

        os::Semaphore g_Prfile2ContextSemaphore(PF_MAX_CONTEXT_NUM, PF_MAX_CONTEXT_NUM);

        const int PathLengthMax = 3 + EntryNameLengthMax; // "a:/" + path

        nn::MemoryResource* g_MemoryResourceForFat = nullptr;

        void* AllocateForFat(size_t size)
        {
            NN_SDK_REQUIRES(g_MemoryResourceForFat != nullptr);
            return g_MemoryResourceForFat->allocate(size);
        }

        void FreeForFat(void* p, size_t size)
        {
            NN_SDK_REQUIRES(g_MemoryResourceForFat != nullptr);
            g_MemoryResourceForFat->deallocate(p, size);
        }

        int AllocateForFatSafe(char drive, PFM_U_LONG size, PFM_U_LONG **pp_bufAddr)
        {
            NN_UNUSED(drive);
            auto sizeWithHeader = size + sizeof(PFM_U_LONG);
            auto p = AllocateForFat(sizeWithHeader);
            if( p != nullptr )
            {
                *reinterpret_cast<PFM_U_LONG*>(p) = static_cast<PFM_U_LONG>(sizeWithHeader);
                *pp_bufAddr = reinterpret_cast<PFM_U_LONG*>(reinterpret_cast<char*>(p) + sizeof(PFM_U_LONG));
                return 0;
            }
            else
            {
                *pp_bufAddr = nullptr;
                return -1;
            }
        }

        int FreeForFatSafe(char drive, PFM_U_LONG *p_bufAddr)
        {
            NN_UNUSED(drive);
            PFM_U_LONG* pWithBuffer = p_bufAddr - 1;
            PFM_U_LONG size = *pWithBuffer;
            FreeForFat(pWithBuffer, size);
            return 0;
        }

        void AllocatePfContext()
        {
            g_Prfile2ContextSemaphore.Acquire();
            int pfResult = pf_regctx();
            if (pfResult != PF_RET_NO_ERR)
            {
                NN_SDK_LOG("[fs] Warning: Failed to register context\n");
            }
        }

        void FreePfContext()
        {
            int pfResult = pf_unregctx();
            if (pfResult != PF_RET_NO_ERR)
            {
                NN_SDK_LOG("[fs] Warning: Failed to unregister context\n");
            }
            g_Prfile2ContextSemaphore.Release();
        }

        class FatFile: public fsa::IFile, public Newable
        {
        public:
            static const int ClusterBufferMax = 64;

            NN_IMPLICIT FatFile(OpenMode mode, PF_FILE* pPfFile, detail::NnResultMaker* pNnResultMaker, int bytesPerCluster, detail::FatStorageAdapter* pFatStorageAdapter) NN_NOEXCEPT
                : m_Mode(mode),
                  m_pPfFile(pPfFile),
                  m_pNnResultMaker(pNnResultMaker),
                  m_Mutex(false),
                  m_BytesPerCluster(bytesPerCluster),
                  m_pFatStorageAdapter(pFatStorageAdapter)
            {
            }

            virtual ~FatFile() NN_NOEXCEPT
            {
                std::lock_guard<os::Mutex> scopedLock(m_Mutex);
                AllocatePfContext();

                pf_flock(m_pPfFile, PF_LOCK_UN);
                pf_fclose(m_pPfFile);

                FreePfContext();
            }

            virtual Result DoRead(
                size_t* outValue,
                int64_t offset,
                void *buffer,
                size_t size,
                const ReadOption& option) NN_NOEXCEPT NN_OVERRIDE;
            virtual Result DoWrite(
                int64_t offset,
                const void *buffer,
                size_t size,
                const WriteOption& option) NN_NOEXCEPT NN_OVERRIDE;
            virtual Result DoFlush() NN_NOEXCEPT NN_OVERRIDE;
            virtual Result DoSetSize(int64_t size) NN_NOEXCEPT NN_OVERRIDE;
            virtual Result DoGetSize(int64_t *outValue) NN_NOEXCEPT NN_OVERRIDE;
            virtual Result DoOperateRange(
                void* outBuffer,
                size_t outBufferSize,
                OperationId operationId,
                int64_t offset,
                int64_t size,
                const void* inBuffer,
                size_t inBufferSize) NN_NOEXCEPT NN_OVERRIDE;

            Result SetClusterLink() NN_NOEXCEPT;

        private:
            const OpenMode m_Mode;
            PF_FILE* const m_pPfFile;
            detail::NnResultMaker* const m_pNnResultMaker;
            os::Mutex      m_Mutex;
            int            m_BytesPerCluster;
            detail::FatStorageAdapter* const m_pFatStorageAdapter;
        };

        Result FatFile::DoRead(size_t* outValue, int64_t offset, void *buffer, size_t size, const ReadOption& option) NN_NOEXCEPT
        {
            if( offset > PfFilePosMaxValueForInt64 )
            {
                return ResultOutOfRange();
            }

            size_t readSize = 0;
            NN_RESULT_DO(DryRead(&readSize, offset, size, option, m_Mode));
            if( readSize == 0 )
            {
                *outValue = 0;
                NN_RESULT_SUCCESS;
            }

            auto pos = static_cast<PF_FPOS_T>(offset);

            std::lock_guard<os::Mutex> scopedLock(m_Mutex);
            AllocatePfContext();

            NN_UTIL_SCOPE_EXIT
            {
                FreePfContext();
            };

            auto pfResult = pf_fsetpos(m_pPfFile, &pos);
            if( pfResult != PF_RET_NO_ERR )
            {
                return m_pNnResultMaker->GetNnResultFromPfFileError(m_pPfFile, "fsetpos");
            }

            auto readResult = pf_fread(buffer, 1, static_cast<PF_SIZE_T>(readSize), m_pPfFile);
            *outValue = readResult;

            if( readResult != readSize )
            {
                NN_RESULT_DO(m_pNnResultMaker->GetNnResultFromPfFileError(m_pPfFile, "fread"));
            }

            NN_RESULT_SUCCESS;
        }

        Result FatFile::DoWrite(int64_t offset, const void *buffer, size_t size, const WriteOption& option) NN_NOEXCEPT
        {

            NN_FSP_REQUIRES( (m_Mode & nn::fs::OpenMode_Write) != 0, fs::ResultInvalidOperationForOpenMode() );

            if( offset < 0 || offset > PfFilePosMaxValueForInt64)
            {
                return ResultOutOfRange();
            }

            int64_t currentSize;
            NN_RESULT_DO(GetSize(&currentSize));

            bool IsAppended = (static_cast<int64_t>(offset + size) > currentSize || static_cast<int64_t>(offset + size) < offset);
            if ((m_Mode & OpenMode_AllowAppend) == 0)
            {
                NN_FSP_REQUIRES(!IsAppended, fs::ResultFileExtensionWithoutOpenModeAllowAppend(), "OpenMode_AllowAppend is required for implicit extension of file size by WriteFile().");
            }

            if (size != 0)
            {
                auto pos = static_cast<PF_FPOS_T>(offset);

                std::lock_guard<os::Mutex> scopedLock(m_Mutex);
                AllocatePfContext();

                NN_UTIL_SCOPE_EXIT
                {
                    FreePfContext();
                };

                auto pfResult = pf_fsetpos(m_pPfFile, &pos);
                if( pfResult != PF_RET_NO_ERR )
                {
                    return m_pNnResultMaker->GetNnResultFromPfFileError(m_pPfFile, "pf_fsetpos");
                }

                // todo: const
                PF_SIZE_T writeResult;
                if (IsAppended)
                {
                    writeResult = pf_fwrite(const_cast<void*>(buffer), 1, size, m_pPfFile);
                }
                else
                {
                    writeResult = pf_fwrite_without_append(const_cast<void*>(buffer), 1, size, m_pPfFile);
                }
                if( writeResult != size )
                {
                    return m_pNnResultMaker->GetNnResultFromPfFileError(m_pPfFile, "fwrite");
                }
            }

            if (((option.flags & WriteOptionFlag_Flush) != 0) || IsAppended)
            {
                NN_RESULT_DO(Flush());
            }

            NN_RESULT_SUCCESS;
        }

        Result FatFile::DoFlush() NN_NOEXCEPT
        {
            if( (m_Mode & nn::fs::OpenMode_Write) == 0 )
            {
                NN_RESULT_SUCCESS;
            }

            std::lock_guard<os::Mutex> scopedLock(m_Mutex);
            AllocatePfContext();

            NN_UTIL_SCOPE_EXIT
            {
                FreePfContext();
            };

            auto pfResult = pf_fsync(m_pPfFile);
            if( pfResult != PF_RET_NO_ERR )
            {
                NN_RESULT_DO(m_pNnResultMaker->GetNnResultFromPfFileError(m_pPfFile, "fsync"));
                NN_RESULT_SUCCESS;
            }

            NN_RESULT_SUCCESS;
        }

        Result FatFile::DoSetSize(int64_t size) NN_NOEXCEPT
        {
            NN_RESULT_DO(DrySetSize(size, m_Mode));

            if( size > PfFilePosMaxValueForInt64)
            {
                return ResultOutOfRange();
            }

            std::lock_guard<os::Mutex> scopedLock(m_Mutex);
            AllocatePfContext();

            NN_UTIL_SCOPE_EXIT
            {
                FreePfContext();
            };

            int pfResult;
            if ((m_Mode & OpenMode_AllowAppend) == 0)
            {
                pfResult = pf_fsetclstlink(m_pPfFile, PF_CLST_LINK_DISABLE, nullptr);
                if (pfResult != PF_RET_NO_ERR)
                {
                    return m_pNnResultMaker->GetNnResultFromPfError("fsetclstlink");
                }
            }

            NN_UTIL_SCOPE_EXIT
            {
                if ((m_Mode & OpenMode_AllowAppend) == 0)
                {
                    SetClusterLink();
                }
            };

            pfResult = pf_ftruncate(m_pPfFile, static_cast<PF_FPOS_T>(size));
            if( pfResult != PF_RET_NO_ERR )
            {
                return m_pNnResultMaker->GetNnResultFromPfFileError(m_pPfFile, "ftruncate");
            }
            pfResult = pf_fsync(m_pPfFile);
            if( pfResult != PF_RET_NO_ERR )
            {
                return m_pNnResultMaker->GetNnResultFromPfFileError(m_pPfFile, "fsync");
            }

            NN_RESULT_SUCCESS;
        }

        Result FatFile::DoGetSize(int64_t *outValue) NN_NOEXCEPT
        {
            std::lock_guard<os::Mutex> scopedLock(m_Mutex);
            AllocatePfContext();

            NN_UTIL_SCOPE_EXIT
            {
                FreePfContext();
            };

            PF_DTA dta;
            auto pfResult = pf_fgetdta(m_pPfFile, &dta);
            if( pfResult != PF_RET_NO_ERR )
            {
                return m_pNnResultMaker->GetNnResultFromPfError("fgetdta");
            }

            *outValue = dta.FileSize;

            NN_RESULT_SUCCESS;
        }

        Result FatFile::DoOperateRange(
            void* outBuffer,
            size_t outBufferSize,
            OperationId operationId,
            int64_t offset,
            int64_t size,
            const void* inBuffer,
            size_t inBufferSize) NN_NOEXCEPT
        {
            switch( operationId )
            {
            case OperationId::Invalidate:
                NN_RESULT_SUCCESS;

            case OperationId::QueryRange:
                NN_RESULT_DO(m_pFatStorageAdapter->OperateRange(outBuffer, outBufferSize, operationId, offset, size, inBuffer, inBufferSize));
                NN_RESULT_SUCCESS;

            default:
                return nn::fs::ResultUnsupportedOperation();
            }
        }

        Result FatFile::SetClusterLink() NN_NOEXCEPT
        {
            PF_DTA dta;
            auto pfResult = pf_fgetdta(m_pPfFile, &dta);
            if( pfResult != PF_RET_NO_ERR )
            {
                return m_pNnResultMaker->GetNnResultFromPfError("fgetdta");
            }

            if (dta.FileSize != 0)
            {
                int64_t intervalSize = static_cast<int64_t>(m_BytesPerCluster) * FatFile::ClusterBufferMax;
                int interval = static_cast<int>((dta.FileSize + intervalSize - 1) / intervalSize - 1);
                PF_CLUSTER_FILE pfClusterFile = {nullptr, FatFile::ClusterBufferMax, static_cast<PF_U_SHORT>(std::min(USHRT_MAX, interval))};
                pfResult = pf_fsetclstlink(m_pPfFile, PF_CLST_LINK_ENABLE, &pfClusterFile);
                if (pfResult != PF_RET_NO_ERR)
                {
                    NN_SDK_LOG("[fs] Warning: Failed to set cluster link on prfile2 (%d)\n", pfResult);
                    return m_pNnResultMaker->GetNnResultFromPfError("fsetclstlink");
                }
            }

            NN_RESULT_SUCCESS;
        }

        class FatDirectory : public fsa::IDirectory, public Newable
        {
        public:
            FatDirectory(OpenDirectoryMode mode, const char* path, PF_DIR* pPfDir, detail::NnResultMaker* pNnResultMaker) NN_NOEXCEPT
                : m_Mode(mode),
                  m_FirstRead(true),
                  m_pPfDir(pPfDir),
                  m_pNnResultMaker(pNnResultMaker),
                  m_Mutex(false)
            {
                NN_ABORT_UNLESS(strnlen(path, sizeof(m_Path)) + 2 + 1 <= sizeof(m_Path));

                bool hasTailBackSlash = (path[strnlen(path, sizeof(m_Path)) - 1] == '\\');

                if( hasTailBackSlash )
                {
                    nn::util::SNPrintf(m_Path, sizeof(m_Path), "%s*", path);
                }
                else
                {
                    nn::util::SNPrintf(m_Path, sizeof(m_Path), "%s\\*", path);
                }
            }

            FatDirectory(FatDirectory* pOriginal, PF_DIR* pPfDir) NN_NOEXCEPT
                : m_Mode(pOriginal->m_Mode),
                  m_FirstRead(true),
                  m_pPfDir(pPfDir),
                  m_pNnResultMaker(pOriginal->m_pNnResultMaker),
                  m_Mutex(false)
            {
                nn::util::SNPrintf(m_Path, sizeof(m_Path), "%s", pOriginal->m_Path);
            }

            virtual ~FatDirectory() NN_NOEXCEPT
            {
                if( m_pPfDir != nullptr )
                {
                    std::lock_guard<os::Mutex> scopedLock(m_Mutex);
                    AllocatePfContext();

                    pf_closedir(m_pPfDir);

                    FreePfContext();
                }
            }

            virtual Result DoRead(int64_t *outValue, DirectoryEntry *entryBuffer, int64_t entryBufferCount) NN_NOEXCEPT;
            virtual Result DoGetEntryCount(int64_t *outValue) NN_NOEXCEPT;

        private:
            const OpenDirectoryMode m_Mode;
            char                    m_Path[PathLengthMax + 2 + 1]; // path + "/*"
            PF_DTA                  m_Dta;
            bool                    m_FirstRead;
            PF_DIR* const           m_pPfDir;
            detail::NnResultMaker* const m_pNnResultMaker;
            os::Mutex               m_Mutex;

            bool IsReadTarget(const PF_DTA& dirent) const NN_NOEXCEPT;
            Result FindNextEntry(nn::util::optional<PF_DTA>* outValue) NN_NOEXCEPT;
        };

        bool FatDirectory::IsReadTarget(const PF_DTA& dta) const NN_NOEXCEPT
        {
            // Skip "." and ".."
            if( strncmp(reinterpret_cast<const char*>(dta.FileName), "." , 2) == 0 ||
                strncmp(reinterpret_cast<const char*>(dta.FileName), "..", 3) == 0)
            {
                return false;
            }

            bool isDirectory = (dta.Attribute & ATTR_DIR) != 0;
            if( (  isDirectory && (m_Mode & OpenDirectoryMode_Directory) ) ||
                ( !isDirectory && (m_Mode & OpenDirectoryMode_File)      )    )
            {
                return true;
            }
            return false;
        }

namespace {
        Result VerifyFileName(const char* pFileName, int fileNameLengthMax)
        {
            NN_RESULT_DO(fs::VerifyPath(pFileName, fileNameLengthMax, fileNameLengthMax));

            int i;
            for (i = 0; i < fileNameLengthMax; i++)
            {
                char ch = pFileName[i];

                if (ch == '\0')
                {
                    break;
                }

                if ((ch < 0x20) || (ch >= 0x7f))
                {
                    return nn::fs::ResultInvalidCharacter();
                }
                // FAT の使用不可文字チェック
                if ((ch == '"') || (ch == '*') || (ch == ':') || (ch == '<') || (ch == '>') || (ch == '?') || (ch == '|') || (ch == '/') || (ch == '\\'))
                {
                    return nn::fs::ResultInvalidCharacter();
                }
            }
            if ((i == 0) || (i == fileNameLengthMax))
            {
                return nn::fs::ResultInvalidCharacter();
            }

            NN_RESULT_SUCCESS;
        }

        Result ConvertDirectoryEntry(nn::fs::DirectoryEntry* outValue, const PF_DTA& src)
        {
            NN_SDK_ASSERT(outValue != nullptr);

            bool hasLongName = strnlen(reinterpret_cast<const char*>(src.LongName), sizeof(src.LongName)) > 0;
            const char* pFileName = reinterpret_cast<const char*>( hasLongName ? src.LongName
                                                                               : src.FileName );
            int fileNameLength = hasLongName ? sizeof(src.LongName) : sizeof(src.FileName);
            Result result = VerifyFileName(pFileName, fileNameLength);
            if (result.IsFailure())
            {
                NN_DETAIL_FS_WARN("Invalid entry name [%s]\n", pFileName);
                return result;
            }

            int nameLengthMax = std::min(fileNameLength - 1, nn::fs::EntryNameLengthMax);
            memset(outValue, 0, sizeof(*outValue));
            std::strncpy(outValue->name, pFileName, nameLengthMax);
            outValue->name[nameLengthMax] = '\0';

            outValue->directoryEntryType = static_cast<int8_t>(
                (src.Attribute & ATTR_DIR) != 0 ? nn::fs::DirectoryEntryType_Directory
                                                : nn::fs::DirectoryEntryType_File
                                           );
            outValue->fileSize           =
                (src.Attribute & ATTR_DIR) != 0 ? 0 : src.FileSize;

            // reserved フィールドを間借り
            outValue->reserved1[0]       = ((src.Attribute & ATTR_ARCH) != 0 ? Attribute::Attribute_Archive
                                                                             : 0) |
                                           ((src.Attribute & ATTR_DIR)  != 0 ? Attribute::Attribute_Directory
                                                                             : 0);
            NN_RESULT_SUCCESS;
        }
}

        Result FatDirectory::FindNextEntry(nn::util::optional<PF_DTA>* outValue) NN_NOEXCEPT
        {
            while(NN_STATIC_CONDITION(true))
            {
                int pfResult;

                if(m_FirstRead)
                {
                    m_FirstRead = false;
                    const unsigned char targetAttribute = (ATTR_RDONLY | ATTR_HIDDEN | ATTR_SYSTEM | ATTR_DIR | ATTR_ARCH | ATTR_NONE);
                    pfResult = pf_fsfirst(m_Path, targetAttribute, &m_Dta);
                }
                else
                {
                    pfResult = pf_fsnext(&m_Dta);
                }

                if( pfResult != PF_RET_NO_ERR )
                {
                    auto pfError = pf_errnum();
                    if( pfError == PF_ERR_ENOENT )
                    {
                        NN_RESULT_SUCCESS;
                    }
                    else
                    {
                        return m_pNnResultMaker->GetNnResultFromPfError("fsnext");
                    }
                }

                if( IsReadTarget(m_Dta) )
                {
                    *outValue = m_Dta;
                    NN_RESULT_SUCCESS;
                }
            }
        }

        Result FatDirectory::DoRead(int64_t *outValue, DirectoryEntry *entryBuffer, int64_t entryBufferCount) NN_NOEXCEPT
        {
            // TORIAEZU: スタック消費量大

            std::lock_guard<os::Mutex> scopedLock(m_Mutex);
            AllocatePfContext();

            NN_UTIL_SCOPE_EXIT
            {
                FreePfContext();
            };

            int64_t entryCount = 0;
            while( entryCount < entryBufferCount )
            {
                nn::util::optional<PF_DTA> dta;
                NN_RESULT_DO(FindNextEntry(&dta));
                if( dta == nullptr )
                {
                    break;
                }

                Result result = ConvertDirectoryEntry(&entryBuffer[entryCount], *dta);
                if (result.IsSuccess())
                {
                    ++entryCount;
                }
            }
            *outValue = entryCount;

            NN_RESULT_SUCCESS;
        }

        Result FatDirectory::DoGetEntryCount(int64_t *outValue) NN_NOEXCEPT
        {
            // TORIAEZU: スタック消費量大

            FatDirectory clone(this, nullptr);

            std::lock_guard<os::Mutex> scopedLock(m_Mutex);
            AllocatePfContext();

            NN_UTIL_SCOPE_EXIT
            {
                FreePfContext();
            };

            int64_t entryCount = 0;
            while(NN_STATIC_CONDITION(true))
            {
                nn::util::optional<PF_DTA> dta;
                NN_RESULT_DO(clone.FindNextEntry(&dta));
                if( dta == nullptr )
                {
                    break;
                }

                ++entryCount;
            }
            *outValue = entryCount;

            NN_RESULT_SUCCESS;
        }

    }

    void FatFileSystem::SetAllocatorForFat(nn::MemoryResource* pMemoryResource) NN_NOEXCEPT
    {
        g_MemoryResourceForFat = pMemoryResource;
    }


    size_t FatFileSystem::GetCacheBufferSize() NN_NOEXCEPT
    {
        size_t cacheCount = FatCachePageSize * FatCachePageCount + DataCachePageSize * DataCachePageCount;
        return sizeof(PF_CACHE_PAGE) * cacheCount + sizeof(PF_SECTOR_BUF) * cacheCount;
    }

    class FatFileSystem::DriveHandle : public Newable
    {
    public:
        DriveHandle()
            : driveLetter(0)
        {
        }

    public:
        char driveLetter;
        PDM_DISK* pDiskTable;
        PDM_PARTITION* pPartitionTable;

        PDM_INIT_DISK diskInitTbl;
    };

    class FatFileSystem::FatSafeContext : public Newable
    {
    public:
        PF_SAFE_INIT safeInit;
    };

    FatFileSystem::FatFileSystem() NN_NOEXCEPT
        : m_Mutex(false),
          m_DriveHandle(nullptr),
          m_pNnResultMaker(nullptr),
          m_IsMounted(false)
    {
    }

    FatFileSystem::~FatFileSystem() NN_NOEXCEPT
    {
        Finalize();
    }

    class FatFileSystem::TailBuffer : public Newable
    {
    public:
        static const int TailBufferMax = 313; /* (10000 / 32) + 1 = 313 */

        PF_TAIL_BUF buffer[TailBufferMax];
    };

    Result FatFileSystem::Initialize(std::shared_ptr<IStorage> pBaseStorage, void* cacheBuffer, size_t cacheBufferSize) NN_NOEXCEPT
    {
        FatAttribute attrs = {false, false, true};
        std::unique_ptr<FatErrorInfoSetter> pFatErrorInfoSetter(new FatErrorInfoSetter());
        NN_RESULT_THROW_UNLESS(pFatErrorInfoSetter != nullptr, nn::fs::ResultAllocationMemoryFailedNew());

        return Initialize(pBaseStorage, cacheBuffer, cacheBufferSize, &attrs, std::move(pFatErrorInfoSetter), ResultInvalidFatFormat(), ResultUsableSpaceNotEnough());
    }

    Result FatFileSystem::Initialize(IStorage* pBaseStorage, void* cacheBuffer, size_t cacheBufferSize) NN_NOEXCEPT
    {
        FatAttribute attrs = {false, false, true};
        std::unique_ptr<FatErrorInfoSetter> pFatErrorInfoSetter(new FatErrorInfoSetter());
        NN_RESULT_THROW_UNLESS(pFatErrorInfoSetter != nullptr, nn::fs::ResultAllocationMemoryFailedNew());

        return Initialize(pBaseStorage, cacheBuffer, cacheBufferSize, &attrs, std::move(pFatErrorInfoSetter), ResultInvalidFatFormat(), ResultUsableSpaceNotEnough());
    }

    Result FatFileSystem::Initialize(std::shared_ptr<IStorage> pBaseStorage, void* cacheBuffer, size_t cacheBufferSize, FatAttribute* pFatAttribute, std::unique_ptr<FatErrorInfoSetter>&& pFatErrorInfoSetter, Result resultInvalidFatFormat, Result resultUsableSpaceNotEnough) NN_NOEXCEPT
    {
        m_FatStorageAdapter.Initialize(std::move(pBaseStorage));
        return InitializeImpl(cacheBuffer, cacheBufferSize, pFatAttribute, std::move(pFatErrorInfoSetter), resultInvalidFatFormat, resultUsableSpaceNotEnough);
    }

    Result FatFileSystem::Initialize(IStorage* pBaseStorage, void* cacheBuffer, size_t cacheBufferSize, FatAttribute* pFatAttribute, std::unique_ptr<FatErrorInfoSetter>&& pFatErrorInfoSetter, Result resultInvalidFatFormat, Result resultUsableSpaceNotEnough) NN_NOEXCEPT
    {
        m_FatStorageAdapter.Initialize(pBaseStorage);
        return InitializeImpl(cacheBuffer, cacheBufferSize, pFatAttribute, std::move(pFatErrorInfoSetter), resultInvalidFatFormat, resultUsableSpaceNotEnough);
    }

    Result FatFileSystem::InitializeImpl(void* cacheBuffer, size_t cacheBufferSize, FatAttribute* pFatAttribute, std::unique_ptr<FatErrorInfoSetter>&& pFatErrorInfoSetter, Result resultInvalidFatFormat, Result resultUsableSpaceNotEnough) NN_NOEXCEPT
    {
        PDM_ERROR pdmResult;
        int       pfResult;

        PF_DRV_TBL drv_tbl_entry = {0};
        PF_DRV_TBL *drv_tbl[2] = {0};

        PF_VOL_CFG vol_config = {0};

        NN_SDK_REQUIRES(cacheBufferSize >= GetCacheBufferSize());
        NN_UNUSED(cacheBufferSize);

        std::lock_guard<os::Mutex> scopedLock(m_Mutex);

        m_DriveHandle.reset(new DriveHandle());
        NN_RESULT_THROW_UNLESS(m_DriveHandle != nullptr, nn::fs::ResultAllocationMemoryFailedInFatFileSystemC());

        m_DriveHandle->diskInitTbl.p_func = detail::FatStorageAdapterInitDriveTable;
        m_DriveHandle->diskInitTbl.ui_ext = reinterpret_cast<uintptr_t>(&m_FatStorageAdapter);

        // TODO: 対象デバイスに応じてカスタマイズ
        bool isInitializeSuccess = false;

        std::unique_ptr<detail::NnResultMaker> pNnResultMaker(new detail::NnResultMaker());
        NN_RESULT_THROW_UNLESS(pNnResultMaker != nullptr, ResultAllocationMemoryFailedInFatFileSystemD());
        m_pNnResultMaker = std::move(pNnResultMaker);
        m_pNnResultMaker->Initialize(&m_FatStorageAdapter, std::move(pFatErrorInfoSetter), resultInvalidFatFormat, resultUsableSpaceNotEnough);
        NN_UTIL_SCOPE_EXIT
        {
            if (!isInitializeSuccess)
            {
                m_pNnResultMaker->Finalize();
            }
        };

        m_Attribute = *pFatAttribute;

        // TORIAEZU: 個別インスタンス生成時で行う処理ではない
        if( !g_IsPrfile2Initialized )
        {

            // Initializle Disk Manager
            pdmResult = pdm_init_diskmanager(0, NULL);
            if(pdmResult != PDM_OK)
            {
                return m_pNnResultMaker->GetNnResultFromPdmError(pdmResult, "init_diskmanager");
            }

            // Initialize PrFile2
            pfResult = pf_init_prfile2(PF_CHAR_CHECK_ENABLE, NULL);
            if(pfResult != PF_RET_NO_ERR)
            {
                return fs::ResultFatFsInternalError();
            }

            pf_set_allocator(PF_ALLOCATE_TYPE_SMALL, Allocate, Deallocate);
#if defined(NN_BUILD_CONFIG_HARDWARE_JETSONTK1) || defined(NN_BUILD_CONFIG_HARDWARE_JETSONTK2) || defined(NN_BUILD_CONFIG_HARDWARE_NX)
            pf_set_allocator(PF_ALLOCATE_TYPE_WORK, AllocateForFat, FreeForFat);
#else
            /* Windowsビルドだと FatFileSystemCreator で SetAllocatorForFat() が呼ばれない */
            pf_set_allocator(PF_ALLOCATE_TYPE_WORK, Allocate, Deallocate);
#endif
            pf_set_timestamp_callback(GetPfTimeStamp);

            g_IsPrfile2Initialized = true;
        }

        // Open disk
        PDM_DISK* pDiskTable;
        pdmResult = pdm_open_disk(&m_DriveHandle->diskInitTbl, &pDiskTable);
        if (pdmResult != PDM_OK)
        {
            return m_pNnResultMaker->GetNnResultFromPdmError(pdmResult, "open_disk");
        }

        NN_UTIL_SCOPE_EXIT
        {
            if( !isInitializeSuccess )
            {
                pdm_close_disk(pDiskTable);
            }
        };

        // Open partition
        PDM_PARTITION* pPartitionTable;
        pdmResult = pdm_open_partition(pDiskTable, 0, &pPartitionTable);
        if (pdmResult != PDM_OK)
        {
            return m_pNnResultMaker->GetNnResultFromPdmError(pdmResult, "open_partition");
        }

        NN_UTIL_SCOPE_EXIT
        {
            if( !isInitializeSuccess )
            {
                pdm_close_partition(pPartitionTable);
            }
        };


        // Attach
        PF_CACHE_SETTING cacheSet;
        size_t pagesSize = sizeof(PF_CACHE_PAGE) * (FatCachePageSize * FatCachePageCount + DataCachePageSize * DataCachePageCount);
        cacheSet.pages   = static_cast<PF_CACHE_PAGE*>(cacheBuffer);
        cacheSet.buffers = reinterpret_cast<PF_SECTOR_BUF*>(static_cast<char*>(cacheBuffer) + pagesSize);
        cacheSet.num_fat_pages  = FatCachePageSize * FatCachePageCount;
        cacheSet.num_data_pages = DataCachePageSize * DataCachePageCount;
        cacheSet.num_fat_buf_size = FatCachePageSize;
        cacheSet.num_data_buf_size = DataCachePageSize;

        drv_tbl_entry.p_part = pPartitionTable;
        drv_tbl_entry.cache = &cacheSet;

        drv_tbl[0] = &drv_tbl_entry;
        drv_tbl[1] = nullptr;

        AllocatePfContext();

        NN_UTIL_SCOPE_EXIT
        {
            FreePfContext();
        };

        pfResult = pf_attach((PF_DRV_TBL**)drv_tbl);
        if (pfResult != PF_RET_NO_ERR)
        {
            return m_pNnResultMaker->GetNnResultFromPfError("attach");
        }

        NN_UTIL_SCOPE_EXIT
        {
            if( !isInitializeSuccess )
            {
                pf_detach(drv_tbl_entry.drive);
            }
        };

        // pf_attach()後はマウント状態で返ってくる場合があるので一旦アンマウント
        // もしマウント状態でない場合pf_unmount()は何もせずOKで返る
        pfResult = pf_unmount(drv_tbl_entry.drive, PF_UNMOUNT_FORCE);
        if (pfResult != PF_RET_NO_ERR)
        {
            return m_pNnResultMaker->GetNnResultFromPfError("unmount");
        }

        m_pNnResultMaker->StoreDriveLetter(drv_tbl_entry.drive);

        // Configure
        pfResult = pf_getvolcfg(drv_tbl_entry.drive, &vol_config);
        if (pfResult != PF_RET_NO_ERR)
        {
            return m_pNnResultMaker->GetNnResultFromPfError("getvolcfg");
        }

        vol_config.file_config &= ~PF_UPDATE_ACCESS_DATE_ENABLE;
        vol_config.file_config |= PF_UPDATE_ACCESS_DATE_DISABLE;
        if (!m_Attribute.isTimeStampUpdated)
        {
            vol_config.file_config |= PF_WRITE_TIMESTMP_NOUPDATE;
        }

        pfResult = pf_setvolcfg(drv_tbl_entry.drive, &vol_config);
        if (pfResult != PF_RET_NO_ERR)
        {
            return m_pNnResultMaker->GetNnResultFromPfError("setvolcfg");
        }

        m_DriveHandle->driveLetter = drv_tbl_entry.drive;

        m_DriveHandle->pPartitionTable = pPartitionTable;
        m_DriveHandle->pDiskTable = pDiskTable;

        isInitializeSuccess = true;

        NN_RESULT_SUCCESS;
    } // NOLINT(impl/function_size)

    void FatFileSystem::Finalize() NN_NOEXCEPT
    {
        std::lock_guard<os::Mutex> scopedLock(m_Mutex);
        AllocatePfContext();

        NN_UTIL_SCOPE_EXIT
        {
            FreePfContext();
        };

        if (!m_DriveHandle || m_DriveHandle->driveLetter == 0)
        {
            return;
        }

        if( m_IsMounted )
        {
            Unmount();
        }

        int pfResult;

        pfResult = pf_detach(m_DriveHandle->driveLetter);
        if (pfResult != PF_RET_NO_ERR)
        {
            Result result = m_pNnResultMaker->GetNnResultFromPfError("detach");
            NN_ABORT("Failed to detach %c drive prfile2 (Module:%d, Description:%d)\n", m_DriveHandle->driveLetter, result.GetModule(), result.GetDescription());
            NN_UNUSED(result);
        }
        m_DriveHandle->driveLetter = 0;

        auto pdmResult = pdm_close_partition(m_DriveHandle->pPartitionTable);
        if (pdmResult != PDM_OK)
        {
            Result result = m_pNnResultMaker->GetNnResultFromPdmError(pdmResult, "close_partition");
            NN_ABORT("Failed to close partition on prfile2 (Module:%d, Description:%d)\n", result.GetModule(), result.GetDescription());
            NN_UNUSED(result);
        }

        pdmResult = pdm_close_disk(m_DriveHandle->pDiskTable);
        if (pdmResult != PDM_OK)
        {
            Result result = m_pNnResultMaker->GetNnResultFromPdmError(pdmResult, "close_disk");
            NN_ABORT("Failed to close disk on prfile2 (Module:%d, Description:%d)\n", result.GetModule(), result.GetDescription());
            NN_UNUSED(result);
        }
    }

    Result FatFileSystem::ResolvePath(char* outPath, size_t size, const char* path) const NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(path[0] == '/');
        util::SNPrintf(outPath, size, "%c:\\%s", m_DriveHandle->driveLetter, path + 1);

        // 入力Pathのバッファ長判定
        if (nn::fs::EntryNameLengthMax < strlen(path))
        {
            return nn::fs::ResultTooLongPath();
        }

        auto length = strnlen(outPath, size);
        if( length < size && outPath[length - 1] == '/' )
        {
            outPath[length - 1] = '\0'; // remove '/'
        }

        static const auto ResolvedPathLengthMax = 260 - 3 - 1;
        static const auto ResolvedNameLengthMax = 255;
        return fs::VerifyPath(outPath + 3, ResolvedPathLengthMax, ResolvedNameLengthMax);
    }

    Result FatFileSystem::Format() NN_NOEXCEPT
    {
        FatFormatParam fatFormatParam = {false, 0, ResultFatFsWriteVerifyError()};
        return Format(&fatFormatParam);
    }

    Result FatFileSystem::Format(FatFormatParam* pFatFormatParam) NN_NOEXCEPT
    {
        int pfResult;
        uint8_t* pBuffer = nullptr;

        std::lock_guard<os::Mutex> scopedLock(m_Mutex);
        AllocatePfContext();

        NN_UTIL_SCOPE_EXIT
        {
            FreePfContext();
        };

        if (pFatFormatParam->isSdCard)
        {
            pBuffer = static_cast<uint8_t*>(AllocateForFat(FatFormatWorkBufferSize));
            NN_RESULT_THROW_UNLESS(pBuffer != nullptr, nn::fs::ResultAllocationMemoryFailedInFatFileSystemD());

            pFatFormatParam->pWorkBuffer = pBuffer;
            pFatFormatParam->workBufferSize = FatFormatWorkBufferSize;
        }

        NN_UTIL_SCOPE_EXIT
        {
            if (pBuffer != nullptr)
            {
                FreeForFat(pBuffer, FatFormatWorkBufferSize);
            }

            // pf_format()後はフォーマットの成否に関わらず、マウント状態で返ってくる場合があるためアンマウント
            pfResult = pf_unmount(m_DriveHandle->driveLetter, PF_UNMOUNT_FORCE);
            if (pfResult != PF_RET_NO_ERR)
            {
                NN_SDK_LOG("[fs] Warning: Failed to unmount after format (%d)\n", pfResult);
            }
        };

        pfResult = pf_format_flags(m_DriveHandle->driveLetter, reinterpret_cast<const char*>(pFatFormatParam), false, !pFatFormatParam->isSdCard);
        if (pfResult != PF_RET_NO_ERR)
        {
            return m_pNnResultMaker->GetNnResultFromPfFormatError();
        }

        NN_RESULT_SUCCESS;
    }

    bool FatFileSystem::IsExFatSupported() NN_NOEXCEPT
    {
#if PF_EXFAT_SUPPORT
        return true;
#else /* !PF_EXFAT_SUPPORT */
        return false;
#endif /* !PF_EXFAT_SUPPORT */
    }

    Result FatFileSystem::Mount() NN_NOEXCEPT
    {
        bool isFatSafeEnabled = m_Attribute.isFatSafeEnabled;

        std::lock_guard<os::Mutex> scopedLock(m_Mutex);
        AllocatePfContext();

        NN_UTIL_SCOPE_EXIT
        {
            FreePfContext();
        };

        if (isFatSafeEnabled)
        {
            NN_RESULT_TRY(MountWithFatSafe())
                NN_RESULT_CATCH(ResultFatFileSystemCorrupted)
                {
                   isFatSafeEnabled = false;
                }
            NN_RESULT_END_TRY
        }

        if (!isFatSafeEnabled)
        {
            auto pfResult = pf_mount(m_DriveHandle->driveLetter);
            if (pfResult != PF_RET_NO_ERR)
            {
                return m_pNnResultMaker->GetNnResultFromPfError("mount");
            }
        }

        NN_UTIL_SCOPE_EXIT
        {
            if( !m_IsMounted )
            {
                Unmount();
            }
        };

        PF_DEV_INF dev_inf;
        auto pfResult = pf_getdev_light(m_DriveHandle->driveLetter, &dev_inf);
        if (pfResult != PF_RET_NO_ERR)
        {
            return m_pNnResultMaker->GetNnResultFromPfError("getdev_light");
        }
        m_BytesPerCluster = dev_inf.spc * dev_inf.bps;

        if (m_Attribute.isFatFormatNormalized)
        {
            if (dev_inf.fmt == PF_FMT_FAT32)
            {
                /* デフォルトの状態で、既に PF_FSI_TRUST_DISABLE, PF_FSI_REFRESH_DISABLE になっている */
                pfResult = pf_setupfsi(m_DriveHandle->driveLetter, PF_FSI_WRITE_UNKNOWN_ONCE);
                if (pfResult != PF_RET_NO_ERR)
                {
                    return m_pNnResultMaker->GetNnResultFromPfError("setupfsi");
                }
            }
#if PF_EXFAT_SUPPORT
            else if (dev_inf.fmt == PF_FMT_EXFAT)
            {
                /* exFATの場合、BootSectorのPercentInUseを書き込む */
                pfResult = pf_setupfsi(m_DriveHandle->driveLetter, PF_FSI_WRITE_PERINUSE_NOT_AVAILABLE_ONCE);
                if (pfResult != PF_RET_NO_ERR)
                {
                    return m_pNnResultMaker->GetNnResultFromPfError("setupfsi");
                }
            }
            else
            {
                /* 何もしない */
            }
#endif /* PF_EXFAT_SUPPORT */
        }

        if (!isFatSafeEnabled)
        {
            pfResult = pf_buffering(m_DriveHandle->driveLetter, PF_EJECT_SAFE_WITH_WRITE_BACK);
            if (pfResult != PF_RET_NO_ERR)
            {
                return m_pNnResultMaker->GetNnResultFromPfError("buffering");
            }
        }

        if (dev_inf.fmt != PF_FMT_EXFAT)
        {
            std::unique_ptr<TailBuffer> pTailBuffer(new TailBuffer());
            NN_RESULT_THROW_UNLESS(pTailBuffer != nullptr, ResultAllocationMemoryFailedInFatFileSystemD());

            pfResult = pf_settailbuf(m_DriveHandle->driveLetter, TailBuffer::TailBufferMax, pTailBuffer->buffer);
            if (pfResult != PF_RET_NO_ERR)
            {
                return m_pNnResultMaker->GetNnResultFromPfError("settailbuf");
            }
            m_TailBuffer = std::move(pTailBuffer);
        }

        m_IsMounted = true;
        NN_RESULT_SUCCESS;
    }


    Result FatFileSystem::MountWithFatSafe() NN_NOEXCEPT
    {
        int pfResult;

        std::unique_ptr<FatSafeContext> pFatSafeContext(new FatSafeContext());
        NN_RESULT_THROW_UNLESS(pFatSafeContext != nullptr, ResultAllocationMemoryFailedInFatFileSystemE());

        // Initialize FatSafe
        pFatSafeContext->safeInit.si_malloc = AllocateForFatSafe;
        pFatSafeContext->safeInit.si_free   = FreeForFatSafe;
        pfResult = pf_safe_initialize(m_DriveHandle->driveLetter, &pFatSafeContext->safeInit);
        if (pfResult != PF_RET_NO_ERR)
        {
            return m_pNnResultMaker->GetNnResultFromPfError("safe_initialize");
        }

        bool isSuccess = false;
        NN_UTIL_SCOPE_EXIT
        {
            if( !isSuccess )
            {
                pfResult = pf_safe_finalize(m_DriveHandle->driveLetter);
                if (pfResult != PF_RET_NO_ERR)
                {
                    // SIGLO-60683: 出荷状態のユーザパーティションに正しくアクセスできない件の W/A: エラーを無視する
                    NN_SDK_LOG("Failed to finalize fat safe for %c drive on prfile2 (%d)\n", m_DriveHandle->driveLetter, pf_errnum());
                }
            }
        };

        pfResult = pf_mount(m_DriveHandle->driveLetter);
        if (pfResult != PF_RET_NO_ERR)
        {
            return m_pNnResultMaker->GetNnResultFromPfError("mount");
        }

        NN_UTIL_SCOPE_EXIT
        {
            if( !isSuccess )
            {
                auto pfResult = pf_unmount(m_DriveHandle->driveLetter, PF_UNMOUNT_FORCE);
                if (pfResult != PF_RET_NO_ERR)
                {
                    NN_SDK_LOG("Failed to unmount %c drive on prfile2 (%d)\n", m_DriveHandle->driveLetter, pfResult);
                }
            }
        };

        // Recover corruption
        PF_SAFE_STAT safeStat;
        pfResult = pf_safe_check(m_DriveHandle->driveLetter, &safeStat);
        if (pfResult != PF_RET_NO_ERR)
        {
            return m_pNnResultMaker->GetNnResultFromPfError("safe_check");
        }

        if (safeStat.ss_stat == PF_SAFE_STAT_ABNORMAL)
        {
            pfResult = pf_safe_recover(m_DriveHandle->driveLetter);
            if (pfResult != PF_RET_NO_ERR)
            {
                return m_pNnResultMaker->GetNnResultFromPfError("safe_recover");
            }
        }
        else if (safeStat.ss_stat != PF_SAFE_STAT_NORMAL) // PF_SAFE_STAT_ERROR or PF_SAFE_STAT_UNSUPPORTED
        {
            return m_pNnResultMaker->GetResultInvalidfatFormat(safeStat.ss_stat, "safe_check");
        }

        // Enable FatSafe
        pfResult = pf_safe_backup_on(m_DriveHandle->driveLetter);
        if (pfResult != PF_RET_NO_ERR)
        {
            return m_pNnResultMaker->GetNnResultFromPfError("safe_backup_on");
        }


        m_FatSafeContext = std::move(pFatSafeContext);
        isSuccess = true;

        NN_RESULT_SUCCESS;
    }

    void FatFileSystem::Unmount() NN_NOEXCEPT
    {
        NN_UTIL_SCOPE_EXIT
        {
            // アンマウントにエラーが発生しても、マウント済みフラグは落とします
            m_IsMounted = false;
        };

        int pfResult;

        if( m_FatSafeContext != nullptr )
        {
            pfResult = pf_safe_backup_off(m_DriveHandle->driveLetter);
            if (pfResult != PF_RET_NO_ERR)
            {
                Result result = m_pNnResultMaker->GetNnResultFromPfError("safe_backup_off");
                NN_ABORT("Failed to disable fat safe on %c drive on prfile2 (Module:%d, Description:%d)\n", m_DriveHandle->driveLetter, result.GetModule(), result.GetDescription());
                NN_UNUSED(result);
            }
        }

        pfResult = pf_unmount(m_DriveHandle->driveLetter, PF_UNMOUNT_FORCE);
        if (pfResult == PF_RET_UNMOUNT_ERR)
        {
            Result result = m_pNnResultMaker->GetNnResultFromPfError("unmount");
            NN_SDK_LOG("[fs] Warning: forced unmount %c drive on prfile2 (Module:%d, Description:%d)\n", m_DriveHandle->driveLetter, result.GetModule(), result.GetDescription());
            NN_UNUSED(result);
        }
        else if (pfResult != PF_RET_NO_ERR)
        {
            Result result = m_pNnResultMaker->GetNnResultFromPfError("unmount");
            NN_ABORT("Failed to unmount %c drive on prfile2 (Module:%d, Description:%d)\n", m_DriveHandle->driveLetter, result.GetModule(), result.GetDescription());
            NN_UNUSED(result);
        }

        if( m_FatSafeContext != nullptr )
        {
            pfResult = pf_safe_finalize(m_DriveHandle->driveLetter);
            if (pfResult != PF_RET_NO_ERR)
            {
                Result result = m_pNnResultMaker->GetNnResultFromPfError("safe_finalize");
                NN_ABORT("Failed to finalize fat safe for %c drive on prfile2 (Module:%d, Description:%d)\n", m_DriveHandle->driveLetter, result.GetModule(), result.GetDescription());
                NN_UNUSED(result);
            }
        }
    }

    Result FatFileSystem::PfGetEntryType(DirectoryEntryType* outValue, const char* pFullPath) NN_NOEXCEPT
    {
        PF_STAT stat;
        auto pfResult = pf_fstat(pFullPath, &stat);
        if (pfResult != PF_RET_NO_ERR)
        {
            return m_pNnResultMaker->GetNnResultFromPfError("fstat");
        }

        if ((stat.fstat & ATTR_DIR) != 0)
        {
            *outValue = DirectoryEntryType_Directory;
        }
        else
        {
            *outValue = DirectoryEntryType_File;
        }

        NN_RESULT_SUCCESS;
    }

    Result FatFileSystem::DoGetEntryType(DirectoryEntryType* outValue, const char *path) NN_NOEXCEPT
    {
        char fullPath[PathLengthMax + 1];
        NN_RESULT_DO(ResolvePath(fullPath, sizeof(fullPath), path));

        std::lock_guard<os::Mutex> scopedLock(m_Mutex);
        AllocatePfContext();

        NN_UTIL_SCOPE_EXIT
        {
            FreePfContext();
        };

        return PfGetEntryType(outValue, fullPath);
    }

    Result FatFileSystem::DoGetFreeSpaceSize(int64_t* outValue, const char *path) NN_NOEXCEPT
    {
        NN_UNUSED(path);
        PF_DEV_INF dev_inf;

        std::lock_guard<os::Mutex> scopedLock(m_Mutex);
        AllocatePfContext();

        NN_UTIL_SCOPE_EXIT
        {
            FreePfContext();
        };

        auto pfResult = pf_getdev(m_DriveHandle->driveLetter, &dev_inf);
        if (pfResult != PF_RET_NO_ERR)
        {
            return m_pNnResultMaker->GetNnResultFromPfError("getdev");
        }

        const int64_t bytes_per_cluster = static_cast<int64_t>(dev_inf.spc) * dev_inf.bps;
        *outValue = dev_inf.ecl * bytes_per_cluster;

        NN_RESULT_SUCCESS;
    }

    Result FatFileSystem::DoGetTotalSpaceSize(int64_t* outValue, const char *path) NN_NOEXCEPT
    {
        NN_UNUSED(path);
        PF_DEV_INF dev_inf;

        std::lock_guard<os::Mutex> scopedLock(m_Mutex);
        AllocatePfContext();

        NN_UTIL_SCOPE_EXIT
        {
            FreePfContext();
        };

        auto pfResult = pf_getdev_light(m_DriveHandle->driveLetter, &dev_inf);
        if (pfResult != PF_RET_NO_ERR)
        {
            return m_pNnResultMaker->GetNnResultFromPfError("getdev_light");
        }

        const int64_t bytes_per_cluster = static_cast<int64_t>(dev_inf.spc) * dev_inf.bps;
        *outValue = dev_inf.cls * bytes_per_cluster;

        NN_RESULT_SUCCESS;
    }

namespace
{
    uint64_t ToPosixTimeValueFromPfTimeStmp(PF_SYS_DATE* pDate, PF_SYS_TIME* pTime)
    {
        nn::time::CalendarTime calendarTime;
        calendarTime.year   = pDate->sys_year;
        calendarTime.month  = static_cast<int8_t>(pDate->sys_month);
        calendarTime.day    = static_cast<int8_t>(pDate->sys_day);
        if ((pTime != nullptr) &&
            (pTime->sys_hour != PF_SETSTMP_NONE) &&
            (pTime->sys_min != PF_SETSTMP_NONE) &&
            (pTime->sys_sec != PF_SETSTMP_NONE) &&
            (pTime->sys_ms != PF_SETSTMP_NONE))
        {
            calendarTime.hour   = static_cast<int8_t>(pTime->sys_hour);
            calendarTime.minute = static_cast<int8_t>(pTime->sys_min);
            calendarTime.second = static_cast<int8_t>(pTime->sys_sec + pTime->sys_ms / 100);
        }
        else
        {
            calendarTime.hour   = 0;
            calendarTime.minute = 0;
            calendarTime.second = 0;
        }

        nn::time::PosixTime posixTime = nn::time::ToPosixTimeFromUtc(calendarTime);
        return posixTime.value;
    }
}

    Result FatFileSystem::GetEntryTimeStamp(nn::fs::FileTimeStampRaw* outTimeStamp, const char* pFullPath) NN_NOEXCEPT
    {
        PF_TIMESTMP timestmp;
        auto pfResult = pf_getstamp(pFullPath, &timestmp);
        if (pfResult != PF_RET_NO_ERR)
        {
            return m_pNnResultMaker->GetNnResultFromPfError("getstamp");
        }

        outTimeStamp->create = ToPosixTimeValueFromPfTimeStmp(&timestmp.cre_date, &timestmp.cre_time);
        outTimeStamp->modify = ToPosixTimeValueFromPfTimeStmp(&timestmp.mod_date, &timestmp.mod_time);
#if PF_EXFAT_SUPPORT
        outTimeStamp->access = ToPosixTimeValueFromPfTimeStmp(&timestmp.acc_date, &timestmp.acc_time);
#else /* !PF_EXFAT_SUPPORT */
        outTimeStamp->access = ToPosixTimeValueFromPfTimeStmp(&timestmp.acc_date, nullptr);
#endif /* !PF_EXFAT_SUPPORT */
        outTimeStamp->isLocalTime = true;

        NN_RESULT_SUCCESS;
    }

    Result FatFileSystem::DoGetFileTimeStampRaw(nn::fs::FileTimeStampRaw* outTimeStamp, const char* path) NN_NOEXCEPT
    {
        char fullPath[PathLengthMax + 1];
        NN_RESULT_DO(ResolvePath(fullPath, sizeof(fullPath), path));

        std::lock_guard<os::Mutex> scopedLock(m_Mutex);
        AllocatePfContext();

        NN_UTIL_SCOPE_EXIT
        {
            FreePfContext();
        };

        PF_STAT stat;
        auto pfResult = pf_fstat(fullPath, &stat);
        if (pfResult != PF_RET_NO_ERR)
        {
            return m_pNnResultMaker->GetNnResultFromPfError("fstat");
        }
        if ((stat.fstat & ATTR_DIR) != 0)
        {
            return ResultPathNotFound();
        }

        NN_RESULT_DO(GetEntryTimeStamp(outTimeStamp, fullPath));

        NN_RESULT_SUCCESS;
    }

    Result FatFileSystem::DoFlush() NN_NOEXCEPT
    {
        std::lock_guard<os::Mutex> scopedLock(m_Mutex);
        AllocatePfContext();

        NN_UTIL_SCOPE_EXIT
        {
            FreePfContext();
        };

        auto pfResult = pf_sync(m_DriveHandle->driveLetter, PF_NINVALIDATE);
        if (pfResult != PF_RET_NO_ERR)
        {
            return m_pNnResultMaker->GetNnResultFromPfError("sync");
        }

        NN_RESULT_SUCCESS;
    }

    Result FatFileSystem::DoOpenFile(std::unique_ptr<fsa::IFile>* outValue, const char* path, OpenMode mode) NN_NOEXCEPT
    {
        bool unlockOnExit = false;

        NN_FSP_REQUIRES( (mode & (nn::fs::OpenMode_Read | nn::fs::OpenMode_Write)) != 0, fs::ResultInvalidArgument(), "OpenMode_Read or OpenMode_Write is required.");

        char fullPath[PathLengthMax + 1];
        NN_RESULT_DO(ResolvePath(fullPath, sizeof(fullPath), path));

        PF_FILE* pPfFile = nullptr;

        std::lock_guard<os::Mutex> scopedLock(m_Mutex);
        AllocatePfContext();

        NN_UTIL_SCOPE_EXIT
        {
            if( unlockOnExit && pPfFile != nullptr)
            {
                pf_flock(pPfFile, PF_LOCK_UN);
            }

            if( pPfFile != nullptr )
            {
                pf_fclose(pPfFile);
            }

            FreePfContext();
        };


        const char readOnlyMode[] = "r";
        const char readWriteMode[] = "r+";
        if (mode & OpenMode_Write || mode & OpenMode_AllowAppend)
        {
            pPfFile = pf_fopen(fullPath, readWriteMode);
        }
        else
        {
            pPfFile = pf_fopen(fullPath, readOnlyMode);
        }
        if (pPfFile == nullptr)
        {
            return m_pNnResultMaker->GetNnResultFromPfError("fopen");
        }

        auto lockMode = PF_LOCK_NB | (((mode & OpenMode_Write) != 0) ? PF_LOCK_EX
                                                                     : PF_LOCK_SH);

        auto pfResult = pf_flock(pPfFile, lockMode);
        if (pfResult != PF_RET_NO_ERR)
        {
            return m_pNnResultMaker->GetNnResultFromPfError("flock");
        }

        unlockOnExit = true;

        std::unique_ptr<FatFile> file(new FatFile(mode, pPfFile, m_pNnResultMaker.get(), m_BytesPerCluster, &m_FatStorageAdapter));
        if (file == nullptr)
        {
            return ResultAllocationMemoryFailedInFatFileSystemF();
        }

        if ((mode & OpenMode_AllowAppend) == 0)
        {
            file->SetClusterLink();
        }

        *outValue = std::move(file);

        unlockOnExit = false;
        pPfFile = nullptr;

        NN_RESULT_SUCCESS;
    }

    Result FatFileSystem::DoOpenDirectory(std::unique_ptr<fsa::IDirectory>* outValue, const char* path, OpenDirectoryMode mode) NN_NOEXCEPT
    {
        char fullPath[PathLengthMax + 1];
        NN_RESULT_DO(ResolvePath(fullPath, sizeof(fullPath), path));

        PF_DIR* pPfDir;

        {
            std::lock_guard<os::Mutex> scopedLock(m_Mutex);
            AllocatePfContext();

            NN_UTIL_SCOPE_EXIT
            {
                FreePfContext();
            };

            // check existence
            pPfDir = pf_opendir(fullPath);
            if (pPfDir == nullptr)
            {
                return m_pNnResultMaker->GetNnResultFromPfError("opendir");
            }
        }

        std::unique_ptr<FatDirectory> directory(new FatDirectory(mode, fullPath, pPfDir, m_pNnResultMaker.get()));
        if (directory == nullptr)
        {
            return ResultAllocationMemoryFailedInFatFileSystemH();
        }

        *outValue = std::move(directory);

        NN_RESULT_SUCCESS;
    }

    Result FatFileSystem::DoCreateFile(const char *path, int64_t size, int option) NN_NOEXCEPT
    {
        NN_UNUSED(option);
        char fullPath[PathLengthMax + 1];
        NN_RESULT_DO(ResolvePath(fullPath, sizeof(fullPath), path));

        if( size < 0 || size > PfFilePosMaxValueForInt64)
        {
            return ResultOutOfRange();
        }

        std::lock_guard<os::Mutex> scopedLock(m_Mutex);
        AllocatePfContext();

        NN_UTIL_SCOPE_EXIT
        {
            FreePfContext();
        };

        auto pPfFile = pf_create(fullPath, 0); // TBD: PF_OPEN_MODE_CONT_CLUSTER?
        if (pPfFile == nullptr)
        {
            return m_pNnResultMaker->GetNnResultFromPfError("create");
        }

        int pfResult = pf_ftruncate(pPfFile, static_cast<PF_FPOS_T>(size));
        if( pfResult != PF_RET_NO_ERR )
        {
            Result result = m_pNnResultMaker->GetNnResultFromPfError("ftruncate");
            pf_fclose(pPfFile);
            pf_remove(fullPath);
            return result;
        }

        pfResult = pf_fclose(pPfFile);
        if( pfResult != PF_RET_NO_ERR )
        {
            /* pf_fclose()失敗時はオープン状態のままなため、pf_remove()できない */
            return m_pNnResultMaker->GetNnResultFromPfError("fclose");
        }

        NN_RESULT_SUCCESS;
    }

    Result FatFileSystem::DoDeleteFile(const char *path) NN_NOEXCEPT
    {
        char fullPath[PathLengthMax + 1];
        NN_RESULT_DO(ResolvePath(fullPath, sizeof(fullPath), path));

        std::lock_guard<os::Mutex> scopedLock(m_Mutex);
        AllocatePfContext();

        NN_UTIL_SCOPE_EXIT
        {
            FreePfContext();
        };

        auto pfResult = pf_remove(fullPath);
        if( pfResult != PF_RET_NO_ERR )
        {
            int pfError = pf_errnum();
            if( pfError == PF_ERR_EACCES )
            {
                // ディレクトリを消そうとした or ファイルがオープン中
                DirectoryEntryType type = DirectoryEntryType_File;
                NN_RESULT_DO(PfGetEntryType(&type, fullPath));
                if( type == DirectoryEntryType_Directory )
                {
                    return ResultPathNotFound();
                }
            }

            return m_pNnResultMaker->GetNnResultFromPfError("remove");
        }

        NN_RESULT_SUCCESS;
    }

    Result FatFileSystem::CreateDirectory(const char *path, int attribute) NN_NOEXCEPT
    {
        char fullPath[PathLengthMax + 1];
        NN_RESULT_DO(ResolvePath(fullPath, sizeof(fullPath), path));

        std::lock_guard<os::Mutex> scopedLock(m_Mutex);
        AllocatePfContext();

        NN_UTIL_SCOPE_EXIT
        {
            FreePfContext();
        };

        auto pfResult = pf_mkdir(fullPath);
        if( pfResult != PF_RET_NO_ERR )
        {
            return m_pNnResultMaker->GetNnResultFromPfError("mkdir");
        }

        if( attribute & Attribute::Attribute_Archive )
        {
            pfResult = pf_chdmod(fullPath, ATTR_ARCH | ATTR_DIR);
            if( pfResult != PF_RET_NO_ERR )
            {
                return m_pNnResultMaker->GetNnResultFromPfError("chdmod");
            }
        }

        NN_RESULT_SUCCESS;

    }

    Result FatFileSystem::DoCreateDirectory(const char *path) NN_NOEXCEPT
    {
        return CreateDirectory(path, 0);
    }

    Result FatFileSystem::DoDeleteDirectory(const char *path) NN_NOEXCEPT
    {
        char fullPath[PathLengthMax + 1];
        NN_RESULT_DO(ResolvePath(fullPath, sizeof(fullPath), path));

        std::lock_guard<os::Mutex> scopedLock(m_Mutex);
        AllocatePfContext();

        NN_UTIL_SCOPE_EXIT
        {
            FreePfContext();
        };

        auto pfResult = pf_rmdir(fullPath);
        if( pfResult != PF_RET_NO_ERR )
        {
            return m_pNnResultMaker->GetNnResultFromPfError("rmdir");
        }

        NN_RESULT_SUCCESS;
    }

    Result FatFileSystem::DoDeleteDirectoryRecursively(const char *path) NN_NOEXCEPT
    {
        char fullPath[PathLengthMax + 1];
        NN_RESULT_DO(ResolvePath(fullPath, sizeof(fullPath), path));

        std::lock_guard<os::Mutex> scopedLock(m_Mutex);
        AllocatePfContext();

        NN_UTIL_SCOPE_EXIT
        {
            FreePfContext();
        };

        auto pfResult = pf_deletedir(fullPath);
        if( pfResult != PF_RET_NO_ERR )
        {
            return m_pNnResultMaker->GetNnResultFromPfError("deletedir");
        }

        NN_RESULT_SUCCESS;
    }

    Result FatFileSystem::DoCleanDirectoryRecursively(const char* path) NN_NOEXCEPT
    {
        char fullPath[PathLengthMax + 1];
        NN_RESULT_DO(ResolvePath(fullPath, sizeof(fullPath), path));

        std::lock_guard<os::Mutex> scopedLock(m_Mutex);
        AllocatePfContext();

        NN_UTIL_SCOPE_EXIT
        {
            FreePfContext();
        };

        unsigned long count = 0;
        auto pfResult = pf_cleandir(fullPath, "*", PF_CLEANDIR_TREE | ATTR_FILE_DIR, &count);
        if( pfResult != PF_RET_NO_ERR )
        {
            return m_pNnResultMaker->GetNnResultFromPfError("cleandir");
        }

        NN_RESULT_SUCCESS;
    }

    Result FatFileSystem::DoRenameFile(const char *currentPath, const char *newPath) NN_NOEXCEPT
    {
        char currentFullPath[PathLengthMax];
        char newFullPath[PathLengthMax];
        NN_RESULT_DO(ResolvePath(currentFullPath, sizeof(currentFullPath), currentPath));
        NN_RESULT_DO(ResolvePath(newFullPath,     sizeof(newFullPath),     newPath));

        std::lock_guard<os::Mutex> scopedLock(m_Mutex);
        AllocatePfContext();

        NN_UTIL_SCOPE_EXIT
        {
            FreePfContext();
        };

        DirectoryEntryType type = DirectoryEntryType_Directory;
        NN_RESULT_DO(PfGetEntryType(&type, currentFullPath));

        if( type != DirectoryEntryType_File )
        {
            return ResultPathNotFound();
        }

        auto pfResult = pf_move(currentFullPath, newFullPath);
        if( pfResult != PF_RET_NO_ERR )
        {
            return m_pNnResultMaker->GetNnResultFromPfError("move");
        }

        NN_RESULT_SUCCESS;
    }

    Result FatFileSystem::DoRenameDirectory(const char *currentPath, const char *newPath) NN_NOEXCEPT
    {
        char currentFullPath[PathLengthMax];
        char newFullPath[PathLengthMax];
        NN_RESULT_DO(ResolvePath(currentFullPath, sizeof(currentFullPath), currentPath));
        NN_RESULT_DO(ResolvePath(newFullPath,     sizeof(newFullPath),     newPath));

        std::lock_guard<os::Mutex> scopedLock(m_Mutex);
        AllocatePfContext();

        NN_UTIL_SCOPE_EXIT
        {
            FreePfContext();
        };

        DirectoryEntryType type = DirectoryEntryType_File;
        NN_RESULT_DO(PfGetEntryType(&type, currentFullPath));

        if( type != DirectoryEntryType_Directory )
        {
            return ResultPathNotFound();
        }

        auto pfResult = pf_move(currentFullPath, newFullPath);
        if( pfResult != PF_RET_NO_ERR )
        {
            return m_pNnResultMaker->GetNnResultFromPfError("move");
        }

        NN_RESULT_SUCCESS;
    }

    Result FatFileSystem::DoCommit() NN_NOEXCEPT
    {
        NN_RESULT_SUCCESS;
    }

    Result FatFileSystem::GetAttribute(int* outValue, const char *path) NN_NOEXCEPT
    {
        char fullPath[PathLengthMax + 1];
        NN_RESULT_DO(ResolvePath(fullPath, sizeof(fullPath), path));

        std::lock_guard<os::Mutex> scopedLock(m_Mutex);
        AllocatePfContext();

        NN_UTIL_SCOPE_EXIT
        {
            FreePfContext();
        };

        PF_STAT stat;
        auto pfResult = pf_fstat(fullPath, &stat);
        if (pfResult != PF_RET_NO_ERR)
        {
            return m_pNnResultMaker->GetNnResultFromPfError("fstat");
        }

        *outValue = ((stat.fstat & ATTR_ARCH) != 0 ? Attribute::Attribute_Archive
                                                   : 0) |
                    ((stat.fstat & ATTR_DIR)  != 0 ? Attribute::Attribute_Directory
                                                   : 0);

        NN_RESULT_SUCCESS;
    }

    Result FatFileSystem::SetAttribute(const char *path, int attribute) NN_NOEXCEPT
    {
        char fullPath[PathLengthMax + 1];
        NN_RESULT_DO(ResolvePath(fullPath, sizeof(fullPath), path));

        std::lock_guard<os::Mutex> scopedLock(m_Mutex);
        AllocatePfContext();

        NN_UTIL_SCOPE_EXIT
        {
            FreePfContext();
        };

        int fatAttribute = (attribute & Attribute::Attribute_Archive) ? ATTR_ARCH : 0;
        auto pfResult = pf_chdmod(fullPath, fatAttribute);
        if (pfResult != PF_RET_NO_ERR)
        {
            return m_pNnResultMaker->GetNnResultFromPfError("chdmod");
        }

        NN_RESULT_SUCCESS;
    }

    Result FatFileSystem::GetFileSize(int64_t* outValue, const char *path) NN_NOEXCEPT
    {
        char fullPath[PathLengthMax + 1];
        NN_RESULT_DO(ResolvePath(fullPath, sizeof(fullPath), path));

        std::lock_guard<os::Mutex> scopedLock(m_Mutex);
        AllocatePfContext();

        NN_UTIL_SCOPE_EXIT
        {
            FreePfContext();
        };

        PF_STAT stat;
        auto pfResult = pf_fstat(fullPath, &stat);
        if (pfResult != PF_RET_NO_ERR)
        {
            return m_pNnResultMaker->GetNnResultFromPfError("fstat");
        }

        *outValue = stat.fstfz;
        NN_RESULT_SUCCESS;
    }

}}
