﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstring>
#include <nn/util/util_BitUtil.h>
#include <nn/fs/fs_IStorage.h>
#include <nn/fat/fat_FatFileSystem.h>
#include "prfile2/pf_config.h"
#include "prfile2/pdm_driver.h"
#include "prfile2/pdm_uplyrif.h"
#include "prfile2/pdm_common.h"
#include "prfile2/plugin/safe/fatsafe/pf_sm_safe_config.h"
#include "pf_system.h"
#include "../fatfs/pf_entry_p.h"
#include "../dskmng/pdm_mbr_p.h"
#include "../dskmng/pdm_bpb_p.h"
#if PF_EXFAT_SUPPORT
#if PF_EXFAT_FORMAT_SUPPORT
#include "../dskmng/pdm_exfat_bss_p.h"
#include "pf_entry.h"
#include "pf_exfat_entry.h"
#include "prfile2/pdm_exfat_common.h"
#endif /* PF_EXFAT_FORMAT_SUPPORT */
#endif /* PF_EXFAT_SUPPORT */

namespace {

using namespace nne::prfile2;

static const size_t SectorSize = 512;

static const PDM_U_CHAR PDM_MBR_PARTTYPE_FAT12      = 0x01;
static const PDM_U_CHAR PDM_MBR_PARTTYPE_FAT16      = 0x04;
static const PDM_U_CHAR PDM_MBR_PARTTYPE_FAT16_OVER32MB = 0x06;
static const PDM_U_CHAR PDM_MBR_PARTTYPE_FAT32      = 0x0b;
static const PDM_U_CHAR PDM_MBR_PARTTYPE_FAT32_LBA  = 0x0c;
#if PF_EXFAT_SUPPORT && PF_EXFAT_FORMAT_SUPPORT
static const PDM_U_CHAR PDM_MBR_PARTTYPE_EXFAT      = 0x07;
#endif /* PF_EXFAT_SUPPORT && PF_EXFAT_FORMAT_SUPPORT */

static const PDM_U_CHAR PDM_BPB_FAT_BIT_EXFAT       = 32;

#if PF_EXFAT_SUPPORT && PF_EXFAT_FORMAT_SUPPORT
static const PDM_U_LONG EXFAT_BSS_EXTEND_BOOT_SECTOR    = 1;
static const PDM_U_LONG EXFAT_BSS_OEM_PARAM_SECTOR  = 9;
static const PDM_U_LONG EXFAT_BSS_CHECKSUM_SECTOR   = 11;
#endif /* PF_EXFAT_SUPPORT && PF_EXFAT_FORMAT_SUPPORT */

static const PDM_U_SHORT FAT_BPB_FSINFO_SECTOR      = 1; /* for FAT32 only */
static const PDM_U_SHORT FAT_BPB_RESERVED_FOR_BOOT_SECTOR   = 2; /* for FAT32 only */
static const PDM_U_SHORT FAT_BPB_BACKUP_SECTOR      = 6; /* for FAT32 only */

static const PDM_U_SHORT FAT_BPB_RESERVED_SEC       = 1; /* for FAT12/FAT16 only */
static const PDM_U_SHORT FAT_BPB_RESERVED_SEC32     = 9; /* for FAT32 only */

/* BPB data size */
static const PF_SIZE_T FAT_BPB_SIZE                 = 512;

/* FSINFO data size */
static const PF_SIZE_T FAT_FSINFO_SIZE              = 512;

static const PF_CHAR FAT_BPB_FS_TYPE_12[]           = "FAT12   ";
static const PF_CHAR FAT_BPB_FS_TYPE_16[]           = "FAT16   ";
static const PF_CHAR FAT_BPB_FS_TYPE_32[]           = "FAT32   ";
#if PF_EXFAT_SUPPORT && PF_EXFAT_FORMAT_SUPPORT
static const PF_CHAR FAT_BPB_FS_TYPE_EXFAT[]        = "EXFAT   ";
#endif /* PF_EXFAT_SUPPORT && PF_EXFAT_FORMAT_SUPPORT */

static const PDM_U_CHAR FAT_BPB_EXT_BOOT_SIG        = 0x29;
static const PDM_U_CHAR FAT_BPB_DRIVE_NON_REMOVABLE     = 0x80;
static const PDM_U_SHORT FAT_BPB_ROOT_ENT_CNT32     = 0; /* for FAT12/FAT16 only */

static const PDM_U_CHAR FAT_BPB_NUM_FATS            = 2;
#if PF_EXFAT_SUPPORT && PF_EXFAT_FORMAT_SUPPORT
static const PDM_U_CHAR FAT_BPB_NUM_FATS_EXFAT      = 1;
#endif /* PF_EXFAT_SUPPORT && PF_EXFAT_FORMAT_SUPPORT */

static const PDM_U_CHAR NANDDRV_MEDIA_DESCRIPTOR    = PDM_BPB_MEDIA_0xF8;

static const PDM_U_LONG NAND_BPB_VOL_ID             = 0;
static const PDM_U_SHORT NAND_BPB_FS_VER            = 0;
static const PDM_U_SHORT NAND_BPB_ROOT_ENT_CNT      = 512;
static const PF_CHAR NAND_BPB_DEFAULT_VOL_LABEL[]   = "NO NAME    ";
static const PDM_U_LONG NAND_BOUNDARY_UNIT_SECTORS  = 32;

static const PDM_U_LONG FAT_SIZE_ALIGNMENT_SECTORS  = 32;

/* BPB Sectors per track */
static const PDM_U_SHORT NANDDRV_SECTORS_PER_TRACK  = 63;

/* BPB Heads */
static const PDM_U_SHORT NANDDRV_HEADS              = 16;


#define KBYTES_TO_SECTORS(v)                        ((v) * 1024ULL / SectorSize)
#define MBYTES_TO_SECTORS(v)                        ((v) * 1024ULL / SectorSize * 1024ULL)
#define GBYTES_TO_SECTORS(v)                        ((v) * 1024ULL / SectorSize * 1024ULL * 1024ULL)

static const PDM_U_CHAR SD_FIXED_VALUE_CHS_HEAD     = 0xfe;
static const PDM_U_CHAR SD_FIXED_VALUE_CHS_SECTOR   = 0x3f;
static const PDM_U_SHORT SD_FIXED_VALUE_CHS_CYLINDER    = 0x3ff;

static const PDM_U_CHAR SDDRV_MEDIA_DESCRIPTOR      = PDM_BPB_MEDIA_0xF8;

static const PDM_U_SHORT SDDRV_BPB_FS_VER           = 0;
static const PDM_U_SHORT SD_BPB_ROOT_ENT_CNT        = 512;
static const PF_CHAR SD_BPB_DEFAULT_VOL_LABEL[]     = "NO NAME    ";

static const PDM_U_LONG SD_CEIL_ROOT_DIR_ENTRIES    = 32 * SD_BPB_ROOT_ENT_CNT / SectorSize;

/* SDHC CHS  8032.5MB = Sector Disp */
static const PDM_U_LONG SDDRV_MAX_CHS_VALUE         = KBYTES_TO_SECTORS(8225280);

/* Maximum Cluster Number is computed as following */
#define MAX_CLUSTER_NUM(ts, nom, ssa, sc)           (((ts) - (nom) - (ssa)) / (sc) + 1)

/* Sectors per FAT(SF') is recalculated as following */
#define ROUNDUP_DIV(x, y)                           (((x) + (y) - 1) / (y))
#define SECTORS_PER_FAT_DASH(max, bits, ss)         ROUNDUP_DIV((2 + ((max) - 1)) * (bits), (ss) * PDM_BPB_BITS_PER_BYTE)

#if PF_EXFAT_SUPPORT && PF_EXFAT_FORMAT_SUPPORT
static const PDM_U_SHORT SDDRV_EXFAT_FS_REVISION    = 0x0100;
#endif /* PF_EXFAT_SUPPORT && PF_EXFAT_FORMAT_SUPPORT */

typedef uint64_t MEDIA_SECTORS_WIDTH;

typedef struct {
    MEDIA_SECTORS_WIDTH total_sectors_min;
    MEDIA_SECTORS_WIDTH total_sectors_max;
    PDM_U_LONG          boundary_unit;
    PDM_U_SHORT         number_of_heads;
    PDM_U_SHORT         sectors_per_track;
    PDM_U_LONG          sectors_per_cluster;
    PDM_U_LONG          fat_bits;
    PDM_FAT_TYPE        fat_type;
} SD_SPEC_PARAM;

nn::fat::detail::FatStorageAdapter* GetFatStorageAdapter(PDM_DISK* pDisk)
{
    return reinterpret_cast<nn::fat::detail::FatStorageAdapter*>(pdm_disk_get_extended_info(pDisk));
}


PDM_ERROR GetTotalSectors(PF_U_LONG* pOutTotalSectors, PDM_DISK* pDisk)
{
    int64_t size;
    auto result = GetFatStorageAdapter(pDisk)->GetSize(&size);
    if (result.IsFailure())
    {
        return PDM_ERR_DRIVER_ERROR;
    }

    *pOutTotalSectors = static_cast<PF_U_LONG>(size / SectorSize);

    return PDM_OK;
}

nn::Result PhysicalWrite(PDM_DISK* pDisk, uint32_t block, uint32_t num_blocks, const void* buf)
{
    NN_RESULT_DO(GetFatStorageAdapter(pDisk)->Write(static_cast<int64_t>(block) * SectorSize, buf, num_blocks * SectorSize));
    NN_RESULT_SUCCESS;
}

nn::Result PhysicalRead(PDM_DISK* pDisk, uint32_t block, uint32_t num_blocks, void* buf)
{
    NN_RESULT_DO(GetFatStorageAdapter(pDisk)->Read(static_cast<int64_t>(block) * SectorSize, buf, num_blocks * SectorSize));
    NN_RESULT_SUCCESS;
}

nn::Result PhysicalWriteVerify(PDM_DISK* p_disk, uint32_t block, uint32_t num_blocks, const void* buf)
{
    NN_RESULT_DO(PhysicalWrite(p_disk, block, num_blocks, buf));

    uint8_t readBuf[SectorSize];
    uint32_t i;

    for (i = 0; i < num_blocks; i++)
    {
        NN_RESULT_DO(PhysicalRead(p_disk, block + i, 1, readBuf));

        if (std::memcmp(static_cast<const uint8_t*>(buf) + i * SectorSize, readBuf, SectorSize) != 0)
        {
            return nn::fs::ResultFatFsWriteVerifyError();
        }
    }

    NN_RESULT_SUCCESS;
}

/*---------------------------------------------------------------------------
   Build up FSInfo Sector into specfied buffer.
  ---------------------------------------------------------------------------*/
nn::Result  BuildUpFSInfoSector(PDM_U_CHAR* buf)
{
    PDM_FSINFO  fsinfo;

    /* Reset the FSINFO field */
    pf_memset(buf, 0x0, FAT_FSINFO_SIZE);

    /* Set up fields of FSInfo */
    fsinfo.free_count = PDM_FSI_FREE_COUNT_UNKNOWN;
    fsinfo.next_free = PDM_FSI_NEXT_FREE_UNKNOWN;
    if (pdm_bpb_set_fsinfo_information(&fsinfo, buf) != PDM_OK)
    {
        /* pdm_bpb_set_fsinfo_information()が返すエラーはPDM_ERR_INVALID_PARAMETERのみ */
        return nn::fs::ResultFatFsFormatInvalidParameter();
    }

    NN_RESULT_SUCCESS;
}


void  nanddrv_AdjustBpb(PDM_BPB* pBpb)
{
    PDM_U_LONG remain;
    PDM_U_LONG fat_region_sectors;
    PDM_U_LONG first_data_sector;
    PDM_U_LONG sectors_per_FAT = pBpb->sectors_per_FAT;

    remain = sectors_per_FAT % FAT_SIZE_ALIGNMENT_SECTORS;
    if (remain != 0)
    {
        sectors_per_FAT += FAT_SIZE_ALIGNMENT_SECTORS - remain;
        pBpb->sectors_per_FAT = sectors_per_FAT;
        pBpb->sectors_per_FAT32 = sectors_per_FAT;
    }

    fat_region_sectors = pBpb->num_FATs * sectors_per_FAT;
    first_data_sector = pBpb->num_reserved_sectors + fat_region_sectors;
    remain = first_data_sector % NAND_BOUNDARY_UNIT_SECTORS;
    if (remain != 0)
    {
        first_data_sector += NAND_BOUNDARY_UNIT_SECTORS - remain;
        pBpb->num_reserved_sectors = (PDM_U_SHORT)(first_data_sector - fat_region_sectors);
    }
}

/*---------------------------------------------------------------------------
   Build up Boot Sector into specfied buffer.
  ---------------------------------------------------------------------------*/
PDM_ERROR  nanddrv_BuildUpBootSector(PDM_U_CHAR* buf, PDM_FAT_TYPE* type, PDM_U_LONG total_sectors)
{
    PDM_ERROR       err;
    PDM_BPB_FORMAT  format;
    PDM_BPB         bpb;
    PDM_FAT_TYPE    fat_type;

    /* Reset the BPB field */
    pf_memset(buf, 0x0, FAT_BPB_SIZE);

    format.bytes_per_sector = SectorSize;
    format.total_sectors = total_sectors;

    // BPB に適当な値を埋め込むが、PrFILE2 では参照されない
    format.sector_per_track = NANDDRV_SECTORS_PER_TRACK;
    format.num_heads = NANDDRV_HEADS;

    /* get_disk_infoで取得した値から容量を算出し、「FAT種別」と「クラスタあたりのセクタ数」を決定 */
    if (format.total_sectors < (32U * 1024 * 1024 / format.bytes_per_sector))  //32MB未満 → FAT12、sectors_per_cluster:32
    {
        fat_type = PDM_FAT_12;
        pf_memcpy(format.fs_type, FAT_BPB_FS_TYPE_12, pf_strlen(FAT_BPB_FS_TYPE_12));
        format.sectors_per_cluster = 32;
    }
    else if (format.total_sectors < (1024U * 1024 * 1024 / format.bytes_per_sector))  //1GB未満 → FAT32、sectors_per_cluster:1
    {
        fat_type = PDM_FAT_32;
        pf_memcpy(format.fs_type, FAT_BPB_FS_TYPE_32, pf_strlen(FAT_BPB_FS_TYPE_32));
        format.sectors_per_cluster = 1;
    }
    else //1GB以上 → FAT32、sectors_per_cluster:32
    {
        fat_type = PDM_FAT_32;
        pf_memcpy(format.fs_type, FAT_BPB_FS_TYPE_32, pf_strlen(FAT_BPB_FS_TYPE_32));
        format.sectors_per_cluster = 32;
    }
    format.fat_type = fat_type;

    /* build basic BPB informations */
    format.jump_boot[0] = PDM_BPB_JC0_0xEB;
    format.jump_boot[1] = PDM_BPB_JC0_0xE9;
    format.jump_boot[2] = PDM_BPB_JC2_0x90;
    pf_memcpy(format.vol_label, NAND_BPB_DEFAULT_VOL_LABEL, pf_strlen(NAND_BPB_DEFAULT_VOL_LABEL));
    pf_memset(format.oem_name, 0x0, 8);
    format.num_fats             = FAT_BPB_NUM_FATS;
    format.media                = NANDDRV_MEDIA_DESCRIPTOR;
    format.drive                = FAT_BPB_DRIVE_NON_REMOVABLE;
    format.num_hidden_sectors   = 0x00000000;
    format.vol_id               = NAND_BPB_VOL_ID;
    format.fs_version           = NAND_BPB_FS_VER;
    format.boot_sig             = FAT_BPB_EXT_BOOT_SIG;


    if (format.fat_type != PDM_FAT_32)
    {
        format.num_reserved_sectors = FAT_BPB_RESERVED_SEC;
        format.num_root_dir_entries = NAND_BPB_ROOT_ENT_CNT;
        format.root_dir_cluster     = 0x00000000;
        format.fs_info_sector       = 0x0000;
        format.ext_flags            = 0x0000;
        format.backup_boot_sector   = 0x0000;
    }
    else //FAT32
    {
        format.num_reserved_sectors = FAT_BPB_RESERVED_SEC32;
        format.num_root_dir_entries = FAT_BPB_ROOT_ENT_CNT32;
        format.root_dir_cluster     = 0x00000002;
        format.fs_info_sector       = FAT_BPB_FSINFO_SECTOR;
        format.ext_flags            = 0x0000;
        format.backup_boot_sector   = FAT_BPB_BACKUP_SECTOR;
    }

    err = pdm_bpb_build_boot_sector(&format, &bpb);
    if (err == PDM_OK)
    {
        if (format.fat_type == PDM_FAT_32)
        {
            nanddrv_AdjustBpb(&bpb);
        }
        err = pdm_bpb_set_bpb_information(&bpb, buf);
        if (err == PDM_OK)
        {
            /* set FAT type */
            *type = format.fat_type;
        }
    }

    return err;
}

/*---------------------------------------------------------------------------
  nanddrv_format - Do physical formatting of media on the specified drive.

  [Description]
    This function does phtsical formatting of the volume specified by 'p_disk'.
  ---------------------------------------------------------------------------*/
nn::Result  nanddrv_format(PDM_DISK* p_disk, PDM_U_LONG totalSectors)
{
    PDM_U_CHAR    buf[SectorSize];
    PDM_FAT_TYPE  type;
    PDM_ERROR     err;

    if(!p_disk){
        return nn::fs::ResultFatFsFormatInvalidParameter();
    }

    /* Buile boot block image */
    err = nanddrv_BuildUpBootSector(buf, &type, totalSectors);
    if (err)
    {
        return nn::fs::ResultFatFsFormatInvalidBpb();
    }
    NN_RESULT_DO(PhysicalWrite(p_disk, 0, 1, buf));

    //FAT32であれば、FSInfoも処理を行う
    if (type == PDM_FAT_32)
    {
        PDM_U_SHORT     backup_boot_sector = PDM_LOAD_BPB_BAK_BS_SECTOR(buf);
        NN_RESULT_DO(PhysicalWrite(p_disk, backup_boot_sector, 1, buf));

        PDM_U_SHORT     fs_info_sector = PDM_LOAD_BPB_FS_INFO_SECTOR(buf);
        /* Buildl FSINFO block image */
        NN_RESULT_DO(BuildUpFSInfoSector(buf));
        NN_RESULT_DO(PhysicalWrite(p_disk, fs_info_sector, 1, buf));
        NN_RESULT_DO(PhysicalWrite(p_disk, backup_boot_sector + 1, 1, buf));

        // Zero-fill sector for FatSafe
        pf_memset(buf, 0, SectorSize);
        NN_RESULT_DO(PhysicalWrite(p_disk, PFSAFE_META_DATA_SECTOR, 1, buf));
    }

    NN_RESULT_SUCCESS;
}

/* CardCapacity の小さいものから定義しないといけない */
const SD_SPEC_PARAM  sd_spec_param_table[] = {
    {
        /*     0    < CardCapacity <=     2MB */
        1,                                 /* 1) Total Sectors (min)               */
        MBYTES_TO_SECTORS(2),              /* 2) Total Sectors (max)               */
        KBYTES_TO_SECTORS(8),              /* 3) Boundary Unit                     */
        2,                                 /* 4) Number Of Heads[Number Of Sides]  */
        16,                                /* 5) Sectors Per Track                 */
        16,                                /* 6) Sectors Per Cluster               */
        PDM_BPB_FAT_BIT_FAT12,             /* 7) FAT bits                          */
        PDM_FAT_12                         /* 8) FAT TYPE                          */
    },
    {
        /*     2MB  < CardCapacity <=     8MB */
        MBYTES_TO_SECTORS(2),              /* 1) Total Sectors (min)               */
        MBYTES_TO_SECTORS(8),              /* 2) Total Sectors (max)               */
        KBYTES_TO_SECTORS(8),              /* 3) Boundary Unit                     */
        2,                                 /* 4) Number Of Heads[Number Of Sides]  */
        32,                                /* 5) Sectors Per Track                 */
        16,                                /* 6) Sectors Per Cluster               */
        PDM_BPB_FAT_BIT_FAT12,             /* 7) FAT bits                          */
        PDM_FAT_12                         /* 8) FAT TYPE                          */
    },
    {
        /*     8MB  < CardCapacity <=    16MB */
        MBYTES_TO_SECTORS(8),              /* 1) Total Sectors (min)               */
        MBYTES_TO_SECTORS(16),             /* 2) Total Sectors (max)               */
        KBYTES_TO_SECTORS(16),             /* 3) Boundary Unit                     */
        2,                                 /* 4) Number Of Heads[Number Of Sides]  */
        32,                                /* 5) Sectors Per Track                 */
        32,                                /* 6) Sectors Per Cluster               */
        PDM_BPB_FAT_BIT_FAT12,             /* 7) FAT bits                          */
        PDM_FAT_12                         /* 8) FAT TYPE                          */
    },
    {
        /*    16MB  < CardCapacity <=    32MB */
        MBYTES_TO_SECTORS(16),             /* 1) Total Sectors (min)               */
        MBYTES_TO_SECTORS(32),             /* 2) Total Sectors (max)               */
        KBYTES_TO_SECTORS(16),             /* 3) Boundary Unit                     */
        4,                                 /* 4) Number Of Heads[Number Of Sides]  */
        32,                                /* 5) Sectors Per Track                 */
        32,                                /* 6) Sectors Per Cluster               */
        PDM_BPB_FAT_BIT_FAT12,             /* 7) FAT bits                          */
        PDM_FAT_12                         /* 8) FAT TYPE                          */
    },
    {
        /*    32MB  < CardCapacity <=    64MB */
        MBYTES_TO_SECTORS(32),             /* 1) Total Sectors (min)               */
        MBYTES_TO_SECTORS(64),             /* 2) Total Sectors (max)               */
        KBYTES_TO_SECTORS(16),             /* 3) Boundary Unit                     */
        8,                                 /* 4) Number Of Heads[Number Of Sides]  */
        32,                                /* 5) Sectors Per Track                 */
        32,                                /* 6) Sectors Per Cluster               */
        PDM_BPB_FAT_BIT_FAT12,             /* 7) FAT bits                          */
        PDM_FAT_12                         /* 8) FAT TYPE                          */
    },
    {
        /*    64MB  < CardCapacity <=   128MB */
        MBYTES_TO_SECTORS(64),             /* 1) Total Sectors (min)               */
        MBYTES_TO_SECTORS(128),            /* 2) Total Sectors (max)               */
        KBYTES_TO_SECTORS(32),             /* 3) Boundary Unit                     */
        8,                                 /* 4) Number Of Heads[Number Of Sides]  */
        32,                                /* 5) Sectors Per Track                 */
        32,                                /* 6) Sectors Per Cluster               */
        PDM_BPB_FAT_BIT_FAT16,             /* 7) FAT bits                          */
        PDM_FAT_16                         /* 8) FAT TYPE                          */
    },
    {
        /*   128MB  < CardCapacity <=   256MB */
        MBYTES_TO_SECTORS(128),            /* 1) Total Sectors (min)               */
        MBYTES_TO_SECTORS(256),            /* 2) Total Sectors (max)               */
        KBYTES_TO_SECTORS(32),             /* 3) Boundary Unit                     */
        16,                                /* 4) Number Of Heads[Number Of Sides]  */
        32,                                /* 5) Sectors Per Track                 */
        32,                                /* 6) Sectors Per Cluster               */
        PDM_BPB_FAT_BIT_FAT16,             /* 7) FAT bits                          */
        PDM_FAT_16                         /* 8) FAT TYPE                          */
    },
    {
        /*   256MB  < CardCapacity <=   504MB */
        MBYTES_TO_SECTORS(256),            /* 1) Total Sectors (min)               */
        MBYTES_TO_SECTORS(504),            /* 2) Total Sectors (max)               */
        KBYTES_TO_SECTORS(64),             /* 3) Boundary Unit                     */
        16,                                /* 4) Number Of Heads[Number Of Sides]  */
        63,                                /* 5) Sectors Per Track                 */
        32,                                /* 6) Sectors Per Cluster               */
        PDM_BPB_FAT_BIT_FAT16,             /* 7) FAT bits                          */
        PDM_FAT_16                         /* 8) FAT TYPE                          */
    },
    {
        /*   504MB  < CardCapacity <=  1008MB */
        MBYTES_TO_SECTORS(504),            /* 1) Total Sectors (min)               */
        MBYTES_TO_SECTORS(1008),           /* 2) Total Sectors (max)               */
        KBYTES_TO_SECTORS(64),             /* 3) Boundary Unit                     */
        32,                                /* 4) Number Of Heads[Number Of Sides]  */
        63,                                /* 5) Sectors Per Track                 */
        32,                                /* 6) Sectors Per Cluster               */
        PDM_BPB_FAT_BIT_FAT16,             /* 7) FAT bits                          */
        PDM_FAT_16                         /* 8) FAT TYPE                          */
    },
    {
        /*  1008MB  < CardCapacity <=  1024MB */
        MBYTES_TO_SECTORS(1008),           /* 1) Total Sectors (min)               */
        MBYTES_TO_SECTORS(1024),           /* 2) Total Sectors (max)               */
        KBYTES_TO_SECTORS(64),             /* 3) Boundary Unit                     */
        64,                                /* 4) Number Of Heads[Number Of Sides]  */
        63,                                /* 5) Sectors Per Track                 */
        32,                                /* 6) Sectors Per Cluster               */
        PDM_BPB_FAT_BIT_FAT16,             /* 7) FAT bits                          */
        PDM_FAT_16                         /* 8) FAT TYPE                          */
    },
    {
        /*  1024MB  < CardCapacity <=  2016MB */
        MBYTES_TO_SECTORS(1024),           /* 1) Total Sectors (min)               */
        MBYTES_TO_SECTORS(2016),           /* 2) Total Sectors (max)               */
        KBYTES_TO_SECTORS(64),             /* 3) Boundary Unit                     */
        64,                                /* 4) Number Of Heads[Number Of Sides]  */
        63,                                /* 5) Sectors Per Track                 */
        64,                                /* 6) Sectors Per Cluster               */
        PDM_BPB_FAT_BIT_FAT16,             /* 7) FAT bits                          */
        PDM_FAT_16                         /* 8) FAT TYPE                          */
    },
    {
        /*  2016MB  < CardCapacity <=  2048MB */
        MBYTES_TO_SECTORS(2016),           /* 1) Total Sectors (min)               */
        MBYTES_TO_SECTORS(2048),           /* 2) Total Sectors (max)               */
        KBYTES_TO_SECTORS(64),             /* 3) Boundary Unit                     */
        128,                               /* 4) Number Of Heads[Number Of Sides]  */
        63,                                /* 5) Sectors Per Track                 */
        64,                                /* 6) Sectors Per Cluster               */
        PDM_BPB_FAT_BIT_FAT16,             /* 7) FAT bits                          */
        PDM_FAT_16                         /* 8) FAT TYPE                          */
    },
    {
        /*  2088.5MB<=CardCapacity <=  4032MB */
        KBYTES_TO_SECTORS(2138624),        /* 1) Total Sectors (min)               */
        MBYTES_TO_SECTORS(4032),           /* 2) Total Sectors (max)               */
        MBYTES_TO_SECTORS(4),              /* 3) Boundary Unit                     */
        128,                               /* 4) Number Of Heads[Number Of Sides]  */
        63,                                /* 5) Sectors Per Track                 */
        64,                                /* 6) Sectors Per Cluster               */
        PDM_BPB_FAT_BIT_FAT32,             /* 7) FAT bits                          */
        PDM_FAT_32                         /* 8) FAT TYPE                          */
    },
    {
        /*  4032MB  < CardCapacity <= 32768MB */
        MBYTES_TO_SECTORS(4032),           /* 1) Total Sectors (min)               */
        MBYTES_TO_SECTORS(32768),          /* 2) Total Sectors (max)               */
        MBYTES_TO_SECTORS(4),              /* 3) Boundary Unit                     */
        255,                               /* 4) Number Of Heads[Number Of Sides]  */
        63,                                /* 5) Sectors Per Track                 */
        64,                                /* 6) Sectors Per Cluster               */
        PDM_BPB_FAT_BIT_FAT32,             /* 7) FAT bits                          */
        PDM_FAT_32                         /* 8) FAT TYPE                          */
    },
    {
        /* 32896MB  <=CardCapacity <=   128GB */
        MBYTES_TO_SECTORS(32896),           /* 1) Total Sectors (min)               */
        GBYTES_TO_SECTORS(128),             /* 2) Total Sectors (max)               */
        MBYTES_TO_SECTORS(16),              /* 3) Boundary Unit                     */
        255,                                /* 4) Number Of Heads[Number Of Sides]  */
        63,                                 /* 5) Sectors Per Track                 */
        256,                                /* 6) Sectors Per Cluster               */
        PDM_BPB_FAT_BIT_EXFAT,              /* 7) FAT bits                          */
        PDM_EXFAT                           /* 8) FAT TYPE                          */
    },
    {
        /*   128GB  < CardCapacity <=   512GB */
        GBYTES_TO_SECTORS(128),            /* 1) Total Sectors (min)               */
        GBYTES_TO_SECTORS(512),            /* 2) Total Sectors (max)               */
        MBYTES_TO_SECTORS(32),             /* 3) Boundary Unit                     */
        255,                               /* 4) Number Of Heads[Number Of Sides]  */
        63,                                /* 5) Sectors Per Track                 */
        512,                               /* 6) Sectors Per Cluster               */
        PDM_BPB_FAT_BIT_EXFAT,             /* 7) FAT bits                          */
        PDM_EXFAT                          /* 8) FAT TYPE                          */
    },
    {
        /*   512GB  < CardCapacity <=     2TB */
        GBYTES_TO_SECTORS(512),            /* 1) Total Sectors (min)               */
        GBYTES_TO_SECTORS(2048),           /* 2) Total Sectors (max)               */
        MBYTES_TO_SECTORS(64),             /* 3) Boundary Unit                     */
        255,                               /* 4) Number Of Heads[Number Of Sides]  */
        63,                                /* 5) Sectors Per Track                 */
        1024,                              /* 6) Sectors Per Cluster               */
        PDM_BPB_FAT_BIT_EXFAT,             /* 7) FAT bits                          */
        PDM_EXFAT                          /* 8) FAT TYPE                          */
    }
};


PDM_LOCAL nn::Result sddrv_get_value_with_total_sectors(MEDIA_SECTORS_WIDTH media_sectors, const SD_SPEC_PARAM** p_param_tbl);
PDM_LOCAL nn::Result sddrv_calc_part_tbl_chs(PDM_PART_TBL* p_part_tbl, const SD_SPEC_PARAM* p_param_tbl);
PDM_LOCAL nn::Result sddrv_get_timestamp(PDM_U_SHORT* p_date, PDM_U_SHORT* p_time);
PDM_LOCAL nn::Result sddrv_format_param_computation_fat1216(PDM_BPB_FORMAT* p_format_param, const SD_SPEC_PARAM* p_param_tbl, PDM_PART_TBL* p_part_tbl, PDM_U_LONG partition_sector, PDM_U_LONG sectors_per_fat, PDM_U_LONG* p_user_start_sector);
PDM_LOCAL nn::Result sddrv_format_param_computation_fat32(PDM_BPB_FORMAT* p_format_param, const SD_SPEC_PARAM* p_param_tbl, PDM_PART_TBL* p_part_tbl, PDM_U_LONG partition_sector, PDM_U_LONG sectors_per_fat, PDM_U_LONG* p_user_start_sector);
PDM_LOCAL nn::Result sddrv_calc_format_param(PDM_U_LONG partition_sector, PDM_BPB_FORMAT* p_format_param, const SD_SPEC_PARAM* p_param_tbl, PDM_PART_TBL* p_part_tbl, PDM_U_LONG* p_user_start_sector);
PDM_LOCAL nn::Result sddrv_set_partition_type(PDM_PART_TBL* p_part_tbl, PDM_U_LONG partition_sector, PDM_FAT_TYPE fat_type);
PDM_LOCAL PDM_ERROR sddrv_build_boot_sector_information(PDM_BPB_FORMAT* p_bpb_format, PDM_U_CHAR* p_buf);
#if PF_EXFAT_SUPPORT
#if PF_EXFAT_FORMAT_SUPPORT
PDM_LOCAL nn::Result sddrv_format_param_computation_exfat(PDM_BPB_FORMAT* p_format_param, const SD_SPEC_PARAM* p_param_tbl, PDM_PART_TBL* p_part_tbl, PDM_U_LONG partition_sector);
PDM_LOCAL PDM_U_LONG sddrv_calc_exfat_BootChecksum(PDM_U_CHAR* buf, PDM_U_LONG num_of_sector, PDM_U_LONG* Checksum);
PDM_LOCAL nn::Result sddrv_build_exfat_system_area(PDM_DISK* p_disk, PDM_U_LONG boundary_unit, PDM_U_LONG start_sector, PDM_U_LONG offset);
#endif /* PF_EXFAT_FORMAT_SUPPORT */
#endif /* PF_EXFAT_SUPPORT */
PDM_LOCAL nn::Result sddrv_write_aligned_zero_pad_sectors(PDM_DISK* p_disk, const PDM_BPB_FORMAT* p_format_param, PDM_U_LONG lba_start_sector, PDM_U_LONG user_start_sector, uint8_t* buffer, size_t buffer_size);

/*---------------------------------------------------------------------------
sddrv_get_value_with_total_sectors

[Calling Procedure]
nn::Result sddrv_get_value_with_total_sectors

[Explanation]
This function gets the value depends on total_sectors.
----------------------------------------------------------------------------*/
PDM_LOCAL nn::Result sddrv_get_value_with_total_sectors(MEDIA_SECTORS_WIDTH media_sectors, const SD_SPEC_PARAM** p_param_tbl)
{
    PDM_U_LONG  i;

#if PDM_PARAM_CHECK_ENABLE
    if (p_param_tbl == NULL)
    {
        return nn::fs::ResultFatFsFormatInvalidParameter();
    }
#endif /* PDM_PARAM_CHECK_ENABLE */
    for (i = 0; i < (sizeof(sd_spec_param_table) / sizeof(sd_spec_param_table[0])); i++)
    {
        if ((sd_spec_param_table[i].total_sectors_min <= media_sectors)
            && (sd_spec_param_table[i].total_sectors_max >= media_sectors))
        {
            *p_param_tbl = &sd_spec_param_table[i];
            NN_RESULT_SUCCESS;
        }
    }

    /* Not Found */
    return nn::fs::ResultFatFsFormatUnsupportedSize();
}
/*---------------------------------------------------------------------------
sddrv_calc_part_tbl_chs - Set partition table information(Starting/Ending (Head,Sector,Cylinder)).
---------------------------------------------------------------------------*/
PDM_LOCAL nn::Result sddrv_calc_part_tbl_chs(
    PDM_PART_TBL*    p_part_tbl,
    const SD_SPEC_PARAM* p_param_tbl)
{
    PDM_U_LONG cyl_size;

#if PDM_PARAM_CHECK_ENABLE
    if (p_part_tbl == NULL || p_param_tbl == NULL)
    {
        return nn::fs::ResultFatFsFormatInvalidParameter();
    }
#endif /* PDM_PARAM_CHECK_ENABLE */
    cyl_size = p_param_tbl->number_of_heads * p_param_tbl->sectors_per_track;

    /* It is judged whether the starting position of a partition exceeds FAT_SDDRV_MAX_CHS_VALUE (8032.5 MB). */
    if (p_part_tbl->lba_start_sector > SDDRV_MAX_CHS_VALUE)
    {
        /* if exceeds SDDRV_MAX_CHS_VALUE (8032.5 MB), */
        /* Starting Head/Sector/Cylinder is fixed value    */
        p_part_tbl->s_head = SD_FIXED_VALUE_CHS_HEAD;
        p_part_tbl->s_sector = SD_FIXED_VALUE_CHS_SECTOR;
        p_part_tbl->s_cylinder = SD_FIXED_VALUE_CHS_CYLINDER;
    }
    else
    {
        /* if not exceeds SDDRV_MAX_CHS_VALUE (8032.5 MB),*/
        /* Starting Head/Sector/Cylinder is calculated.       */
        p_part_tbl->s_head =(PDM_U_CHAR)((p_part_tbl->lba_start_sector % cyl_size)
            / p_param_tbl->sectors_per_track);
        p_part_tbl->s_sector = (PDM_U_CHAR)((p_part_tbl->lba_start_sector % p_param_tbl->sectors_per_track) + 1);
        p_part_tbl->s_cylinder = (PDM_U_SHORT)(p_part_tbl->lba_start_sector / cyl_size);
    }

    /* It is judged whether the end position of a partition exceeds FAT_SDDRV_MAX_CHS_VALUE (8032.5 MB). */
    if (p_part_tbl->lba_num_sectors > SDDRV_MAX_CHS_VALUE)
    {
        /* if exceeds SDDRV_MAX_CHS_VALUE(8032.5MB), */
        /* End Head/Sector/Cylinder is fixed value       */
        p_part_tbl->e_head = SD_FIXED_VALUE_CHS_HEAD;
        p_part_tbl->e_sector = SD_FIXED_VALUE_CHS_SECTOR;
        p_part_tbl->e_cylinder = SD_FIXED_VALUE_CHS_CYLINDER;
    }
    else
    {
        PDM_U_LONG end_sector = p_part_tbl->lba_start_sector + p_part_tbl->lba_num_sectors - 1;
        /* if not exceeds SDDRV_MAX_CHS_VALUE (8032.5 MB),*/
        /* Starting Head/Sector/Cylinder is calculated.       */
        p_part_tbl->e_head = (PDM_U_CHAR)((end_sector % cyl_size) / p_param_tbl->sectors_per_track);
        p_part_tbl->e_sector = (PDM_U_CHAR)((end_sector % p_param_tbl->sectors_per_track) + 1);
        p_part_tbl->e_cylinder = (PDM_U_SHORT)(end_sector / cyl_size);
    }
    NN_RESULT_SUCCESS;
}
/*---------------------------------------------------------------------------
sddrv_get_timestamp - get current date and time, and covert into timestamp format.

[Description]
Get current date and time with proper format.
----------------------------------------------------------------------------*/
PDM_LOCAL nn::Result sddrv_get_timestamp(PDM_U_SHORT* p_date, PDM_U_SHORT* p_time)
{
    PF_SYS_DATE     sys_date;
    PF_SYS_TIME     sys_time;

#if PDM_PARAM_CHECK_ENABLE
    if (p_date == NULL || p_time == NULL)
    {
        return nn::fs::ResultFatFsFormatInvalidParameter();
    }
#endif /* PDM_PARAM_CHECK_ENABLE */
    /* get current date & time */
    PFSYS_TimeStamp(&sys_date, &sys_time);

    *p_date = (PDM_U_SHORT)(SET_SYS_DATE(sys_date.sys_year, sys_date.sys_month, sys_date.sys_day));
    *p_time = (PDM_U_SHORT)(SET_SYS_TIME(sys_time.sys_hour, sys_time.sys_min, sys_time.sys_sec));

    NN_RESULT_SUCCESS;
}
/*---------------------------------------------------------------------------
sddrv_format_param_computation_fat1216 - Format Parameter Comuputation for FAT12/16.
---------------------------------------------------------------------------*/
PDM_LOCAL nn::Result sddrv_format_param_computation_fat1216(
    PDM_BPB_FORMAT* p_format_param,
    const SD_SPEC_PARAM* p_param_tbl,
    PDM_PART_TBL*   p_part_tbl,
    PDM_U_LONG      partition_sector,
    PDM_U_LONG      sectors_per_fat,
    PDM_U_LONG*     p_user_start_sector
)
{
    PDM_U_LONG      n;
    PDM_U_LONG      mbr_sectors_num = 0;
    PDM_U_LONG      fat_bits;
    PDM_U_LONG      boundary_unit;
    PDM_U_LONG      sectors_per_fat_dash = 0;
    PDM_U_LONG      system_area_sectors = 0;
    PDM_U_LONG      max_cluster_num;
    PDM_U_LONG      local_sectors_per_fat;

#if PDM_PARAM_CHECK_ENABLE
    if (p_format_param == NULL || p_param_tbl == NULL || p_part_tbl == NULL || p_user_start_sector == NULL)
    {
        return nn::fs::ResultFatFsFormatInvalidParameter();
    }
#endif /* PDM_PARAM_CHECK_ENABLE */
    fat_bits = p_param_tbl->fat_bits;
    boundary_unit = p_param_tbl->boundary_unit;
    local_sectors_per_fat = sectors_per_fat;

    /* Set jump boot */
    p_format_param->jump_boot[0] = PDM_BPB_JC0_0xEB;
    p_format_param->jump_boot[1] = PDM_BPB_JC0_0xE9;
    p_format_param->jump_boot[2] = PDM_BPB_JC2_0x90;
    p_format_param->num_reserved_sectors = FAT_BPB_RESERVED_SEC;
    p_format_param->num_root_dir_entries = SD_BPB_ROOT_ENT_CNT;
    p_format_param->num_fats = FAT_BPB_NUM_FATS;
    p_format_param->root_dir_cluster = 0x00000000;
    p_format_param->ext_flags = 0x0000;
    p_format_param->backup_boot_sector = 0x0000;
    if (p_format_param->fat_type == PDM_FAT_12)
    {
        pf_memcpy(p_format_param->fs_type, FAT_BPB_FS_TYPE_12, PDM_BS_FILE_SYS_TYPE_SIZE);
    }
    else
    {
        pf_memcpy(p_format_param->fs_type, FAT_BPB_FS_TYPE_16, PDM_BS_FILE_SYS_TYPE_SIZE);
    }

    /* this loop is caluculation sectors per FAT */
    for (; local_sectors_per_fat > 0 ;)
    {
        /* SSA(= Number of sectors in the system area)(system_area_sectors_num)
        : Reserved Sector Count + (2 * Sectors per FAT) + The Size of Root Directory */
        system_area_sectors = FAT_BPB_RESERVED_SEC + FAT_BPB_NUM_FATS * local_sectors_per_fat + SD_CEIL_ROOT_DIR_ENTRIES;

        /* NOM(= Number of Sectors in MBR)(mbr_sectors_num) : Boundary Unit * n - SSA > 0 */
        for (n = 1; ; n++)
        {
            if (boundary_unit * n > system_area_sectors)
            {
                mbr_sectors_num = boundary_unit * n - system_area_sectors;
                /* If NOM isn't equal BU, NOM is added to BU */
                if (mbr_sectors_num != boundary_unit)
                {
                    mbr_sectors_num += boundary_unit;
                }
                break;
            }
        }

        for (; mbr_sectors_num < partition_sector;)
        {
            max_cluster_num = MAX_CLUSTER_NUM(partition_sector, mbr_sectors_num, system_area_sectors, p_param_tbl->sectors_per_cluster);
            sectors_per_fat_dash = SECTORS_PER_FAT_DASH(max_cluster_num, fat_bits, SectorSize);

            /* The result of calculation of the 1st time and a re-calculation is compared. */
            if (sectors_per_fat_dash <= local_sectors_per_fat)
            {
                break;
            }
            /* NOM is add BU , Return "MAX" Re-Calculation */
            mbr_sectors_num += boundary_unit;
        }
        /* if infinite loop terminate condition meet, then this function return Failed. */
        if (mbr_sectors_num >= partition_sector) {
            return nn::fs::ResultFatFsFormatIllegalSectorsA();
        }

        /* If SF' isn't equal to SF, SF' is used as SF. */
        if (sectors_per_fat_dash == local_sectors_per_fat)
        {
            break;    /* sectors per FAT is decided */
        }
        /* Back to for loop, then "SSA" Re-Calculation */
        local_sectors_per_fat = sectors_per_fat_dash;
    }
    /* if infinite loop terminate condition meet, then this function return Failed. */
    if (local_sectors_per_fat <= 0) {
        return nn::fs::ResultFatFsFormatIllegalSectorsB();
    }

    p_format_param->num_hidden_sectors = mbr_sectors_num;
    p_format_param->total_sectors = partition_sector - mbr_sectors_num;

    p_part_tbl->lba_start_sector = mbr_sectors_num;
    p_part_tbl->lba_num_sectors = partition_sector - mbr_sectors_num;
    *p_user_start_sector = mbr_sectors_num + system_area_sectors;

    NN_RESULT_SUCCESS;
}
/*---------------------------------------------------------------------------
sddrv_format_param_computation_fat32 - Format Parameter Comuputation for FAT32.
---------------------------------------------------------------------------*/
PDM_LOCAL nn::Result sddrv_format_param_computation_fat32(
    PDM_BPB_FORMAT* p_format_param,
    const SD_SPEC_PARAM* p_param_tbl,
    PDM_PART_TBL*   p_part_tbl,
    PDM_U_LONG      partition_sector,
    PDM_U_LONG      sectors_per_fat,
    PDM_U_LONG*     p_user_start_sector
)
{
    PDM_U_SHORT     n;
    PDM_U_LONG      mbr_sectors_num;
    PDM_U_LONG      fat_bits;
    PDM_U_SHORT     boundary_unit;
    PDM_U_LONG      sectors_per_fat_dash = 0;
    PDM_U_SHORT     reserve_sectors_num = 0;
    PDM_U_LONG      system_area_sectors = 0;
    PDM_U_LONG      max_cluster_num;
    PDM_U_SHORT     local_sectors_per_fat;

#if PDM_PARAM_CHECK_ENABLE
    if (p_format_param == NULL || p_param_tbl == NULL || p_part_tbl == NULL || p_user_start_sector == NULL)
    {
        return nn::fs::ResultFatFsFormatInvalidParameter();
    }
#endif /* PDM_PARAM_CHECK_ENABLE */
    fat_bits = p_param_tbl->fat_bits;
    boundary_unit = (PDM_U_SHORT)p_param_tbl->boundary_unit;
    local_sectors_per_fat = (PDM_U_SHORT)sectors_per_fat;

    /* Set jump boot */
    p_format_param->jump_boot[0] = PDM_BPB_JC0_0xEB;
    p_format_param->jump_boot[1] = PDM_BPB_JC0_0xE9;
    p_format_param->jump_boot[2] = PDM_BPB_JC2_0x90;
    p_format_param->num_root_dir_entries = FAT_BPB_ROOT_ENT_CNT32;
    p_format_param->backup_boot_sector = FAT_BPB_BACKUP_SECTOR;
    p_format_param->fs_info_sector = FAT_BPB_FSINFO_SECTOR;
    p_format_param->num_fats = FAT_BPB_NUM_FATS;
    p_format_param->ext_flags = 0x0000;
    p_format_param->root_dir_cluster = 0x0002;
    pf_memcpy(p_format_param->fs_type, FAT_BPB_FS_TYPE_32, PDM_BS_FILE_SYS_TYPE_SIZE);

    mbr_sectors_num = boundary_unit;

    /* this loop is caluculation sectors per FAT */
    for (; local_sectors_per_fat > 0;)
    {
        /* RSC(= Resered Sectors Count) : Boundary Unit * n - 2*SF > 0 */
        for (n = 1; ; n++)
        {
            if ((boundary_unit * n) > (FAT_BPB_NUM_FATS * local_sectors_per_fat))
            {
                reserve_sectors_num = boundary_unit * n - FAT_BPB_NUM_FATS * local_sectors_per_fat;
                if (reserve_sectors_num < FAT_BPB_RESERVED_SEC32)
                {
                    reserve_sectors_num += boundary_unit;
                }
                break;
            }
        }

        /* SSA(= Number of sectors in the system area)(system_area_sectors_num)
        : Reserved Sector Count + (2 * Sectors per FAT) */
        system_area_sectors = reserve_sectors_num + FAT_BPB_NUM_FATS * local_sectors_per_fat;

        for (; system_area_sectors < partition_sector;)
        {
            max_cluster_num = MAX_CLUSTER_NUM(partition_sector, mbr_sectors_num, system_area_sectors, p_param_tbl->sectors_per_cluster);
            sectors_per_fat_dash = SECTORS_PER_FAT_DASH(max_cluster_num, fat_bits, SectorSize);

            /* The result of calculation of the 1st time and a re-calculation is compared. */
            if (sectors_per_fat_dash <= local_sectors_per_fat)
            {
                break;
            }
            /* SSA, and RSC are added BU, then "MAX" Re-Calculation */
            system_area_sectors += boundary_unit;
            reserve_sectors_num += boundary_unit;
        }
        /* if infinite loop terminate condition meet, then this function return Failed. */
        if (system_area_sectors >= partition_sector) {
            return nn::fs::ResultFatFsFormatIllegalSectorsC();
        }

        /* If SF` isn't equal to SF, SF-1 is used as new SF. */
        if (sectors_per_fat_dash == local_sectors_per_fat)
        {
            break;    /* sectors per FAT is decided */
        }
        /* SF -1 and , then "SSA" Re-Calculation */
        local_sectors_per_fat--;
    }
    /* if infinite loop terminate condition meet, then this function return Failed. */
    if (local_sectors_per_fat <= 0) {
        return nn::fs::ResultFatFsFormatIllegalSectorsD();
    }

    p_format_param->num_reserved_sectors = reserve_sectors_num;
    p_format_param->num_hidden_sectors = mbr_sectors_num;
    p_format_param->total_sectors = partition_sector - mbr_sectors_num;

    p_part_tbl->lba_start_sector = mbr_sectors_num;
    p_part_tbl->lba_num_sectors = partition_sector - mbr_sectors_num;
    *p_user_start_sector = mbr_sectors_num + system_area_sectors;

    NN_RESULT_SUCCESS;
}

#if PF_EXFAT_SUPPORT
#if PF_EXFAT_FORMAT_SUPPORT
/*---------------------------------------------------------------------------
sddrv_format_param_computation_exfat - Format Parameter Comuputation for exFAT.
---------------------------------------------------------------------------*/
PDM_LOCAL nn::Result sddrv_format_param_computation_exfat(
    PDM_BPB_FORMAT* p_format_param,
    const SD_SPEC_PARAM* p_param_tbl,
    PDM_PART_TBL*   p_part_tbl,
    PDM_U_LONG      partition_sector
)
{
    PDM_U_LONG      boundary_unit;

#if PDM_PARAM_CHECK_ENABLE
    if (p_format_param == NULL || p_param_tbl == NULL || p_part_tbl == NULL)
    {
        return nn::fs::ResultFatFsFormatInvalidParameter();
    }
#endif /* PDM_PARAM_CHECK_ENABLE */
    boundary_unit = p_param_tbl->boundary_unit;

    p_format_param->jump_boot[0] = PDM_EXFAT_BSS_JC0_0xEB;
    p_format_param->jump_boot[1] = PDM_EXFAT_BSS_JC1_0x76;
    p_format_param->jump_boot[2] = PDM_EXFAT_BSS_JC2_0x90;
    p_format_param->num_root_dir_entries = 0;
    p_format_param->num_fats = FAT_BPB_NUM_FATS_EXFAT;
    pf_memcpy(p_format_param->oem_name, FAT_BPB_FS_TYPE_EXFAT, PDM_BS_OEM_NAME_SIZE);
    pf_memcpy(p_format_param->fs_type, FAT_BPB_FS_TYPE_EXFAT, PDM_BS_FILE_SYS_TYPE_SIZE);
    p_format_param->partition_offset = boundary_unit;
    p_format_param->vol_length = partition_sector - boundary_unit;
    p_format_param->fat_offset = boundary_unit / 2;
    p_format_param->fat_length = boundary_unit / 2;
    p_format_param->cluster_heap_offset = boundary_unit;
    p_format_param->cluster_count = (partition_sector - boundary_unit * 2) / p_param_tbl->sectors_per_cluster;
    p_format_param->num_hidden_sectors = boundary_unit;
    p_format_param->total_sectors = partition_sector - boundary_unit;
    p_format_param->sectors_per_cluster_exfat = p_param_tbl->sectors_per_cluster;
    p_format_param->fs_version = SDDRV_EXFAT_FS_REVISION;

    p_part_tbl->lba_start_sector = boundary_unit;
    p_part_tbl->lba_num_sectors = partition_sector - boundary_unit;

    NN_RESULT_SUCCESS;
}
#endif /* PF_EXFAT_FORMAT_SUPPORT */
#endif /* PF_EXFAT_SUPPORT */

/*---------------------------------------------------------------------------
sddrv_calc_format_param - Format Parameter Comuputation.
---------------------------------------------------------------------------*/
PDM_LOCAL nn::Result sddrv_calc_format_param(
    PDM_U_LONG      partition_sector,
    PDM_BPB_FORMAT* p_format_param,
    const SD_SPEC_PARAM* p_param_tbl,
    PDM_PART_TBL*   p_part_tbl,
    PDM_U_LONG*     p_user_start_sector
)
{
    PDM_U_LONG      fat_bits;
    PDM_U_LONG      total_clusters;
    PDM_U_LONG      sectors_per_fat;

#if PDM_PARAM_CHECK_ENABLE
    if(p_format_param == NULL || p_param_tbl == NULL || p_part_tbl == NULL || p_user_start_sector == NULL)
    {
        return nn::fs::ResultFatFsFormatInvalidParameter();
    }
#endif /* PDM_PARAM_CHECK_ENABLE */

    {
        p_format_param->fat_type = p_param_tbl->fat_type;
        /* Common */
        p_format_param->bytes_per_sector = SectorSize;
        if (p_format_param->fat_type != PDM_EXFAT)
        {
            p_format_param->sectors_per_cluster = (PDM_U_CHAR)p_param_tbl->sectors_per_cluster;
        }
        p_format_param->sector_per_track = p_param_tbl->sectors_per_track;
        p_format_param->media = SDDRV_MEDIA_DESCRIPTOR;
        p_format_param->drive = FAT_BPB_DRIVE_NON_REMOVABLE;
        p_format_param->num_heads = p_param_tbl->number_of_heads;
        p_format_param->boot_sig = FAT_BPB_EXT_BOOT_SIG;
        {
            PDM_U_SHORT date;
            PDM_U_SHORT time;

            /* get current date & time */
            NN_RESULT_DO(sddrv_get_timestamp(&date, &time));
            p_format_param->vol_id = (PDM_U_LONG)((date << 16) + time);
        }
        p_format_param->fs_version = SDDRV_BPB_FS_VER;

        pf_memcpy(p_format_param->vol_label, SD_BPB_DEFAULT_VOL_LABEL, pf_strnlen((PF_CONST PF_CHAR*)SD_BPB_DEFAULT_VOL_LABEL, PDM_BS_VOLUME_LABEL_SIZE));
        pf_memset(p_format_param->oem_name, 0x0, PDM_BS_OEM_NAME_SIZE);

        fat_bits = p_param_tbl->fat_bits;
        total_clusters = partition_sector / p_param_tbl->sectors_per_cluster;
        sectors_per_fat = total_clusters * fat_bits / (PDM_BPB_BITS_PER_BYTE * SectorSize);
        if ((total_clusters * fat_bits % (PDM_BPB_BITS_PER_BYTE * SectorSize)) != 0)    /* has remain */
        {
            sectors_per_fat++;                                            /* upvaluation */
        }
        if ((p_format_param->fat_type == PDM_FAT_12) || (p_format_param->fat_type == PDM_FAT_16))
        {
            NN_RESULT_DO(sddrv_format_param_computation_fat1216(
                p_format_param,
                p_param_tbl,
                p_part_tbl,
                partition_sector,
                sectors_per_fat,
                p_user_start_sector
            ));
        }
        else if (p_format_param->fat_type == PDM_FAT_32)
        {
            NN_RESULT_DO(sddrv_format_param_computation_fat32(
                p_format_param,
                p_param_tbl,
                p_part_tbl,
                partition_sector,
                sectors_per_fat,
                p_user_start_sector
            ));
        }
        else if (p_format_param->fat_type == PDM_EXFAT)
        {
#if PF_EXFAT_SUPPORT && PF_EXFAT_FORMAT_SUPPORT
            NN_RESULT_DO(sddrv_format_param_computation_exfat(
                p_format_param,
                p_param_tbl,
                p_part_tbl,
                partition_sector
            ));
            *p_user_start_sector = p_part_tbl->lba_start_sector + p_format_param->fat_offset + p_format_param->fat_length;
#else /* !PF_EXFAT_SUPPORT || !PF_EXFAT_FORMAT_SUPPORT */
            return nn::fs::ResultExFatUnavailable();
#endif /* !PF_EXFAT_SUPPORT || !PF_EXFAT_FORMAT_SUPPORT */
        }
        else /* Error*/
        {
            return nn::fs::ResultFatFsFormatInvalidParameter();
        }

        /* Determine Starting/Ending (Head,Sector,Cylinder) */
        NN_RESULT_DO(sddrv_calc_part_tbl_chs(p_part_tbl, p_param_tbl));
        NN_RESULT_DO(sddrv_set_partition_type(p_part_tbl, partition_sector, p_param_tbl->fat_type));
    }
    NN_RESULT_SUCCESS;
}
/*---------------------------------------------------------------------------
sddrv_set_partition_type - Set partition type.
---------------------------------------------------------------------------*/
PDM_LOCAL nn::Result sddrv_set_partition_type(
    PDM_PART_TBL* p_part_tbl,
    PDM_U_LONG partition_sector,
    PDM_FAT_TYPE    fat_type
)
{
#if PDM_PARAM_CHECK_ENABLE
    if (p_part_tbl == NULL)
    {
        return nn::fs::ResultFatFsFormatInvalidParameter();
    }
#endif /* PDM_PARAM_CHECK_ENABLE */

    p_part_tbl->boot_flag = PDM_MBR_BOOTFLAG_DISABLE;

    if ((fat_type == PDM_FAT_12) || (fat_type == PDM_FAT_16))
    {
        if (partition_sector < KBYTES_TO_SECTORS(16340)) /* (16MB-4kB)/512 = 32680 sectors */
        {
            p_part_tbl->partition_type = PDM_MBR_PARTTYPE_FAT12;
        }
        else if (partition_sector < MBYTES_TO_SECTORS(32)) /* 32MB/512 = 65536 sectors */
        {
            p_part_tbl->partition_type = PDM_MBR_PARTTYPE_FAT16;
        }
        else
        {
            p_part_tbl->partition_type = PDM_MBR_PARTTYPE_FAT16_OVER32MB;
        }
    }
    else if (fat_type == PDM_FAT_32)
    {
        if (partition_sector < SDDRV_MAX_CHS_VALUE)
        {
            p_part_tbl->partition_type = PDM_MBR_PARTTYPE_FAT32;
        }
        else
        {
            p_part_tbl->partition_type = PDM_MBR_PARTTYPE_FAT32_LBA;
        }
    }
#if PF_EXFAT_SUPPORT
#if PF_EXFAT_FORMAT_SUPPORT
    else if (fat_type == PDM_EXFAT)
    {
        p_part_tbl->partition_type = PDM_MBR_PARTTYPE_EXFAT;
    }
#endif /* PF_EXFAT_FORMAT_SUPPORT */
#endif /* PF_EXFAT_SUPPORT */
    else
    {
        return nn::fs::ResultFatFsFormatInvalidParameter();
    }

    NN_RESULT_SUCCESS;
}

#if PF_EXFAT_SUPPORT
#if PF_EXFAT_FORMAT_SUPPORT
/*---------------------------------------------------------------------------
sddrv_calc_BootChecksum -
---------------------------------------------------------------------------*/
PDM_LOCAL PDM_U_LONG sddrv_calc_exfat_BootChecksum(
    PDM_U_CHAR*     buf,
    PDM_U_LONG      num_of_sector,
    PDM_U_LONG*     Checksum
)
{
    PDM_U_LONG     i;

    for (i = 0; i < SectorSize; i++)
    {
        if (num_of_sector == 0)
        {
            if ((i == PDM_BSS_VOLUME_FLAGS) || (i == (PDM_BSS_VOLUME_FLAGS + 1)) || (i == PDM_BSS_PERCENT_IN_USE))
            {
                continue;
            }
        }

        *Checksum = ((*Checksum & 1) ? 0x80000000 : 0) + (*Checksum >> 1) + buf[i];
    }

    return 0;
}
#endif /* PF_EXFAT_FORMAT_SUPPORT */
#endif /* PF_EXFAT_SUPPORT */

/*---------------------------------------------------------------------------
sddrv_build_boot_sector_information - Build BPB informations.
---------------------------------------------------------------------------*/
PDM_LOCAL PDM_ERROR sddrv_build_boot_sector_information(PDM_BPB_FORMAT* p_bpb_format, PDM_U_CHAR* p_buf)
{
    PF_ERROR        err;
    PDM_BPB         pdm_bpb;

#if PDM_PARAM_CHECK_ENABLE
    if ((p_bpb_format == NULL) || (p_buf == NULL))
    {
        return PDM_ERR_INVALID_PARAMETER;
    }
#endif /* PDM_PARAM_CHECK_ENABLE */
    pf_memset(p_buf, 0, SectorSize);

#if PF_EXFAT_SUPPORT
#if PF_EXFAT_FORMAT_SUPPORT
    if (p_bpb_format->fat_type == PDM_EXFAT)
    {
        err = pdm_bpb_build_boot_sector_exfat(p_bpb_format, &pdm_bpb);
        if (err != PDM_OK)
        {
            return err;
        }

        err = pdm_bpb_set_bpb_information_exfat(&pdm_bpb, p_buf);
    }
    else
#endif /* PF_EXFAT_FORMAT_SUPPORT */
#endif /* PF_EXFAT_SUPPORT */
    {
        err = pdm_bpb_build_boot_sector(p_bpb_format, &pdm_bpb);
        if (err != PDM_OK)
        {
            return err;
        }

        err = pdm_bpb_set_bpb_information(&pdm_bpb, p_buf);
    }
    return err;
}

#if PF_EXFAT_SUPPORT
#if PF_EXFAT_FORMAT_SUPPORT
/*---------------------------------------------------------------------------
sddrv_build_exfat_system_area - make EXFAT Sytem Area Layout.
---------------------------------------------------------------------------*/
PDM_LOCAL nn::Result sddrv_build_exfat_system_area(
    PDM_DISK* p_disk,
    PDM_U_LONG boundary_unit,
    PDM_U_LONG start_sector,
    PDM_U_LONG offset
)
{
    PDM_U_CHAR buf[SectorSize];
    PDM_U_LONG i;
    PDM_U_LONG Checksum = 0;

    /* ExtendedBootSector */
    pf_memset(buf, 0, SectorSize);
    PDM_STORE_BS_TRAIL_SIG1(buf, PDM_BS_TRAIL_SIG1_VALUE);
    PDM_STORE_BS_TRAIL_SIG2(buf, PDM_BS_TRAIL_SIG2_VALUE);
    for (i = EXFAT_BSS_EXTEND_BOOT_SECTOR; i < EXFAT_BSS_OEM_PARAM_SECTOR; i++)
    {
        NN_RESULT_DO(PhysicalWrite(p_disk, offset + start_sector + i, 1, buf));
    }

    pf_memset(buf, 0, SectorSize);
    /* Flash Parameter GUID set (The following codes have to be little endian.) */
    PDM_STORE_INTO_LE32_FIELD(buf, 0, PDM_OEM_PARAM_F_GUID_0);
    PDM_STORE_INTO_LE32_FIELD(buf, 4, PDM_OEM_PARAM_F_GUID_1);
    PDM_STORE_INTO_LE32_FIELD(buf, 8, PDM_OEM_PARAM_F_GUID_2);
    PDM_STORE_INTO_LE32_FIELD(buf, 12, PDM_OEM_PARAM_F_GUID_3);
    /* Erase Block Size is the half of the Boundary Size */
    PDM_U_LONG erase_block_size = boundary_unit * SectorSize / 2;
    PDM_STORE_INTO_LE32_FIELD(buf, PDM_OEM_PARAM_F_ERASE_BLOCK_SIZE_OFFSET, erase_block_size);

    NN_RESULT_DO(PhysicalWrite(p_disk, offset + start_sector + EXFAT_BSS_OEM_PARAM_SECTOR, 1, buf));

    /*
     * EXFATブートセクタの10セクタ目は予約セクタで全てゼロを書かなければならない。
     * sddrv_write_aligned_zero_pad_sectors()で既にゼロ埋めされているので、ここでは何もしない。
     */

    /* BootChecksum */
    for (i = 0; i < EXFAT_BSS_CHECKSUM_SECTOR; i++)
    {
        NN_RESULT_DO(PhysicalRead(p_disk, offset + start_sector + i, 1, buf));

        sddrv_calc_exfat_BootChecksum(buf, i, &Checksum);
    }

    for (i = 0; i < SectorSize; i += sizeof(Checksum))
    {
        PDM_STORE_INTO_LE32_FIELD(buf, i, Checksum);
    }
    NN_RESULT_DO(PhysicalWrite(p_disk, offset + start_sector + EXFAT_BSS_CHECKSUM_SECTOR, 1, buf));

    NN_RESULT_SUCCESS;
}
#endif /* PF_EXFAT_FORMAT_SUPPORT */
#endif /* PF_EXFAT_SUPPORT */

/*---------------------------------------------------------------------------
sddrv_write_aligned_zero_pad_sectors - write zeros to system area.
---------------------------------------------------------------------------*/
PDM_LOCAL nn::Result sddrv_write_aligned_zero_pad_sectors(PDM_DISK* p_disk, const PDM_BPB_FORMAT* p_format_param, PDM_U_LONG lba_start_sector, PDM_U_LONG user_start_sector, uint8_t* buffer, size_t buffer_size)
{
    PDM_U_LONG start = lba_start_sector;
    PDM_U_LONG end = user_start_sector;
    PDM_U_LONG unused_area_start = 0;
    PDM_U_LONG unused_area_end = 0;
    PDM_U_LONG sector;
    NN_SDK_ASSERT(buffer_size == nn::fat::FatFormatWorkBufferSize);

    pf_memset(buffer, 0, buffer_size);

    if (p_format_param->fat_type == PDM_FAT_32)
    {
        unused_area_start = lba_start_sector + FAT_BPB_RESERVED_SEC32;
        unused_area_end = lba_start_sector + p_format_param->num_reserved_sectors;
    }
#if PF_EXFAT_SUPPORT
#if PF_EXFAT_FORMAT_SUPPORT
    else if (p_format_param->fat_type == PDM_EXFAT)
    {
        unused_area_start = lba_start_sector + PDM_BSS_BACKUP_BOOT_SECTOR * 2;
        unused_area_end = lba_start_sector + p_format_param->fat_offset;
    }
#endif /* PF_EXFAT_FORMAT_SUPPORT */
#endif /* PF_EXFAT_SUPPORT */
    else
    {
        /* do nothing */
    }

    start = nn::util::align_down(start, FAT_SIZE_ALIGNMENT_SECTORS);
    end = nn::util::align_up(end, FAT_SIZE_ALIGNMENT_SECTORS);

    for (sector = start; sector < end; sector += FAT_SIZE_ALIGNMENT_SECTORS)
    {
        if ((sector >= unused_area_start) &&
            ((sector + FAT_SIZE_ALIGNMENT_SECTORS) <= unused_area_end))
        {
            continue;
        }
        NN_RESULT_DO(PhysicalWrite(p_disk, sector, FAT_SIZE_ALIGNMENT_SECTORS, buffer));
    }

    NN_RESULT_SUCCESS;
}

/*---------------------------------------------------------------------------
sddrv_full_format

[Calling Procedure]
nn::Result sddrv_full_format

[Explanation]
This function formats system area of SD card.
----------------------------------------------------------------------------*/
nn::Result sddrv_full_format(PDM_DISK* p_disk, PDM_U_LONG data_area_sector, PDM_U_LONG protected_area_sector, uint8_t* buffer, size_t buffer_size)
{
    PDM_ERROR          err;
    const SD_SPEC_PARAM* param_tbl;
    PDM_PART_TBL       part_tbl;
    PDM_MBR            mbr_tbl;
    PDM_BPB_FORMAT     format_param;
    PDM_U_CHAR         buf[SectorSize];
    PDM_U_LONG    i;
    PDM_U_LONG    offset;
    PDM_U_LONG    user_start_sector;
    MEDIA_SECTORS_WIDTH media_total_sector = (MEDIA_SECTORS_WIDTH)data_area_sector + (MEDIA_SECTORS_WIDTH)protected_area_sector;

#if PDM_PARAM_CHECK_ENABLE
    if (p_disk == NULL)
    {
        return nn::fs::ResultFatFsFormatInvalidParameter();
    }
#endif /* PDM_PARAM_CHECK_ENABLE */
    /* Get Parameters */
    NN_RESULT_DO(sddrv_get_value_with_total_sectors(media_total_sector, &param_tbl));

    /* Format Parameter Comuputation */
    NN_RESULT_DO(sddrv_calc_format_param(data_area_sector, &format_param, param_tbl, &part_tbl, &user_start_sector));

    NN_RESULT_DO(sddrv_write_aligned_zero_pad_sectors(p_disk, &format_param, part_tbl.lba_start_sector, user_start_sector, buffer, buffer_size));

    /* Insert part_tbl to MBR */
    mbr_tbl.epbr_base_sector = PDM_MBR_MBR_DUMMY_SECTOR;
    mbr_tbl.current_sector = PDM_MBR_MBR_START_SECTOR;
    pf_memcpy(&mbr_tbl.partition_table[0], &part_tbl, sizeof(PDM_PART_TBL));
    for (i = 1; i < PDM_MBR_MAX_PARTTBL; i++)
    {
        pf_memset(&mbr_tbl.partition_table[i], 0, sizeof(PDM_PART_TBL));
    }
    /* Set MBR */
    pf_memset(buf, 0, SectorSize);
    PDM_STORE_BS_TRAIL_SIG1(buf, PDM_MBR_SIGNATURE1);
    PDM_STORE_BS_TRAIL_SIG2(buf, PDM_MBR_SIGNATURE2);
    err = pdm_mbr_set_table(&mbr_tbl, mbr_tbl.current_sector, buf);
    if (err != PDM_OK)
    {
        /* pdm_mbr_set_table()が返すエラーはPDM_ERR_INVALID_PARAMETERのみ */
        return nn::fs::ResultFatFsFormatInvalidParameter();
    }

    offset = 0;
    NN_RESULT_DO(PhysicalWriteVerify(p_disk, offset, 1, buf));

    /* Build BPB informations */
    err = sddrv_build_boot_sector_information(&format_param, buf);
    if (err != PDM_OK)
    {
        return nn::fs::ResultFatFsFormatInvalidBpb();
    }
    offset = mbr_tbl.partition_table[0].lba_start_sector;
    /* FAT12/16/32:BPB, exFAT:Main Boot Sector */
    NN_RESULT_DO(PhysicalWriteVerify(p_disk, offset, 1, buf));

    /* FAT32 Sytem Area Layout */
    if (param_tbl->fat_type == PDM_FAT_32)
    {
        /* BackUp BPB */
        NN_RESULT_DO(PhysicalWrite(p_disk, offset + FAT_BPB_BACKUP_SECTOR, 1, buf));

        /* Build FSINFO block image */
        NN_RESULT_DO(BuildUpFSInfoSector(buf));
        /* FsInfo */
        NN_RESULT_DO(PhysicalWrite(p_disk, offset + FAT_BPB_FSINFO_SECTOR, 1, buf));
        /* BackUp FsInfo */
        NN_RESULT_DO(PhysicalWrite(p_disk, offset + FAT_BPB_FSINFO_SECTOR + FAT_BPB_BACKUP_SECTOR, 1, buf));

        /* Reserved for Boot Sector */
        pf_memset(buf, 0, SectorSize);
        PDM_STORE_BS_TRAIL_SIG1(buf, PDM_BS_TRAIL_SIG1_VALUE);
        PDM_STORE_BS_TRAIL_SIG2(buf, PDM_BS_TRAIL_SIG2_VALUE);
        NN_RESULT_DO(PhysicalWrite(p_disk, offset + FAT_BPB_RESERVED_FOR_BOOT_SECTOR, 1, buf));
        /* BackUp Reserved for Boot Sector */
        NN_RESULT_DO(PhysicalWrite(p_disk, offset + FAT_BPB_RESERVED_FOR_BOOT_SECTOR + FAT_BPB_BACKUP_SECTOR, 1, buf));
    }
#if PF_EXFAT_SUPPORT
#if PF_EXFAT_FORMAT_SUPPORT
    else if (param_tbl->fat_type == PDM_EXFAT)
    {
        /* BackUp Boot Sector */
        NN_RESULT_DO(PhysicalWrite(p_disk, offset + PDM_BSS_BACKUP_BOOT_SECTOR, 1, buf));

        /* EXFAT Sytem Area Layout (Main) */
        NN_RESULT_DO(sddrv_build_exfat_system_area(p_disk, param_tbl->boundary_unit, 0, offset));
        /* EXFAT Sytem Area Layout (BackUp) */
        NN_RESULT_DO(sddrv_build_exfat_system_area(p_disk, param_tbl->boundary_unit, PDM_BSS_BACKUP_BOOT_SECTOR, offset));
    }
#endif /* PF_EXFAT_FORMAT_SUPPORT */
#endif /* PF_EXFAT_SUPPORT */
    else
    {
        // FAT12, FAT16では特有のフィールドはないので、何もしない
    }
    // FAT, RDEはこの後のPrFILE2側の処理で初期化される

    NN_RESULT_SUCCESS;
}

PDM_ERROR  StorageAdapterInit(PDM_DISK* pDisk)
{
    pdm_disk_notify_media_insert(pDisk);
    return PDM_OK;
}


PDM_ERROR  StorageAdapterMount(PDM_DISK* pDisk)
{
    NN_UNUSED(pDisk);
    return PDM_OK;
}


PDM_ERROR  StorageAdapterFormat(PDM_DISK* pDisk, const PDM_U_CHAR* pParam)
{
    NN_ABORT_UNLESS_NOT_NULL(pParam);
    PDM_ERROR ret;
    PDM_U_LONG dataAreaSectors;
    const nn::fat::FatFormatParam* pFatFormatParam = reinterpret_cast<const nn::fat::FatFormatParam*>(pParam);
    nn::Result result;

    GetFatStorageAdapter(pDisk)->SetFormatErrorResult(nn::ResultSuccess());

    ret = GetTotalSectors(&dataAreaSectors, pDisk);
    if (ret != PDM_OK)
    {
        return ret;
    }

    pdm_disk_notify_media_eject(pDisk);

    if (pFatFormatParam->isSdCard)
    {
        result = sddrv_full_format(pDisk, dataAreaSectors, pFatFormatParam->protectedAreaSectors, pFatFormatParam->pWorkBuffer, pFatFormatParam->workBufferSize);
    }
    else
    {
        result = nanddrv_format(pDisk, dataAreaSectors);
    }

    pdm_disk_notify_media_insert(pDisk);

    if (result.IsFailure())
    {
        if (nn::fs::ResultExFatUnavailable::Includes(result))
        {
            ret = PDM_ERR_EXFAT_FORMAT_UNSUPPORTED;
        }
        else
        {
            if (nn::fs::ResultFatFsWriteVerifyError::Includes(result))
            {
                result = pFatFormatParam->writeVerifyErrorResult;
            }
            GetFatStorageAdapter(pDisk)->SetFormatErrorResult(result);
            ret = PDM_ERR_DRIVER_ERROR;
        }
    }
    return ret;
}


PDM_ERROR  StorageAdapterPread(PDM_DISK* pDisk, PDM_U_CHAR* buffer, PDM_U_LONG lba,
                               PDM_U_LONG blockCount, PDM_U_LONG* pNumSuccess)
{
    auto result = GetFatStorageAdapter(pDisk)->Read(static_cast<int64_t>(lba) * SectorSize, buffer, blockCount * SectorSize);
    if( result.IsSuccess() )
    {
        *pNumSuccess = blockCount;
        return PDM_OK;
    }
    else
    {
        return PDM_ERR_DRIVER_ERROR;
    }
}


PDM_ERROR  StorageAdapterPwrite(PDM_DISK* pDisk, const PDM_U_CHAR* buffer, PDM_U_LONG lba,
                                PDM_U_LONG blockCount, PDM_U_LONG* pCountSuccess)
{
    auto result = GetFatStorageAdapter(pDisk)->Write(static_cast<int64_t>(lba) * SectorSize, buffer, blockCount * SectorSize);
    if( result.IsSuccess() )
    {
        *pCountSuccess = blockCount;
        return PDM_OK;
    }
    else
    {
        return PDM_ERR_DRIVER_ERROR;
    }
}


PDM_ERROR  StorageAdapterUnmount(PDM_DISK* pDisk)
{
    NN_UNUSED(pDisk);
    return PDM_OK;
}


PDM_ERROR  StorageAdapterFinalize(PDM_DISK* pDisk)
{
    NN_UNUSED(pDisk);
    return PDM_OK;
}


PDM_ERROR  StorageAdapterGetDiskInfo(PDM_DISK* pDisk, PDM_DISK_INFO* pDiskInfo)
{
    // Set Media Attribute
    pDiskInfo->media_attr = PDM_MEDIA_ATTR_REMOVABLE;

    // Set the number of Total Sector
    PDM_U_LONG totalSectors;
    PDM_ERROR err = GetTotalSectors(&totalSectors, pDisk);
    if (err != PDM_OK)
    {
        return err;
    }
    pDiskInfo->total_sectors = totalSectors;

    // Set Byte Per Sector
    pDiskInfo->bytes_per_sector = SectorSize;

    // Set the number of cylinders (PrFILE2 では参照されない)
    pDiskInfo->cylinders = 0;

    // Set the number of translated heads (PrFILE2 では参照されない)
    pDiskInfo->heads = 0;

    // Set the number of sectors per track (PrFILE2 では参照されない)
    pDiskInfo->sectors_per_track = 0;

    pDiskInfo->format_param = NULL;

    return PDM_OK;
}


PDM_FUNCTBL       g_StorageAdapterFuncTable = {
                        StorageAdapterInit,
                        StorageAdapterFinalize,
                        StorageAdapterMount,
                        StorageAdapterUnmount,
                        StorageAdapterFormat,
                        StorageAdapterPread,
                        StorageAdapterPwrite,
                        StorageAdapterGetDiskInfo,
                    };

} // namespace

namespace nn { namespace fat { namespace detail {


PDM_ERROR  FatStorageAdapterInitDriveTable(PDM_DISK_TBL* pDiskTable, uintptr_t ui_ext)
{
    pDiskTable->p_func = &g_StorageAdapterFuncTable;
    pDiskTable->ui_ext = ui_ext;

    return PDM_OK;
}

nn::Result FormatDryRun(uint32_t userAreaSectors, uint32_t protectedAreaSectors)
{
    const SD_SPEC_PARAM* paramTable;
    PDM_PART_TBL         partitionTable;
    PDM_BPB_FORMAT       formatParam;
    PDM_U_LONG           userStartSector;
    MEDIA_SECTORS_WIDTH  totalSectors = static_cast<MEDIA_SECTORS_WIDTH>(userAreaSectors) + static_cast<MEDIA_SECTORS_WIDTH>(protectedAreaSectors);

    NN_RESULT_DO(sddrv_get_value_with_total_sectors(totalSectors, &paramTable));
    NN_RESULT_DO(sddrv_calc_format_param(userAreaSectors, &formatParam, paramTable, &partitionTable, &userStartSector));
    NN_RESULT_SUCCESS;
}

}}}
