﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/os.h>
#include <nn/nn_Abort.h>
#include <nn/nn_SdkLog.h>
#include <nn/util/util_IntrusiveList.h>
#include <nn/usb/usb_Host.h>

#include "../eth_Allocator.h"
#include "../eth_Packet.h"

namespace nn     {
namespace eth    {
namespace device {
namespace usb    {

struct EndPointDescriptor
{
    nn::usb::UsbEndpointType      type;
    nn::usb::UsbEndpointDirection direction;
};

enum
{
    UT_WRITE  = 0x00,
    UT_READ   = 0x80,
    UT_VENDOR = 0x40,
    UT_DEVICE = 0x00,
    UT_READ_VENDOR_DEVICE  = UT_READ  | UT_VENDOR | UT_DEVICE,
    UT_WRITE_VENDOR_DEVICE = UT_WRITE | UT_VENDOR | UT_DEVICE,
};

class UsbDevice;

typedef void(*FinalizeFunction)(UsbDevice*, void*);

class UsbDevice :
    public Allocator,
    public nn::util::IntrusiveListBaseNode<UsbDevice>
{

private:
    const size_t BulkInMaxSize      = (64 * 1024);
    const size_t BulkOutMaxSize     = (64 * 1024);
    const size_t InterruptInMaxSize = (4 * 1024);
    const uint32_t MaxInPacketCount = 4;

    const EndPointDescriptor*   EpDescriptors;
    const int                   EpCount;

    nn::os::MultiWaitHolderType m_MultiWaitHolder;
    nn::os::SystemEventType*    m_pEvent;
    nn::usb::HostEndpoint*      m_EpSession;
    nn::usb::HostInterface      m_IfSession;
    void*                       m_FinalizeArgument;
    FinalizeFunction            m_FinalizeFunction;

    nn::Result AssociateEndpoints(nn::usb::InterfaceProfile* pProfile) NN_NOEXCEPT;

public:
    static nn::util::IntrusiveList<UsbDevice, nn::util::IntrusiveListBaseNodeTraits<UsbDevice>> m_DeviceList;

    nn::os::SystemEventType* GetTerminateEventPointer()
    NN_NOEXCEPT
    {
        return m_pEvent;
    }

    FinalizeFunction GetFinalizeFunction()
    NN_NOEXCEPT
    {
        return m_FinalizeFunction;
    }

    void* GetFinalizeArgument()
    NN_NOEXCEPT
    {
        return m_FinalizeArgument;
    }

    nn::Result SubmitControlOutRequest(
                    size_t*  pLengthOut,
                    void*    pData,
                    uint8_t  request,
                    uint8_t  type,
                    uint16_t value,
                    uint16_t index,
                    uint16_t length) NN_NOEXCEPT;

    nn::Result SubmitControlInRequest(
                    size_t*  pLengthOut,
                    void*    pData,
                    uint8_t  request,
                    uint8_t  type,
                    uint16_t value,
                    uint16_t index,
                    uint16_t length) NN_NOEXCEPT;

    nn::Result SubmitInRequest(
                    size_t*  pLengthOut,
                    int      ep,
                    void*    pData,
                    size_t   dataLength) NN_NOEXCEPT;

    nn::Result SubmitOutRequest(
                    size_t*  p_LengthOut,
                    int      ep,
                    void*    pData,
                    size_t   dataLength) NN_NOEXCEPT;

    nn::Result RunAsyncInOperation(
                    Packet*  pPacket,
                    uint32_t packetCount,
                    std::function<void(Packet*)> function,
                    int ep) NN_NOEXCEPT;

    nn::Result Initialize(
                    nn::usb::Host* pHost,
                    nn::usb::InterfaceProfile* pProfile,
                    FinalizeFunction finalizeFunction,
                    void* finalizeArgument) NN_NOEXCEPT;

    nn::Result Finalize() NN_NOEXCEPT;

    void CancelTransfers() NN_NOEXCEPT;

    UsbDevice(const EndPointDescriptor* descriptors, int count) NN_NOEXCEPT;

    ~UsbDevice() NN_NOEXCEPT;
};

}}}}
